<?php
/**
 * Kalium WordPress Theme
 *
 * Theme header functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Show or hide header.
 *
 * @return bool
 */
function kalium_show_header() {
	return apply_filters( 'kalium_show_header', true );
}

/**
 * Check if header is fullwidth.
 *
 * @return bool
 */
function kalium_is_fullwidth_header() {
	$return = boolval( kalium_get_theme_option( 'header_fullwidth' ) );

	if ( get_queried_object() instanceof WP_Post && ( $queried_object_id = kalium_get_queried_object_id() ) ) {
		$header_fullwidth = kalium_get_field( 'header_fullwidth', $queried_object_id );

		if ( in_array( $header_fullwidth, [ 'yes', 'no' ] ) ) {
			$return = 'yes' === $header_fullwidth;
		}
	}

	return $return;
}

/**
 * Check if header has top bar menu.
 *
 * @return bool
 */
function kalium_header_has_top_bar() {
	if ( ! kalium_get_theme_option( 'top_header_bar' ) ) {
		return false;
	}

	$top_bar_content = kalium_header_get_top_bar_content();

	// No top bar content
	if ( ! $top_bar_content['left'] && ! $top_bar_content['right'] ) {
		return false;
	}

	// Content blocks
	$content_left  = $top_bar_content['left'];
	$content_right = $top_bar_content['right'];

	// If both blocks are invisible
	if ( ( $content_left && ! $content_left['visible'] ) && ( $content_right && ! $content_right['visible'] ) ) {
		return false;
	}

	// If any of content blocks has elements
	if ( ! empty( $content_left['elements'] ) || ! empty( $content_right['elements'] ) ) {
		return true;
	}

	return false;
}

/**
 * Get header option.
 *
 * @param string $option
 *
 * @return mixed|null
 */
function kalium_header_get_option( $option ) {
	if ( ! ( get_queried_object() instanceof WP_Post ) ) {
		return null;
	}

	$queried_object_id = kalium_get_queried_object_id();

	switch ( $option ) {

		// Position
		case 'position':
			$header_transparent = kalium_get_theme_option( 'header_transparent' );
			$post_meta_field    = 'header_position';

			$return = $header_transparent ? 'absolute' : 'static';
			break;

		// Spacing
		case 'spacing':
			$post_meta_field = 'header_spacing';

			// Ignore post meta field when header position is not defined in post meta
			if ( 'inherit' === kalium_get_field( 'header_position', $queried_object_id ) ) {
				$post_meta_field = null;
			}

			$return = kalium_get_theme_option( 'header_spacing' );
			break;

		// Default
		default:
			return null;
	}

	// Get value from page settings
	if ( $queried_object_id > 0 && $post_meta_field ) {
		$page_value = kalium_get_field( $post_meta_field, $queried_object_id );

		if ( 'inherit' !== $page_value ) {
			$return = $page_value;
		}
	}

	return $return;
}

/**
 * New implementation of Sticky Header.
 *
 * @return array
 */
function kalium_get_sticky_header_options() {

	// Options
	$options = [

		// Sticky menu type: standard, autohide
		'type'                      => 'standard',

		// Sticky container selector
		'containerElement'          => '.site-header',

		// Logo selector
		'logoElement'               => '.logo-image',

		// Trigger hook offset relative to screen
		'triggerOffset'             => 0,

		// Initial offset for all animation scenes
		'offset'                    => 0,

		// Initial offset for animation scene (sticky type: standard)
		'animationOffset'           => 10,

		// Spacer to cover up the header height
		'spacer'                    => true,

		// Animate duration with scroll
		'animateProgressWithScroll' => true,

		// Animate duration (if not null, overwrites animation duration on timeline)
		'animateDuration'           => null,

		// Tween changes for animation scene
		'tweenChanges'              => false,

		// Sticky header class name definitions
		'classes'                   => [

			// Element name
			'name'        => 'site-header',

			// Class prefix for sticky header options
			'prefix'      => 'sticky',

			// Initialized class
			'init'        => 'initialized',

			// Menu is set to fixed
			'fixed'       => 'fixed',

			// Menu is set to absolute
			'absolute'    => 'absolute',

			// Sticky header spacer class
			'spacer'      => 'spacer',

			// Sticky header is active
			'active'      => 'active',

			// Sticky header is fully active
			'fullyActive' => 'fully-active',
		],

		// Autohide options
		'autohide'                  => [

			// Animation type
			'animationType' => 'fade-slide-top',

			// Animate duration
			'duration'      => 0.3,

			// Threshold distance before menu is shown/hide
			'threshold'     => 100,
		],

		// Props to animate
		'animateScenes'             => [],

		// Alternate logos
		'alternateLogos'            => [

			// Sticky logo
			// 'sticky'   => [
			// 'name'  => 'sticky',
			// 'image' => wp_get_attachment_image( 1, 'original' ),
			// 'css'   => [
			// 'width'  => 30,
			// 'height' => 30,
			// ],
			// ],
		],

		// Supported on devices
		'supportedOn'               => [

			// Desktop
			'desktop' => true,

			// Tablet
			'tablet'  => false,

			// Mobile
			'mobile'  => true,
		],

		// Other options
		'other'                     => [

			// Menu skin on sticky mode
			'menuSkin' => null,
		],
	];

	// Animate scenes
	$animate_scenes = [];

	// Padding animation
	// $animate_scenes['padding'] = [
	// 'name'     => 'padding',
	// 'selector' => 'self',
	// 'props'    => [
	// 'paddingTop',
	// 'paddingBottom',
	// ],
	// 'css'      => [
	// 'default' => [
	// 'paddingTop'    => 10,
	// 'paddingBottom' => 10,
	// ],
	// ],
	// ];

	// Background animation
	// $animate_scenes['background'] = [
	// 'name'     => 'background',
	// 'selector' => 'self',
	// 'props'    => [
	// 'backgroundColor',
	// ],
	// 'css'      => [
	// 'default' => [
	// 'backgroundColor' => '#c0c0c0',
	// ],
	// ],
	// ];

	// Resize logo animation
	// $animate_scenes['resizeLogo'] = [
	// 'name'     => 'logo-resize',
	// 'selector' => '.logo-image',
	// 'props'    => [
	// 'width',
	// 'height',
	// ],
	// 'css'      => [
	// 'default' => [
	// 'width'  => 30,
	// 'height' => 30,
	// ],
	// ],
	// ];

	// Sticky logo animation
	// $animate_scenes['stickyLogo'] = [
	// 'name'     => 'sticky-logo',
	// 'selector' => '.logo-image',
	// 'props'    => [
	// 'width',
	// 'height',
	// ],
	// 'data'     => [
	// 'type'          => 'alternate-logo',
	// 'alternateLogo' => 'sticky',
	// ],
	// 'duration' => 0.6,
	// 'css'      => [
	// 'width'  => 40,
	// 'height' => 30,
	// ],
	// ];

	// Add animate scene array
	// $options['animateScenes'] = $animate_scenes;

	/**
	 * Hook: kalium_sticky_header_options.
	 *
	 * @hooked kalium_sticky_header_options_default - 10
	 * @hooked kalium_sticky_options_for_static_header_type - 20
	 */
	return apply_filters( 'kalium_sticky_header_options', $options );
}

/**
 * Get logo switch sections.
 *
 * @return array
 */
function kalium_get_logo_switch_sections() {
	$sections = [];

	if ( is_singular() && kalium_get_field( 'section_logo_switch' ) ) {
		$sections = kalium_get_field( 'logo_switch_sections' );
	}

	return apply_filters( 'kalium_sticky_logo_switch_sections', $sections );
}

/**
 * Parse content builder field.
 *
 * @param string $str
 *
 * @return array
 * @deprecated
 */
function kalium_parse_content_builder_field( $str ) {

	// Parsed JSON
	$parsed = json_decode( $str, true );

	// Empty JSON to array
	if ( ! is_array( $parsed ) ) {
		$parsed = [];
	}

	// Entries list
	$entries = kalium_get_array_key( $parsed, 'entries', [] );

	// Map to properly format entries
	// $entries = array_map( 'kalium_parse_content_builder_entry', $entries );

	// Options array
	$options = [

		// Alignment
		'alignment' => kalium_get_array_key( $parsed, 'alignment' ),
	];

	return [
		'options' => $options,
		'entries' => $entries,
	];
}

/**
 * Get header content.
 *
 * @return array
 * @since 4.0
 */
function kalium_header_get_content() {
	$header_content = kalium_get_theme_option( 'header_content' );

	// Default header content
	if ( empty( $header_content ) ) {
		$header_content = laborator_builder_get_content_type( 'legacy_header' )->get_default_content();
	}

	return apply_filters(
		'kalium_header_custom_header_content',
		[
			'left'   => $header_content['elements'][0] ?? null,
			'right'  => $header_content['elements'][1] ?? null,
			'bottom' => $header_content['elements'][2] ?? null,
		]
	);
}

/**
 * Get all header elements.
 *
 * @return array
 * @since 4.0
 */
function kalium_header_get_all_elements() {
	$elements = [];

	foreach ( kalium_header_get_content() as $content_block ) {
		if ( ! empty( $content_block['elements'] ) ) {
			$elements = array_merge( $elements, $content_block['elements'] );
		}
	}
	return $elements;
}

/**
 * Get top bar content.
 *
 * @return array
 * @since 4.0
 */
function kalium_header_get_top_bar_content() {
	$top_bar_content = get_theme_mod( 'top_bar_content' );

	$content_left  = $top_bar_content['elements'][0] ?? null;
	$content_right = $top_bar_content['elements'][1] ?? null;

	return [
		'left'  => $content_left,
		'right' => $content_right,
	];
}

/**
 * Check if header contains top or side menu.
 *
 * @param string $menu_type
 *
 * @return bool
 * @since 4.0
 */
function kalium_header_has_menu_type( $menu_type ) {
	foreach ( kalium_header_get_all_elements() as $element ) {
		$trigger_type = null;

		if ( 'legacy_header_menu_trigger' === $element['name'] ) {
			switch ( $menu_type ) {

				// Fullscreen menu check
				case 'fullscreen':
					$trigger_type = 'fullscreen';
					break;

				// Off-Canvas side check
				case 'off-canvas-side':
					$trigger_type = 'off-canvas-side';
					break;

				// Off-Canvas top check
				case 'off-canvas-top':
					$trigger_type = 'off-canvas-top';
					break;
			}

			if ( $trigger_type && ! empty( $element['attributes'] ) ) {
				foreach ( $element['attributes'] as $attribute ) {
					if ( 'trigger_type' === $attribute['name'] && $trigger_type === $attribute['value'] ) {
						return true;
					}
				}
			}
		}
	}

	return false;
}

/**
 * Append or prepend standard menu toggle in the beginning or end of row.
 *
 * @param array $entries
 *
 * @return array
 * @since 4.0
 */
function kalium_header_standard_menu_toggle_setup( $content_block ) {
	if ( ! empty( $content_block['elements'] ) ) {

		// Create a copy of elements array
		$elements = $content_block['elements'];

		foreach ( $content_block['elements'] as $index => $element ) {

			/**
			 * Menu Trigger element.
			 *
			 * Prepend or append menu in the content block.
			 */
			if ( 'legacy_header_menu_trigger' === $element['name'] ) {
				$trigger_type      = laborator_builder_get_attribute_value( $element, 'trigger_type', 'standard' );
				$menu_position     = laborator_builder_get_attribute_value( $element, 'standard_menu_position', 'start' );
				$menu_item_spacing = laborator_builder_get_attribute_value( $element, 'standard_menu_item_spacing' );

				if ( ! empty( $element['visible'] ) && 'standard' === $trigger_type && in_array( $menu_position, [ 'start', 'end' ] ) ) {
					$toggle_index = 'end' === $menu_position ? count( $content_block['elements'] ) : 0;
					$toggle_id    = mt_rand( 1000, 9999 );

					// Menu Element
					$menu_element = [
						'name'       => 'legacy_header_menu_trigger',
						'attributes' => array_merge(
							$element['attributes'],
							[
								[
									'name'  => 'toggle_id',
									'value' => $toggle_id,
								],
							]
						),
					];

					// Assign Trigger ID attribute
					$elements[ $index ]['attributes'][] = [
						'name'  => 'trigger_menu_id',
						'value' => $toggle_id,
					];

					// Insert menu element
					array_splice( $elements, $toggle_index, 0, [ $menu_element ] );
				}
			}

			/**
			 * Menu element.
			 *
			 * Prepend or append mobile menu toggle for regular navigation element.
			 */
			elseif ( 'legacy_header_menu' === $element['name'] ) {
				$mobile_menu_toggle = laborator_builder_get_attribute_value( $element, 'mobile_menu_toggle', true );

				if ( $mobile_menu_toggle ) {
					$toggle_position = laborator_builder_get_attribute_value( $element, 'mobile_menu_toggle_position', 'current' );
					$toggle_index    = $index + 1;

					if ( 'start' === $toggle_position ) {
						$toggle_index = 0;
					} elseif ( 'end' === $toggle_position ) {
						$toggle_index = count( $elements );
					}

					// Menu Element
					$toggle_element = [
						'name'       => 'legacy_header_menu_trigger',
						'attributes' => array_merge(
							$element['attributes'],
							[
								[
									'name'  => 'mobile_menu_toggle',
									'value' => true,
								],
							]
						),
					];

					// Insert mobile toggle
					array_splice( $elements, $toggle_index, 0, [ $toggle_element ] );
				}
			}
		}

		// Re-assign new elements array
		$content_block['elements'] = $elements;
	}

	return $content_block;
}

/**
 * Header classes.
 *
 * @return array
 * @since 4.0
 */
function kalium_header_get_classes() {
	$header_classes = [
		'site-header',
		'main-header',
	];

	// Header block vars
	$header_block_vars = [];

	// Bottom spacing
	$has_bottom_spacing = false;

	// Header styling
	$header_border            = kalium_to_border_box( kalium_get_theme_option( 'header_border' ) );
	$header_padding           = kalium_get_theme_option( 'header_vpadding' );
	$header_bottom_spacing    = kalium_get_theme_option( 'header_bottom_spacing' );
	$header_menu_item_spacing = kalium_get_theme_option( 'header_menu_item_spacing' );

	// Submenu styling
	$submenu_border     = kalium_get_theme_option( 'header_style_submenu_border' );
	$submenu_padding    = kalium_get_theme_option( 'header_style_submenu_padding' );
	$submenu_radius     = kalium_get_theme_option( 'header_style_submenu_border_radius' );
	$submenu_box_shadow = kalium_get_theme_option( 'header_style_submenu_shadow' );

	// Menu type header class
	$header_classes[] = 'menu-type-' . kalium_get_theme_option( 'main_menu_type' );

	// Fullwidth header
	if ( kalium_is_fullwidth_header() ) {
		$header_classes[] = 'fullwidth-header';
	}

	// Sticky header
	if ( kalium_get_theme_option( 'sticky_header' ) ) {

		// Sticky header is active
		$header_classes[] = 'is-sticky';

		kalium_js_define_vars(
			[
				// Custom header sticky menu
				'stickyHeaderOptions'  => kalium_get_sticky_header_options(),

				// Logo switch on sections
				'logoSwitchOnSections' => kalium_get_logo_switch_sections(),
			]
		);
	}

	// Border
	if ( kalium_has_border( $header_border ) ) {
		$header_block_vars['header-border-width'] = $header_border['width'];
		$header_block_vars['header-border-style'] = $header_border['style'];
		$header_block_vars['header-border-color'] = $header_border['color'];

		$has_bottom_spacing = true;
	}

	// Padding top
	if ( isset( $header_padding['top'] ) && is_numeric( $header_padding['top'] ) ) {
		$header_block_vars['header-padding-top'] = kalium_get_multi_numeric_prop( $header_padding, 'top' );
	}

	// Padding bottom
	if ( isset( $header_padding['bottom'] ) && is_numeric( $header_padding['bottom'] ) ) {
		$header_block_vars['header-padding-bottom'] = kalium_get_multi_numeric_prop( $header_padding, 'bottom' );
	}

	// Menu item spacing
	if ( $header_menu_item_spacing ) {
		$header_block_vars['header-elements-spacing'] = $header_menu_item_spacing;
	}

	// Submenu border
	if ( ! empty( $submenu_border ) ) {
		$submenu_border = kalium_to_border_box( $submenu_border );

		$header_block_vars['submenu-border-width'] = $submenu_border['width'] ?? null;
		$header_block_vars['submenu-border-style'] = $submenu_border['style'] ?? null;
		$header_block_vars['submenu-border-color'] = $submenu_border['color'] ?? null;
	}

	// Submenu Padding
	if ( is_array( $submenu_padding ) ) {
		$header_block_vars['submenu-padding'] = kalium_to_length( $submenu_padding );
	}

	// Submenu Border Radius
	if ( is_array( $submenu_radius ) ) {
		$header_block_vars['submenu-radius'] = kalium_to_length( $submenu_radius );
	}

	// Submenu Box Shadow
	$header_block_vars['submenu-shadow'] = kalium_to_box_shadow( $submenu_box_shadow, '0 0 5px' );

	// Bottom spacing
	if ( ! kalium_is_empty_responsive( $header_bottom_spacing ) ) {
		$header_block_vars['header-bottom-spacing'] = kalium_map_responsive_value(
			$header_bottom_spacing,
			function ( $margin_bottom ) {
				return $margin_bottom . 'px';
			}
		);

		$has_bottom_spacing = false;
	}

	// Bottom spacing if background or border is assigned
	if ( $has_bottom_spacing ) {
		$header_classes[] = 'header-bottom-spacing';
	}

	// Print header-block styles
	kalium_print_inline_style(
		[
			'id'       => 'site-header',
			'selector' => [
				'.site-header',
				'.sidebar-menu-wrapper',
				kalium_conditional( $has_bottom_spacing, '.site-header.header-bottom-spacing' ),
			],
			'vars'     => $header_block_vars,
		]
	);

	return $header_classes;
}

/**
 * Get fullscreen menu options.
 *
 * @return array
 * @since 4.0
 */
function kalium_header_get_fullscreen_menu_options() {
	$options = [
		'menu_id'      => null,
		'align'        => kalium_get_theme_option( 'menu_full_bg_alignment' ),
		'search_field' => kalium_get_theme_option( 'menu_full_bg_search_field' ),
		'custom_bg'    => kalium_get_theme_option( 'fullscreen_menu_bg' ),
		'background'   => [
			'color'       => kalium_get_theme_option( 'fullscreen_menu_bg_color' ),
			'image'       => kalium_get_attachment_id_from_theme_option( 'fullscreen_menu_bg_image' ),
			'position'    => kalium_get_theme_option( 'fullscreen_menu_bg_position' ),
			'repeat'      => kalium_get_theme_option( 'fullscreen_menu_bg_repeat' ),
			'size'        => kalium_get_theme_option( 'fullscreen_menu_bg_size' ),
			'size_custom' => kalium_get_theme_option( 'fullscreen_menu_bg_size_custom' ),
		],
	];

	foreach ( kalium_header_get_all_elements() as $element ) {
		if ( 'legacy_header_menu_trigger' === $element['name'] && 'fullscreen' === laborator_builder_get_attribute_value( $element, 'trigger_type' ) ) {
			$options['menu_id']      = laborator_builder_get_attribute_value( $element, 'menu_id', $options['menu_id'] );
			$options['align']        = laborator_builder_get_attribute_value( $element, 'fullscreen_alignment' );
			$options['search_field'] = laborator_builder_get_attribute_value( $element, 'fullscreen_search', true );
		}
	}

	return $options;
}

/**
 * Get off canvas side menu options.
 *
 * @return array
 * @since 4.0
 */
function kalium_header_get_off_canvas_side_options() {
	$options = [
		'menu_id'   => kalium_get_theme_option( 'menu_sidebar_menu_id' ),
		'alignment' => kalium_get_theme_option( 'menu_sidebar_alignment' ),
		'widgets'   => kalium_get_theme_option( 'menu_sidebar_show_widgets' ),
	];

	foreach ( kalium_header_get_all_elements() as $element ) {
		if ( 'legacy_header_menu_trigger' === $element['name'] && 'off-canvas-side' === laborator_builder_get_attribute_value( $element, 'trigger_type' ) ) {
			$options['menu_id']   = laborator_builder_get_attribute_value( $element, 'menu_id', $options['menu_id'] );
			$options['alignment'] = laborator_builder_get_attribute_value( $element, 'off_canvas_side_alignment', 'right' );
			$options['widgets']   = laborator_builder_get_attribute_value( $element, 'off_canvas_side_show_widgets', true );
		}
	}

	return $options;
}

/**
 * Get off canvas top menu options.
 *
 * @return array
 * @since 4.0
 */
function kalium_header_get_off_canvas_top_options() {
	$options = [
		'menu_id'                 => kalium_get_theme_option( 'menu_top_menu_id' ),
		'items_per_row'           => kalium_get_theme_option( 'menu_top_items_per_row' ),
		'widgets'                 => kalium_get_theme_option( 'menu_top_show_widgets' ),
		'widgets_per_row'         => kalium_get_theme_option( 'menu_top_widgets_per_row' ),
		'widgets_container_width' => kalium_get_theme_option( 'menu_top_widgets_container_width' ),
	];

	foreach ( kalium_header_get_all_elements() as $element ) {
		if ( 'legacy_header_menu_trigger' === $element['name'] && 'off-canvas-top' === laborator_builder_get_attribute_value( $element, 'trigger_type' ) ) {
			$options['menu_id']                 = laborator_builder_get_attribute_value( $element, 'menu_id', $options['menu_id'] );
			$options['items_per_row']           = laborator_builder_get_attribute_value( $element, 'off_canvas_top_items_per_row', $options['items_per_row'] );
			$options['widgets']                 = laborator_builder_get_attribute_value( $element, 'off_canvas_top_widgets' );
			$options['widgets_per_row']         = laborator_builder_get_attribute_value( $element, 'off_canvas_top_widgets_per_row', $options['widgets_per_row'] );
			$options['widgets_container_width'] = laborator_builder_get_attribute_value( $element, 'off_canvas_top_widgets_container_width', $options['widgets_container_width'] );
		}
	}

	return $options;
}

/**
 * Get menu list as choices array.
 *
 * @return array
 * @since 4.0
 */
function kalium_header_get_menu_choices() {
	static $choices;

	if ( is_null( $choices ) ) {
		$choices = array_map(
			function ( $nav ) {
				return [
					'value' => $nav->term_id,
					'label' => $nav->name,
				];
			},
			wp_get_nav_menus()
		);
	}

	return $choices;
}
