<?php
/**
 * Kalium WordPress Theme
 *
 * Header hook functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Custom header style options on page.
 */
function kalium_header_custom_style_on_page() {
	if ( ! is_singular() && ! is_front_page() || kalium_is_maintenance_mode() ) {
		return;
	}

	$queried_object_id = kalium_get_queried_object_id();

	// Header styling
	switch ( kalium_get_field( 'page_header_styling', $queried_object_id ) ) {
		case 'yes':
			$header_style = kalium_get_field( 'header_style', $queried_object_id );

			if ( $container_background = kalium_get_array_key( $header_style, 'container_background' ) ) {
				add_filter( 'kalium_theme_option_header_background', kalium_hook_return_value( $container_background ) );
			}

			if ( $header_border_color = kalium_get_array_key( $header_style, 'container_border_color' ) ) {
				add_filter(
					'kalium_theme_option_header_border',
					function ( $border ) use ( &$header_border_color ) {
						if ( ! empty( $border['color'] ) ) {
							$border['color'] = $header_border_color;
						} elseif ( is_array( $border ) ) {
							foreach ( $border as $prop => $border_entry ) {
								if ( ! empty( $border_entry['color'] ) ) {
									$border[ $prop ]['color'] = $header_border_color;
								}
							}
						}

						return $border;
					}
				);
			}

			if ( ! empty( $header_padding = array_filter( kalium_get_array_key( $header_style, 'container_padding', [] ), 'is_numeric' ) ) ) {
				$header_padding = array_merge(
					$header_padding,
					[
						'unit' => 'px',
					]
				);

				add_filter( 'kalium_theme_option_header_vpadding', kalium_hook_array_merge( $header_padding ) );
			}

			if ( is_numeric( $container_bottom_spacing = kalium_get_array_key( $header_style, 'container_bottom_spacing' ) ) ) {
				add_filter( 'kalium_theme_option_header_bottom_spacing', kalium_hook_return_value( $container_bottom_spacing ) );
			}

			if ( ! empty( $header_links = kalium_filter_array_items( kalium_get_array_key( $header_style, 'links' ) ) ) ) {
				add_filter( 'kalium_theme_option_header_style_links', kalium_hook_array_merge( $header_links ) );
			}

			if ( ! empty( $header_hamburger_menu = kalium_filter_array_items( kalium_get_array_key( $header_style, 'hamburger_menu' ) ) ) ) {
				add_filter( 'kalium_theme_option_header_style_menu_trigger', kalium_hook_array_merge( $header_hamburger_menu ) );
			}

			if ( $header_text = kalium_get_array_key( $header_style, 'text' ) ) {
				add_filter( 'kalium_theme_option_header_style_text', kalium_hook_return_value( $header_text ) );
			}

			if ( ! empty( $header_pill_background = kalium_filter_array_items( kalium_get_array_key( $header_style, 'pill_background' ) ) ) ) {
				add_filter( 'kalium_theme_option_header_menu_pill_background', kalium_hook_array_merge( $header_pill_background ) );
			}

			if ( ! empty( $header_pill_color = kalium_filter_array_items( kalium_get_array_key( $header_style, 'pill_color' ) ) ) ) {
				add_filter( 'kalium_theme_option_header_menu_pill_color', kalium_hook_array_merge( $header_pill_color ) );
			}

			if ( $submenu_bg = kalium_get_array_key( $header_style, 'submenu_background' ) ) {
				add_filter( 'kalium_theme_option_header_style_submenu_bg', kalium_hook_return_value( $submenu_bg ) );
			}

			if ( ! empty( $header_submenu_links = kalium_filter_array_items( kalium_get_array_key( $header_style, 'submenu_links' ) ) ) ) {
				add_filter( 'kalium_theme_option_header_style_submenu_links', kalium_hook_array_merge( $header_submenu_links ) );
			}

			if ( ! empty( $header_submenu_pill_bg = kalium_filter_array_items( kalium_get_array_key( $header_style, 'submenu_pill_background' ) ) ) ) {
				add_filter( 'kalium_theme_option_header_menu_pill_submenu_bg', kalium_hook_array_merge( $header_submenu_pill_bg ) );
			}

			if ( $header_submenu_border_color = kalium_get_array_key( $header_style, 'submenu_border_color' ) ) {
				add_filter(
					'kalium_theme_option_header_border',
					function ( $border ) use ( &$header_submenu_border_color ) {
						if ( ! empty( $border['color'] ) ) {
							$border['color'] = $header_submenu_border_color;
						} elseif ( is_array( $border ) ) {
							foreach ( $border as $prop => $border_entry ) {
								if ( ! empty( $border_entry['color'] ) ) {
									$border[ $prop ]['color'] = $header_submenu_border_color;
								}
							}
						}

						return $border;
					}
				);
			}

			// Set custom colors for Mini Cart
			if ( ! empty( $header_links['normal'] ) && kalium()->is->woocommerce_active() && ! kalium()->style->is_dark_color( $header_links['normal'] ) ) {
				add_action(
					'wp_body_open',
					function () {
						kalium_print_inline_style(
							[
								'styles' => [
									[
										'selector' => '.header-block .mini-cart-trigger .items-count',
										'vars'     => [
											'badge-color' => kalium_css_var_reference( 'heading-color' ),
										],
									],
									[
										'selector' => '.header-block .mini-cart-trigger .mini-cart-popup',
										'vars'     => [
											'link-color' => kalium_css_var_reference( 'heading-color' ),
											'link-hover-color' => kalium_css_var_reference( 'heading-color' ),
										],
									],
								],
							]
						);
					}
				);
			}

			// Mobile menu
			if ( ! empty( $mobile_menu_links = kalium_filter_array_items( kalium_get_array_key( $header_style, 'mobile_menu_links' ) ) ) ) {
				add_filter( 'kalium_theme_option_mobile_menu_link_colors', kalium_hook_array_merge( $mobile_menu_links ) );
			}

			if ( $mobile_menu_text = kalium_get_array_key( $header_style, 'mobile_menu_text' ) ) {
				add_filter( 'kalium_theme_option_mobile_menu_text', kalium_hook_return_value( $mobile_menu_text ) );
			}

			if ( $mobile_menu_close_button = kalium_get_array_key( $header_style, 'mobile_menu_close_button' ) ) {
				add_filter(
					'kalium_theme_option_mobile_menu_close_button_color',
					kalium_hook_return_value(
						[
							'normal' => $mobile_menu_close_button,
							'hover'  => $mobile_menu_close_button,
						]
					)
				);
			}
			break;

		case 'no':
			add_filter( 'kalium_theme_option_header_background', '__return_null' );
			add_filter( 'kalium_theme_option_header_border', '__return_null' );
			add_filter( 'kalium_theme_option_header_bottom_spacing', '__return_null' );
			break;
	}
}

add_action( 'wp', 'kalium_header_custom_style_on_page' );

/**
 * Header position and spacing.
 *
 * @return void
 */
function kalium_header_position_spacing_action() {
	$queried_object_id = kalium_get_queried_object_id();

	$header_position = kalium_header_get_option( 'position' );
	$header_spacing  = kalium_header_get_option( 'spacing' );

	// Header position
	if ( 'absolute' === $header_position && ! post_password_required( $queried_object_id ) && ! kalium_is_maintenance_mode() ) {
		kalium_append_custom_css( '.site-header', 'position: absolute; left: 0; right: 0;' );

		// Header spacing
		if ( $header_spacing ) {
			$header_spacing = str_replace( 'px', '', $header_spacing );
			kalium_append_custom_css( '.wrapper', "padding-top: {$header_spacing}px" );
		}
	}
}

add_action( 'wp_head', 'kalium_header_position_spacing_action' );

/**
 * Absolute header body class.
 *
 * @param array $classes
 *
 * @return array
 */
function kalium_absolute_header_body_class( $classes ) {
	$queried_object_id = kalium_get_queried_object_id();
	$header_position   = kalium_header_get_option( 'position' );

	if ( 'absolute' === $header_position && ! post_password_required( $queried_object_id ) ) {
		$classes[] = 'header-absolute';
	}

	return $classes;
}

add_filter( 'body_class', 'kalium_absolute_header_body_class' );

/**
 * Custom logo and menu options for current page.
 */
function kalium_header_custom_logo_and_menu_on_page() {
	if ( ! is_singular() || kalium_is_maintenance_mode() ) {
		return;
	}

	$queried_object_id = kalium_get_queried_object_id();

	// Custom sticky logo
	if ( $custom_sticky_logo  = kalium_get_field( 'custom_sticky_logo', $queried_object_id ) ) {
		add_filter( 'kalium_theme_option_sticky_header_custom_logo', '__return_true' );
		add_filter( 'kalium_theme_option_sticky_header_logo', kalium_hook_return_value( $custom_sticky_logo ) );
	}

	// Overwrite sticky header option for current page
	$sticky_menu_on_page = kalium_get_field( 'sticky_menu_on_page', $queried_object_id );

	if ( in_array( $sticky_menu_on_page, [ 'enable', 'disable' ] ) ) {
		add_filter( 'kalium_theme_option_sticky_header', ( 'enable' === $sticky_menu_on_page ? '__return_true' : '__return_false' ) );

		// Add unit
		$add_unit = function ( &$arr, $props ) {
			foreach ( $props as $prop ) {
				if ( isset( $arr[ $prop ] ) && is_numeric( $arr[ $prop ] ) ) {
					$arr[ $prop ] .= 'px';
				}
			}
		};

		$sticky_style = kalium_get_field( 'sticky_header_style', $queried_object_id );

		if ( $sticky_bg = kalium_get_array_key( $sticky_style, 'container_background' ) ) {
			add_filter( 'kalium_theme_option_sticky_header_background_color', kalium_hook_return_value( $sticky_bg ) );
		}

		if ( ! empty( $sticky_border = kalium_filter_array_items( kalium_get_array_key( $sticky_style, 'container_bottom_border' ) ) ) ) {
			$add_unit( $sticky_border, [ 'width' ] );

			if ( ! empty( $sticky_border['color'] ) || ! empty( $sticky_border['width'] ) ) {
				add_filter( 'kalium_theme_option_sticky_header_border', kalium_hook_array_merge( $sticky_border ) );
			}
		}

		if ( ! empty( $sticky_shadow = kalium_filter_array_items( kalium_get_array_key( $sticky_style, 'container_shadow' ) ) ) ) {
			$add_unit( $sticky_shadow, [ 'offset_x', 'offset_y', 'blur_radius', 'spread_radius' ] );

			add_filter( 'kalium_theme_option_sticky_header_shadow', kalium_hook_array_merge( $sticky_shadow ) );
		}

		if ( ! empty( $sticky_padding = kalium_filter_array_items( kalium_get_array_key( $sticky_style, 'container_padding' ) ) ) ) {
			$sticky_padding = array_merge(
				$sticky_padding,
				[
					'unit' => 'px',
				]
			);

			add_filter( 'kalium_theme_option_sticky_header_vertical_padding', kalium_hook_array_merge( $sticky_padding ) );
		}

		if ( ! empty( $sticky_links = kalium_filter_array_items( kalium_get_array_key( $sticky_style, 'links' ) ) ) ) {
			add_filter( 'kalium_theme_option_sticky_header_links_color', kalium_hook_array_merge( $sticky_links ) );
		}

		if ( ! empty( $sticky_menu_trigger = kalium_filter_array_items( kalium_get_array_key( $sticky_style, 'menu_trigger' ) ) ) ) {
			add_filter( 'kalium_theme_option_sticky_header_menu_trigger', kalium_hook_array_merge( $sticky_menu_trigger ) );
		}

		if ( $sticky_text = kalium_get_array_key( $sticky_style, 'text' ) ) {
			add_filter( 'kalium_theme_option_sticky_header_text_color', kalium_hook_return_value( $sticky_text ) );
		}

		if ( ! empty( $sticky_pill_bg = kalium_filter_array_items( kalium_get_array_key( $sticky_style, 'pill_background' ) ) ) ) {
			add_filter( 'kalium_theme_option_sticky_header_pill_bg_color', kalium_hook_array_merge( $sticky_pill_bg ) );
		}

		if ( ! empty( $sticky_pill_color = kalium_filter_array_items( kalium_get_array_key( $sticky_style, 'pill_color' ) ) ) ) {
			add_filter( 'kalium_theme_option_sticky_header_pill_color', kalium_hook_array_merge( $sticky_pill_color ) );
		}
	}

	// Custom logo
	if ( is_numeric( $custom_logo = kalium_get_field( 'custom_logo', $queried_object_id ) ) ) {
		$custom_logo_width = kalium_get_field( 'custom_logo_width' );

		add_filter( 'kalium_theme_option_use_uploaded_logo', '__return_true' );
		add_filter( 'kalium_theme_option_custom_logo_image', kalium_hook_return_value( $custom_logo ) );

		if ( is_numeric( $custom_logo_width ) && $custom_logo_width > 0 ) {
			add_filter( 'kalium_theme_option_custom_logo_max_width', kalium_hook_return_value( $custom_logo_width ) );
		}
	}
}

add_action( 'wp', 'kalium_header_custom_logo_and_menu_on_page' );

/**
 * Default sticky options.
 *
 * @param array $options
 *
 * @return array
 */
function kalium_sticky_header_options_default( $options ) {

	// Sticky vars
	$vertical_padding    = kalium_get_theme_option( 'sticky_header_vertical_padding' );
	$background_color    = kalium_get_theme_option( 'sticky_header_background_color' );
	$border_color        = kalium_get_theme_option( 'sticky_header_border' );
	$shadow_color        = kalium_get_theme_option( 'sticky_header_shadow' );
	$is_autohide         = kalium_get_theme_option( 'sticky_header_autohide' );
	$autohide_type       = kalium_get_theme_option( 'sticky_header_autohide_animation_type' );
	$autohide_duration   = kalium_get_theme_option( 'sticky_header_autohide_duration' );
	$logo_width          = kalium_get_theme_option( 'sticky_header_logo_width' );
	$use_custom_logo     = kalium_get_theme_option( 'sticky_header_custom_logo' );
	$custom_logo         = kalium_get_attachment_id_from_theme_option( 'sticky_header_logo' );
	$animate_with_scroll = kalium_get_theme_option( 'sticky_header_animate_duration' );
	$tween_changes       = kalium_get_theme_option( 'sticky_header_tween_changes' );
	$animate_duration    = kalium_get_theme_option( 'sticky_header_duration' );
	$animate_offset      = kalium_get_theme_option( 'sticky_header_initial_offset' );
	$responsive          = kalium_get_theme_option( 'sticky_header_responsive' );

	$supported_on = [
		'desktop' => in_array( 'desktop', $responsive ),
		'tablet'  => in_array( 'tablet', $responsive ),
		'mobile'  => in_array( 'mobile', $responsive ),
	];

	$css_vars = $css_props = [];

	/**
	 * Initial offset for scenes to exclude top header bar.
	 */
	$options['offset'] = '.top-header-bar';

	/**
	 * Animation scene offset.
	 */
	if ( is_numeric( $animate_offset ) ) {
		$options['animationOffset'] = $animate_offset;
	}

	/**
	 * Animate animation progress with scroll.
	 */
	$options['animateProgressWithScroll'] = kalium_validate_boolean( $animate_with_scroll );

	/**
	 * Tween changes with scroll progress.
	 */
	$options['tweenChanges'] = boolval( $tween_changes );

	/**
	 * Animate duration in seconds.
	 */
	if ( is_numeric( $animate_duration ) ) {
		$options['animateDuration'] = $animate_duration;
	}

	/**
	 * Generate spacer or not.
	 */
	$options['spacer'] = 'absolute' !== kalium_header_get_option( 'position' );

	/**
	 * Autohide sticky type.
	 */
	if ( $is_autohide ) {

		// Sticky type
		$options['type'] = 'autohide';

		// Animation type
		$options['autohide']['animationType'] = $autohide_type;

		// Duration
		if ( is_numeric( $autohide_duration ) ) {
			$options['autohide']['duration'] = $autohide_duration;
		}
	}

	/**
	 * Supported on device types.
	 */
	$options['supportedOn'] = $supported_on;

	/**
	 * Animation scenes.
	 */
	$scene_name_padding     = 'padding';
	$scene_name_styling     = 'styling';
	$scene_name_sticky_logo = 'sticky-logo';
	$scene_name_resize_logo = 'resize-logo';

	$animation_scenes = [];

	// Padding animation scene
	if ( isset( $vertical_padding['top'] ) && isset( $vertical_padding['bottom'] ) ) {
		$animation_scene_padding = [
			'name'     => 'padding',
			'selector' => '.header-block',
			'props'    => [
				'paddingTop',
				'paddingBottom',
			],
			'css'      => [
				'default' => [
					'paddingTop'    => $vertical_padding['top'],
					'paddingBottom' => $vertical_padding['bottom'],
				],
			],
		];

		$animation_scenes[ $scene_name_padding ] = $animation_scene_padding;
	}

	// Background, shadow and border animation scene
	$animation_scene_style = [
		'name'     => 'style',
		'selector' => '.header-block',
		'props'    => [],
		'css'      => [
			'default' => [],
		],
		'data'     => [
			'tags' => [
				'transparent-header', // required!
			],
		],
	];

	if ( $background_color ) {
		$animation_scene_style['props'][]                           = 'backgroundColor';
		$animation_scene_style['css']['default']['backgroundColor'] = kalium_replace_color_references( $background_color );
	}

	// Bottom border
	$has_border = ! empty( $border_color ) || ! empty( $shadow_color );

	if ( $has_border ) {

		// Bottom border color
		if ( $border_color ) {
			$border = kalium_to_border_box( $border_color );

			if ( ! empty( $border['width'] ) ) {
				$border_width = $border['width'];
				$border_style = $border['style'] ?: 'solid';
				$border_color = $border['color'];

				$animation_scene_style['props'][]                        = 'borderBottom';
				$animation_scene_style['css']['default']['borderBottom'] = sprintf( '%s %s %s', $border_width, $border_style, $border_color );

				if ( ! kalium_has_border( kalium_get_theme_option( 'header_border' ) ) ) {
					$css_props['border-bottom'] = "{$border_width} {$border_style} transparent";
				}
			}
		}

		// Shadow color
		if ( $shadow_color ) {
			$shadow = kalium_to_box_shadow( $shadow_color );

			$animation_scene_style['props'][]                     = 'boxShadow';
			$animation_scene_style['css']['default']['boxShadow'] = $shadow;

			$css_props['box-shadow'] = '0px 0px 0px transparent';
		}
	}

	// Register styling scene if there are props to animate
	if ( ! empty( $animation_scene_style['css']['default'] ) ) {
		$animation_scenes[ $scene_name_styling ] = $animation_scene_style;
	}

	// Custom logo animation scene
	$sticky_logo_width = $sticky_logo_height = null;

	if ( $use_custom_logo && ( $sticky_logo = wp_get_attachment_image_src( $custom_logo, 'full' ) ) ) {
		list( $src, $width, $height ) = $sticky_logo;

		// Alternate logo
		$options['alternateLogos']['sticky'] = [
			'name'  => 'sticky',
			'image' => wp_get_attachment_image( $custom_logo, 'full' ),
		];

		// Custom logo dimensions
		if ( is_numeric( $logo_width ) ) {
			$new_dimensions     = kalium()->helpers->resize_by_width( $width, $height, $logo_width );
			$sticky_logo_width  = $new_dimensions[0];
			$sticky_logo_height = $new_dimensions[1];
		}

		// Custom sticky logo
		$animation_scene_sticky_logo = [
			'name'     => 'sticky-logo',
			'selector' => 'logo',
			'props'    => [
				'width',
				'height',
			],
			'css'      => [
				'width'  => $sticky_logo_width,
				'height' => $sticky_logo_height,
			],
			'data'     => [
				'type'          => 'alternate-logo',
				'alternateLogo' => 'sticky',
				'tags'          => [
					'logo-switch', // required!
				],
			],
		];

		// Logo switch animation scene
		$animation_scenes[ $scene_name_sticky_logo ] = $animation_scene_sticky_logo;
	} elseif ( $logo_width && ( $custom_logo_src = wp_get_attachment_image_src( kalium_get_attachment_id_from_theme_option( 'custom_logo_image' ), 'full' ) ) ) {
		$custom_logo_width  = $custom_logo_src[1];
		$custom_logo_height = $custom_logo_src[2];
		$new_dimensions     = kalium()->helpers->resize_by_width( $custom_logo_width, $custom_logo_height, $logo_width );

		$sticky_logo_width  = $new_dimensions[0];
		$sticky_logo_height = $new_dimensions[1];

		// Resize logo animation scene
		$animation_scenes[ $scene_name_resize_logo ] = [
			'name'     => 'logo-resize',
			'selector' => '.logo-image',
			'props'    => [
				'width',
				'height',
			],
			'css'      => [
				'default' => [
					'width'  => $sticky_logo_width,
					'height' => $sticky_logo_height,
				],
			],
			'data'     => [
				'tags' => [
					'logo-resize', // required!
				],
			],
		];
	}

	/**
	 * Animation chaining.
	 */
	$scene_duration_base = 0.3;

	// Set animation chaining
	if ( isset( $animation_chain ) ) {

		foreach ( $animation_chain as $i => $scenes ) {

			$scenes = ! is_array( $scenes ) ? [ $scenes ] : $scenes;

			foreach ( $scenes as $scene_name ) {

				if ( isset( $animation_scenes[ $scene_name ] ) ) {
					$position = $i * $scene_duration_base;

					// Duration on timeline
					if ( ! isset( $animation_scenes[ $scene_name ]['duration'] ) ) {
						$animation_scenes[ $scene_name ]['duration'] = $scene_duration_base;
					}

					// Position on timeline
					$animation_scenes[ $scene_name ]['position'] = $position;
				}
			}
		}
	}

	// Sort animation scenes by their position
	uasort(
		$animation_scenes,
		function ( $a, $b ) {
			if ( isset( $a['position'] ) && isset( $b['position'] ) ) {
				return $a['position'] < $b['position'] ? - 1 : 1;
			}
		}
	);

	// Update animation scenes property
	$options['animateScenes'] = $animation_scenes;

	/**
	 * Element styling on sticky state.
	 */
	$sticky_state_class = '.site-header--sticky-fully-active';

	// All styles
	$styles = [
		[
			'selector' => '.header-block',
			'vars'     => $css_vars,
			'props'    => $css_props,
		],
	];

	if ( $style_links = kalium_get_theme_option( 'sticky_header_links_color', true ) ) {
		$styles[] = [
			'selector' => $sticky_state_class,
			'vars'     => [
				'header-link-color'        => kalium_get_array_key( $style_links, 'normal' ),
				'header-link-hover-color'  => kalium_get_array_key( $style_links, 'hover' ),
				'header-link-active-color' => kalium_get_array_key( $style_links, 'active' ),
			],
		];
	}

	if ( $style_menu_trigger = kalium_get_theme_option( 'sticky_header_menu_trigger', true ) ) {
		$styles[] = [
			'selector' => $sticky_state_class,
			'vars'     => [
				'header-menu-trigger-color'        => kalium_get_array_key( $style_menu_trigger, 'normal' ),
				'header-menu-trigger-hover-color'  => kalium_get_array_key( $style_menu_trigger, 'hover' ),
				'header-menu-trigger-active-color' => kalium_get_array_key( $style_menu_trigger, 'active' ),
			],
		];
	}

	if ( $style_text  = kalium_get_theme_option( 'sticky_header_text_color', true ) ) {
		$styles[] = [
			'selector' => $sticky_state_class,
			'vars'     => [
				'header-color' => $style_text,
			],
		];
	}

	if ( $style_pill_bg = kalium_get_theme_option( 'sticky_header_pill_bg_color', true ) ) {
		$styles[] = [
			'selector' => "{$sticky_state_class} .nav-pills",
			'vars'     => [
				'pill-bg'        => kalium_get_array_key( $style_pill_bg, 'normal' ),
				'pill-bg-hover'  => kalium_get_array_key( $style_pill_bg, 'hover' ),
				'pill-bg-active' => kalium_get_array_key( $style_pill_bg, 'active' ),
			],
		];
	}

	if ( $style_pill_color = kalium_get_theme_option( 'sticky_header_pill_color', true ) ) {
		$styles[] = [
			'selector' => "{$sticky_state_class} .nav-pills",
			'vars'     => [
				'pill-color'        => kalium_get_array_key( $style_pill_color, 'normal' ),
				'pill-color-hover'  => kalium_get_array_key( $style_pill_color, 'hover' ),
				'pill-color-active' => kalium_get_array_key( $style_pill_color, 'active' ),
			],
		];
	}

	// Print styles
	kalium_print_inline_style(
		[
			'styles' => $styles,
		]
	);

	return $options;
}

add_action( 'kalium_sticky_header_options', 'kalium_sticky_header_options_default' );

/**
 * Convert header types (standard, fullscreen, off-canvas top and off-canvas side) to Custom Header content.
 *
 * @param array $content
 *
 * @return array
 */
function kalium_header_convert_static_types_to_custom_header_content( $content ) {
	$header_content = [];

	// Vars
	$menu_type      = kalium_get_theme_option( 'main_menu_type' );
	$submenu_arrows = kalium_get_theme_option( 'submenu_dropdown_indicator' );

	if ( 'custom-header' !== $menu_type ) {

		// Language Switcher
		if ( kalium_get_theme_option( 'header_wpml_language_switcher' ) ) {
			$header_content[] = [
				'name'       => 'legacy_header_wpml_language_switcher',
				'attributes' => [
					[
						'name'  => 'flag',
						'value' => kalium_get_theme_option( 'header_wpml_language_flag' ),
					],
					[
						'name'  => 'flag_position',
						'value' => kalium_get_theme_option( 'header_wpml_language_flag_position' ),
					],
					[
						'name'  => 'display_text',
						'value' => kalium_get_theme_option( 'header_wpml_language_switcher_text' ),
					],
					[
						'name'  => 'display_text_format',
						'value' => kalium_get_theme_option( 'header_wpml_language_switcher_text_display_type' ),
					],
				],
			];
		}

		// Search field
		if ( kalium_get_theme_option( 'header_search_field' ) ) {
			$animate_icon = kalium_get_theme_option( 'header_search_field_animate_icon' );

			$header_content[] = [
				'name'       => 'legacy_header_search_field',
				'attributes' => [
					[
						'name'  => 'input_visibility',
						'value' => $animate_icon ? 'click' : 'visible',
					],
					[
						'name'  => 'icon_animation',
						'value' => $animate_icon ? kalium_get_theme_option( 'header_search_field_icon_animation' ) : 'none',
					],
				],
			];
		}

		// Mini Cart
		if ( kalium_get_theme_option( 'header_mini_cart' ) ) {
			$header_content[] = [
				'name'       => 'legacy_header_woocommerce_mini_cart',
				'attributes' => [
					[
						'name'  => 'icon',
						'value' => kalium_get_theme_option( 'header_cart_icon' ),
					],
					[
						'name'  => 'hide_empty',
						'value' => kalium_get_theme_option( 'header_cart_hide_empty' ),
					],
					[
						'name'  => 'counter_badge',
						'value' => kalium_get_theme_option( 'header_cart_counter_badge' ),
					],
					[
						'name'  => 'title',
						'value' => kalium_get_theme_option( 'header_mini_cart_title' ),
					],
					[
						'name'  => 'click_action',
						'value' => kalium_get_theme_option( 'header_cart_click_action' ),
					],
					[
						'name'  => 'popup_show_on',
						'value' => kalium_get_theme_option( 'header_cart_popup_show_on' ),
					],
					[
						'name'  => 'popup_align',
						'value' => kalium_get_theme_option( 'header_mini_popup_align' ),
					],
				],
			];
		}
	}

	switch ( $menu_type ) {

		// Standard menu
		case 'standard-menu':
			$menu_item_spacing = kalium_get_theme_option( 'header_menu_item_spacing' );

			if ( kalium_get_theme_option( 'menu_standard_menu_bar_visible' ) ) {
				$header_content[] = [
					'name'       => 'legacy_header_menu_trigger',
					'visible'    => true,
					'attributes' => [
						[
							'name'  => 'standard_menu_position',
							'value' => 'start',
						],
						[
							'name'  => 'standard_menu_animation',
							'value' => kalium_get_theme_option( 'menu_standard_menu_bar_effect' ),
						],
						[
							'name'  => 'standard_menu_dropdown_caret',
							'value' => $submenu_arrows,
						],
						[
							'name'  => 'menu_item_spacing',
							'value' => $menu_item_spacing,
						],
					],
				];
			} else {
				array_unshift(
					$header_content,
					[
						'name'       => 'legacy_header_menu',
						'visible'    => true,
						'attributes' => [
							[
								'name'  => 'submenu_dropdown_caret',
								'value' => $submenu_arrows,
							],
							[
								'name'  => 'menu_item_spacing',
								'value' => $menu_item_spacing,
							],
							[
								'name'  => 'pill_style',
								'value' => kalium_get_theme_option( 'header_menu_pills' ),
							],
							[
								'name'  => 'pill_background',
								'value' => kalium_get_theme_option( 'header_menu_pill_background' ),
							],
							[
								'name'  => 'pill_color',
								'value' => kalium_get_theme_option( 'header_menu_pill_color' ),
							],
							[
								'name'  => 'pill_padding',
								'value' => kalium_get_theme_option( 'header_menu_pill_padding' ),
							],
							[
								'name'  => 'pill_radius',
								'value' => kalium_get_theme_option( 'header_menu_pill_radius' ),
							],
						],
					]
				);
			}
			break;

		// Fullscreen menu
		case 'full-bg-menu':
			$fullscreen_toggle = [
				'name'       => 'legacy_header_menu_trigger',
				'visible'    => true,
				'attributes' => [
					[
						'name'  => 'trigger_type',
						'value' => 'fullscreen',
					],
					[
						'name'  => 'fullscreen_alignment',
						'value' => kalium_get_theme_option( 'menu_full_bg_alignment' ),
					],
					[
						'name'  => 'fullscreen_search',
						'value' => kalium_get_theme_option( 'menu_full_bg_search_field' ),
					],
				],
			];

			// Custom background
			if ( kalium_get_theme_option( 'fullscreen_menu_bg' ) ) {
				$fullscreen_toggle['attributes'] = array_merge(
					$fullscreen_toggle['attributes'],
					[
						[
							'name'  => 'fullscreen_bg_color',
							'value' => kalium_get_theme_option( 'fullscreen_menu_bg_color' ),
						],
						[
							'name'  => 'fullscreen_bg_image',
							'value' => kalium_get_attachment_id_from_theme_option( 'fullscreen_menu_bg_image' ),
						],
						[
							'name'  => 'fullscreen_bg_position',
							'value' => kalium_get_theme_option( 'fullscreen_menu_bg_position' ),
						],
						[
							'name'  => 'fullscreen_bg_repeat',
							'value' => kalium_get_theme_option( 'fullscreen_menu_bg_repeat' ),
						],
						[
							'name'  => 'fullscreen_bg_size',
							'value' => kalium_get_theme_option( 'fullscreen_menu_bg_size' ),
						],
						[
							'name'  => 'fullscreen_bg_size_custom',
							'value' => kalium_get_theme_option( 'fullscreen_menu_bg_size_custom' ),
						],
					]
				);
			}

			$header_content[] = $fullscreen_toggle;
			break;

		// Off-Canvas Side
		case 'sidebar-menu':
			$header_content[] = [
				'name'       => 'legacy_header_menu_trigger',
				'visible'    => true,
				'attributes' => [
					[
						'name'  => 'trigger_type',
						'value' => 'off-canvas-side',
					],
					[
						'name'  => 'menu_sidebar_menu_id',
						'value' => kalium_get_theme_option( 'menu_sidebar_menu_id' ),
					],
					[
						'name'  => 'off_canvas_side_show_widgets',
						'value' => kalium_get_theme_option( 'menu_sidebar_show_widgets' ),
					],
					[
						'name'  => 'off_canvas_side_alignment',
						'value' => kalium_get_theme_option( 'menu_sidebar_alignment' ),
					],
				],
			];
			break;

		// Off-Canvas Top
		case 'top-menu':
			$header_content[] = [
				'name'       => 'legacy_header_menu_trigger',
				'visible'    => true,
				'attributes' => [
					[
						'name'  => 'trigger_type',
						'value' => 'off-canvas-top',
					],
					[
						'name'  => 'menu_id',
						'value' => kalium_get_theme_option( 'menu_top_menu_id' ),
					],
					[
						'name'  => 'off_canvas_top_items_per_row',
						'value' => kalium_get_theme_option( 'menu_top_items_per_row' ),
					],
					[
						'name'  => 'off_canvas_top_widgets',
						'value' => kalium_get_theme_option( 'menu_top_show_widgets' ),
					],
					[
						'name'  => 'off_canvas_top_widgets_per_row',
						'value' => kalium_get_theme_option( 'menu_top_widgets_per_row' ),
					],
					[
						'name'  => 'off_canvas_top_widgets_container_width',
						'value' => kalium_get_theme_option( 'menu_top_widgets_container_width' ),
					],
				],
			];
			break;
	}

	// Modified header content
	if ( ! empty( $header_content ) ) {
		return array_merge(
			$content,
			[
				'left'   => null,
				'bottom' => null,
				'right'  => array_merge(
					$content['right'] ?? [],
					[
						'visible'    => true,
						'attributes' => [],
						'elements'   => $header_content,
					]
				),
			]
		);
	}

	return $content;
}

add_filter( 'kalium_header_custom_header_content', 'kalium_header_convert_static_types_to_custom_header_content' );

/**
 * Standard menu toggle setup hook.
 *
 * @param array $content
 *
 * @return array
 * @since 4.0
 */
function kalium_header_standard_menu_toggle_fix( $content ) {
	return array_map( 'kalium_header_standard_menu_toggle_setup', $content );
}

add_filter( 'kalium_header_custom_header_content', 'kalium_header_standard_menu_toggle_fix' );

/**
 * Mobile menu content.
 */
function kalium_mobile_menu_content() {
	laborator_builder_render(
		[
			'content_type'   => 'mobile_menu',
			'content_source' => 'mobile_menu_content',
			'wrapper'        => '<div class="mobile-menu__content">%s</div>',
		],
		[
			'off_canvas_view' => true,
		],
	);
}

add_action( 'kalium_mobile_menu', 'kalium_mobile_menu_content' );

/**
 * Mobile menu breakpoint.
 */
function kalium_mobile_menu_breakpoint() {
	$breakpoint = kalium_get_mobile_menu_breakpoint();

	// Outside mobile breakpoint
	$outside_breakpoint = [
		'selector' => [
			'.mobile-menu',
			'.mobile-menu-close',
			'.mobile-menu-overlay',
			'.header-block .header-block__item--mobile-menu-toggle',
		],
		'props'    => [
			'display' => 'none',
		],
		'media'    => sprintf( '(min-width: %dpx)', $breakpoint + 1 ),
	];

	// Inside mobile breakpoint
	$inside_breakpoint = [
		'selector' => [
			'.header-block .header-block__item--standard-menu',
		],
		'props'    => [
			'display' => 'none',
		],
		'media'    => sprintf( '(max-width: %dpx)', $breakpoint ),
	];

	// Style
	kalium_print_inline_style(
		[
			'id'     => 'mobile-menu-breakpoint',
			'styles' => [
				$outside_breakpoint,
				$inside_breakpoint,
			],
		]
	);
}

add_action( 'wp_head', 'kalium_mobile_menu_breakpoint' );
