<?php
/**
 * Kalium WordPress Theme
 *
 * Other hooks functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Define debug mode in body class.
 *
 * @param array $classes
 *
 * @return array
 */
function kalium_debug_mode_body_class( $classes ) {
	if ( kalium()->is->debugging() ) {
		$classes[] = 'kalium-debug';
	}

	return $classes;
}

add_filter( 'body_class', 'kalium_debug_mode_body_class', 10 );

/**
 * Homepage hashtags links fix.
 * Remove multiple current menu items with hashtags.
 *
 * @param array   $classes
 * @param WP_Post $item
 *
 * @return array
 */
function kalium_unique_hashtag_url_base_menu_item( $classes, $item ) {
	$url = $item->url;

	// Only hashtag links
	if ( false !== strpos( $url, '#' ) ) {
		$url_md5 = ( preg_replace( '/#.*/', '', $url ) );

		// Skip first item only
		if ( ! isset( $GLOBALS['kalium_hashtag_links'][ $url_md5 ] ) ) {
			$GLOBALS['kalium_hashtag_links'][ $url_md5 ] = true;

			return $classes;
		}

		$remove_classes = [
			'current_page_item',
			'current-menu-item',
			'current-menu-ancestor',
			'current_page_ancestor',
		];

		foreach ( $remove_classes as $class_to_remove ) {
			$current_item_index = array_search( $class_to_remove, $classes );

			if ( false !== $current_item_index ) {
				unset( $classes[ $current_item_index ] );
			}
		}
	}

	return $classes;
}

add_filter( 'nav_menu_css_class', 'kalium_unique_hashtag_url_base_menu_item', 10, 2 );

/**
 * Homepage hashtags reset skipped item.
 *
 * @param array $args Array of wp_nav_menu() arguments.
 *
 * @return array
 */
function kalium_unique_hashtag_url_base_reset( $args ) {
	$GLOBALS['kalium_hashtag_links'] = [];

	return $args;
}

add_filter( 'wp_nav_menu_args', 'kalium_unique_hashtag_url_base_reset' );

/**
 * Custom sidebars params.
 *
 * @param array $sidebar
 *
 * @return array
 */
function kalium_custom_sidebars_params( $sidebar ) {

	// Widget wrappers
	$before_widget = '<div id="%1$s" class="widget %2$s">';
	$after_widget  = '</div>';
	$before_title  = '<h3 class="widget-title">';
	$after_title   = '</h3>';

	$sidebar['before_widget'] = $before_widget;
	$sidebar['after_widget']  = $after_widget;
	$sidebar['before_title']  = $before_title;
	$sidebar['after_title']   = $after_title;

	return $sidebar;
}

add_filter( 'cs_sidebar_params', 'kalium_custom_sidebars_params', 10 );

/**
 * Password protected post form.
 *
 * @param string $output
 *
 * @return string
 */
function kalium_password_form_assign_button_class( $output ) {
	$output = str_replace( 'type="submit"', sprintf( 'type="submit" %s', 'class="button button-small"' ), $output );

	return $output;
}

add_filter( 'the_password_form', 'kalium_password_form_assign_button_class', 10 );

/**
 * Product Filter plugin AJAX fix with WPBakery.
 *
 * @return void
 */
function kalium_prdctfltr_map_wpb_shortcodes_fix() {
	if ( class_exists( 'WPBMap' ) ) {
		add_action( 'admin_init', 'WPBMap::addAllMappedShortcodes' );
	}
}

add_action( 'before_prdctfltr_init', 'kalium_prdctfltr_map_wpb_shortcodes_fix' );

/**
 * Maintenance mode.
 *
 * @return void
 */
function kalium_page_maintenance_mode() {
	if ( ! kalium_is_maintenance_mode() ) {
		return;

	}

	$classes = [
		'maintenance-mode',
	];

	// Description
	$page_title       = kalium_get_theme_option( 'maintenance_mode_title' );
	$page_description = kalium_get_theme_option( 'maintenance_mode_description' );

	// Logo
	if ( kalium_get_theme_option( 'maintenance_mode_use_uploaded_logo' ) ) {
		$max_logo_width = kalium_get_theme_option( 'maintenance_mode_custom_logo_max_width' );

		ob_start();

		if ( $custom_logo = kalium_get_attachment_id_from_theme_option( 'maintenance_mode_custom_logo_image' ) ) {
			kalium_logo_element( $custom_logo, $max_logo_width );
		} else {
			kalium_logo_element( null, $max_logo_width );
		}

		$site_logo = ob_get_clean();

	}

	// Countdown
	$countdown      = kalium_get_theme_option( 'maintenance_mode_countdown' );
	$countdown_date = strtolower( kalium_get_theme_option( 'maintenance_mode_countdown_date' ) . ' ' . kalium_get_theme_option( 'maintenance_mode_countdown_time' ) . ':00' );

	if ( $countdown ) {
		$countdown_plugin = sprintf( '<script>%s</script>', file_get_contents( kalium()->assets_path( 'vendors/date-countdown.js' ) ) );
		add_action( 'wp_print_footer_scripts', kalium_hook_echo_value( $countdown_plugin ) );
	}

	// Social networks
	$social_networks = kalium_get_theme_option( 'maintenance_mode_social_networks' );

	// Style
	$css_vars = [];

	if ( $custom_text = kalium_get_theme_option( 'maintenance_mode_custom_text_color' ) ) {
		$css_vars['mm-color'] = $custom_text;
	}

	if ( $custom_bg = kalium_get_theme_option( 'maintenance_mode_custom_bg_color' ) ) {
		$css_vars['mm-bg-color'] = $custom_bg;
	}

	if ( $custom_bg_id = kalium_get_attachment_id_from_theme_option( 'maintenance_mode_custom_bg_id' ) ) {
		$image = wp_get_attachment_image_src( $custom_bg_id, 'full' );

		$css_vars['mm-bg-image'] = sprintf( 'url("%s")', esc_url( $image[0] ) );
		$css_vars['mm-bg-size']  = kalium_get_theme_option( 'maintenance_mode_custom_bg_size' );

		if ( kalium_get_theme_option( 'maintenance_mode_custom_bg_color_overlay' ) ) {
			$classes[] = 'maintenance-mode--bg-overlay';
		}
	}

	// Hide header and footer
	add_filter( 'kalium_show_header', '__return_false' );
	add_filter( 'kalium_show_footer', '__return_false' );

	// Remove demo store notice
	remove_action( 'wp_footer', 'woocommerce_demo_store' );

	// Document title
	if ( $page_title ) {
		add_filter( 'document_title_parts', kalium_hook_add_array_value( $page_title, 'title' ) );
	}

	// Body classes
	kalium()->helpers->add_body_class( $classes );

	// Theme header
	get_header();

	// Styles
	kalium_print_inline_style(
		[
			'id'       => 'maintenance-mode',
			'selector' => '.maintenance-mode',
			'vars'     => $css_vars,
		]
	);

	// Load page template
	kalium_get_template(
		'pages/maintenance.php',
		[
			'site_logo'        => $site_logo ?? null,
			'page_title'       => $page_title,
			'page_description' => $page_description,
			'countdown'        => $countdown,
			'countdown_date'   => $countdown_date,
			'social_networks'  => $social_networks,
			'aov_args'         => [
				'animation' => 'fade-in',
				'duration'  => 1,
				'stagger'   => 1,
			],
		]
	);

	// Theme footer
	get_footer();

	exit;
}

add_action( 'template_redirect', 'kalium_page_maintenance_mode' );

/**
 * Google Meta Theme Color (Phone).
 *
 * @return void
 */
function kalium_google_theme_color() {
	$google_theme_color = kalium_get_theme_option( 'google_theme_color' );

	if ( kalium_is_color_reference( $google_theme_color ) ) {
		$google_theme_color = kalium_color_reference_value( $google_theme_color );
	}

	if ( $google_theme_color ) {
		printf( '<meta name="theme-color" content="%s">', esc_attr( $google_theme_color ) );
	}
}

add_action( 'wp_head', 'kalium_google_theme_color' );

/**
 * Text line below username on single post page.
 *
 * @param WP_User $profileuser
 */
function kalium_user_custom_text( $profileuser ) {
	$user_custom_text = isset( $profileuser->_user_custom_text ) ? $profileuser->_user_custom_text : '';
	?>
	<tr>
		<th scope="row">
			User custom text
		</th>
		<td>
			<input type="text" name="user_custom_text" id="user_custom_text" value="<?php echo esc_attr( $user_custom_text ); ?>" class="regular-text"/>
			<span class="description">Enter text to display below user name in single post. Default: user role.</span>
		</td>
	</tr>
	<?php
}

add_action( 'personal_options', 'kalium_user_custom_text' );

/**
 * Display user custom text on single post page.
 *
 * @param string $text
 *
 * @return string
 */
function kalium_user_custom_text_display( $text ) {
	global $post;

	if ( $user_custom_text = get_user_meta( $post->post_author, '_user_custom_text', true ) ) {
		return $user_custom_text;
	}

	return $text;
}

add_filter( 'kalium_blog_single_post_author_info_subtitle', 'kalium_user_custom_text_display' );

/**
 * Save custom user text.
 *
 * @param int $user_id
 */
function kalium_user_custom_text_save( $user_id ) {
	if ( current_user_can( 'edit_user', $user_id ) ) {
		$user_custom_text = kalium()->request->input( 'user_custom_text' );
		update_user_meta( $user_id, '_user_custom_text', $user_custom_text );
	}
}

add_action( 'personal_options_update', 'kalium_user_custom_text_save' );
add_action( 'edit_user_profile_update', 'kalium_user_custom_text_save' );

/**
 * Like feature for post items.
 *
 * @return void
 */
function kalium_update_likes_for_post() {
	$response = [
		'liked' => false,
		'count' => 0,
	];

	$post_id = kalium()->request->request( 'post_id' );
	$user_ip = kalium_get_user_ip();

	// Only if post exists
	if ( $post = get_post( $post_id ) ) {
		$likes = $post->post_likes;
		$likes = is_array( $likes ) ? $likes : [];
		$liked = in_array( $user_ip, $likes );

		// Like post
		if ( ! $liked ) {
			$likes[] = $user_ip;

			$response['liked'] = true;
			$response['count'] = count( $likes );
		} else { // Unlike post
			$like_index = array_search( $user_ip, $likes );

			if ( false !== $like_index ) {
				unset( $likes[ $like_index ] );
			}

			$response['liked'] = false;
			$response['count'] = count( $likes );
		}

		// Update post meta
		update_post_meta( $post_id, 'post_likes', $likes );

		// Delete post cache
		if ( function_exists( 'wp_cache_post_change' ) ) {
			wp_cache_post_change( $post_id );
		}
	}

	wp_send_json( $response );
}

add_action( 'wp_ajax_kalium_post_like', 'kalium_update_likes_for_post' );
add_action( 'wp_ajax_nopriv_kalium_post_like', 'kalium_update_likes_for_post' );

/**
 * Like button.
 *
 * @param array $atts
 *
 * @return string|null
 */
function kalium_shortcode_ajax_like_button( $atts ) {
	global $post;

	// If post doesn't exists
	if ( ! ( $post instanceof WP_Post ) && ! isset( $atts['id'] ) ) {
		return null;
	}

	// Post ID
	$id = kalium_get_array_key( $atts, 'id', $post->ID );

	return kalium_like_button(
		[
			'display_count' => true,
			'post_id'       => $id,
		]
	);
}

add_shortcode( 'kalium_ajax_like_button', 'kalium_shortcode_ajax_like_button' );

/**
 * Recommended by ACF.
 *
 * @see https://www.advancedcustomfields.com/blog/acf-6-0-3-release-security-changes-to-the-acf-shortcode-and-ui-improvements/#acf-shortcode
 */
function kalium_set_acf_settings() {
	acf_update_setting( 'enable_shortcode', false );
}

add_action( 'acf/init', 'kalium_set_acf_settings' );

/**
 * Set Include libraries globally to "Off" initially.
 *
 * @param array $settings
 *
 * @return array
 * @since 3.7.1
 */
function kalium_rs_get_global_settings( $settings ) {
	if ( ! is_array( $settings ) ) {
		$settings = [
			'allinclude'   => false,
			'fontdownload' => 'preload',
		];
	}

	return $settings;
}

add_filter( 'rs_get_global_settings', 'kalium_rs_get_global_settings' );

/**
 * LayerSlider Conditional script loading.
 *
 * @since 3.7.1
 */
add_filter( 'ls_conditional_script_loading', '__return_true' );

/**
 * RECaptcha support in Contact Form.
 */
function kalium_contact_form_recaptcha_support( $forms ) {
	$forms['kalium-contact-form'] = [
		'form_name' => 'Kalium Contact Form',
	];

	return $forms;
}

add_action( 'gglcptch_add_custom_form', 'kalium_contact_form_recaptcha_support' );

/**
 * Jetpack remove YouTube and Vimeo embed.
 */
function kalium_jetpack_remove_youtube_vimeo_shortcodes( $shortcodes ) {
	$jetpack_shortcodes_dir = WP_CONTENT_DIR . '/plugins/jetpack/modules/shortcodes/';

	$shortcodes_to_unload = [
		'youtube.php',
		'vimeo.php',
	];

	foreach ( $shortcodes_to_unload as $shortcode ) {
		if ( $key = array_search( $jetpack_shortcodes_dir . $shortcode, $shortcodes ) ) {
			unset( $shortcodes[ $key ] );
		}
	}

	return $shortcodes;
}

add_filter( 'jetpack_shortcodes_to_include', 'kalium_jetpack_remove_youtube_vimeo_shortcodes' );

/**
 * Remove LayerSlider plugin notices.
 */
if ( defined( 'LS_PLUGIN_BASE' ) ) {
	remove_action( 'after_plugin_row_' . LS_PLUGIN_BASE, 'layerslider_plugins_purchase_notice', 10 );
}

/**
 * LayerSlider hide Notice
 */
add_filter( 'option_layerslider-authorized-site', '__return_true', 1000 );

/**
 * Portfolio like and share.
 *
 * @return string
 */
function shortcode_lab_portfolio_like_share() {
	return 'DEV IN PROGRESS'; // TMP
}

add_shortcode( 'lab_portfolio_like_share', 'shortcode_lab_portfolio_like_share' );

/**
 * Remove the one-week support notice for Ninja Forms that shows right after installation.
 *
 * @param array $notices
 *
 * @return array
 * @since 4.0.9
 */
function kalium_nf_disable_one_week_support_notice( $notices ) {
	unset( $notices['one_week_support'] );

	return $notices;
}

add_action( 'ninja_forms_admin_notices', 'kalium_nf_disable_one_week_support_notice' );

/**
 * Site info shortcode.
 *
 * @param array $atts
 *
 * @return string
 * @since 4.0
 */
function kalium_site_info_shortcode( $atts ) {
	$atts = shortcode_atts(
		[
			'display' => '{copyright} {year} {site_title} – {theme_credits}',
		],
		$atts
	);

	$display = $atts['display'];

	$placeholders = [
		'copyright'     => '&copy;',
		'year'          => wp_date( 'Y' ),
		'site_title'    => get_bloginfo( 'name' ),
		'theme_credits' => 'WordPress theme by <a href="https://laborator.co">Laborator</a>',
	];

	foreach ( $placeholders as $placeholder => $value ) {
		$display = str_replace( "{{$placeholder}}", $value, $display );
	}

	return $display;
}

add_shortcode( 'kalium_site_info', 'kalium_site_info_shortcode' );

/**
 * Skip onboarding for Elementor (breaks starter site importing).
 *
 * @since 4.0
 */
function kalium_elementor_skip_onboarding() {
	add_option( 'elementor_onboarded', true );
}

add_action( 'activate_elementor/elementor.php', 'kalium_elementor_skip_onboarding' );

/**
 * Use legacy form style on contact form.
 *
 * @since 4.0
 */
function kalium_maybe_use_legacy_form_style( $use ) {
	return get_theme_mod( 'legacy_form_style', $use );
}

add_filter( 'kalium_use_legacy_form_style', 'kalium_maybe_use_legacy_form_style' );

/**
 * Set WooCommerce Product Filter to use as theme. Temporary fix to
 * set the plugin as theme bundled plugin.
 *
 * @param array $args
 *
 * @return array|string
 */
function kalium_prdctfltr_set_as_theme( $args ) {
	if ( is_bool( $args ) ) {
		return 'true';
	}

	$args['product_filter']['key'] = 'true';

	return $args;
}

add_filter( 'svx_plugins_settings', 'kalium_prdctfltr_set_as_theme', 100 );
add_filter( 'pre_option_xforwc_key_product_filter', 'kalium_prdctfltr_set_as_theme' );
