<?php
/**
 * Kalium WordPress Theme
 *
 * Theme style hook functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Print theme colors.
 */
function kalium_print_theme_color_vars() {
	$flattened_color_vars = $vars = [];

	/**
	 * Check if color variant is unchanged.
	 *
	 * @param string $color
	 *
	 * @return bool
	 */
	$is_color_variant_unchanged = function ( $color ) use ( &$is_color_variant_unchanged ) {
		// Extract color from color variant
		if ( kalium_is_color_variant( $color ) && preg_match( KALIUM_COLOR_VARIANT_REGEX, $color, $matches ) ) {
			return $is_color_variant_unchanged( $matches['color'] );
		}
		// Check color reference match
		elseif ( kalium_is_color_reference( $color ) && preg_match( KALIUM_COLOR_REFERENCE_REGEX, $color, $matches ) ) {
			$path       = explode( '.', $matches['path'] );
			$setting_id = array_shift( $path );

			$value         = kalium_get_theme_option( $setting_id );
			$default_value = kalium_get_default_theme_option( $setting_id );

			while ( $sub_key = array_shift( $path ) ) {
				$value         = kalium_get_array_key( $value, $sub_key );
				$default_value = kalium_get_array_key( $default_value, $sub_key );
			}

			// Changed
			if ( $value !== $default_value ) {
				return false;
			}
			// Continue checking color references
			elseif ( kalium_is_color_reference( $value ) ) {
				return $is_color_variant_unchanged( $value );
			}
		}

		return true;
	};

	// Flatten color vars
	foreach ( kalium_get_theme_color_vars() as $color_var ) {
		// Theme mod
		if ( isset( $color_var['id'] ) ) {
			$setting_id  = $color_var['id'];
			$css_var     = $color_var['css_var'];
			$transparent = $color_var['transparent'] ?? false;
			$value       = kalium_get_theme_option( $setting_id, null );
			$default     = kalium_get_default_theme_option( $setting_id );

			if ( is_string( $css_var ) ) {
				$flattened_color_vars[] = [
					'css_var'     => $css_var,
					'value'       => $value,
					'is_default'  => is_null( $value ) || $default === $value,
					'transparent' => $transparent,
				];
			} elseif ( is_array( $css_var ) ) {
				foreach ( $css_var as $key => $css_var_name ) {
					$entry_value   = $value[ $key ] ?? null;
					$entry_default = kalium_get_array_key( $default, $key );

					if ( is_null( $entry_value ) ) {
						$entry_value = $entry_default;
					}

					$flattened_color_vars[] = [
						'css_var'     => $css_var_name,
						'value'       => $entry_value,
						'is_default'  => kalium_get_array_key( $default, $key ) === $entry_value,
						'transparent' => $transparent,
					];
				}
			}
		}
		// Generate
		elseif ( ! empty( $color_var['generate'] ) ) {
			if ( $source = kalium_get_array_key( $color_var['generate'], 'source' ) ) {
				$color_entry = [
					'css_var'   => $color_var['css_var'],
					'value'     => null,
					'generated' => true,
				];

				// Shift
				if ( array_key_exists( 'shift', $color_var['generate'] ) ) {
					$color_entry['value'] = kalium_color_variant( kalium_color_reference( $source ), [ "shift({$color_var['generate']['shift']})" ] );
				}

				// Add generated color entry
				if ( ! empty( $color_entry['value'] ) ) {
					$flattened_color_vars[] = $color_entry;
				}
			}
		}
	}

	// Register color vars
	foreach ( $flattened_color_vars as $color_var ) {
		$css_var = $color_var['css_var'];
		$value   = $color_var['value'];

		$is_default       = ! empty( $color_var['is_default'] );
		$is_color_variant = kalium_is_color_variant( $value );
		$is_transparent   = ! $value && $color_var['transparent'];

		// Skip defaults
		if ( $is_default && ! $is_color_variant && ! $is_transparent ) {
			continue;
		} elseif ( $is_color_variant && $is_color_variant_unchanged( $value ) ) {
			continue;
		}

		if ( $value ) {

			// Process value
			$parsed_value = kalium_replace_color_references( $value );

			// Register color var
			$vars[ $css_var ] = $parsed_value;

			// RGB variant of color
			$vars[ $css_var . '-rgb' ] = kalium_parse_color_rgb( $value );
		} elseif ( $is_transparent ) {
			$vars[ $css_var ]          = 'transparent';
			$vars[ $css_var . '-rgb' ] = '0, 0, 0';
		}
	}

	// Register CSS vars
	if ( is_array( $vars ) ) {
		kalium_print_inline_style(
			[
				'id'       => 'color-vars',
				'selector' => kalium_get_root_selector(),
				'vars'     => $vars,
			]
		);
	}
}

add_action( 'wp_head', 'kalium_print_theme_color_vars', 15 );

/**
 * Nav menu widget.
 *
 * @param array $args
 *
 * @return array
 */
function kalium_widget_nav_menu_link_style( $args ) {
	$args['container_class'] = kalium_tokenize_list(
		[
			kalium_get_array_key( $args, 'container_class' ),
			kalium_get_link_block_class(),
			kalium_get_link_text_class(),
		]
	);

	return $args;
}

add_filter( 'widget_nav_menu_args', 'kalium_widget_nav_menu_link_style' );

/**
 * Adjust link styles application based on their type and use.
 *
 * @param bool   $should_apply
 * @param string $context
 *
 * @return bool
 */
function kalium_link_styles_adjust_by_context( $should_apply, $context ) {
	static $link_style;

	if ( is_null( $link_style ) ) {
		$link_style = kalium_get_theme_option( 'link_style' );
	}

	// Swap and fill link style
	if ( in_array( $link_style, [ 'swap', 'fill' ] ) ) {
		global $portfolio_args;

		$portfolio_layout_type = $portfolio_args['layout_type'] ?? null;

		// Disable style on portfolio item type 2
		if ( 'portfolio-loop-item-categories' === $context && 'type-2' === $portfolio_layout_type ) {
			$should_apply = false;
		}
	}

	return $should_apply;
}

add_action( 'kalium_link_style_class_by_context', 'kalium_link_styles_adjust_by_context', 10, 2 );
