<?php
/**
 * Kalium WordPress Theme
 *
 * WooCommerce template functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

if ( ! function_exists( 'kalium_woocommerce_cart_totals_widget' ) ) {

	/**
	 * Cart totals widget used on header.
	 */
	function kalium_woocommerce_cart_totals_widget( $args = [] ) {

		// If WooCommerce is not installed
		if ( ! kalium()->is->woocommerce_active() ) {
			return;
		}

		// Args
		$args = wp_parse_args(
			$args,
			[
				'total_price' => '',
				'text_before' => '',
				'hide_empty'  => true,
			]
		);

		// Totals
		$totals = WC()->cart->get_totals();

		switch ( $args['total_price'] ) {

			// Sub total
			case 'cart-subtotal':
				$total_display_price = $totals['subtotal'];
				break;

			// Cart total excluding tax
			case 'cart-total-ex-tax':
				$total_display_price = $totals['total'] - $totals['total_tax'];
				break;

			// Default total price
			default:
				$total_display_price = $totals['total'];
		}

		// Classes
		$classes = [
			'cart-totals-widget',
			kalium_get_link_style_class_by_context( 'header-cart-totals', kalium_get_link_plain_class() ),
		];

		// Hide when empty
		if ( $args['hide_empty'] ) {
			$classes[] = 'cart-totals-widget--hide-empty';

			if ( 0 === kalium_woocommerce_get_items_in_cart() ) {
				$classes[] = 'cart-totals-widget--hidden';
			}
		}
		?>
		<div <?php kalium_class_attr( $classes ); ?> data-total-price="<?php echo esc_attr( $args['total_price'] ? $args['total_price'] : 'cart-total' ); ?>">

			<?php if ( $args['text_before'] ) : ?>
				<span class="text-before">
					<?php echo esc_html( $args['text_before'] ); ?>
				</span>
			<?php endif; ?>


			<?php
			// Price total
			kalium_element(
				[
					'tag_name'   => 'a',
					'attributes' => [
						'href'  => wc_get_cart_url(),
						'class' => 'cart-total',
					],
					'content'    => wc_price( $total_display_price ), // Wrap price with currency
					'echo'       => true,
				]
			);
			?>

		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_my_account_dashboard_heading' ) ) {

	/**
	 * Dashboard page heading.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_my_account_dashboard_heading() {
		kalium_woocommerce_my_account_page_heading(
			__( 'Hello', 'kalium' ),
			__( 'Manage your account, addresses and view your orders', 'kalium' ),
			[ 'dashboard-heading' ]
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_my_account_orders_heading' ) ) {

	/**
	 * My orders page title.
	 */
	function kalium_woocommerce_my_account_orders_heading( $has_orders ) {

		// No orders yet
		if ( 0 === kalium_woocommerce_get_orders_count() ) {
			kalium_woocommerce_no_entries(
				kalium_xtranslate( '__', 'No order has been made yet.', 'woocommerce' ),
				__( 'Go to the shop page and make your first purchase!', 'kalium' ),
				'kalium-icon-shopping-bag-empty'
			);

			return;
		}

		// Orders heading
		kalium_woocommerce_my_account_page_heading(
			esc_html__( 'My Orders', 'kalium' ),
			esc_html__( 'Your recent orders are displayed in the table below.', 'kalium' )
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_my_account_downloads_heading' ) ) {

	/**
	 * My downloads page title.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_my_account_downloads_heading( $has_orders ) {

		// No downloads yet
		if ( ! WC()->customer->get_downloadable_products() ) {
			kalium_woocommerce_no_entries(
				kalium_xtranslate( '__', 'No downloads available yet.', 'woocommerce' ),
				__( 'Go to the shop page and browse for downloadable products!', 'kalium' ),
				'kalium-icon-folder-empty'
			);

			return;
		}

		// Downloads heading
		kalium_woocommerce_my_account_page_heading(
			esc_html__( 'My Downloads', 'kalium' ),
			esc_html__( 'Your digital downloads are displayed in the table below.', 'kalium' )
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_myaccount_edit_account_heading' ) ) {

	/**
	 * Edit account heading.
	 */
	function kalium_woocommerce_myaccount_edit_account_heading() {
		kalium_woocommerce_my_account_page_heading(
			esc_html__( 'My account', 'kalium' ),
			esc_html__( 'Edit your account details or change your password', 'kalium' )
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_myaccount_edit_address_heading' ) ) {

	/**
	 * My account addresses page heading.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_myaccount_edit_address_heading() {
		kalium_woocommerce_my_account_page_heading(
			kalium_xtranslate( '_n', 'Addresses', 'Address', (int) wc_shipping_enabled(), 'woocommerce' ),
			esc_html__( 'Manage your billing and/or shipping addresses', 'kalium' )
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_myaccount_add_payment_method_heading' ) ) {

	/**
	 * My account add payment heading.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_myaccount_add_payment_method_heading() {
		kalium_woocommerce_my_account_page_heading(
			kalium_xtranslate( '__', 'Payment methods', 'woocommerce' ),
			esc_html__( 'Manage your payment methods', 'kalium' )
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_myaccount_form_pay_heading' ) ) {

	/**
	 * My account form pay heading.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_myaccount_form_pay_heading() {
		kalium_woocommerce_my_account_page_heading(
			kalium_xtranslate( '__', 'Pay for order', 'woocommerce' ),
			esc_html__( 'Review your order and proceed to payment', 'kalium' )
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_no_entries' ) ) {

	/**
	 * No entries template.
	 *
	 * @param string $title
	 * @param string $description
	 * @param string $icon
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_no_entries( $title, $description, $icon = '' ) {
		?>
		<div class="no-entries">
			<?php if ( $icon ) : ?>
				<i class="<?php echo esc_attr( $icon ); ?>"></i>
			<?php endif; ?>

			<h4><?php echo esc_html( $title ); ?></h4>
			<p><?php echo esc_html( $description ); ?></p>
			<a href="<?php echo esc_url( apply_filters( 'woocommerce_return_to_shop_redirect', wc_get_page_permalink( 'shop' ) ) ); ?>" class="button"><?php kalium_xtranslate( 'esc_html_e', 'Browse products', 'woocommerce' ); ?></a>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_get_lightbox_trigger_button' ) ) {

	/**
	 * Trigger lightbox button.
	 *
	 * @param int $attachment_id
	 */
	function kalium_woocommerce_get_lightbox_trigger_button( $attachment_id ) {
		return sprintf( '<button class="product-gallery-lightbox-trigger" data-id="%s" title="%s"><i class="kalium-icon-zoom"></i></button>', $attachment_id, esc_attr__( 'View full size', 'kalium' ) );
	}
}

if ( ! function_exists( 'kalium_woocommerce_get_product_gallery_container_alignment' ) ) {

	/**
	 * Single product images alignment.
	 */
	function kalium_woocommerce_get_product_gallery_container_alignment() {
		$image_alignment = kalium_get_theme_option( 'shop_single_image_alignment' );

		return 'right' === $image_alignment ? 'right' : 'left';
	}
}

if ( ! function_exists( 'kalium_woocommerce_top_rated_products_widget_rating' ) ) {

	/**
	 * Show rating below top rated products widget.
	 */
	function kalium_woocommerce_top_rated_products_widget_rating( $args ) {
		global $product;

		if ( ! empty( $args['show_rating'] ) && $product->get_average_rating() ) :
			?>
			<p class="rating">
				<i class="kalium-icon-star-filled"></i>
				<?php echo $product->get_average_rating(); ?>
			</p>
			<?php
		endif;
	}
}

if ( ! function_exists( 'kalium_woocommerce_go_back_link' ) ) {

	/**
	 * WooCommerce Go back link.
	 *
	 * @param string $link
	 * @param string $title
	 */
	function kalium_woocommerce_go_back_link( $link, $title = '' ) {
		if ( empty( $title ) ) {
			$title = esc_html__( 'Go back', 'kalium' );
		}

		$icon = '<i class="kalium-icon-long-arrow-left"></i>';

		printf( '<a href="%1$s" class="button go-back-link">%3$s%2$s</a>', esc_url( $link ), esc_html( $title ), $icon );
	}
}

if ( ! function_exists( 'kalium_woocommerce_empty_cart_message' ) ) {

	/**
	 * Empty cart text.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_empty_cart_message() {

		?>
		<div class="cart-empty wc-empty-cart-message">

			<div class="cart-empty__icon">
				<i class="kalium-icon-shopping-bag-empty"></i>
			</div>

			<h1 class="cart-empty__title">
				<?php kalium_xtranslate( 'esc_html_e', 'Your cart is currently empty!', 'woocommerce' ); ?>
			</h1>

			<p class="cart-empty__message">
				<?php esc_html_e( 'Browse our store, find products, and happy shopping!', 'kalium' ); ?>
			</p>

		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_cross_sell_display' ) ) {

	/**
	 * Cross-sells on cart page.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_cross_sell_display() {
		$args = apply_filters(
			'kalium_woocommerce_cross_sell_args',
			[
				'limit'   => 4,
				'columns' => 4,
			]
		);

		woocommerce_cross_sell_display( $args['limit'], $args['columns'] );
	}
}

if ( ! function_exists( 'kalium_woocommerce_cart_checkout_heading' ) ) {

	/**
	 * Cart and checkout heading.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_cart_checkout_heading() {

		// Hide cart heading when default heading title is enabled
		if ( kalium_get_field( 'heading_title', get_queried_object_id() ) ) {
			return;
		}

		// Number of items in cart
		$cart_items = kalium_woocommerce_get_items_in_cart();

		// Template args
		$args = [
			'container'   => false,
			'heading_tag' => 'h1',
			'title'       => get_the_title(),
			'description' => sprintf( _n( 'You\'ve got 1 item in the cart', 'You\'ve got %d items in the cart', $cart_items, 'kalium' ), $cart_items ),
		];

		// Load page heading template
		kalium_page_heading( $args );
	}
}

if ( ! function_exists( 'kalium_woocommerce_coupon_form' ) ) {


	/**
	 * Coupon form in order details table.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_coupon_form() {
		if ( wc_coupons_enabled() ) {
			?>
			<tr>
				<td colspan="2">
					<div <?php kalium_class_attr( [ 'coupon-form', kalium_get_link_text_class(), kalium_get_link_plain_class() ] ); ?>>
						<a href="#" class="coupon-form__link">
							<i class="coupon-icon kalium-icon-coupon-cut"></i>
							<?php echo kalium_xtranslate( 'esc_html__', 'Have a coupon?', 'woocommerce' ); ?>
						</a>
						<div class="coupon-form__coupon-code row">
							<div class="col">
								<input type="text" class="coupon-code" name="coupon_code" placeholder="<?php kalium_xtranslate( 'esc_attr_e', 'Coupon code', 'woocommerce' ); ?>">
							</div>
							<div class="col-auto">
								<input type="submit" class="button" name="apply_coupon" value="<?php kalium_xtranslate( 'esc_attr_e', 'Apply', 'woocommerce' ); ?>">
							</div>
							<?php do_action( 'woocommerce_cart_coupon' ); ?>
						</div>
					</div>
				</td>
			</tr>
			<?php
		} else {
			?>
			<tr>
				<td colspan="2">
					<div class="invoice-details__separator"></div>
				</td>
			</tr>
			<?php
		}
	}
}

if ( ! function_exists( 'kalium_woocommerce_checkout_login_form_toggle' ) ) {

	/**
	 * Login form toggle on checkout page.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_checkout_login_form_toggle() {
		if ( kalium_woocommerce_has_checkout_login_form() ) {
			?>
			<div class="page-heading__action-container col-auto">
				<a href="#" class="checkout-login-form">
					<span class="checkout-login-form__icon">
						<i class="kalium-icon-login"></i>
					</span>
					<span class="checkout-login-form__text">
						<span class="checkout-login-form__text-title"><?php echo apply_filters( 'woocommerce_checkout_login_message', kalium_xtranslate( 'esc_html__', 'Returning customer?', 'woocommerce' ) ); ?></span>
						<span class="checkout-login-form__text-description"><?php kalium_xtranslate( 'esc_html_e', 'Click here to login', 'woocommerce' ); ?></span>
					</span>
				</a>
			</div>
			<?php
		}
	}
}

if ( ! function_exists( 'kalium_woocommerce_checkout_login_form' ) ) {

	/**
	 * Checkout login form.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_checkout_login_form() {
		if ( kalium_woocommerce_has_checkout_login_form() ) {
			woocommerce_login_form(
				[
					'message'  => kalium_xtranslate( 'esc_html__', 'If you have shopped with us before, please enter your details below. If you are a new customer, please proceed to the Billing section.', 'woocommerce' ),
					'redirect' => wc_get_checkout_url(),
					'hidden'   => true,
				]
			);
		}
	}
}

if ( ! function_exists( 'kalium_woocommerce_cart_wrapper_start' ) ) {

	/**
	 * Cart wrapper start.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_cart_wrapper_start() {
		echo '<div class="cart-wrapper">';
		echo '<div class="row">';
	}
}

if ( ! function_exists( 'kalium_woocommerce_cart_wrapper_end' ) ) {

	/**
	 * Cart wrapper end.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_cart_wrapper_end() {
		echo '</div>';
		echo '</div>';
	}
}

if ( ! function_exists( 'kalium_woocommerce_cart_form_wrapper_start' ) ) {

	/**
	 * Cart form wrapper start.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_cart_form_wrapper_start() {
		echo '<div class="col cart-wrapper__cart-form">';
	}
}

if ( ! function_exists( 'kalium_woocommerce_cart_form_wrapper_end' ) ) {

	/**
	 * Cart form wrapper end.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_cart_form_wrapper_end() {
		echo '</div>';
	}
}

if ( ! function_exists( 'kalium_woocommerce_cart_collaterals_wrapper_start' ) ) {

	/**
	 * Cart collaterals wrapper start.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_cart_collaterals_wrapper_start() {
		$classes = [
			'invoice-details',
		];

		if ( 'invoice-cut' === kalium_get_theme_option( 'shop_order_review_style' ) ) {
			$classes[] = 'invoice-details--invoice-cut';
		}

		if ( kalium_get_theme_option( 'shop_order_review_sticky' ) ) {
			$classes[] = 'invoice-details--sticky';
		}

		echo '<div class="col cart-wrapper__cart-collaterals">';
		echo '<div ' . kalium_class_attr( $classes, false ) . '>';
	}
}

if ( ! function_exists( 'kalium_woocommerce_cart_collaterals_wrapper_end' ) ) {

	/**
	 * Cart collaterals wrapper start.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_cart_collaterals_wrapper_end() {
		echo '</div>';
		echo '</div>';
	}
}

if ( ! function_exists( 'kalium_woocommerce_checkout_order_review_wrapper_start' ) ) {

	/**
	 * Order review wrapper start.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_checkout_order_review_wrapper_start() {
		$classes = [
			'invoice-details',
		];

		if ( 'invoice-cut' === kalium_get_theme_option( 'shop_order_review_style' ) ) {
			$classes[] = 'invoice-details--invoice-cut';
		}

		echo '<div ' . kalium_class_attr( $classes, false ) . '>';
	}
}

if ( ! function_exists( 'kalium_woocommerce_checkout_order_review_wrapper_end' ) ) {

	/**
	 * Order review wrapper end.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_checkout_order_review_wrapper_end() {
		echo '</div>';
	}
}

if ( ! function_exists( 'kalium_woocommerce_order_details_separator' ) ) {

	/**
	 * Order details table separator.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_order_details_separator() {
		?>
		<tr>
			<td colspan="2">
				<div class="invoice-details__separator invoice-details__separator--solid"></div>
			</td>
		</tr>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_order_review_cart_items_separator' ) ) {

	/**
	 * Order review cart items separator with order totals.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_order_review_cart_items_separator() {
		?>
		<tr class="separator">
			<td colspan="2">
				<div class="invoice-details__separator-dashed"></div>
			</td>
		</tr>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_my_account_create_account_button' ) ) {

	/**
	 * Create account button in WooCommerce login tab.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_my_account_create_account_button() {
		if ( 'yes' !== get_option( 'woocommerce_enable_myaccount_registration' ) || is_checkout() ) {
			return;
		}

		printf( '<button type="button" class="button button-secondary create-account">%1$s</button>', kalium_xtranslate( 'esc_html__', 'Create an account?', 'woocommerce' ) );
	}
}

if ( ! function_exists( 'kalium_woocommerce_my_account_back_to_login_button' ) ) {

	/**
	 * Back to login form button in WooCommerce my account page.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_my_account_back_to_login_button() {
		printf(
			'<button type="button" class="button button-link back-to-login-form">%s%s</button>',
			'<i class="kalium-icon-login"></i>',
			kalium_xtranslate( 'esc_html__', 'Log in', 'woocommerce' )
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_customer_login_form_wrapper_start' ) ) {

	/**
	 * Customer login form wrapper start.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_customer_login_form_wrapper_start() {
		if ( ! kalium_get_theme_option( 'shop_my_account_login_image' ) ) {
			return;
		}

		$classes = [
			'login-wrapper',
			'login-wrapper--image-' . kalium_get_theme_option( 'shop_my_account_login_image_align' ),
		];

		?>
		<div <?php kalium_class_attr( $classes ); ?>>
			<div class="login-wrapper__form">
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_customer_login_form_wrapper_end' ) ) {

	/**
	 * Customer login form wrapper end.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_customer_login_form_wrapper_end() {
		if ( ! kalium_get_theme_option( 'shop_my_account_login_image' ) ) {
			return;
		}

		?>
			</div><?php // End of .login-wrapper__form ?>
			<div class="login-wrapper__image">
				<?php
				if ( $attachment = wp_get_attachment_image( kalium_get_attachment_id_from_theme_option( 'shop_my_account_login_image' ), 'full' ) ) {
					echo $attachment;
				} else {
					// Placeholder image
					echo kalium_element(
						'img',
						[
							'src'    => kalium()->assets_url( 'images/woocommerce-my-account.png' ),
							'alt'    => 'woocommerce-my-account',
							'width'  => 1431,
							'height' => 1061,
						]
					);
				}
				?>
			</div>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_lost_password_form_heading' ) ) {

	/**
	 * Lost password form heading.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_lost_password_form_heading() {
		kalium_page_heading(
			[
				'heading_tag' => 'h2',
				'title'       => esc_html__( 'Reset password', 'kalium' ),
				'description' => esc_html__( 'Recover your lost password', 'kalium' ),
				'container'   => false,
			]
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_lost_password_wrapper_start' ) ) {

	/**
	 * Lost password wrapper start.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_lost_password_wrapper_start() {
		echo '<div class="lost-password-wrapper">';
	}
}

if ( ! function_exists( 'kalium_woocommerce_lost_password_wrapper_end' ) ) {

	/**
	 * Lost password wrapper end.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_lost_password_wrapper_end() {
		echo '</div>';
	}
}

if ( ! function_exists( 'kalium_woocommerce_myaccount_forgot_password_back_link' ) ) {

	/**
	 * Go back link for Forgot Password page.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_lost_password_back_link() {
		kalium_woocommerce_go_back_link( wc_get_account_endpoint_url( 'dashboard' ) );
	}
}

if ( ! function_exists( 'kalium_woocommerce_myaccount_navigation_user_profile' ) ) {

	/**
	 * User profile detail in my account page navigation.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_myaccount_navigation_user_profile() {
		$user = wp_get_current_user();

		?>
		<div class="user-profile">
			<div class="user-profile__avatar">
				<?php
				echo get_avatar( $user, 128 );
				?>
			</div>
			<div <?php kalium_class_attr( [ 'user-profile__info', kalium_get_link_text_class(), kalium_get_link_plain_class() ] ); ?>>
				<h5><?php echo esc_html( $user->display_name ); ?></h5>
				<a href="<?php echo esc_url( wc_get_endpoint_url( 'edit-account', '', wc_get_page_permalink( 'myaccount' ) ) ); ?>"><?php kalium_xtranslate( 'esc_html_e', 'Edit account', 'woocommerce' ); ?></a>
			</div>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_my_account_page_heading' ) ) {

	/**
	 * My account page heading.
	 *
	 * @param string $title
	 * @param string $description
	 * @param array  $classes
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_my_account_page_heading( $title, $description, $classes = [] ) {
		kalium_page_heading(
			[
				'heading_tag' => 'h1',
				'container'   => false,
				'classes'     => $classes,
				'title'       => $title,
				'description' => $description,
			]
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_downloads_column_product_name' ) ) {

	/**
	 * Download product name.
	 *
	 * @param array $download
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_downloads_column_product_name( $download ) {
		$product_image = '';

		// Assign image to product name
		if ( $attachment_id = get_post_thumbnail_id( $download['product_id'] ) ) {
			$image = wp_get_attachment_image( $attachment_id, [ 100, 100 ] );

			if ( ! empty( $image ) ) {
				$product_image = sprintf( '<span class="product-image">%1$s</span>', $image );
			}
		}

		// Product name
		if ( $download['product_url'] ) {
			printf( '<a href="%s">%s</a>', esc_url( $download['product_url'] ), $product_image . esc_html( $download['product_name'] ) );
		} else {
			echo $product_image . esc_html( $download['product_name'] );
		}
	}
}

if ( ! function_exists( 'kalium_woocommerce_order_thankyou_wrapper_start' ) ) {

	/**
	 * Thank you page wrapper start.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_order_thankyou_wrapper_start() {
		echo '<div class="thankyou-wrapper">';
	}
}

if ( ! function_exists( 'kalium_woocommerce_order_thankyou_wrapper_end' ) ) {

	/**
	 * Thank you page wrapper end.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_order_thankyou_wrapper_end() {
		echo '</div>';
	}
}

if ( ! function_exists( 'kalium_woocommerce_checkout_steps' ) ) {

	/**
	 * Checkout steps breadcrumb.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_checkout_steps() {
		if ( ! kalium_get_theme_option( 'shop_checkout_steps' ) || ! ( is_cart() || is_checkout() ) || ( is_cart() && 0 === kalium_woocommerce_get_items_in_cart() ) ) {
			return;
		}

		// Step nums
		$step_numbers = kalium_get_theme_option( 'shop_checkout_step_numbers' );

		// If enabled in theme options
		$checkout_steps = [
			'cart'           => [
				'title'   => __( 'Shopping cart', 'kalium' ),
				'link'    => wc_get_page_permalink( 'cart' ),
				'current' => true,
			],
			'checkout'       => [
				'title'   => kalium_xtranslate( '__', 'Checkout', 'woocommerce' ),
				'link'    => wc_get_page_permalink( 'checkout' ),
				'current' => is_checkout(),
			],
			'order-complete' => [
				'title'   => __( 'Order complete', 'kalium' ),
				'link'    => null,
				'current' => is_order_received_page(),
			],
		];

		// Disable links on order receive page
		if ( is_order_received_page() ) {
			foreach ( $checkout_steps as $i => $entry ) {
				$checkout_steps[ $i ]['link'] = null;
			}
		}
		?>
		<div <?php kalium_class_attr( [ 'checkout-steps', kalium_get_link_plain_class() ] ); ?>>
			<div class="container">
				<ul class="checkout-steps__list">
					<?php
					$step_num = 1;
					foreach ( $checkout_steps as $step_id => $entry ) :

						// Classes
						$classes = [
							'checkout-steps__list-entry',
							'checkout-steps__list-entry--' . $step_id,
						];

						// Current
						if ( $entry['current'] ) {
							$classes[] = 'current';
						}

						// Title
						$title = kalium_element(
							'span',
							[
								'class' => 'title',
							],
							esc_html( $entry['title'] )
						);

						// Content
						$step = $title;

						// Step num
						if ( $step_numbers ) {
							$num = kalium_element(
								'span',
								[
									'class' => 'num',
								],
								$step_num
							);

							$step = $num . $title;
						}

						// Link
						if ( ! empty( $entry['link'] ) ) {
							$step = kalium_element(
								'a',
								[
									'href' => $entry['link'],
								],
								$step
							);
						}
						?>
						<li <?php kalium_class_attr( $classes ); ?>>
							<?php echo $step; ?>
						</li>
						<?php
						++$step_num;
					endforeach;
					?>
				</ul>
			</div>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_second_image_on_hover' ) ) {

	/**
	 * Second image on hover element.
	 *
	 * @param int   $first_attachment_id
	 * @param int   $second_attachment_id
	 * @param array $args
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_second_image_on_hover( $first_attachment_id, $second_attachment_id, $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'hover_transition' => 'fade',
			]
		);

		// Container classes
		$classes = [
			'image-set',
		];

		// Hover transition
		if ( in_array( $args['hover_transition'], [ 'fade' ] ) ) {
			$classes[] = 'image-set--hover-transition-' . $args['hover_transition'];
		}

		// Image args
		$image_args = [];

		// Pass image args
		foreach ( [ 'link' ] as $image_arg ) {
			if ( isset( $args[ $image_arg ] ) ) {
				$image_args[ $image_arg ] = $args[ $image_arg ];
			}
		}
		?>
		<div <?php kalium_class_attr( $classes ); ?>>
			<div class="image-set__entry image-set__entry--hover-invisible">
				<?php kalium_woocommerce_loop_product_image( $first_attachment_id, $image_args ); ?>
			</div>
			<div class="image-set__entry image-set__entry--overlay image-set__entry--hover-visible">
				<?php kalium_woocommerce_loop_product_image( $second_attachment_id, $image_args ); ?>
			</div>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_images_gallery_on_hover' ) ) {

	/**
	 * Images gallery on hover element.
	 *
	 * @param int[] $attachment_ids
	 * @param array $args
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_images_gallery_on_hover( $attachment_ids, $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'hover_transition'           => 'fade',
				'pagination'                 => true,
				'pagination_style'           => 'bars',
				'pagination_hover'           => false,
				'pagination_hover_animation' => 'fade',
			]
		);

		// Total attachments
		$total_attachments = count( $attachment_ids );

		// Container classes
		$classes = [
			'image-set',
			'image-set--hover-entries',
			'image-set--hover-entries-' . $total_attachments,
		];

		// Hover transition
		if ( in_array( $args['hover_transition'], [ 'fade' ] ) ) {
			$classes[] = 'image-set--hover-transition-' . $args['hover_transition'];
		}

		// Pagination
		if ( $args['pagination'] ) {
			$classes[] = 'image-set--has-pagination';

			// Pagination style
			if ( in_array( $args['pagination_style'], [ 'bars', 'dots' ] ) ) {
				$classes[] = 'image-set--pagination-style-' . $args['pagination_style'];
			}

			// Pagination visible on hover
			if ( $args['pagination_hover'] ) {
				$classes[] = 'image-set--pagination-visible-on-hover';
				$classes[] = 'image-set--pagination-animation-' . $args['pagination_hover_animation'];
			}
		}

		// Link open
		if ( $args['link'] ) {
			printf( '<a href="%s" aria-label="%s">', esc_url( kalium_woocommerce_get_product_link() ), esc_attr( get_the_title() ) );
		}

		?>
		<div <?php kalium_class_attr( $classes ); ?>>
			<?php
			foreach ( $attachment_ids as $i => $attachment_id ) :
				$entry_classes = [
					'image-set__entry',
				];

				$hover_entry_classes = [
					'image-set__hover-entry',
				];

				// Overlay entry after first one
				if ( 0 < $i ) {
					$entry_classes[]       = 'image-set__entry--overlay';
					$hover_entry_classes[] = 'image-set__hover-entry--offset-' . $i;
				} else {
					$entry_classes[] = 'image-set__entry--hover-invisible';
				}

				?>
				<div <?php kalium_class_attr( $hover_entry_classes ); ?>></div>
				<div <?php kalium_class_attr( $entry_classes ); ?>>
					<?php
					kalium_woocommerce_loop_product_image(
						$attachment_id,
						[
							'link' => false,
						]
					);
					?>
				</div>
				<?php
			endforeach;

			// Pagination
			if ( $args['pagination'] ) {
				?>
				<div class="image-set__pagination">
					<?php
					foreach ( $attachment_ids as $i => $attachment_id ) :
						?>
						<a href="#" class="image-set__pagination__link" data-index="<?php echo esc_attr( $i ); ?>"></a>
						<?php
					endforeach;
					?>
				</div>
				<?php
			}
			?>
		</div>
		<?php

		// Link close
		if ( $args['link'] ) {
			echo '</a>';
		}
	}
}

if ( ! function_exists( 'kalium_woocommerce_images_slider' ) ) {

	/**
	 * Images slider.
	 *
	 * @param int[] $attachment_ids
	 * @param array $args
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_images_slider( $attachment_ids, $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				// Slider options
				'loop'                        => false,
				'effect'                      => null,

				// Navigation arrows
				'navigation'                  => true,
				'navigation_visible_on_hover' => false,
				'navigation_size'             => null,
				'navigation_type'             => null,
				'navigation_style'            => null,
				'navigation_style_hover'      => null,
				'navigation_border_radius'    => null,

				// Pagination
				'pagination'                  => true,
				'pagination_style'            => 'bars',
				'pagination_hover'            => false,
				'pagination_hover_animation'  => 'fade',
			]
		);

		// Swiper args
		$swiper_args = [
			'navigation' => $args['navigation'],
			'loop'       => $args['loop'],
			'effect'     => $args['effect'],
		];

		// Navigation arrows
		$arrow_icon_prev = 'kalium-icon-arrow-left';
		$arrow_icon_next = 'kalium-icon-arrow-right';

		switch ( $args['navigation_type'] ) {
			case 'type-2':
				$arrow_icon_prev = 'kalium-icon-long-arrow-left';
				$arrow_icon_next = 'kalium-icon-long-arrow-right';
				break;
		}

		// Container classes
		$classes = [
			'image-set',
			'image-set--slider',
			'swiper-container',
		];

		// Arrows navigation classes
		if ( $args['navigation'] ) {
			$classes[] = 'image-set--navigation-arrows';

			// Visible on hover
			if ( $args['navigation_visible_on_hover'] ) {
				$classes[] = 'image-set--navigation-arrows-visible-on-hover';
			}
		}

		// Arrows size
		if ( $args['navigation_size'] ) {
			$classes[] = 'image-set--navigation-arrows-size-' . $args['navigation_size'];
		}

		// Pagination
		if ( $args['pagination'] ) {
			$classes[] = 'image-set--has-pagination';

			// Pagination style
			if ( in_array( $args['pagination_style'], [ 'bars', 'dots' ] ) ) {
				$classes[] = 'image-set--pagination-style-' . $args['pagination_style'];
			}

			// Pagination visible on hover
			if ( $args['pagination_hover'] ) {
				$classes[] = 'image-set--pagination-visible-on-hover';
				$classes[] = 'image-set--pagination-animation-' . $args['pagination_hover_animation'];
			}
		}

		?>
		<div <?php kalium_class_attr( $classes ); ?> data-slider-args="<?php echo esc_attr( wp_json_encode( $swiper_args ) ); ?>">
			<div class="swiper-wrapper">
				<?php
				foreach ( $attachment_ids as $attachment_id ) :
					$entry_classes = [
						'swiper-slide',
						'image-set__entry',
					];
					?>
					<div <?php kalium_class_attr( $entry_classes ); ?>>
						<?php kalium_woocommerce_loop_product_image( $attachment_id ); ?>
					</div>
					<?php
				endforeach;
				?>
			</div>

			<?php
			// Navigation arrows
			if ( $args['navigation'] ) :
				?>
				<div class="image-set__navigation-button image-set__navigation-button--prev">
					<i class="<?php echo esc_attr( $arrow_icon_prev ); ?>"></i>
				</div>
				<div class="image-set__navigation-button image-set__navigation-button--next">
					<i class="<?php echo esc_attr( $arrow_icon_next ); ?>"></i>
				</div>
				<?php
			endif;

			// Pagination
			if ( $args['pagination'] ) {
				echo '<div class="image-set__pagination"></div>';
			}
			?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_product_badge' ) ) {

	/**
	 * Product badge flash.
	 *
	 * @param WC_Product|false $product
	 * @param array            $args
	 */
	function kalium_woocommerce_product_badge( $product = false, $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'style'       => null, // badge,circle,triangle
				'outline'     => null,
				'rotation'    => null, // applied for style:triangle
				'sale_as_pct' => null, // applied to sale badges
				'uppercase'   => null, // uppercase text
			]
		);

		// Current product
		if ( ! $product ) {
			$product = $GLOBALS['product'];
		}

		// Compatibility with YITH Badge Management plugin
		if ( defined( 'YITH_WCBM' ) && has_action( 'yith_wcbm_theme_badge_container_start' ) ) {
			do_action( 'yith_wcbm_theme_badge_container_start' );
			do_action( 'yith_wcbm_theme_badge_container_end' );
			return;
		}

		// Default globals
		if ( is_null( $args['style'] ) ) {
			$args['style'] = kalium_get_theme_option( 'shop_badges_style' );
		}

		if ( is_null( $args['uppercase'] ) ) {
			$args['uppercase'] = kalium_get_theme_option( 'shop_badges_upper_case' );
		}

		if ( is_null( $args['outline'] ) ) {
			$args['outline'] = kalium_get_theme_option( 'shop_badges_outline' );
		}

		if ( is_null( $args['sale_as_pct'] ) ) {
			$args['sale_as_pct'] = kalium_get_theme_option( 'shop_badges_sale_as_pct' );
		}

		// Classes
		$classes = [
			'product-badge',
		];

		// Priority index
		$priority = 0;

		// Badges
		$badges = apply_filters(
			'kalium_woocommerce_product_badges',
			[
				[
					'type'     => 'out-of-stock',
					'text'     => kalium_xtranslate( 'esc_html__', 'Out of stock', 'woocommerce' ),
					'active'   => ! $product->is_in_stock(),
					'priority' => $priority++,
				],
				[
					'type'     => 'backorder',
					'text'     => kalium_xtranslate( 'esc_html__', 'Available on backorder', 'woocommerce' ),
					'active'   => $product->managing_stock() && $product->is_on_backorder( 1 ) && $product->backorders_require_notification(),
					'priority' => $priority++,
				],
				[
					'type'     => 'backorder',
					'text'     => kalium_xtranslate( 'esc_html__', 'Available on backorder', 'woocommerce' ),
					'active'   => ! $product->managing_stock() && $product->is_on_backorder( 1 ),
					'priority' => $priority++,
				],
				[
					'type'     => 'sale',
					'text'     => kalium_xtranslate( 'esc_html__', 'Sale!', 'woocommerce' ),
					'active'   => $product->is_on_sale(),
					'priority' => $priority++,
				],
				[
					'type'     => 'featured',
					'text'     => esc_html__( 'Featured', 'kalium' ),
					'active'   => $product->is_featured(),
					'priority' => $priority++,
				],
			]
		);

		// Sort by priority
		usort(
			$badges,
			function ( $a, $b ) {
				return $a['priority'] - $b['priority'];
			}
		);

		// Select first matching badge
		foreach ( $badges as $badge_entry ) {
			if ( $badge_entry['active'] ) {
				$badge = $badge_entry;
				break;
			}
		}

		// Show badge
		if ( isset( $badge ) ) {
			$style = $args['style'];
			$type  = $badge['type'];
			$text  = $badge['text'];

			// Style
			$classes[] = 'product-badge--style-' . $style;
			$classes[] = 'badge';

			if ( 'badge' === $style ) {

				// Outline
				if ( $args['outline'] ) {
					$classes[] = 'badge-outline';
				}
			}

			// Type
			$classes[] = 'product-badge--type-' . $type;

			switch ( $type ) {

				// Out of stock
				case 'out-of-stock':
					$classes[] = 'badge-light-gray';
					break;

				// Backorder
				case 'backorder':
					$classes[] = 'badge-warning';
					break;

				// Default badge style
				default:
					$classes[] = 'badge-primary';
			}

			// Triangle rotation
			if ( 'triangle' === $style && $args['rotation'] ) {
				$classes[] = 'product-badge--style-triangle-rotation-' . $args['rotation'];
			}

			// Sale as percentage
			if ( $args['sale_as_pct'] && $product->is_on_sale() ) {

				// Variable product
				if ( $product instanceof WC_Product_Variable ) {
					$product = array_reduce(
						array_map( 'wc_get_product', $product->get_children() ),
						function ( $compare, $variation ) {
							if ( $compare && $compare->is_on_sale() && $variation->is_on_sale() ) {
								$discount_a = $variation->get_regular_price() - $variation->get_sale_price();
								$discount_b = $compare->get_regular_price() - $compare->get_sale_price();

								return $discount_a > $discount_b ? $variation : $compare;
							}

							return $compare && $compare->is_on_sale() ? $compare : $variation;
						}
					);
				}

				$regular_price = wc_get_price_to_display( $product, [ 'price' => $product->get_regular_price() ] );
				$sale_price    = wc_get_price_to_display( $product );

				if ( is_numeric( $regular_price ) && is_numeric( $sale_price ) ) {
					$percent = round( 100 * ( $regular_price - $sale_price ) / $regular_price );

					$text = sprintf( '-%s%%', $percent );
				}
			}

			// Text transform
			if ( $args['uppercase'] ) {
				$classes[] = 'text-uppercase';
			}

			?>
			<div <?php kalium_class_attr( $classes ); ?>>
				<div class="product-badge__text"><?php echo esc_html( $text ); ?></div>
			</div>
			<?php
		}
	}
}

if ( ! function_exists( 'kalium_woocommerce_product_image' ) ) {

	/**
	 * Get product image.
	 *
	 * @param int|array $attachment_id
	 * @param array     $args (
	 * @type string $image_size
	 * )
	 * @return string|void
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_product_image( $attachment_id = 0, $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'size'            => 'main', // main|thumbnail
				'echo'            => true,
				'link'            => true,
				'loading_eager'   => false,
				'lightbox_button' => false,
			]
		);

		// Image size
		$main_image     = 'thumbnail' !== $args['size'];
		$image_size     = 'woocommerce_single';
		$thumbnail_size = wc_get_image_size( 'gallery_thumbnail' );

		// Thumbnail size
		if ( false === $main_image ) {
			$image_size = apply_filters( 'woocommerce_gallery_thumbnail_size', [ $thumbnail_size['width'], $thumbnail_size['height'] ] );
		}

		// Vars
		$attachment_id = empty( $attachment_id ) ? get_post_thumbnail_id() : $attachment_id;
		$image_size    = apply_filters( 'woocommerce_gallery_image_size', $image_size );
		$full_size     = apply_filters( 'woocommerce_gallery_full_size', apply_filters( 'woocommerce_product_thumbnails_large_size', 'full' ) );
		$full_src      = wp_get_attachment_image_src( $attachment_id, $full_size );
		$thumb_src     = wp_get_attachment_image_src( $attachment_id, [ $thumbnail_size['width'], $thumbnail_size['height'] ] );

		// Image
		$image = call_user_func(
			kalium_conditional( $args['loading_eager'], 'kalium_get_attachment_image_eager', 'kalium_get_attachment_image' ),
			$attachment_id,
			$image_size,
			apply_filters(
				'woocommerce_gallery_image_html_attachment_image_params',
				[
					'title'                 => _wp_specialchars( get_post_field( 'post_title', $attachment_id ), ENT_QUOTES, 'UTF-8', true ),
					'data-caption'          => get_post_field( 'post_excerpt', $attachment_id ),
					'data-thumb-image'      => esc_url( $thumb_src ? $thumb_src[0] : null ),
					'data-full-image'       => esc_url( $full_src ? $full_src[0] : null ),
					'data-full-image-width' => $full_src ? $full_src[1] : null,
					'class'                 => esc_attr( $main_image ? 'wp-post-image' : '' ),
				],
				$attachment_id,
				$image_size,
				$main_image
			)
		);

		// Image with link
		if ( $args['link'] ) {
			$image = sprintf( '<a href="%s">%s</a>', esc_url( $full_src ? $full_src[0] : null ), $image );
		}

		// Lightbox button
		if ( $args['lightbox_button'] ) {
			$image .= '<button class="product-gallery__item__enlarge" type="button"><i class="kalium-icon-zoom"></i></button>';
		}

		// Image wrapped
		$image_wrapped = sprintf( '<div class="product-gallery__item product-gallery__item-image">%1$s</div>', $image );

		// Print images
		if ( $args['echo'] ) {
			echo $image_wrapped;

			return;
		}

		return $image_wrapped;
	}
}

if ( ! function_exists( 'kalium_woocommerce_loop_product_image' ) ) {

	/**
	 * Loop product image.
	 *
	 * @param int|array $attachment_id
	 * @param array     $args (
	 * @type bool $show_placeholder
	 * @type bool $link
	 * @type string $image_size
	 * )
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_loop_product_image( $attachment_id = 0, $args = [] ) {
		if ( is_array( $attachment_id ) ) {
			$args          = $attachment_id;
			$attachment_id = null;
		}

		// Parse args
		$args = wp_parse_args(
			$args,
			[
				'show_placeholder' => true,
				'link'             => true,
				'image_size'       => apply_filters( 'single_product_archive_thumbnail_size', 'woocommerce_thumbnail' ),
			]
		);

		// Attachment ID
		$attachment_id = empty( $attachment_id ) ? get_post_thumbnail_id() : $attachment_id;

		// Product image
		$image = kalium_get_attachment_image(
			$attachment_id,
			$args['image_size'],
			null,
			[
				'class' => 'loop-product-image',
			]
		);

		// Image placeholder
		if ( ! $image && $args['show_placeholder'] ) {
			$image = kalium_get_attachment_image( wc_placeholder_img_src() );
		}

		// Link
		if ( $args['link'] && $image ) {
			$image = sprintf( '<a href="%s" aria-label="%s">%s</a>', esc_attr( kalium_woocommerce_get_product_link() ), esc_attr( get_the_title() ), $image );
		}

		// Show product image
		echo $image;
	}
}

if ( ! function_exists( 'kalium_woocommerce_loop_product_second_image_on_hover' ) ) {

	/**
	 * Loop product second featured image on hover.
	 *
	 * @param array $args
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_loop_product_second_image_on_hover( $args = [] ) {

		// Product images
		$images = kalium_woocommerce_get_product_images( $args );

		// First and second attachment IDs
		$first_attachment_id = $second_attachment_id = 0;

		// Get second image (or both)
		if ( count( $images ) > 0 ) {
			do {
				$attachment_id = array_shift( $images );

				if ( ! $first_attachment_id ) {
					$first_attachment_id = $attachment_id;
				} else {
					$second_attachment_id = $attachment_id;
				}
			} while ( ! empty( $images ) && ( ! $first_attachment_id || ! $second_attachment_id ) );
		}

		// If both first and second images exist
		if ( $first_attachment_id && $second_attachment_id ) {
			kalium_woocommerce_second_image_on_hover( $first_attachment_id, $second_attachment_id, $args );
		} else { // There is no second image
			kalium_woocommerce_loop_product_image( $first_attachment_id, $args );
		}
	}
}

if ( ! function_exists( 'kalium_woocommerce_loop_product_gallery_on_hover' ) ) {

	/**
	 * Product images gallery on hover.
	 *
	 * @param array $args
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_loop_product_gallery_on_hover( $args = [] ) {

		// Product images
		$images = kalium_woocommerce_get_product_images( $args );

		// If there are sufficient images
		if ( 1 < count( $images ) ) {
			kalium_woocommerce_images_gallery_on_hover( $images, $args );
		} // Only one image on array
		elseif ( 1 === count( $images ) ) {
			kalium_woocommerce_loop_product_image( $images[0] );
		} // Show default image (if present)
		else {
			kalium_woocommerce_loop_product_image();
		}
	}
}

if ( ! function_exists( 'kalium_woocommerce_loop_product_gallery_slider' ) ) {

	/**
	 * Product images slider (carousel).
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_loop_product_gallery_slider( $args = [] ) {

		// Product images
		$images = kalium_woocommerce_get_product_images( $args );

		// If there are sufficient images
		if ( 1 < count( $images ) ) {
			kalium_woocommerce_images_slider( $images, $args );
		} else {
			kalium_woocommerce_loop_product_image( 1 === count( $images ) ? $images[0] : null );
		}
	}
}

if ( ! function_exists( 'kalium_woocommerce_loop_add_to_cart' ) ) {

	/**
	 * Add to cart button (loop).
	 *
	 * @param array $args
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_loop_add_to_cart( $args = [] ) {
		global $product;

		if ( $product ) {
			$args = wp_parse_args(
				$args,
				[
					'button'        => true, // display as button
					'button_block'  => false,
					'show_icon'     => true,
					'show_text'     => true,
					'icon'          => 'kalium-icon-plus',
					'icon_svg'      => null,
					'icon_svg_size' => null,
					'tooltip'       => null,
				]
			);

			// Class list
			$classes = [
				'add-to-cart',
				'link-button',
				'product_type_' . $product->get_type(),
				$product->is_purchasable() && $product->is_in_stock() ? 'add_to_cart_button' : '',
				$product->supports( 'ajax_add_to_cart' ) && $product->is_purchasable() && $product->is_in_stock() ? 'ajax_add_to_cart' : '',
			];

			// Button
			if ( $args['button'] ) {
				array_unshift( $classes, 'button' );

				// Block button
				if ( $args['button_block'] ) {
					$classes[] = 'link-button--block';
				}
			}

			// Text only
			if ( ! $args['show_icon'] ) {
				$classes[] = 'link-button--text-only';
			}

			// Custom icon
			if ( 'custom' === $args['icon'] ) {
				$icon_svg      = $args['icon_svg'];
				$icon_svg_size = $args['icon_svg_size'];

				if ( empty( $icon_svg_size ) ) {
					$icon_svg_size = 16;
				}

				if ( ! empty( $icon_svg ) ) {
					$args['icon_custom'] = kalium_get_svg_file( get_attached_file( $icon_svg ), null, $icon_svg_size );
				}
			}

			// Custom attributes
			$attributes = [];

			// Tooltip
			if ( $args['tooltip'] && $args['show_icon'] && ! $args['show_text'] ) {
				$tooltip_args = wp_parse_args(
					$args['tooltip'],
					[
						'hover'     => true,
						'added'     => false,
						'placement' => 'top',
					]
				);

				// Hover tooltip
				if ( $tooltip_args['hover'] ) {
					$attributes['data-tooltip'] = $product->add_to_cart_text();
				}

				// After product is added to cart
				if ( $tooltip_args['added'] ) {
					$attributes['data-tooltip-added'] = __( 'Added to cart', 'kalium' );
				}

				// Tooltip placement
				if ( $tooltip_args['hover'] || $tooltip_args['added'] ) {
					$attributes['data-tooltip-placement'] = $tooltip_args['placement'];
				}
			}

			// WooCommerce defaults
			$args = wp_parse_args(
				$args,
				[
					'quantity'   => 1,
					'class'      => implode( ' ', array_filter( $classes ) ),
					'attributes' => array_merge(
						[
							'data-product_id'  => $product->get_id(),
							'data-product_sku' => $product->get_sku(),
							'aria-label'       => $product->add_to_cart_description(),
							'rel'              => 'nofollow',
						],
						$attributes
					),
				]
			);

			$args = apply_filters( 'woocommerce_loop_add_to_cart_args', $args, $product );

			if ( isset( $args['attributes']['aria-label'] ) ) {
				$args['attributes']['aria-label'] = wp_strip_all_tags( $args['attributes']['aria-label'] );
			}

			kalium_get_template( 'woocommerce/loop/add-to-cart.php', $args );
		}
	}
}

if ( ! function_exists( 'kalium_woocommerce_loop_add_to_wishlist' ) ) {

	/**
	 * Add to wishlist (loop).
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_loop_add_to_wishlist( $args = [] ) {
		global $product;

		if ( $product && defined( 'YITH_WCWL' ) ) {
			$args = wp_parse_args(
				$args,
				[
					'button'       => true, // display as button
					'button_block' => false,
					'show_icon'    => true,
					'show_text'    => true,
					'tooltip'      => null,
				]
			);

			// Class list
			$classes = [
				'add-to-wishlist',
				'link-button',
			];

			// Button
			if ( $args['button'] ) {
				array_unshift( $classes, 'button' );

				// Block button
				if ( $args['button_block'] ) {
					$classes[] = 'link-button--block';
				}
			}

			// Details
			$current_product_id     = $product->get_id();
			$exists                 = yith_wcwl_wishlists()->is_product_in_wishlist( $current_product_id );
			$add_to_wishlist_text   = apply_filters( 'yith_wcwl_button_label', get_option( 'yith_wcwl_add_to_wishlist_text' ) );
			$added_to_wishlist_text = apply_filters( 'yith_wcwl_product_added_to_wishlist_message_button', get_option( 'yith_wcwl_product_added_text' ) );
			$browse_wishlist_text   = apply_filters( 'yith_wcwl_browse_wishlist_label', get_option( 'yith_wcwl_browse_wishlist_text' ) );

			// Base URL
			$base_url     = yith_wcwl_get_current_url();
			$wishlist_url = wp_nonce_url( add_query_arg( 'add_to_wishlist', $current_product_id, $base_url ), 'add_to_wishlist' );

			// Button label
			$wishlist_label = $add_to_wishlist_text;

			if ( $exists ) {
				$wishlist_url   = YITH_WCWL()->get_wishlist_url();
				$wishlist_label = $browse_wishlist_text;
				$classes[]      = 'link-button--filled exists';
			}

			// Custom attributes
			$attributes = [];

			// Tooltip
			if ( $args['tooltip'] && $args['show_icon'] && ! $args['show_text'] ) {
				$tooltip_args = wp_parse_args(
					$args['tooltip'],
					[
						'hover'     => true,
						'added'     => false,
						'placement' => 'top',
					]
				);

				// Hover tooltip
				if ( $tooltip_args['hover'] ) {
					$attributes['data-tooltip'] = $exists ? $browse_wishlist_text : $add_to_wishlist_text;
				}

				// After product is added to cart
				if ( $tooltip_args['added'] ) {
					$attributes['data-tooltip-added'] = $added_to_wishlist_text;
				}

				// Tooltip placement
				if ( $tooltip_args['hover'] || $tooltip_args['added'] ) {
					$attributes['data-tooltip-placement'] = $tooltip_args['placement'];
				}
			}

			// Element args
			$args = wp_parse_args(
				$args,
				[
					'class'          => implode( ' ', array_filter( $classes ) ),
					'wishlist_url'   => $wishlist_url,
					'wishlist_label' => $wishlist_label,
					'attributes'     => array_merge(
						[
							'data-product-id'          => $current_product_id,
							'data-product-type'        => $product->get_type(),
							'data-original-product-id' => $product->get_parent_id() ? $product->get_parent_id() : $current_product_id,
							'rel'                      => 'nofollow',
						],
						$attributes
					),
				]
			);

			kalium_get_template( 'woocommerce/loop/add-to-wishlist.php', $args );
		}
	}
}

if ( ! function_exists( 'kalium_woocommerce_pagination' ) ) {

	/**
	 * WooCommerce pagination.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_pagination() {
		if ( ! wc_get_loop_prop( 'is_paginated' ) || ! woocommerce_products_will_display() ) {
			return;
		}

		$total_pages = wc_get_loop_prop( 'total_pages' );

		if ( $total_pages <= 1 ) {
			return;
		}

		// Pagination settings
		$pagination_settings = kalium()->customize->get_pagination_settings( 'shop_pagination' );

		// Load more pagination
		if ( 'load-more' === $pagination_settings['type'] ) {
			kalium_load_more_pagination(
				[
					'align'          => $pagination_settings['align'],
					'loading_style'  => $pagination_settings['infinite_scroll_loading_effect'],
					'last_page_text' => esc_html__( 'No more products to show', 'kalium' ),
					'options'        => [
						'item'            => '.product',
						'infinite_scroll' => $pagination_settings['infinite_scroll_auto_reveal'],
					],
				]
			);
		}

		// Pagination
		kalium_pagination(
			[
				'align'           => $pagination_settings['align'],
				'display'         => $pagination_settings['numbers_display'],
				'pagination_args' => [
					'base'     => esc_url_raw(
						str_replace(
							999999999,
							'%#%',
							remove_query_arg( 'add-to-cart', get_pagenum_link( 999999999, false ) )
						)
					),
					'format'   => '',
					'add_args' => false,
					'current'  => max( 1, wc_get_loop_prop( 'current_page' ) ),
					'total'    => $total_pages,
				],
			]
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_filters_toggle' ) ) {

	/**
	 * Filters toggle.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_filters_toggle() {
		$expand_text   = esc_html__( 'Show Filters', 'kalium' );
		$collapse_text = esc_html__( 'Hide Filters', 'kalium' );
		$is_collapsed  = 'hide' === kalium_get_theme_option( 'shop_sidebar_toggle_default_state' );

		// Custom toggle text
		if ( $show_text = kalium_get_theme_option( 'shop_sidebar_toggle_show_text' ) ) {
			$expand_text = $show_text;
		}

		if ( $hide_text = kalium_get_theme_option( 'shop_sidebar_toggle_hide_text' ) ) {
			$collapse_text = $hide_text;
		}

		kalium_sidebar_toggle(
			[
				'content_wrapper_id' => 'shop_archive',
				'expand_text'        => $expand_text,
				'collapse_text'      => $collapse_text,
				'animate'            => ! kalium_get_theme_option( 'shop_masonry_mode' ) && kalium_get_theme_option( 'shop_sidebar_toggle_animate' ),
				'is_collapsed'       => $is_collapsed,
			]
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_archive_wrapper_start' ) ) {

	/**
	 * Shop archive wrapper start.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_archive_wrapper_start() {
		if ( is_product() ) {
			return;
		}

		kalium_content_wrapper_start(
			'shop_archive',
			[
				'classes'           => [
					'shop',
				],

				// Sidebar
				'sidebar'           => kalium_get_theme_option( 'shop_sidebar' ),
				'sidebar_align'     => kalium_get_theme_option( 'shop_sidebar_align' ),
				'sidebar_collapsed' => kalium_get_theme_option( 'shop_sidebar_toggle' ) && 'hide' === kalium_get_theme_option( 'shop_sidebar_toggle_default_state' ),
			]
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_archive_wrapper_end' ) ) {

	/**
	 * Shop archive wrapper end.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_archive_wrapper_end() {
		if ( is_product() ) {
			return;
		}

		kalium_content_wrapper_end( 'shop_archive' );
	}
}

if ( ! function_exists( 'kalium_woocommerce_single_wrapper_start' ) ) {

	/**
	 * Shop single wrapper start.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_single_wrapper_start() {
		if ( ! is_product() ) {
			return;
		}

		$classes = [
			'single-product',
		];

		$css_vars = [];

		// Gallery width
		$gallery_size = kalium_get_theme_option( 'shop_single_gallery_size' );

		if ( in_array( $gallery_size, [ 'small', 'large', 'container', 'full' ] ) ) {
			$classes[] = 'single-product--gallery-size-' . $gallery_size;
		}

		// Inline gallery options
		if ( in_array( $gallery_size, [ 'small', 'medium', 'large' ] ) ) {

			// Gallery inverted
			if ( kalium_get_theme_option( 'shop_single_gallery_inverted' ) ) {
				$classes[] = 'single-product--gallery-inverted';
			}

			// Sticky description
			if ( kalium_get_theme_option( 'shop_single_sticky_description' ) ) {
				$classes[] = 'single-product--sticky-description';
			}

			// Product summary gap
			$product_summary_gap = kalium_get_theme_option( 'shop_single_product_summary_gap' );

			if ( ! kalium_is_empty_responsive( $product_summary_gap ) ) {
				$css_vars['product-summary-gallery-gap'] = $product_summary_gap;
			}
		}

		// Vars
		if ( ! empty( $css_vars ) ) {
			kalium_print_inline_style(
				[
					'selector' => '.single-product',
					'vars'     => $css_vars,
				]
			);
		}

		kalium_content_wrapper_start(
			'shop_single',
			[
				'classes'       => $classes,

				// Sidebar
				'sidebar'       => kalium_get_theme_option( 'shop_single_sidebar' ),
				'sidebar_align' => kalium_get_theme_option( 'shop_single_sidebar_align' ),
			]
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_single_wrapper_end' ) ) {

	/**
	 * Shop archive wrapper end.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_single_wrapper_end() {
		if ( ! is_product() ) {
			return;
		}

		kalium_content_wrapper_end( 'shop_single' );
	}
}

if ( ! function_exists( 'kalium_woocommerce_archive_sidebar' ) ) {

	/**
	 * Shop archive sidebar.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_archive_sidebar() {
		if ( is_product() ) {
			return;
		}

		if ( kalium_get_theme_option( 'shop_sidebar' ) ) :
			?>
			<aside <?php kalium_class_attr( kalium_get_sidebar_classes( 'shop-sidebar' ) ); ?>>
				<?php
				// Show widgets
				kalium_dynamic_sidebar( 'shop_sidebar' );
				?>
			</aside>
			<?php

		endif;
	}
}

if ( ! function_exists( 'kalium_woocommerce_single_sidebar' ) ) {

	/**
	 * Shop archive sidebar.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_single_sidebar() {
		if ( ! is_product() ) {
			return;
		}

		if ( kalium_get_theme_option( 'shop_single_sidebar' ) ) :

			?>
			<aside <?php kalium_class_attr( kalium_get_sidebar_classes( 'single-product-sidebar' ) ); ?>>
				<?php
				// Show widgets
				kalium_dynamic_sidebar( is_active_sidebar( 'shop_sidebar_single' ) ? 'shop_sidebar_single' : 'shop_sidebar' );
				?>
			</aside>
			<?php

		endif;
	}
}

if ( ! function_exists( 'kalium_woocommerce_shop_heading' ) ) {

	/**
	 * Shop heading.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_shop_heading() {
		if ( is_product() ) {
			return;
		}

		?>
		<div class="shop-heading">
			<div class="container">
				<?php
				/**
				 * Hook: kalium_woocommerce_shop_heading
				 */
				do_action( 'kalium_woocommerce_shop_heading' );
				?>
			</div>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_single_summary_wrapper_start' ) ) {

	/**
	 * Product summary wrapper start.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_single_summary_wrapper_start() {
		echo '<div class="single-product__summary-container">';
	}
}

if ( ! function_exists( 'kalium_woocommerce_single_summary_wrapper_end' ) ) {

	/**
	 * Product summary wrapper end.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_single_summary_wrapper_end() {
		echo '</div>';
	}
}

if ( ! function_exists( 'kalium_woocommerce_single_gallery_wrapper_start' ) ) {

	/**
	 * Product gallery wrapper start.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_single_gallery_wrapper_start() {
		echo '<div class="single-product__gallery">';
	}
}

if ( ! function_exists( 'kalium_woocommerce_single_gallery_wrapper_end' ) ) {

	/**
	 * Product gallery wrapper end.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_single_gallery_wrapper_end() {
		echo '</div>';
	}
}

if ( ! function_exists( 'kalium_woocommerce_shop_toolbar' ) ) {

	/**
	 * Shop loop toolbar.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_shop_toolbar() {

		?>
		<div class="shop-toolbar row align-items-center">
			<?php
			if ( has_filter( 'kalium_woocommerce_shop_toolbar_left' ) ) :
				?>
				<div class="shop-toolbar__column shop-toolbar__column--left col-auto">
					<?php
					/**
					 * Hook: kalium_woocommerce_shop_toolbar_left
					 */
					do_action( 'kalium_woocommerce_shop_toolbar_left' );
					?>
				</div>
				<?php
			endif;
			?>
			<div class="shop-toolbar__column shop-toolbar__column--right col-auto">
				<?php
				/**
				 * Hook: kalium_woocommerce_shop_toolbar_right
				 */
				do_action( 'kalium_woocommerce_shop_toolbar_right' );
				?>
			</div>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_page_title_display' ) ) {

	/**
	 * Shop title.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_page_title_display() {
		if ( ! kalium_get_theme_option( 'shop_title' ) ) {
			return;
		}
		?>
		<h1 class="woocommerce-products-header__title page-title"><?php woocommerce_page_title(); ?></h1>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_archive_description_display' ) ) {

	/**
	 * Shop description.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_archive_description_display() {
		do_action( 'woocommerce_archive_description' );
	}
}

if ( ! function_exists( 'kalium_woocommerce_breadcrumb' ) ) {

	/**
	 * WooCommerce breadcrumb.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_breadcrumb() {
		$classes = kalium_tokenize_list( [ 'woocommerce-breadcrumb', kalium_get_link_style_class_by_context( 'woocommerce-breadcrumb' ) ] );

		woocommerce_breadcrumb(
			[
				'wrap_before' => sprintf( '<nav class="%s">', $classes ),
			]
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_breadcrumb_display' ) ) {

	/**
	 * Breadcrumb.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_breadcrumb_display() {
		if ( ! kalium_get_theme_option( 'shop_breadcrumb' ) ) {
			return;
		}

		kalium_woocommerce_breadcrumb();
	}
}

if ( ! function_exists( 'kalium_woocommerce_single_breadcrumb_display' ) ) {

	/**
	 * Single product breadcrumb.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_single_breadcrumb_display() {
		if ( ! kalium_get_theme_option( 'shop_single_breadcrumb' ) ) {
			return;
		}

		kalium_woocommerce_breadcrumb();
	}
}

if ( ! function_exists( 'kalium_woocommerce_result_count_display' ) ) {

	/**
	 * Results count display.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_result_count_display() {
		if ( ! kalium_get_theme_option( 'shop_results_count' ) ) {
			return;
		}

		woocommerce_result_count();
	}
}

if ( ! function_exists( 'kalium_woocommerce_filters_toggle_display' ) ) {

	/**
	 * Filters toggle.
	 */
	function kalium_woocommerce_filters_toggle_display() {
		if ( ! kalium_get_theme_option( 'shop_sidebar' ) || ! kalium_get_theme_option( 'shop_sidebar_toggle' ) || ! woocommerce_products_will_display() ) {
			return;
		}

		kalium_woocommerce_filters_toggle();
	}
}

if ( ! function_exists( 'kalium_woocommerce_columns_switcher_display' ) ) {

	/**
	 * Columns switcher display.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_columns_switcher_display() {
		if ( ! kalium_get_theme_option( 'shop_columns_switcher' ) || ! woocommerce_products_will_display() ) {
			return;
		}

		$products_view = kalium_woocommerce_get_products_view();
		$grid_columns  = kalium_get_theme_option( 'shop_grid_columns' );

		if ( 'list' === $products_view ) {
			$current_view = 'list';
		} else {
			$current_view = kalium_get_responsive_value( $grid_columns, null, 3 );
		}

		kalium_views_switcher(
			[
				'class'   => 'shop-columns-switcher',
				'views'   => kalium_get_theme_option( 'shop_columns_switcher_views' ),
				'current' => $current_view,
			]
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_catalog_ordering_display' ) ) {

	/**
	 * Catalog ordering.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_catalog_ordering_display() {
		if ( ! kalium_get_theme_option( 'shop_catalog_ordering' ) ) {
			return;
		}

		woocommerce_catalog_ordering();
	}
}

if ( ! function_exists( 'kalium_woocommerce_category' ) ) {

	/**
	 * Category card.
	 *
	 * @param WP_Term $category
	 * @param array   $args
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_category( $category, $args ) {
		$args = apply_filters(
			'kalium_woocommerce_category_args',
			wp_parse_args(
				$args,
				[
					'template'     => null,
					'image_link'   => true,
					'show_count'   => true,
					'inline_count' => false,
					'count_format' => _n( '%d product', '%d products', $category->count, 'kalium' ),
				]
			)
		);

		// Block name
		$block_name = 'category-box';

		// Modifiers
		$class_title_overlay         = "{$block_name}--title-overlay";
		$class_inline_count          = "{$block_name}--inline-count";
		$class_title_top             = "{$block_name}--title-top";
		$class_title_center          = "{$block_name}--title-center";
		$class_title_bottom          = "{$block_name}--title-bottom";
		$class_title_filled          = "{$block_name}--title-filled";
		$class_title_inline_centered = "{$block_name}--title-inline-centered";

		// Class
		$classes = [
			$block_name,
		];

		// Template class
		if ( $args['template'] ) {
			$classes[] = $block_name . '--' . $args['template'];
		}

		// Templates
		switch ( $args['template'] ) {
			case 'type-1':
				$classes[] = $class_title_overlay;
				$classes[] = $class_title_bottom;
				break;

			case 'type-2':
				$classes[] = $class_title_overlay;
				$classes[] = $class_title_center;
				break;

			case 'type-3':
				$classes[] = $class_title_overlay;
				$classes[] = $class_title_top;
				break;

			case 'type-5':
				$classes[] = $class_title_inline_centered;
				break;

			case 'type-6':
				$classes[] = $class_title_overlay;
				$classes[] = $class_title_center;
				$classes[] = $class_title_filled;
				break;
		}

		// Inline count
		if ( $args['inline_count'] ) {
			$classes[] = $class_inline_count;
		}

		// Count
		$count = sprintf( '<span class="count">%s</span>', esc_html( sprintf( $args['count_format'], $category->count ) ) );

		?>
		<div <?php kalium_class_attr( $classes ); ?>>
			<?php
			// Image
			if ( $args['image_link'] && ! in_array( $class_title_overlay, $classes ) ) {
				woocommerce_template_loop_category_link_open( $category );
			}

			kalium_woocommerce_category_thumbnail( $category );

			if ( $args['image_link'] && ! in_array( $class_title_overlay, $classes ) ) {
				woocommerce_template_loop_category_link_close();
			}
			?>

			<div <?php kalium_class_attr( [ 'category-box__title', kalium_get_link_plain_class() ] ); ?>>
				<h2>
					<?php
					// Open link
					woocommerce_template_loop_category_link_open( $category );

					// Category name
					echo esc_html( $category->name );

					// Close link
					woocommerce_template_loop_category_link_close();

					// Count
					if ( $args['show_count'] ) {
						echo $count;
					}
					?>
				</h2>
			</div>

			<?php
			// Overlay link
			if ( in_array( $class_title_overlay, $classes ) ) {
				woocommerce_template_loop_category_link_open( $category );
				woocommerce_template_loop_category_link_close();
			}
			?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_category_display' ) ) {

	/**
	 * Display product category.
	 *
	 * @param WP_Term $category
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_category_display( $category ) {
		if ( ! ( $category instanceof WP_Term ) ) {
			return;
		}

		$template   = kalium_get_theme_option( 'shop_category_layout' );
		$show_count = kalium_get_theme_option( 'shop_category_count' );

		kalium_woocommerce_category(
			$category,
			[
				'template'   => $template,
				'show_count' => $show_count,
			]
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_single_product_header' ) ) {

	/**
	 * Product header template.
	 *
	 * @since 4.1.2
	 */
	function kalium_woocommerce_single_product_header() {
		?>
		<header class="product-header">
			<?php
			/**
			 * Hook: kalium_woocommerce_single_product_header
			 *
			 * @hooked woocommerce_template_single_title - 10
			 */
			do_action( 'kalium_woocommerce_single_product_header' );
			?>
		</header>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_single_product_sku_display' ) ) {

	/**
	 * Product SKU display.
	 */
	function kalium_woocommerce_single_product_sku_display() {
		global $product;

		if ( ! kalium_get_theme_option( 'shop_single_meta_sku' ) || ! $product ) {
			return;
		}

		if ( wc_product_sku_enabled() && ( $product->get_sku() || $product->is_type( 'variable' ) ) ) :
			?>
			<span class="sku_wrapper"><span class="meta_title"><?php esc_html_e( 'SKU:', 'woocommerce' ); ?></span> <span class="sku"><?php echo ( $sku = $product->get_sku() ) ? $sku : esc_html__( 'N/A', 'woocommerce' ); ?></span></span>
			<?php
		endif;
	}
}

if ( ! function_exists( 'kalium_woocommerce_single_product_category_display' ) ) {

	/**
	 * Product category display.
	 */
	function kalium_woocommerce_single_product_category_display() {
		global $product;

		if ( ! kalium_get_theme_option( 'shop_single_meta_categories' ) || ! $product ) {
			return;
		}

		echo wc_get_product_category_list( $product->get_id(), ', ', '<span class="' . kalium_tokenize_list( [ 'posted_in', kalium_get_link_text_class(), kalium_get_link_block_class() ] ) . '"><span class="meta_title">' . kalium_xtranslate( '_n', 'Category:', 'Categories:', count( $product->get_category_ids() ), 'woocommerce' ) . '</span> ', '</span>' );
	}
}

if ( ! function_exists( 'kalium_woocommerce_single_product_tags_display' ) ) {

	/**
	 * Product tags display.
	 */
	function kalium_woocommerce_single_product_tags_display() {
		global $product;

		if ( ! kalium_get_theme_option( 'shop_single_meta_tags' ) || ! $product ) {
			return;
		}

		echo wc_get_product_tag_list( $product->get_id(), ', ', '<span class="' . kalium_tokenize_list( [ 'tagged_as', kalium_get_link_text_class(), kalium_get_link_block_class() ] ) . '"><span class="meta_title">' . kalium_xtranslate( '_n', 'Tag:', 'Tags:', count( $product->get_tag_ids() ), 'woocommerce' ) . '</span> ', '</span>' );
	}
}

if ( ! function_exists( 'kalium_woocommerce_single_product_social_share_display' ) ) {

	/**
	 * Social share.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_single_product_social_share_display() {
		global $product;

		if ( ! kalium_get_theme_option( 'shop_single_social_share' ) ) {
			return;
		}

		$share_networks = kalium_get_theme_option( 'shop_single_social_share_networks' );

		// No share networks assigned
		if ( empty( $share_networks['elements'] ) ) {
			return;
		}

		?>
		<div class="social-sharing">
			<span class="social-sharing__title"><?php esc_html_e( 'Share:', 'kalium' ); ?></span>

			<?php
				// Social networks
				kalium_social_networks_share(
					[
						'post_id'  => $product->get_id(),
						'networks' => $share_networks,
						'style'    => kalium_get_theme_option( 'shop_single_social_share_style' ),
					]
				);
			?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_product_tabs' ) ) {

	/**
	 * Product tabs.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_product_tabs() {
		$product_tabs = apply_filters( 'woocommerce_product_tabs', [] );

		// Map WC tabs to properly work with theme tabs
		$tab_entries = array_map( 'kalium_woocommerce_product_tabs_mapper', $product_tabs, array_keys( $product_tabs ) );

		if ( ! empty( $product_tabs ) ) {
			$args = [
				'classes' => [],
				'entries' => $tab_entries,
			];

			// Hide headings
			if ( false === kalium_get_theme_option( 'shop_single_tabs_headings' ) ) {
				$args['classes'][] = 'hide-headings';
			}

			// Tabs type
			$tabs_type = kalium_get_theme_option( 'shop_single_tabs_type' );

			if ( in_array( $tabs_type, [ 'accordion', 'stacked' ] ) ) {
				$args['type'] = $tabs_type;
			}

			// Tabs type
			if ( 'tabs' === $tabs_type ) {
				$tabs_style       = kalium_get_theme_option( 'shop_single_tabs_style' );
				$tabs_orientation = kalium_get_theme_option( 'shop_single_tabs_orientation' );

				if ( in_array( $tabs_style, [ 'style-2' ] ) ) {
					$args['style'] = str_replace( 'style-', '', $tabs_style );
				}

				// Orientation
				if ( 'vertical' === $tabs_orientation ) {
					$args['orientation'] = 'vertical';
				}
			} elseif ( 'accordion' === $tabs_type ) {
				// Collapse all
				if ( kalium_get_theme_option( 'shop_single_accordion_in_summary_collapse' ) ) {
					$args['active'] = false;
				}
			}

			// Content tabs
			kalium_tabs( $args );

			do_action( 'woocommerce_product_after_tabs' );
		}
	}
}

if ( ! function_exists( 'kalium_woocommerce_product_gallery' ) ) {

	/**
	 * Product gallery.
	 *
	 * @param mixed        $the_product
	 * @param array|string $args
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_product_gallery( $the_product, $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'type'                   => 'grid', // grid|slider
				'zoom'                   => kalium_woocommerce_is_product_gallery_zoom_enabled(),
				'lightbox'               => kalium_woocommerce_is_product_gallery_lightbox_enabled(),
				'mouse_behavior'         => 'zoom', // zoom|lightbox|false
				'gap'                    => null,
				'limit'                  => -1,
				'zoom_on'                => null, // mouseover|click

				// Grid options
				'grid_type'              => 1, // 1...4
				'grid_stretch_aside'     => false,

				// Slider options
				'slider_navigation'      => true,
				'slider_pagination'      => false,
				'slider_effect'          => 'slide', // slide|fade
				'slider_rewind'          => false,
				'slider_auto_height'     => false,
				'slider_items_per_view'  => null,
				'slider_items_max_width' => null,
				'slider_centered'        => null,
				'slider_loop'            => null,
				'slider_autoplay'        => null,

				// Thumbnails
				'thumbs'                 => false,
				'thumbs_position'        => 'bottom', // bottom|left|right
				'thumbs_grid'            => false,
				'thumbs_per_view'        => 4,
			]
		);

		$grid_gallery         = 'grid' === $args['type'];
		$image_link           = false !== $args['mouse_behavior'];
		$thumbs_vertical      = in_array( $args['thumbs_position'], [ 'left', 'right' ] );
		$slider_direction     = $thumbs_vertical ? 'vertical' : 'horizontal';
		$gallery_unique_class = 'product-gallery-' . kalium_generate_unique_string();
		$default_gap          = 30;
		$default_thumbs       = 4;

		$gallery_main_slider_selector   = ".{$gallery_unique_class} .product-gallery__main .swiper";
		$gallery_thumbs_slider_selector = ".{$gallery_unique_class} .product-gallery__thumbnails .swiper";

		// Treat responsive props
		$responsive_props = [
			'gap',
			'thumbs_per_view',
			'slider_items_per_view',
			'slider_items_max_width',
		];

		foreach ( $responsive_props as $prop ) {
			$args[ $prop ] = kalium_map_responsive_value(
				$args[ $prop ],
				function ( $value ) {
					return false === $value ? null : $value;
				}
			);

			if ( kalium_is_empty_responsive( $args[ $prop ] ) ) {
				$args[ $prop ] = null;
			}
		}

		// Product
		$the_product = wc_get_product( $the_product );

		// Get product images
		$images = kalium_woocommerce_get_product_images(
			[
				'product'    => $the_product,
				'max_images' => $args['limit'],
			]
		);

		// Disable zoom when mouse/touch behavior is not zoom
		if ( 'zoom' !== $args['mouse_behavior'] ) {
			$args['zoom'] = false;
		}

		// Classes
		$classes = [
			$gallery_unique_class,
			'product-gallery',
			'product-gallery--' . $args['type'],
		];

		// In stretch mode, set initially hidden
		if ( $args['grid_stretch_aside'] ) {
			$classes[] = 'product-gallery--stretched';
		}

		// Styles
		$styles = [];

		kalium_map_responsive_value(
			$args['gap'],
			function ( $gap, $viewport ) use ( &$styles, $gallery_unique_class ) {
				if ( ! empty( $gap ) ) {
					$styles[] = [
						'selector' => '.' . $gallery_unique_class,
						'vars'     => [
							'product-gallery-gap' => $gap,
						],
						'media'    => kalium_get_viewport_media( $viewport ),
					];
				}
			}
		);

		// Gallery options attribute
		$gallery_options = [
			'type'           => $args['type'],
			'enableZoom'     => $args['zoom'],
			'enableLightbox' => $args['lightbox'],
			'zoomOn'         => $args['zoom_on'],
			'mouseBehavior'  => $args['mouse_behavior'],
		];

		// Other classes
		if ( $grid_gallery ) {
			$classes[] = 'product-gallery--grid-type-' . $args['grid_type'];
		}

		// Disable thumbnails when there are less than 2 items
		if ( $args['thumbs'] && 1 >= count( $images ) ) {
			$args['thumbs'] = false;
		}

		// Thumbnails
		if ( $args['thumbs'] ) {
			$classes[] = 'product-gallery--thumbnails';
			$classes[] = 'product-gallery--thumbnails-' . $args['thumbs_position'];

			if ( $thumbs_vertical ) {
				$classes[] = 'product-gallery--thumbnails-vertical';
			}

			// Thumbs gallery options
			$gallery_options['thumbs']     = true;
			$gallery_options['thumbsGrid'] = $args['thumbs_grid'];
		}

		// Centered loop gallery
		if ( $args['slider_centered'] && $args['slider_loop'] ) {
			$classes[] = 'product-gallery--centered-loop';
		}

		// Parse gallery media
		$gallery_media = function ( $args = [] ) use ( $images ) {
			$args = wp_parse_args(
				$args,
				[
					'size'            => null,
					'lightbox_button' => false,
					'link'            => true,
				]
			);

			return array_map(
				function ( $attachment_id, $index ) use ( $args ) {
					return kalium_woocommerce_product_image(
						$attachment_id,
						[
							'echo'            => false,
							'size'            => $args['size'],
							'link'            => $args['link'],
							'lightbox_button' => $args['lightbox_button'],
							'loading_eager'   => 0 === $index,
						]
					);
				},
				$images,
				array_keys( $images )
			);
		};

		// Wrapper attributes
		$attributes = [
			'class'                => $classes,
			'data-gallery-options' => $gallery_options,
		];

		// Print CSS vars
		kalium_print_inline_style(
			[
				'styles' => $styles,
			]
		);

		// Gallery wrapper start
		echo kalium_element( 'div', $attributes, null, false );

		// Main gallery wrapper start
		echo '<div class="product-gallery__main">';

		/**
		 * Before main gallery hook.
		 *
		 * @param array $args
		 */
		do_action( 'kalium_woocommerce_product_gallery_main_before', $args );

		// Grid layout
		if ( $grid_gallery ) {
			echo implode(
				PHP_EOL,
				$gallery_media(
					[
						'lightbox_button' => $args['lightbox'],
						'link'            => $image_link,
					]
				)
			);
		} // Slider layout
		else {
			kalium_slider(
				[
					'slides'           => $gallery_media(
						[
							'link' => $image_link,
						]
					),
					'in_grid'          => true,
					'slides_per_view'  => $args['slider_items_per_view'],
					'slides_max_width' => $args['slider_items_max_width'],
					'space_between'    => kalium_parse_responsive_value( $args['gap'], $default_gap ),
					'navigation'       => $args['slider_navigation'],
					'pagination'       => $args['slider_pagination'],
					'centered'         => $args['slider_centered'],
					'loop'             => $args['slider_loop'],
					'autoplay'         => $args['slider_autoplay'],
					'effect'           => $args['slider_effect'],
					'rewind'           => $args['slider_rewind'],
					'auto_height'      => $args['slider_auto_height'],
					'thumbs'           => $gallery_thumbs_slider_selector, // Controls thumbs slider
					'keyboard'         => true,
				]
			);

			// Lightbox
			if ( $args['lightbox'] ) {
				echo '<button class="product-gallery__item__enlarge" type="button"><i class="kalium-icon-zoom"></i></button>';
			}
		}

		// End: Main gallery wrapper end
		echo '</div>';

		// Thumbnails
		if ( $args['thumbs'] ) {
			$thumbs_per_view = kalium_parse_responsive_value( $args['thumbs_per_view'], $default_thumbs );

			// Thumbnails wrapper start
			echo '<div class="product-gallery__thumbnails">';

			// Thumbnails grid
			if ( $args['thumbs_grid'] ) {
				$grid_classes = [
					'grid',
				];

				// Grid columns
				$grid_classes = array_merge(
					$grid_classes,
					kalium_map_responsive_value(
						$thumbs_per_view,
						function ( $value, $viewport ) {
							if ( kalium_get_default_viewport() !== $viewport ) {
								return 'columns-' . $viewport . '-' . $value;
							}

							return 'columns-' . $value;
						}
					)
				);

				printf(
					'<div %s>%s</div>',
					kalium_class_attr( $grid_classes, false ),
					implode( PHP_EOL, $gallery_media( [ 'size' => 'thumbnail' ] ) )
				);
			} else {
				kalium_slider(
					[
						'slides'          => $gallery_media( [ 'size' => 'thumbnail' ] ),
						'direction'       => [
							'desktop' => $slider_direction,
							'mobile'  => 'horizontal',
						],
						'in_grid'         => true,
						'navigation'      => false,
						'slides_per_view' => $thumbs_per_view,
						'space_between'   => kalium_parse_responsive_value( $args['gap'], $default_gap ),
						'sync_height'     => kalium_conditional( $thumbs_vertical, $gallery_main_slider_selector ),
					]
				);
			}

			// Thumbnails wrapper end
			echo '</div>';
		}

		// Gallery wrapper end
		echo '</div>';

		// Stretch container with JS
		if ( $args['grid_stretch_aside'] ) {
			kalium_js_stretch_container( '.' . $gallery_unique_class );
		}
	}
}

if ( ! function_exists( 'kalium_woocommerce_single_product_gallery_display' ) ) {

	/**
	 * Display product gallery in single product page.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_single_product_gallery_display() {
		global $product;

		$gallery_type = kalium_get_theme_option( 'shop_single_gallery_type' );

		$args = [
			// Gallery type
			'type' => in_array(
				$gallery_type,
				[
					'type-1',
					'type-2',
					'type-3',
					'type-8',
					'type-9',
				]
			) ? 'slider' : 'grid',
		];

		// Grid type
		if ( 'grid' === $args['type'] ) {
			$grid_types = [
				'type-4' => 1,
				'type-5' => 2,
				'type-6' => 3,
				'type-7' => 4,
			];

			$args['grid_type'] = $grid_types[ $gallery_type ] ?? 1;

			if ( kalium_get_theme_option( 'shop_single_gallery_stretch' ) ) {
				$args['grid_stretch_aside'] = true;
			}
		} else {
			$slider_navigation = kalium_get_theme_option( 'shop_single_gallery_navigation' );

			$args['slider_navigation']  = in_array( 'arrows', $slider_navigation );
			$args['slider_pagination']  = in_array( 'dots', $slider_navigation );
			$args['slider_effect']      = kalium_get_theme_option( 'shop_single_gallery_carousel_animation' );
			$args['slider_autoplay']    = kalium_get_theme_option( 'shop_single_gallery_autoplay' );
			$args['slider_rewind']      = true;
			$args['slider_auto_height'] = true;

			// Carousel type
			if ( 'type-9' === $gallery_type ) {
				$args['slider_items_per_view']  = 'auto';
				$args['slider_items_max_width'] = kalium_get_theme_option( 'shop_single_gallery_item_max_width' );
				$args['slider_rewind']          = false;
				$args['slider_loop']            = true;

				unset( $args['slider_effect'] );
			}
		}

		// Gap
		$args['gap'] = kalium_get_theme_option( 'shop_single_gallery_gap' );

		// Thumbnails
		if ( in_array( $gallery_type, [ 'type-1', 'type-2', 'type-3' ] ) ) {
			$thumbnails_position = [
				'type-1' => 'bottom',
				'type-2' => 'right',
				'type-3' => 'left',
			];

			$args['thumbs']          = true;
			$args['thumbs_position'] = $thumbnails_position[ $gallery_type ] ?? null;
			$args['thumbs_grid']     = ! kalium_get_theme_option( 'shop_single_gallery_thumbnails_carousel' );

			// Thumbnails per view
			$thumbnails_per_view = kalium_get_theme_option( 'shop_single_gallery_thumbnails_per_view' );

			if ( in_array( $gallery_type, [ 'type-2', 'type-3' ] ) && kalium_get_theme_option( 'shop_single_gallery_thumbnails_per_view_auto' ) ) {
				$thumbnails_per_view = 'auto';
			}

			if ( $thumbnails_per_view ) {
				$args['thumbs_per_view'] = $thumbnails_per_view;
			}
		}

		// Mouse/touch behavior
		$args['mouse_behavior'] = kalium_get_theme_option( 'shop_single_gallery_image_click' ) ? kalium_get_theme_option( 'shop_single_gallery_image_click_action' ) : false;

		// Items limit
		$items_limit = kalium_get_theme_option( 'shop_single_gallery_items_limit' );

		if ( is_numeric( $items_limit ) ) {
			$args['limit'] = $items_limit;
		}

		// Show gallery
		kalium_woocommerce_product_gallery( $product, $args );
	}
}

if ( ! function_exists( 'kalium_woocommerce_single_product_badge_display' ) ) {

	/**
	 * Display product badge on single product page.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_single_product_badge_display() {
		kalium_woocommerce_product_badge();
	}
}

if ( ! function_exists( 'kalium_woocommerce_mini_cart_popup' ) ) {

	/**
	 * Mini cart popup.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_mini_cart_popup() {
		kalium_get_template(
			'woocommerce/mini-cart-popup.php',
			[
				'is_empty' => 0 === kalium_woocommerce_get_items_in_cart(),
			]
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_mini_cart_drawer' ) ) {

	/**
	 * Mini cart.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_mini_cart_drawer() {
		$classes = [
			'mini-cart-drawer',
			'off-canvas',
		];

		// Animation
		if ( 'fade' === kalium_get_theme_option( 'shop_mini_cart_animation' ) ) {
			$classes[] = 'off-canvas--fade';
		}

		// Empty state
		$is_empty = 0 === kalium_woocommerce_get_items_in_cart();

		if ( $is_empty ) {
			$classes[] = 'mini-cart-drawer--is-empty';
		}

		kalium_get_template(
			'woocommerce/mini-cart-drawer.php',
			[
				'classes'  => $classes,
				'is_empty' => $is_empty,
			]
		);
	}
}

if ( ! function_exists( 'kalium_woocommerce_mini_cart_empty_message' ) ) {

	/**
	 * Mini cart empty message.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_mini_cart_empty_message() {
		?>
		<div class="mini-cart-drawer__empty__message">
			<i class="kalium-icon-shopping-bag-empty"></i>
			<strong><?php kalium_xtranslate( 'esc_html_e', 'Your cart is currently empty!', 'woocommerce' ); ?></strong>
			<p><?php esc_html_e( 'Browse our store, find products, and happy shopping!', 'kalium' ); ?></p>
			<a href="<?php echo esc_url( wc_get_page_permalink( 'shop' ) ); ?>" class="button"><?php kalium_xtranslate( 'esc_html_e', 'Browse products', 'woocommerce' ); ?></a>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_mini_cart_title' ) ) {

	/**
	 * Mini cart title.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_mini_cart_title() {
		$cart_items = sprintf(
			kalium_xtranslate( '_n', '%d item', '%d items', kalium_woocommerce_get_items_in_cart(), 'woocommerce' ),
			kalium_woocommerce_get_items_in_cart()
		);

		printf( '<h2 class="mini-cart-drawer__title">%s (%s)</h2>', kalium_xtranslate( 'esc_html__', 'Your cart', 'woocommerce' ), $cart_items );
	}
}

if ( ! function_exists( 'kalium_woocommerce_mini_cart_close' ) ) {

	/**
	 * Mini cart close.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_mini_cart_close() {
		?>
		<button type="button" class="mini-cart-drawer__close button button-link" aria-label="Close mini cart drawer" data-off-canvas-toggle>
			<i class="kalium-icon-remove"></i>
		</button>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_mini_cart_notices' ) ) {

	/**
	 * WooCommerce Notices.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_mini_cart_notices() {
		echo '<div class="mini-cart-drawer__notices"></div>';
	}
}

if ( ! function_exists( 'kalium_woocommerce_mini_cart_contents' ) ) {

	/**
	 * Displays mini cart contents.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_mini_cart_contents() {
		?>
		<div class="mini-cart-drawer__items">
			<?php
			foreach ( WC()->cart->get_cart() as $cart_item_key => $cart_item ) {
				$_product   = apply_filters( 'woocommerce_cart_item_product', $cart_item['data'], $cart_item, $cart_item_key );
				$product_id = apply_filters( 'woocommerce_cart_item_product_id', $cart_item['product_id'], $cart_item, $cart_item_key );

				$product_name = apply_filters( 'woocommerce_cart_item_name', $_product->get_name(), $cart_item, $cart_item_key );

				if ( $_product && $_product->exists() && $cart_item['quantity'] > 0 && apply_filters( 'woocommerce_cart_item_visible', true, $cart_item, $cart_item_key ) ) {
					$product_permalink = apply_filters( 'woocommerce_cart_item_permalink', $_product->is_visible() ? $_product->get_permalink( $cart_item ) : '', $cart_item, $cart_item_key );
					?>
					<div class="mini-cart-drawer__items__item">
						<div class="mini-cart-drawer__items__item__image">
							<?php
							// Remove link
							echo apply_filters(
								'woocommerce_cart_item_remove_link',
								sprintf(
									'<a href="%s" class="remove" aria-label="%s" data-product_id="%s" data-product_sku="%s">&times;</a>',
									esc_url( wc_get_cart_remove_url( $cart_item_key ) ),
									esc_attr( sprintf( kalium_xtranslate( '__', 'Remove %s from cart', 'woocommerce' ), wp_strip_all_tags( $product_name ) ) ),
									esc_attr( $product_id ),
									esc_attr( $_product->get_sku() )
								),
								$cart_item_key
							);

							// Thumbnail
							$thumbnail = apply_filters( 'woocommerce_cart_item_thumbnail', kalium_image_placeholder_wrap_element( $_product->get_image() ), $cart_item, $cart_item_key );

							if ( ! $product_permalink ) {
								echo $thumbnail;
							} else {
								printf( '<a href="%s">%s</a>', esc_url( $product_permalink ), $thumbnail );
							}
							?>
						</div>
						<div class="mini-cart-drawer__items__item__data">
							<div <?php kalium_class_attr( [ 'mini-cart-drawer__items__item__name', kalium_get_link_title_class(), kalium_get_link_block_class() ] ); ?>>
								<?php
								if ( ! $product_permalink ) {
									echo wp_kses_post( $product_name . '&nbsp;' );
								} else {
									echo wp_kses_post( apply_filters( 'woocommerce_cart_item_name', sprintf( '<a href="%s">%s</a>', esc_url( $product_permalink ), $_product->get_name() ), $cart_item, $cart_item_key ) );
								}

								// Meta data
								echo wc_get_formatted_cart_item_data( $cart_item ); // PHPCS: XSS ok.
								?>
							</div>
							<div class="mini-cart-drawer__items__item__price">
								<?php
								echo apply_filters( 'woocommerce_cart_item_price', WC()->cart->get_product_price( $_product ), $cart_item, $cart_item_key );
								?>
							</div>
							<div class="mini-cart-drawer__items__item__quantity">
								<?php
								if ( $_product->is_sold_individually() ) {
									$min_quantity = 1;
									$max_quantity = 1;
								} else {
									$min_quantity = 0;
									$max_quantity = $_product->get_max_purchase_quantity();
								}

								$product_quantity = woocommerce_quantity_input(
									[
										'input_name'   => "cart[{$cart_item_key}][qty]",
										'input_value'  => $cart_item['quantity'],
										'max_value'    => $max_quantity,
										'min_value'    => $min_quantity,
										'product_name' => $product_name,
									],
									$_product,
									false
								);

								echo apply_filters( 'woocommerce_cart_item_quantity', $product_quantity, $cart_item_key, $cart_item ); // PHPCS: XSS ok.
								?>
							</div>
						</div>
						<div class="mini-cart-drawer__items__item__subtotal">
							<?php
							echo apply_filters( 'woocommerce_cart_item_subtotal', WC()->cart->get_product_subtotal( $_product, $cart_item['quantity'] ), $cart_item, $cart_item_key );
							?>
						</div>
					</div>
					<?php
				}
			}
			?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_mini_cart_subtotal' ) ) {

	/**
	 * Mini cart subtotals.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_mini_cart_subtotal() {
		?>
		<div class="mini-cart-drawer__subtotal">
			<div class="mini-cart-drawer__subtotal__label">
				<?php kalium_xtranslate( 'esc_html_e', 'Subtotal', 'woocommerce' ); ?>
			</div>
			<div class="mini-cart-drawer__subtotal__value">
				<?php wc_cart_totals_subtotal_html(); ?>
			</div>
			<div class="mini-cart-drawer__subtotal__description">
				<?php kalium_xtranslate( 'esc_html_e', 'Shipping, taxes, and discounts calculated at checkout.', 'woocommerce' ); ?>
			</div>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_mini_cart_buttons' ) ) {

	/**
	 * Mini cart checkout and cart page buttons.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_mini_cart_buttons() {
		?>
		<div class="mini-cart-drawer__buttons">
			<?php if ( kalium_get_theme_option( 'shop_mini_cart_page_link' ) ) : ?>
				<a href="<?php echo esc_url( wc_get_cart_url() ); ?>" class="button button-outline"><?php kalium_xtranslate( 'esc_html_e', 'View cart', 'woocommerce' ); ?></a>
			<?php endif; ?>

			<a href="<?php echo esc_url( wc_get_checkout_url() ); ?>" class="button"><?php kalium_xtranslate( 'esc_html_e', 'Checkout', 'woocommerce' ); ?></a>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_mini_cart_nonce' ) ) {

	/**
	 * Cart nonce field.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_mini_cart_nonce() {
		wp_nonce_field( 'woocommerce-cart', 'woocommerce-cart-nonce' );
	}
}

if ( ! function_exists( 'kalium_woocommerce_free_shipping_bar' ) ) {

	/**
	 * Shows free shipping progress bar if there is any eligible free shipping method.
	 *
	 * @param string $location
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_free_shipping_bar( $location = null ) {
		$show_progress_bar = kalium_get_theme_option( 'shop_free_shipping_bar' );

		// Visibility by location
		if ( $show_progress_bar && $location ) {
			$visibility_by_location = kalium_get_theme_option( 'shop_free_shipping_bar_locations' );

			if ( is_array( $visibility_by_location ) && ! in_array( $location, $visibility_by_location, true ) ) {
				$show_progress_bar = false;
			}
		}

		if ( $show_progress_bar ) {
			$min_amount = kalium_woocommerce_get_free_shipping_minimum_amount();

			if ( is_numeric( $min_amount ) ) {
				$subtotal = WC()->cart->get_subtotal();
				$progress = max( 0, min( 1, $subtotal / $min_amount ) );

				$find = [
					'[min_amount]',
					'[amount_left]',
				];

				$replace = [
					wc_price( $min_amount ),
					wc_price( $min_amount - $subtotal ),
				];

				$initial_message   = str_replace( $find, $replace, kalium_get_theme_option( 'shop_free_shipping_bar_initial_message' ) );
				$qualified_message = str_replace( $find, $replace, kalium_get_theme_option( 'shop_free_shipping_bar_qualified_message' ) );
				?>
				<div class="free-shipping-bar">
					<?php if ( 0 < $progress ) { ?>
					<div class="free-shipping-bar__bar" style="<?php echo esc_attr( sprintf( '%s: %s', kalium_css_var_name( 'progress-width' ), $progress * 100 . '%' ) ); ?>">
						<div class="progress-bar"></div>
					</div>
					<?php } ?>

					<div class="free-shipping-bar__text">
						<?php
						if ( 1 === $progress ) {
							echo wp_kses_post( $qualified_message );
						} else {
							echo wp_kses_post( $initial_message );
						}
						?>
					</div>
				</div>
				<?php
			}
		}
	}
}

if ( ! function_exists( 'kalium_woocommerce_mini_cart_free_shipping_bar' ) ) {

	/**
	 * Mini cart free shipping progress bar.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_mini_cart_free_shipping_bar() {
		kalium_woocommerce_free_shipping_bar( 'mini-cart' );
	}
}

if ( ! function_exists( 'kalium_woocommerce_cart_free_shipping_bar' ) ) {

	/**
	 * Free shipping progress bar on cart page.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_cart_free_shipping_bar() {
		kalium_woocommerce_free_shipping_bar( 'cart' );
	}
}

if ( ! function_exists( 'kalium_woocommerce_checkout_free_shipping_bar' ) ) {

	/**
	 * Free shipping progress bar on cart page.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_checkout_free_shipping_bar() {
		kalium_woocommerce_free_shipping_bar( 'checkout' );
	}
}

if ( ! function_exists( 'kalium_woocommerce_single_product_free_shipping_bar' ) ) {

	/**
	 * Free shipping progress bar on cart page.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_single_product_free_shipping_bar() {
		kalium_woocommerce_free_shipping_bar( 'single-product' );
	}
}

if ( ! function_exists( 'kalium_woocommerce_off_canvas_shop_filters_header' ) ) {

	/**
	 * Off-Canvas shop filters header.
	 *
	 * @since 4.0
	 */
	function kalium_woocommerce_off_canvas_shop_filters_header() {
		$title = kalium_get_theme_option( 'shop_sidebar_toggle_off_canvas_title' );
		?>
		<div class="off-canvas-shop-filters-header">
			<?php if ( trim( $title ) ) : ?>
				<h2><?php echo esc_html( $title ); ?></h2>
			<?php endif; ?>

			<button type="button" class="button button-link" aria-label="Close sidebar" data-off-canvas-toggle>
				<i class="kalium-icon-remove"></i>
			</button>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_woocommerce_product_navigation' ) ) {

	/**
	 * Product navigation template.
	 *
	 * @param array $args
	 *
	 * @since 4.1.2
	 */
	function kalium_woocommerce_product_navigation( $args = [] ) {
		$default_args = kalium()->customize->get_post_navigation_settings(
			'product_navigation',
			[
				'post_type' => 'product',
			]
		);

		kalium_post_navigation(
			wp_parse_args(
				$args,
				$default_args
			)
		);
	}
}

/**
 * Replaced WooCommerce template functions.
 */

/**
 * User gravatar.
 *
 * @param array $comment
 *
 * @since 4.0
 */
function woocommerce_review_display_gravatar( $comment ) {
	echo kalium_get_avatar(
		$comment,
		[
			'size' => 50,
		]
	);
}

