<?php
/**
 * Kalium WordPress Theme
 *
 * Kalium Starter Site Importer plugin.
 *
 * @version 2.0
 * @author  Laborator
 * @link    https://kaliumtheme.com
 */
namespace Kalium\Importer;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Importer {

	/**
	 * Minimum required user capability.
	 *
	 * @const string
	 */
	const MIN_CAPABILITY = 'install_plugins';

	/**
	 * Instance of the class, because it is a singleton.
	 *
	 * @var self
	 * @static
	 */
	private static $instance;

	/**
	 * Alternative plugins when required ones are not active.
	 *
	 * @var array
	 */
	public static $alternative_plugins = [
		'elementor-pro' => [
			'slug'     => 'pro-elements',
			'basename' => 'pro-elements/pro-elements.php',
			'source'   => [ self::class, 'alternative_plugin_pro_elements_url' ],
			'message'  => <<<EOF
This starter site requires the <strong>Elementor Pro</strong> plugin. Since it's a paid plugin, you can use an alternative plugin called <strong>PRO Elements</strong>.

<a href="https://proelements.org" target="_blank" rel="noopener noreferrer">PRO Elements</a> is a derivative of the <strong>Elementor Pro</strong> plugin, licensed under the GNU General Public License, version 3 (GPLv3). It is completely free and does not require activation to use.

Would you like to install the alternative plugin?
EOF
			,
		],
	];

	/**
	 * Create and/or return instance.
	 *
	 * @return self
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Returns the URL of latest release of Pro Elements.
	 *
	 * @return string
	 */
	public static function alternative_plugin_pro_elements_url() {
		$url = 'https://api.github.com/repos/proelements/proelements/releases/latest';

		$response = wp_remote_get( $url );

		// Check for WP_Error
		if ( is_wp_error( $response ) ) {
			return null;
		}

		// Parse the JSON response
		$body = wp_remote_retrieve_body( $response );
		$data = kalium()->helpers->json_decode( $body, true );

		if ( isset( $data['assets'][0]['browser_download_url'] ) ) {
			return $data['assets'][0]['browser_download_url'];
		}

		return null;
	}

	/**
	 * Constructor.
	 *
	 * @return void
	 */
	public function __construct() {
		add_action( 'admin_init', [ $this, 'init' ] );
		add_action( 'wp_ajax_kalium_importer_run_action', [ $this, 'run_action' ] );
		add_action( 'wp_ajax_kalium_importer_view_page', [ $this, 'view_page' ] );
		add_filter( 'wp_redirect', [ $this, 'disable_redirects' ], 100000 );
		add_filter( 'woocommerce_create_pages', [ $this, 'disable_creating_woocommerce_pages' ] );
	}

	/**
	 * Check if importer is running.
	 *
	 * @return bool
	 */
	public function is_running() {
		return wp_doing_ajax() && 'kalium_importer_run_action' === kalium()->request->request( 'action' );
	}

	/**
	 * Init.
	 *
	 * @return void
	 */
	public function init() {
		spl_autoload_register(
			kalium_class_autoloader(
				[
					'namespace' => 'Kalium\Importer',
					'base_dir'  => __DIR__ . '/includes',
				]
			)
		);
	}

	/**
	 * Disable redirects when importing starter sites.
	 *
	 * @param string $url
	 *
	 * @return string|bool
	 */
	public function disable_redirects( $url ) {
		if ( $this->is_running() ) {
			return false;
		}

		return $url;
	}

	/**
	 * Disable creating default WooCommerce pages when importer is running.
	 *
	 * @param array $pages
	 *
	 * @return array
	 */
	public function disable_creating_woocommerce_pages( $pages ) {
		return $this->is_running() ? [] : $pages;
	}

	/**
	 * AJAX gate for import actions.
	 *
	 * @return void
	 *
	 * @see Import_Manager
	 * @see Content_Pack
	 */
	public function run_action() {

		// Only allowed users
		if ( current_user_can( self::MIN_CAPABILITY ) ) {

			// Content pack ID
			$content_pack_id = kalium()->request->xhr_request( 'content-pack' );

			/**
			 * Importer run action hook.
			 */
			do_action( 'kalium_importer_run_action' );

			// Hooks for specific content_pack_id
			if ( $content_pack = $this->get_content_pack( $content_pack_id ) ) {

				// Create import manager instance
				$import_manager = Import_Manager::get_instance( $content_pack );

				// Create backup manager instance
				$backup_manager = Backup_Manager::get_instance( $content_pack );

				// Assign import manager and backup manager to content pack
				$content_pack->import_manager( $import_manager );
				$content_pack->backup_manager( $backup_manager );

				/**
				 * Content pack import actions.
				 *
				 * @param Import_Manager $import_manager
				 * @param Backup_Manager $backup_manager
				 */
				do_action( "kalium_importer_run_action_{$content_pack_id}", $import_manager, $backup_manager );
			}
		}
	}

	/**
	 * Wrap source download links with license parameters.
	 *
	 * @param string $url
	 *
	 * @return string
	 */
	public function source_url( $url ) {
		return kalium()->license->wrap_license_key( $url );
	}

	/**
	 * Get starter sites available for install.
	 *
	 * @param string $type
	 *
	 * @return Content_Pack[]
	 */
	public function get_content_packs( $type = null ) {
		$starter_sites = kalium_api( 'sites-list', [ 'cache' => true ] );

		if ( is_wp_error( $starter_sites ) ) {
			return [];
		}

		$starter_sites = array_map( [ Content_Pack::class, 'create_instance' ], $starter_sites );

		// Get filtered by type
		if ( $type ) {
			$starter_sites = array_filter(
				$starter_sites,
				function ( $content_pack ) use ( $type ) {
					return $type === $content_pack->get_type();
				}
			);
		}

		return $starter_sites;
	}

	/**
	 * Get content pack by ID.
	 *
	 * @param string $content_pack_id
	 *
	 * @return Content_Pack|null
	 */
	public function get_content_pack( $content_pack_id ) {

		// Loop through content packs
		foreach ( $this->get_content_packs() as $content_pack ) {
			if ( $content_pack_id === $content_pack->get_base_id() ) {
				return $content_pack;
			}
		}

		return null;
	}

	/**
	 * Get installed content packs.
	 *
	 * @return Content_Pack[]
	 */
	public function get_installed_content_packs() {
		$content_packs = [];

		foreach ( $this->get_content_packs() as $content_pack ) {
			if ( $content_pack->is_installed() ) {
				$content_packs[] = $content_pack;
			}
		}

		return $content_packs;
	}

	/**
	 * Get starter sites types.
	 *
	 * @return array
	 */
	public function get_starter_sites_types() {
		static $type_titles = [
			'elementor' => 'Elementor',
			'wpb'       => 'WPBakery',
		];

		$types  = [];
		$counts = [];

		foreach ( $this->get_content_packs() as $content_pack ) {
			$content_pack_type = $content_pack->get_type();

			// Count
			if ( ! isset( $counts[ $content_pack_type ] ) ) {
				$counts[ $content_pack_type ] = 0;
			}

			++$counts[ $content_pack_type ];

			if ( ! in_array( $content_pack_type, $types ) ) {
				$types[] = $content_pack_type;
			}
		}

		return array_map(
			static function ( $type ) use ( &$type_titles, &$counts ) {
				return [
					'type'  => $type,
					'title' => $type_titles[ $type ],
					'count' => $counts[ $type ],
				];
			},
			$types
		);
	}

	/**
	 * Get current site type.
	 *
	 * @param bool $return_default
	 *
	 * @return string
	 */
	public function get_current_site_type( $return_default = false ) {
		$site_types   = wp_list_pluck( $this->get_starter_sites_types(), 'type' );
		$current_type = kalium()->request->request( 'site-type' );
		$default_type = reset( $site_types );

		if ( $return_default && ! in_array( $current_type, $site_types ) ) {
			return $default_type;
		}

		return $current_type;
	}

	/**
	 * List starter sites.
	 *
	 * @return void
	 */
	public function list_starter_sites() {
		kalium_enqueue( 'importer' );

		// Load template file
		kalium()->require_file(
			__DIR__ . '/includes/views/starter-sites.php',
			[
				'installed'     => $this->get_installed_content_packs(),
				'starter_sites' => $this->get_content_packs( $this->get_current_site_type( true ) ),
			]
		);
	}

	/**
	 * Import content pack page displayed on AJAX.
	 *
	 * @return void
	 */
	public function view_page() {
		if ( current_user_can( self::MIN_CAPABILITY ) ) {
			switch ( kalium()->request->query( 'action-type' ) ) {
				case 'uninstall':
					$this->view_page_uninstall();
					break;

				default:
					$this->view_page_install();
			}
		}

		die();
	}

	/**
	 * Install page.
	 */
	public function view_page_install() {
		$content_pack_id = kalium()->request->query( 'content-pack' );

		// Content pack
		if ( $content_pack = $this->get_content_pack( $content_pack_id ) ) {
			if ( kalium()->license->is_active() ) {
				kalium()->require_file(
					__DIR__ . '/includes/views/content-pack-install.php',
					[
						'content_pack' => $content_pack,
					]
				);
			} else {
				kalium()->require_file( __DIR__ . '/includes/views/no-active-subscription.php' );
			}
		}
	}

	/**
	 * Uninstall page.
	 */
	public function view_page_uninstall() {
		$content_pack_id = kalium()->request->query( 'content-pack' );

		// Content pack
		if ( $content_pack = $this->get_content_pack( $content_pack_id ) ) {
			$import_instance       = $content_pack->get_import_instance();
			$imported_content_type = $import_instance->get_imported_content_type();

			// Content uninstall view
			kalium()->require_file(
				__DIR__ . '/includes/views/content-pack-uninstall.php',
				[
					'content_pack'          => $content_pack,
					'imported_content_type' => $imported_content_type,
				]
			);
		}
	}

	/**
	 * Starter sites types dropdown.
	 */
	public function starter_sites_types_dropdown() {
		$site_types   = $this->get_starter_sites_types();
		$current_type = $this->get_current_site_type( true );

		$current_type_arr = kalium_get_array_first(
			wp_list_filter(
				$site_types,
				[
					'type' => $current_type,
				]
			)
		);

		$starter_site_image = function ( $type ) {
			?>
			<img src="<?php echo esc_url( kalium()->assets_url( 'admin/images/starter-sites/' . $type . '.png' ) ); ?>" width="16" height="16" alt="<?php echo esc_attr( $type ); ?>" />
			<?php
		};

		?>
		<div class="kalium-sites__dropdown">
			<button type="button" class="button button-secondary" aria-haspopup="menu">
				<?php
				$starter_site_image( $current_type );

				echo esc_html( $current_type_arr['title'] );
				?>
			</button>

			<div class="kalium-sites__dropdown__menu" role="menu">
				<?php
				foreach ( $site_types as $starter_sites_type ) :
					$type  = $starter_sites_type['type'];
					$title = $starter_sites_type['title'];
					$url   = add_query_arg( 'site-type', $type );

					?>
					<a href="<?php echo esc_url( $url ); ?>">
						<?php $starter_site_image( $type ); ?>
						<span class="site-name"><?php echo esc_html( $title ); ?></span>
						<span class="sites-count"><?php echo $starter_sites_type['count']; ?></span>
					</a>
					<?php
				endforeach;
				?>
			</div>
		</div>
		<?php
	}

	/**
	 * Debug import of specific content pack and import ID.
	 *
	 * For testing purpose only.
	 *
	 * @param string $content_pack_id
	 * @param string $import_id
	 * @param array  $args
	 */
	public function debug_content_pack_import( $content_pack_id, $import_id, $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'do_download'    => false,
				'do_backup'      => false,
				'do_import'      => false,
				'do_complete'    => false,
				'do_remove'      => false,
				'import_checked' => true,
				'args_values'    => [],
			]
		);

		// Only when DEBUG is enabled
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {

			// Content pack
			$content_pack = $this->get_content_pack( $content_pack_id );

			if ( is_null( $content_pack ) ) {
				printf( 'Starter site "%s" doesn\'t exists', $content_pack_id );
				exit();
			}

			// Import content type
			$import = $content_pack->get_import_by_id( $import_id );

			if ( is_null( $import ) ) {
				printf( 'Import content type "%s" doesn\'t exists', $import_id );
				exit();
			}

			// Import manager
			$import_manager = Import_Manager::get_instance( $content_pack );

			// Backup manager
			$backup_manager = Backup_Manager::get_instance( $content_pack );

			// Register required plugins
			$import_manager->register_required_plugins();

			// Assign import manager and backup manager to content pack
			$content_pack->import_manager( $import_manager );
			$content_pack->backup_manager( $backup_manager );

			// Create/resume import instance
			$import_instance = $content_pack->get_import_instance();

			// Set current import id in import instance
			$import_instance->set_import_id( $import->get_import_id() );

			// Set checked status of import field
			$import->is_checked( $args['import_checked'] );

			// Set args values
			$import->set_args_values( $args['args_values'] );

			// Initialize Filesystem
			kalium()->filesystem->initialize();

			// Import task: Download
			if ( $args['do_download'] ) {

				// Clear errors for the current import type
				$import_instance->clear_errors();

				$import->do_download();
			}

			// Import task: Backup
			if ( $args['do_backup'] ) {
				$import->do_backup();
			}

			// Import task: Import
			if ( $args['do_import'] ) {
				$import->do_import();
			}

			// Import task: Complete
			if ( $args['do_complete'] ) {
				$import->do_complete();
			}

			// Import task: Remove
			if ( $args['do_remove'] ) {
				$import->do_remove();
			}

			// Display errors
			if ( $import->get_errors()->has_errors() ) {
				printf( '<h4>Errors</h4><pre style="padding: 20px; background: #eee;">%s</pre>', $import->get_errors()->get_error_message() );
			}

			// Import instance
			printf( '<h4>Import Instance <small>(%s)</small></h4><pre style="padding: 20px; background: #eee;">%s</pre>', $import->get_name(), print_r( $import_instance->to_array(), true ) );
		}
	}
}

// Instantiate Importer
Importer::instance();
