<?php
/**
 * Kalium WordPress Theme
 *
 * Starter sites class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Importer;

use Kalium\Utilities\System_Status;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Content_Pack {

	/**
	 * Minimum recommended memory limit in megabytes.
	 *
	 * @const int
	 */
	const MINIMUM_MEMORY_LIMIT = 128;

	/**
	 * Minimum recommended max execution time.
	 *
	 * @const int
	 */
	const MINIMUM_EXECUTION_TIME = 90;

	/**
	 * Content pack type.
	 *
	 * @var string
	 */
	public $type;

	/**
	 * Content pack ID.
	 *
	 * @var string
	 */
	public $id;

	/**
	 * Name.
	 *
	 * @var string
	 */
	public $name;

	/**
	 * Thumbnail URL.
	 *
	 * @var string
	 */
	public $thumbnail;

	/**
	 * Imports.
	 *
	 * @var Import_Type[]
	 */
	public $imports = [];

	/**
	 * List of required plugins.
	 *
	 * @var array
	 */
	public $plugins = [];

	/**
	 * Preview URL.
	 *
	 * @var string
	 */
	public $preview_url;

	/**
	 * Badges.
	 *
	 * @var array
	 */
	public $badges;

	/**
	 * Import instance.
	 *
	 * @var Import_Instance
	 */
	protected $import_instance;

	/**
	 * Import manager.
	 *
	 * @var Import_Manager
	 */
	protected $import_manager;

	/**
	 * Backup manager.
	 *
	 * @var Backup_Manager
	 */
	protected $backup_manager;

	/**
	 * Import class mappings.
	 *
	 * @var array
	 */
	private $import_class_mapping = [
		'wordpress-content' => Import_WordPress_Content::class,
		'theme-options'     => Import_Theme_Options::class,
		'child-theme'       => Import_Child_Theme::class,
		'revolution-slider' => Import_Revolution_Slider::class,
		'wordpress-widgets' => Import_WordPress_Widgets::class,
		'typolab'           => Import_Typolab::class,
		'product-filter'    => Import_WooCommerce_Product_Filter::class,
		'ninja-forms'       => Import_Ninja_Forms::class,
	];

	/**
	 * Create instance.
	 *
	 * @param array $content_pack_entry
	 *
	 * @return self
	 */
	public static function create_instance( $content_pack_entry = [] ) {
		return new self( $content_pack_entry );
	}

	/**
	 * Constructor.
	 *
	 * @param array $content_pack_entry
	 *
	 * @return void
	 */
	public function __construct( $content_pack_entry = [] ) {
		$content_pack_entry = wp_parse_args(
			$content_pack_entry,
			[
				'type'    => '',
				'id'      => '',
				'name'    => '',
				'thumb'   => '',
				'import'  => [],
				'plugins' => [],
				'url'     => '',
				'badges'  => [],
			]
		);

		// Set type
		$this->type = $content_pack_entry['type'];

		// Set ID
		$this->id = $content_pack_entry['id'];

		// Name
		$this->name = $content_pack_entry['name'];

		// Thumbnail
		$this->thumbnail = $content_pack_entry['thumb'];

		// Preview URL
		$this->preview_url = $content_pack_entry['url'];

		// Badges
		$this->badges = $content_pack_entry['badges'];

		// Setup imports
		$this->setup_imports( $content_pack_entry['import'] );

		// Required plugins
		$this->setup_plugins( $content_pack_entry['plugins'] );

		// Import instance
		$this->import_instance = new Import_Instance( $this );

		// AJAX import handler
		add_action( "kalium_importer_run_action_{$this->get_base_id()}", [ $this, 'import_handler' ], 10, 2 );
	}

	/**
	 * Get type.
	 *
	 * @return string
	 */
	public function get_type() {
		return $this->type;
	}

	/**
	 * Get ID.
	 *
	 * @return string
	 */
	public function get_id() {
		return $this->id;
	}

	/**
	 * Get base content pack ID.
	 *
	 * @return string
	 */
	public function get_base_id() {
		return $this->get_type() . '-' . $this->get_id();
	}

	/**
	 * Get content pack name.
	 *
	 * @return string
	 */
	public function get_name() {
		return $this->name;
	}

	/**
	 * Get thumbnail URL.
	 *
	 * @return string
	 */
	public function get_thumbnail_url() {
		return $this->thumbnail;
	}

	/**
	 * Get imports.
	 *
	 * @return Import_Type[]
	 */
	public function get_imports() {
		return $this->imports;
	}

	/**
	 * Get import by ID.
	 *
	 * @param string $import_id
	 *
	 * @return Import_Type|null
	 */
	public function get_import_by_id( $import_id ) {
		foreach ( $this->get_imports() as $import ) {
			if ( $import->get_import_id() === $import_id ) {
				return $import;
			}
		}

		return null;
	}

	/**
	 * Get starter site preview URL.
	 *
	 * @return string
	 */
	public function get_preview_url() {
		return $this->preview_url;
	}

	/**
	 * Get badges.
	 *
	 * @return array
	 */
	public function get_badges() {
		return $this->badges;
	}

	/**
	 * Get required plugins.
	 *
	 * @return array
	 */
	public function get_required_plugins() {
		return $this->plugins;
	}

	/**
	 * Get plugin data.
	 *
	 * @param string $plugin_slug
	 *
	 * @return array|null
	 */
	public function get_plugin_data( $plugin_slug ) {
		foreach ( $this->plugins as $plugin_data ) {
			if ( $plugin_slug === $plugin_data['slug'] ) {
				return $plugin_data;
			}
		}

		return null;
	}

	/**
	 * Starter sites link to the import page.
	 *
	 * @param string $action_type
	 *
	 * @return string
	 */
	public function get_link( $action_type = 'install' ) {
		return admin_url( sprintf( 'admin-ajax.php?action=kalium_importer_view_page&action-type=%2$s&content-pack=%1$s', $this->get_base_id(), $action_type ) );
	}

	/**
	 * Get import instance.
	 *
	 * @return Import_Instance
	 */
	public function get_import_instance() {
		return $this->import_instance;
	}

	/**
	 * Get import type class name.
	 *
	 * @param string $import_type
	 *
	 * @return string
	 */
	public function get_import_type_class( $import_type ) {

		if ( isset( $this->import_class_mapping[ $import_type ] ) ) {
			return $this->import_class_mapping[ $import_type ];
		}

		return 'Import_Type';
	}

	/**
	 * Check if current demo is installed.
	 *
	 * @return bool
	 */
	public function is_installed() {
		$import_instance = $this->get_import_instance();

		return $import_instance->is_installed();
	}

	/**
	 * Setup imports related to package.
	 *
	 * @param array $imports
	 *
	 * @return void
	 */
	public function setup_imports( $imports ) {
		foreach ( $imports as $import_id => $import ) {

			// Create based on mapped class name
			$import_type_class = $this->get_import_type_class( $import['type'] );

			// Create instance and add to $this->imports[]
			$this->imports[] = new $import_type_class( $import_id, $import, $this );
		}
	}

	/**
	 * Setup plugins list.
	 *
	 * @param array $plugins
	 */
	public function setup_plugins( $plugins ) {
		foreach ( $plugins as $plugin_slug => $plugin_data ) {
			$this->plugins[] = [
				'slug'     => $plugin_slug,
				'name'     => kalium_get_array_key( $plugin_data, 'name' ),
				'basename' => kalium_get_array_key( $plugin_data, 'basename' ),
			];
		}
	}

	/**
	 * Getter and setter for Import Manager reference.
	 *
	 * @param Import_Manager $new_instance
	 *
	 * @return Import_Manager
	 */
	public function import_manager( $new_instance = null ) {

		// Set new import manager
		if ( ! is_null( $new_instance ) && $new_instance instanceof Import_Manager ) {
			$this->import_manager = $new_instance;
		}

		return $this->import_manager;
	}

	/**
	 * Getter and setter for Backup Manager reference.
	 *
	 * @param Backup_Manager $new_instance
	 *
	 * @return Backup_Manager
	 */
	public function backup_manager( $new_instance = null ) {

		// Set new import manager
		if ( ! is_null( $new_instance ) && $new_instance instanceof Backup_Manager ) {
			$this->backup_manager = $new_instance;
		}

		return $this->backup_manager;
	}

	/**
	 * Import handler.
	 *
	 * Ensured to serve only to authenticated and valid user levels.
	 *
	 * @param Import_Manager $import_manager
	 * @param Backup_Manager $backup_manager
	 *
	 * @return void
	 */
	public function import_handler( $import_manager, $backup_manager ) {

		// Action
		switch ( kalium()->request->xhr_request( 'content-pack-action' ) ) {

			// Check server limits
			case 'check-server-limits':
				$this->import_action_check_server_limits();
				break;

			// Request filesystem credentials
			case 'request-filesystem-credentials':
				$this->request_filesystem_credentials();
				break;

			// Get tasks list
			case 'tasks-list':
				wp_send_json( $this->get_tasks_list() );
				break;

			// Get uninstall tasks list
			case 'uninstall-tasks-list':
				wp_send_json( $this->get_uninstall_tasks_list() );
				break;

			// Install plugin task
			case 'plugin-install':
				// Register required plugins
				$import_manager->register_required_plugins();

				// Initialize Filesystem
				kalium()->filesystem->initialize();

				$plugin_slug = kalium()->request->xhr_request( 'slug' );
				$plugin_data = $this->get_plugin_data( $plugin_slug );

				// No plugin found
				if ( ! $plugin_data ) {
					wp_die();
				}

				// Alternative plugin
				if ( ( ! $plugin_data || false === kalium()->is->plugin_installed( $plugin_data['basename'] ) ) && isset( Importer::$alternative_plugins[ $plugin_slug ] ) ) {
					$plugin_slug = Importer::$alternative_plugins[ $plugin_slug ]['slug'];
					$plugin_data = $GLOBALS['tgmpa']->plugins[ $plugin_slug ];
				}

				// TGMPA $_GET vars
				$tgmpa_action     = kalium()->is->plugin_installed( $plugin_data['basename'] ) ? 'activate' : 'install';
				$tgmpa_action_var = 'tgmpa-' . $tgmpa_action;

				$_GET['plugin']            = $plugin_data['slug'];
				$_GET[ $tgmpa_action_var ] = $tgmpa_action . '-plugin';

				// Install or activate plugin
				kalium()->theme_plugins->install_plugin_ajax();
				break;

			// Import content task
			case 'content-import':
				$import_id     = kalium()->request->xhr_request( 'import-id' );
				$import_action = kalium()->request->xhr_request( 'import-action' );
				$checked       = wp_validate_boolean( kalium()->request->xhr_request( 'checked' ) );
				$args          = (array) json_decode( stripslashes( kalium()->request->xhr_request( 'args' ) ) );

				// Response
				$response = [
					'success' => false,
				];

				// if the import type exists
				if ( $import = $this->get_import_by_id( $import_id ) ) {

					// Import instance
					$import_instance = $this->get_import_instance();

					// Set current import id in import instance
					$import_instance->set_import_id( $import_id );

					// Set checked status of import field
					$import->is_checked( $checked );

					// Set args values
					$import->set_args_values( $args );

					// Initialize Filesystem
					kalium()->filesystem->initialize();

					// Import action
					switch ( $import_action ) {

						// Do download
						case 'do_download':
							// Clear errors for the current import type
							$import_instance->clear_errors();

							// Run import download method
							$import->do_download();
							break;

						// Do backup
						case 'do_backup':
							$import->do_backup();
							break;

						// Do import
						case 'do_import':
							// Run import method
							$import->do_import();
							break;

						// Do import
						case 'do_complete':
							// Run import complete method
							$import->do_complete();
							break;

						// Do remove
						case 'do_remove':
							$import->do_remove();
							break;
					}

					// Set result
					$errors = $import->get_errors();

					if ( $errors->has_errors() ) {
						$errors->success = false;
						$response        = $errors;
					} else {
						$response['success'] = true;
					}
				}

				// Send response as text/html type because content import type may produce content/errors which cannot be handled with string buffer
				die( sprintf( '<script type="text/template" class="kalium-demo-content-import-response">%s</script>', wp_json_encode( $response ) ) );
				break;

			// Regenerate thumbnails
			case 'regenerate-thumbnails':
				$this->regenerate_thumbnails();
				break;

			// Set content pack import state
			case 'set-install-state':
				$import_instance  = $this->get_import_instance();
				$imported_content = $import_instance->get_imported_content_type();

				// Install state
				$is_installed = 0 !== count( $imported_content );

				// Uninstalled
				if ( ! $is_installed ) {
					$backup_manager->delete_backup_options();
				}

				// Set install state
				$import_instance->set_install_state( $is_installed );

				// Response
				wp_send_json(
					[
						'installed' => $is_installed,
					]
				);
				break;
		}
	}

	/**
	 * Get tasks list to import this demo content pack.
	 *
	 * @return Task[]
	 */
	private function get_tasks_list() {
		$tasks = [];

		$import_types = explode( ',', kalium()->request->xhr_request( 'import-types' ) );

		// Selected imports
		$imports = array_filter(
			$this->get_imports(),
			function ( $import ) use ( $import_types ) {
				return in_array( $import->get_type(), $import_types );
			}
		);

		// Required plugins based on selected imports
		$required_plugins = array_filter(
			$this->get_required_plugins(),
			function ( $plugin_data ) use ( $imports ) {
				$plugin_slug = $plugin_data['slug'];

				foreach ( $imports as $import ) {
					if ( in_array( $plugin_slug, $import->get_required_plugins() ) ) {
						return true;
					}
				}

				return false;
			}
		);

		// Required plugins tasks
		foreach ( $required_plugins as $plugin_data ) {
			$plugin_slug     = $plugin_data['slug'];
			$plugin_basename = $plugin_data['basename'];

			$is_installed = kalium()->is->plugin_installed( $plugin_basename );
			$tgmpa_nonce  = wp_create_nonce( 'tgmpa-' . ( $is_installed ? 'activate' : 'install' ) );
			$message      = null;

			if ( isset( Importer::$alternative_plugins[ $plugin_slug ] ) && false === kalium()->is->plugin_active( $plugin_basename ) ) {
				$message = [
					'title'       => 'Confirm: Alternative Plugin Installation',
					'description' => kalium_format_content( Importer::$alternative_plugins[ $plugin_slug ]['message'] ),
					'buttons'     => [
						[
							'class' => 'button button-primary button-proceed',
							'title' => 'Proceed to Install',
						],
						[
							'class' => 'button button-close',
							'title' => 'Abort',
						],
					],
				];
			}

			// Task instance
			$task = new Task(
				'plugin-install',
				[
					'slug'        => $plugin_slug,
					'installed'   => $is_installed,
					'message'     => $message,
					'tgmpa-nonce' => $tgmpa_nonce,
				]
			);

			// Set as completed if plugin is active
			if ( kalium()->is->plugin_active( $plugin_basename ) ) {
				$task->mark_complete( true );
			}

			// Add task
			$tasks[] = $task;
		}

		// Import type tasks
		foreach ( $imports as $import ) {
			if ( $import->is_imported() ) {
				continue;
			}

			// Task instance
			$task = new Task(
				'content-import',
				[
					'import_id'   => $import->get_import_id(),
					'import_type' => $import->get_type(),
					'import_name' => $import->get_name(),
				]
			);

			// Require thumbnails regenerate
			if ( 'wordpress-content' === $import->get_type() ) {
				$require_thumbnails_regenerate = true;
			}

			$tasks[] = $task;
		}

		// Regenerate thumbnails
		if ( apply_filters( 'kalium_importer_regenerate_thumbnails', ! empty( $require_thumbnails_regenerate ) ) ) {
			$tasks[] = new Task( 'regenerate-thumbnails' );
		}

		return $tasks;
	}

	/**
	 * Get tasks list to uninstall demo content type entries.
	 *
	 * @return Task[]
	 */
	private function get_uninstall_tasks_list() {
		$tasks = [];

		// Get imported content types
		foreach ( $this->get_import_instance()->get_imported_content_type() as $content_type ) {

			// Task instance
			$task = new Task(
				'content-import',
				[
					'import_id'   => $content_type['id'],
					'import_type' => $content_type['type'],
					'import_name' => $content_type['name'],
					'uninstall'   => true,
				]
			);

			$tasks[] = $task;
		}

		return $tasks;
	}

	/**
	 * Import action: Check server limits.
	 *
	 * @return void
	 */
	private function import_action_check_server_limits() {

		// Init System Status vars
		System_Status::init_vars();

		// Send JSON response
		wp_send_json(
			[
				// Current server vars
				'filesystem_method'          => get_filesystem_method(),
				'wp_memory_limit'            => System_Status::get_var( 'wp_memory_limit' ),
				'max_execution_time'         => System_Status::get_var( 'max_execution_time' ),
				'domdocument'                => System_Status::get_var( 'domdocument' ),
				'remote_get'                 => wp_remote_get( kalium()->endpoint_url( 'hello' ), [ 'timeout' => 10 ] ),

				// Recommended values
				'recommended_memory_limit'   => self::MINIMUM_MEMORY_LIMIT,
				'recommended_execution_time' => self::MINIMUM_EXECUTION_TIME,
			]
		);
	}

	/**
	 * Request filesystem credentials form.
	 *
	 * @return void
	 */
	private function request_filesystem_credentials() {
		$url = admin_url( sprintf( 'admin-ajax.php?action=%s', kalium()->request->query( 'action' ) ) );

		// JSON response
		wp_send_json(
			[
				'ok'               => kalium()->filesystem->initialize( $url ),
				'credentials'      => kalium()->filesystem->get_credentials(),
				'credentials_form' => kalium()->filesystem->get_credentials_form(),
				'error_message'    => kalium()->filesystem->get_error_message(),
			]
		);
	}

	/**
	 * Regenerate thumbnails.
	 *
	 * @since 4.0
	 */
	private function regenerate_thumbnails() {
		$instance_id   = $this->get_import_instance()->get_id();
		$attachment_id = kalium()->request->xhr_request( 'attachment-id' );

		// Regenerate attachment
		if ( is_numeric( $attachment_id ) ) {
			require_once __DIR__ . '/plugins/class-regeneratethumbnails-regenerator.php';

			$regenerator = \RegenerateThumbnails_Regenerator::get_instance( $attachment_id );
			$regenerate  = $regenerator->regenerate();

			if ( is_wp_error( $regenerate ) ) {
				wp_send_json_error( $regenerate->get_error_messages() );
			}

			wp_send_json(
				[
					'result' => $regenerate,
				]
			);
		}

		$images = new \WP_Query(
			[
				'post_type'      => 'attachment',
				'post_status'    => 'any',
				'post_mime_type' => 'image',
				'posts_per_page' => -1,
				'fields'         => 'ids',
				'meta_query'     => [
					[
						'key'   => '_content_pack_import_id',
						'value' => $instance_id,
					],
				],
			]
		);

		wp_send_json(
			[
				'instance_id' => $instance_id,
				'images_list' => $images->posts,
			]
		);
	}
}
