<?php
/**
 * Kalium WordPress Theme
 *
 * Starter Site Type - WordPress Import class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Importer;

use Elementor\Plugin;
use WP_Term;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Import_WordPress_Content extends Import_Type {

	/**
	 * Tries to convert an attachment URL into a post ID.
	 *
	 * @param string $url
	 *
	 * @return int|null
	 */
	public static function get_attachment_id_from_url( $url ) {
		static $post_ids, $cached_urls = [];
		global $wpdb;

		// Return already cached url
		if ( isset( $cached_urls[ $url ] ) ) {
			return $cached_urls[ $url ];
		}

		// Process file path in URL
		$dir  = wp_get_upload_dir();
		$path = $url;

		$site_url   = parse_url( $dir['url'] );
		$image_path = parse_url( $path );

		// Force the protocols to match if needed.
		if ( isset( $image_path['scheme'] ) && ( $image_path['scheme'] !== $site_url['scheme'] ) ) {
			$path = str_replace( $image_path['scheme'], $site_url['scheme'], $path );
		}

		if ( 0 === strpos( $path, $dir['baseurl'] . '/' ) ) {
			$path = substr( $path, strlen( $dir['baseurl'] . '/' ) );
		}

		// Query current import instance ID
		if ( ! isset( $post_ids ) ) {
			$post_ids_arr = esc_sql( Helpers::get_post_ids() );
			$post_ids     = implode( ',', $post_ids_arr );

			if ( empty( $post_ids ) ) {
				$post_ids = '-1';
			}
		}

		// Query attachment name based on current query
		$sql = $wpdb->prepare( "SELECT post_id FROM {$wpdb->postmeta} WHERE post_id IN ({$post_ids}) AND meta_key = '_wp_attached_file' AND meta_value LIKE %s", '%' . $wpdb->esc_like( $path ) );

		// Matched attachment
		$attachment_id = $wpdb->get_var( $sql );

		if ( is_numeric( $attachment_id ) ) {
			$attachment_id = (int) $attachment_id;
		}

		// Cache url
		$cached_urls[ $url ] = $attachment_id;

		return $attachment_id;
	}

	/**
	 * Get content pack name.
	 *
	 * @return string
	 */
	public function get_name() {

		// Import type name
		if ( ! empty( $this->name ) ) {
			return $this->name;
		}

		return 'Content';
	}

	/**
	 * Backup before import.
	 *
	 * @return void
	 */
	public function do_backup() {
		$backup_manager = $this->get_content_pack()->backup_manager();
		$options        = $this->get_options();

		// Front page
		if ( isset( $options['front_page'] ) ) {
			$backup_manager->update_backup_option( 'page_on_front', get_option( 'page_on_front' ) );
		}

		// Posts page
		if ( isset( $options['posts_page'] ) ) {
			$backup_manager->update_backup_option( 'page_for_posts', get_option( 'page_for_posts' ) );
		}

		// Update show on front option
		if ( isset( $options['front_page'] ) || isset( $options['posts_page'] ) ) {
			$backup_manager->update_backup_option( 'show_on_front', get_option( 'show_on_front' ) );
		}

		// Permalink structure
		$backup_manager->update_backup_option( 'permalink_structure', get_option( 'permalink_structure' ) );

		// Trash default posts
		$default_posts = [
			'sample-page' => 'page',
			'hello-world' => 'post',
		];

		$trashed_posts = [];

		foreach ( $default_posts as $post_path => $post_type ) {
			if ( $post = get_page_by_path( $post_path, OBJECT, $post_type ) ) {
				wp_trash_post( $post->ID );

				$trashed_posts[ $post->ID ] = $post->post_status;
			}
		}

		$backup_manager->update_backup_option( 'trashed_posts', $trashed_posts );

		// Menus
		if ( ! empty( $options['menus'] ) ) {
			$nav_menu_locations        = get_theme_mod( 'nav_menu_locations' );
			$backup_nav_menu_locations = [];

			foreach ( $options['menus'] as $menu_location => $menu_title ) {
				if ( isset( $nav_menu_locations[ $menu_location ] ) ) {
					$backup_nav_menu_locations[ $menu_location ] = $nav_menu_locations[ $menu_location ];
				}
			}

			// Save backup option
			$backup_manager->update_backup_option( 'menus', $backup_nav_menu_locations );
		}

		// WooCommerce options
		if ( $this->is_plugin_required( 'woocommerce' ) && kalium()->is->woocommerce_active() ) {
			$woocommerce = kalium_get_array_key( $options, 'woocommerce', [] );

			// WooCommerce pages
			if ( $pages = kalium_get_array_key( $woocommerce, 'pages' ) ) {
				$woocommerce_pages = [];

				foreach ( $pages as $page_name => $page_title ) {
					$option_name  = "woocommerce_{$page_name}_page_id";
					$option_value = get_option( $option_name, null );

					if ( ! is_null( $option_value ) ) {
						$woocommerce_pages[ $option_name ] = $option_value;
					}
				}

				// Save backup option
				if ( ! empty( $woocommerce_pages ) ) {
					$backup_manager->update_backup_option( 'woocommerce_pages', $woocommerce_pages );
				}
			}

			// WooCommerce thumbnail sizes
			if ( $thumbnails = kalium_get_array_key( $woocommerce, 'thumbnails' ) ) {
				$woocommerce_thumbnails = [];

				foreach ( $thumbnails as $thumbnail_name => $thumbnail_value ) {
					$option_name  = "woocommerce_thumbnail_{$thumbnail_name}";
					$option_value = get_option( $option_name, null );

					if ( ! is_null( $option_name ) ) {
						$woocommerce_thumbnails[ $option_name ] = $option_value;
					}
				}

				// Save backup option
				if ( ! empty( $woocommerce_thumbnails ) ) {
					$backup_manager->update_backup_option( 'woocommerce_thumbnails', $woocommerce_thumbnails );
				}
			}

			// WooCommerce options
			if ( $woo_options = kalium_get_array_key( $woocommerce, 'options' ) ) {
				$woocommerce_options = [];

				foreach ( $woo_options as $option_name => $option_value ) {
					$current_value = get_option( $option_name, null );

					if ( ! is_null( $current_value ) ) {
						$woocommerce_options[ $option_name ] = $current_value;
					}
				}

				// Save backup option
				if ( ! empty( $woocommerce_options ) ) {
					$backup_manager->update_backup_option( 'woocommerce_options', $woocommerce_options );
				}
			}

			// WooCommerce attribute taxonomies
			if ( ! empty( $woocommerce['attribute_taxonomies'] ) ) {
				$current_attribute_taxonomies = wc_get_attribute_taxonomies();
				$attribute_taxonomies         = [];

				foreach ( $current_attribute_taxonomies as $attribute_taxonomy ) {
					$attribute_taxonomies[] = (array) $attribute_taxonomy;
				}

				// Save backup option
				if ( ! empty( $attribute_taxonomies ) ) {
					$backup_manager->update_backup_option( 'woocommerce_attribute_taxonomies', $attribute_taxonomies );
				}
			}
		}

		// Elementor options
		if ( $this->is_plugin_required( 'elementor' ) && kalium()->is->elementor_active() ) {

			// Active kit
			$active_kit = get_post_meta( Plugin::$instance->kits_manager->get_active_id(), '_elementor_page_settings', true );
			$backup_manager->update_backup_option( 'elementor_active_kit', $active_kit );
		}

		// Deactivate WP_Importer plugin
		if ( kalium()->is->plugin_active( 'wordpress-importer/wordpress-importer.php' ) ) {
			deactivate_plugins( 'wordpress-importer/wordpress-importer.php' );
		}
	}

	/**
	 * Import WordPress XML file.
	 *
	 * @return void
	 */
	public function do_import() {

		// Execute parent do_import
		parent::do_import();

		// Do not run if there are errors reported
		if ( $this->errors->has_errors() ) {
			return;
		}

		// Include WP_Importer
		require_once __DIR__ . '/kalium-wp-import.php';

		// Vars
		$content_pack       = $this->get_content_pack();
		$import_manager     = $content_pack->import_manager();
		$backup_manager     = $content_pack->backup_manager();
		$import_instance    = $content_pack->get_import_instance();
		$import_instance_id = $import_instance->get_id();
		$fetch_attachments  = $this->is_checked();
		$import_terms       = [];

		// Disable creating attachment thumbnails
		add_filter( 'intermediate_image_sizes_advanced', '__return_empty_array' );

		// Initialize import terms
		add_action(
			'import_start',
			function () use ( &$import_terms ) {
				$import_terms = $GLOBALS['kalium_import']->terms;
			}
		);

		// Import process end actions
		add_action(
			'import_end',
			function () use ( $import_instance, $backup_manager ) {
				$backup_manager->update_backup_option( 'processed_posts', $GLOBALS['kalium_import']->processed_posts, false, true );
				$backup_manager->update_backup_option( 'processed_terms', $GLOBALS['kalium_import']->processed_terms, false, true );

				$import_instance->set_import_success();
			}
		);

		// Replace Elementor URLs
		add_filter(
			'wp_import_post_meta',
			function ( $post_meta ) {
				return $this->replace_elementor_urls( $post_meta );
			}
		);

		// Try keeping the same term IDs
		add_filter(
			'wp_insert_term_data',
			function ( $data, $taxonomy ) use ( &$import_terms ) {
				$term = kalium_get_array_first(
					wp_list_filter(
						$import_terms,
						[
							'term_taxonomy' => $taxonomy,
							'slug'          => $data['slug'],
						]
					)
				);

				if ( ! empty( $term ) && ! term_exists( $term['term_id'] ) ) {
					$data['term_id'] = $term['term_id'];
				}

				return $data;
			},
			10,
			2
		);

		// Loop through each source
		foreach ( $this->get_sources() as $source ) {

			// Import file
			$import_file = $import_manager->get_content_pack_import_source_path( $source['name'] );

			// WP_Import instance
			$GLOBALS['kalium_import'] = new Kalium_WP_Import( $import_instance_id, $fetch_attachments, $content_pack->get_preview_url() );

			// Check if import file exists
			if ( true === kalium()->filesystem->exists( $import_file ) ) {

				// Start import
				$GLOBALS['kalium_import']->import( $import_file );
			} else {

				// Import file doesn't exists
				$this->errors->add( 'kalium_importer_import_file_not_exists', 'Import file doesn\'t exists!' );
			}
		}
	}

	/**
	 * Hooks and callbacks to execute after import finished.
	 *
	 * @return void
	 */
	public function do_complete() {
		$options = $this->get_options();

		// Front page
		if ( isset( $options['front_page'] ) && ( $front_page = Helpers::get_page_by_title( null, $options['front_page'] ) ) ) {
			update_option( 'page_on_front', $front_page->ID, 'yes' );
		}

		// Posts page
		if ( isset( $options['posts_page'] ) && ( $posts_page = Helpers::get_page_by_title( null, $options['posts_page'] ) ) ) {
			update_option( 'page_for_posts', $posts_page->ID, 'yes' );
		}

		// Update show on front option
		if ( ( isset( $front_page ) && $front_page ) || ( isset( $posts_page ) && $posts_page ) ) {
			update_option( 'show_on_front', 'page', 'yes' );
		}

		// Menus
		$menus = kalium_get_array_key( $options, 'menus', [] );

		if ( ! empty( $menus ) ) {
			foreach ( $menus as $menu_location => $menu_title ) {
				$menu = Helpers::get_term_by( null, 'name', $menu_title, 'nav_menu' );

				if ( $menu instanceof WP_Term ) {
					$nav_menu_locations                   = get_theme_mod( 'nav_menu_locations', [] );
					$nav_menu_locations[ $menu_location ] = $menu->term_id;
					set_theme_mod( 'nav_menu_locations', $nav_menu_locations );
				}
			}
		}

		// WooCommerce options
		if ( $this->is_plugin_required( 'woocommerce' ) && kalium()->is->woocommerce_active() ) {
			$woocommerce = kalium_get_array_key( $options, 'woocommerce', [] );

			// WooCommerce pages
			if ( $pages = kalium_get_array_key( $woocommerce, 'pages' ) ) {
				foreach ( $pages as $page_name => $page_title ) {
					if ( $page = Helpers::get_page_by_title( null, $page_title ) ) {
						update_option( "woocommerce_{$page_name}_page_id", $page->ID, 'yes' );
					}
				}
			}

			// WooCommerce thumbnail sizes
			if ( $thumbnails = kalium_get_array_key( $woocommerce, 'thumbnails' ) ) {
				foreach ( $thumbnails as $thumbnail_name => $thumbnail_value ) {
					update_option( "woocommerce_thumbnail_{$thumbnail_name}", $thumbnail_value, 'yes' );
				}
			}

			// WooCommerce options
			if ( $woocommerce_options = kalium_get_array_key( $woocommerce, 'options' ) ) {
				foreach ( $woocommerce_options as $option_name => $option_value ) {
					update_option( $option_name, $option_value, 'yes' );
				}
			}

			// Map and update WooCommerce taxonomy data
			if ( $taxonomy_data = kalium_get_array_key( $woocommerce, 'taxonomy_data' ) ) {
				$this->set_woocommerce_taxonomy_data( $taxonomy_data );
			}

			// Map and update WooCommerce attribute taxonomies
			if ( $attribute_taxonomies = kalium_get_array_key( $woocommerce, 'attribute_taxonomies' ) ) {
				$this->set_woocommerce_attribute_taxonomies( $attribute_taxonomies );
			}
		}

		// Elementor options
		if ( $this->is_plugin_required( 'elementor' ) && kalium()->is->elementor_active() ) {
			$elementor = kalium_get_array_key( $options, 'elementor', [] );

			// Import kit settings
			if ( $kit_settings = kalium_get_array_key( $elementor, 'kit_settings' ) ) {
				update_post_meta( Plugin::$instance->kits_manager->get_active_id(), '_elementor_page_settings', $kit_settings );
			}

			// Clear Elementor cache
			if ( kalium()->is->elementor_active() ) {
				Plugin::$instance->files_manager->clear_cache();
			}
		}

		// Flush rewrite rules
		flush_rewrite_rules();

		// Mark as successful task
		parent::do_complete();
	}

	/**
	 * Remove imported content from XML file and revert other options.
	 *
	 * @return void
	 */
	public function do_remove() {

		// Required plugins are not active
		if ( ! $this->plugins_are_active() ) {
			$this->errors->add( 'kalium_importer_remove_plugins_not_active', sprintf( 'Required plugins are not active, <strong>%s</strong> cannot be uninstalled.', $this->get_name() ) );
			return;
		}

		// Vars
		$content_pack       = $this->get_content_pack();
		$backup_manager     = $content_pack->backup_manager();
		$backup_options     = $backup_manager->get_backup_options();
		$import_instance    = $content_pack->get_import_instance();
		$import_instance_id = $import_instance->get_id();
		$options            = $this->get_options();

		// Ensure deletion of Elementor default kit
		$_GET['force_delete_kit'] = true;

		// Delete posts
		$post_ids = Helpers::get_post_ids( $import_instance_id );

		foreach ( $post_ids as $post_id ) {
			wp_delete_post( $post_id, true );
		}

		// Delete terms
		$term_ids = Helpers::get_term_ids( $import_instance_id );

		foreach ( $term_ids as $term_id ) {
			$term = get_term( $term_id );

			if ( $term ) {
				wp_delete_term( $term_id, $term->taxonomy );
			}
		}

		/**
		 * Revert options.
		 */

		// Front page
		if ( isset( $backup_options['page_on_front'] ) ) {
			$page_on_front = $backup_options['page_on_front'];

			if ( get_post( $page_on_front ) ) {
				update_option( 'page_on_front', $page_on_front );
			}
		}

		// Posts page
		if ( isset( $backup_options['page_for_posts'] ) ) {
			$page_for_posts = $backup_options['page_for_posts'];

			if ( get_post( $page_for_posts ) ) {
				update_option( 'page_for_posts', $page_for_posts );
			}
		}

		// Update show on front option
		if ( isset( $backup_options['show_on_front'] ) ) {
			update_option( 'show_on_front', $backup_options['show_on_front'] );
		}

		// Permalink structure
		if ( isset( $backup_options['permalink_structure'] ) ) {
			update_option( 'permalink_structure', $backup_options['permalink_structure'] );
		}

		// Menus
		if ( ! empty( $backup_options['menus'] ) ) {
			$nav_menu_locations = get_theme_mod( 'nav_menu_locations' );

			foreach ( $backup_options['menus'] as $menu_location => $nav_menu_id ) {

				// Replace menu if it exists
				if ( get_term( $nav_menu_id ) ) {
					$nav_menu_locations[ $menu_location ] = $nav_menu_id;
				} else {
					unset( $nav_menu_locations[ $menu_location ] );
				}
			}

			// Revert nav menu locations
			set_theme_mod( 'nav_menu_locations', $nav_menu_locations );
		}

		// WooCommerce options
		if ( $this->is_plugin_required( 'woocommerce' ) && kalium()->is->woocommerce_active() ) {
			$woocommerce = kalium_get_array_key( $options, 'woocommerce', [] );

			// Pages
			if ( $pages = kalium_get_array_key( $backup_options, 'woocommerce_pages' ) ) {
				foreach ( $pages as $option_name => $option_value ) {

					if ( $option_value && get_post( $option_value ) ) {
						update_option( $option_name, $option_value );
					} else {
						update_option( $option_name, '' );
					}
				}
			}

			// Thumbnails
			if ( $thumbnails = kalium_get_array_key( $backup_options, 'woocommerce_thumbnails' ) ) {
				foreach ( $thumbnails as $option_name => $option_value ) {
					update_option( $option_name, $option_value, 'yes' );
				}
			}

			// WooCommerce options
			if ( $woo_options = kalium_get_array_key( $backup_options, 'woocommerce_options' ) ) {
				foreach ( $woo_options as $option_name => $option_value ) {
					update_option( $option_name, $option_value, 'yes' );
				}
			}

			// WooCommerce attribute taxonomies
			if ( $attribute_taxonomies = kalium_get_array_key( $woocommerce, 'attribute_taxonomies' ) ) {
				$imported_attribute_taxonomies = json_decode( $attribute_taxonomies, true );
				$previous_attribute_taxonomies = kalium_get_array_key( $backup_options, 'woocommerce_attribute_taxonomies', [] );

				if ( is_array( $imported_attribute_taxonomies ) ) {
					foreach ( wc_get_attribute_taxonomies() as $attribute_taxonomy ) {

						// Check if current attribute taxonomy is imported
						foreach ( $imported_attribute_taxonomies as $imported_attribute_taxonomy ) {

							// Matched attribute
							if (
								$attribute_taxonomy->attribute_name === $imported_attribute_taxonomy['attribute_name'] ||
								$attribute_taxonomy->attribute_label === $imported_attribute_taxonomy['attribute_label']
							) {

								// Check if this attribute was previously imported
								$previously_imported = false;

								foreach ( $previous_attribute_taxonomies as $previous_attribute_taxonomy ) {
									if (
										$imported_attribute_taxonomy['attribute_name'] === $previous_attribute_taxonomy['attribute_name'] ||
										$imported_attribute_taxonomy['attribute_label'] === $previous_attribute_taxonomy['attribute_label']
									) {
										$previously_imported = true;
										break;
									}
								}

								// If not previously imported, delete it
								if ( ! $previously_imported ) {
									wc_delete_attribute( $attribute_taxonomy->attribute_id );
								}
							}
						}
					}
				}
			}
		}

		// Elementor options
		if ( $this->is_plugin_required( 'elementor' ) && kalium()->is->elementor_active() ) {

			// Default kit
			if ( $kit_settings = kalium_get_array_key( $backup_options, 'elementor_active_kit' ) ) {

				// Active kit
				update_post_meta( Plugin::$instance->kits_manager->get_active_id(), '_elementor_page_settings', $kit_settings );
			}
		}

		// Restore default posts
		foreach ( $backup_options['trashed_posts'] as $post_id => $post_status ) {
			if ( 'publish' === $post_status ) {
				wp_publish_post( $post_id );
			} else {
				wp_untrash_post( $post_id );
			}
		}

		// Flush rewrite rules
		flush_rewrite_rules();

		// Mark as removed
		parent::do_remove();
	}

	/**
	 * Replace Elementor URLs.
	 *
	 * @param array $post_meta
	 *
	 * @return array
	 */
	private function replace_elementor_urls( $post_meta ) {
		static $uploads_dir;

		if ( is_null( $uploads_dir ) ) {
			$uploads_dir = wp_upload_dir();
		}

		return array_map(
			function ( $meta ) use ( &$uploads_dir ) {
				if ( '_elementor_data' !== $meta['key'] || empty( $uploads_dir['baseurl'] ) ) {
					return $meta;
				}

				$from = $this->get_content_pack()->get_preview_url() . '/wp-content/uploads';
				$to   = $uploads_dir['baseurl'];

				$meta['value'] = str_replace( str_replace( '/', '\\/', $from ), str_replace( '/', '\\/', $to ), $meta['value'] );

				echo '<pre>' . $meta['value'] . '</pre>';

				return $meta;
			},
			$post_meta
		);
	}

	/**
	 * Map and update WooCommerce taxonomy data.
	 *
	 * @param array $taxonomy_entries
	 *
	 * @return void
	 */
	private function set_woocommerce_taxonomy_data( $taxonomy_entries ) {
		foreach ( $taxonomy_entries as $taxonomy_entry ) {
			if ( $matched_term = Helpers::get_term_by( null, 'slug', $taxonomy_entry['slug'], $taxonomy_entry['taxonomy'] ) ) {
				$term_id  = $matched_term->term_id;
				$taxonomy = $matched_term->taxonomy;

				// Data to modify
				$thumbnail = kalium_get_array_key( $taxonomy_entry, 'thumbnail' );

				// Update thumbnail
				if ( $thumbnail && ( $attachment_id = self::get_attachment_id_from_url( $thumbnail ) ) ) {
					update_term_meta( $term_id, 'thumbnail_id', $attachment_id );
				}
			}
		}
	}

	/**
	 * Map and update WooCommerce attribute taxonomies.
	 *
	 * @param array $attribute_taxonomies
	 *
	 * @return void
	 */
	private function set_woocommerce_attribute_taxonomies( $attribute_taxonomies ) {
		$current_attribute_taxonomies = wc_get_attribute_taxonomies();

		foreach ( $current_attribute_taxonomies as $current_attribute_taxonomy ) {
			foreach ( $attribute_taxonomies as $attribute_taxonomy ) {
				if ( $attribute_taxonomy['attribute_name'] === $current_attribute_taxonomy->attribute_name ) {
					wc_update_attribute(
						$current_attribute_taxonomy->attribute_id,
						[
							'name' => $attribute_taxonomy['attribute_label'],
						]
					);
					break;
				}
			}
		}
	}
}
