<?php
/**
 * WooCommerce Product Attribute element.
 */

namespace Laborator_Builder\Elements;

use Laborator_Builder\Element;
use Laborator_Builder\Attribute;
use Laborator_Builder\Attribute_Sets\Advanced_Border_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Custom_CSS_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Dimensions_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Layout_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Text_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Visibility_Hover_Set;
use Laborator_Builder\Attribute_Sets\Product_Common_Attributes_Set;
use WC_Product_Attribute;

class WooCommerce_Product_Attributes extends Element {

	/**
	 * Name.
	 *
	 * @var string
	 */
	public static $name = 'woocommerce_product_attribute';

	/**
	 * Title.
	 *
	 * @var string
	 */
	public static $title = 'Attribute';

	/**
	 * Icon.
	 *
	 * @var string
	 */
	public static $icon = 'attribute';

	/**
	 * Get attribute sets.
	 *
	 * @return array
	 */
	public function get_attribute_sets() {
		return array_merge(
			parent::get_attribute_sets(),
			[
				Advanced_Layout_Set::class,
				Advanced_Visibility_Hover_Set::class,
				Advanced_Border_Set::class,
				Advanced_Text_Set::class,
				Advanced_Dimensions_Set::class,
				Advanced_Custom_CSS_Set::class,
				Product_Common_Attributes_Set::class,
			]
		);
	}

	/**
	 * Attributes.
	 *
	 * @return Attribute[]
	 */
	public function create_attributes() {
		$attributes = [];

		// Populate product attributes
		if ( function_exists( 'wc_get_attribute_taxonomy_labels' ) ) {
			$attributes = array_merge(
				wc_get_attribute_taxonomy_labels(),
				[
					'custom' => ':: Custom ::',
				]
			);
		}

		return array_merge(
			parent::create_attributes(),
			[
				// Content / Content
				Attribute::tab_section(
					Attribute::TAB_CONTENT,
					Attribute::SECTION_CONTENT,
					[
						Attribute::create(
							'attribute_name',
							[
								'type'         => 'select',
								'title'        => 'Product attribute',
								'inline_label' => true,
								'choices'      => $attributes,
							]
						),
						Attribute::create(
							'custom_attribute_name',
							[
								'type'         => 'text',
								'title'        => 'Custom attribute',
								'description'  => 'Custom attributes are added on product level only. Specified attribute will be shown only if they exist.',
								'inline_label' => true,
								'dependency'   => [
									'attribute_name' => 'custom',
								],
							]
						),
						Attribute::create(
							'show_label',
							[
								'type'         => 'toggle',
								'title'        => 'Show label',
								'inline_label' => true,
								'default'      => true,
							]
						),
					]
				),
			]
		);
	}

	/**
	 * Get selected attribute.
	 *
	 * @param string $attribute_name
	 * @param string $custom_attribute_name
	 *
	 * @return array (
	 *               title: string
	 *               value: string
	 * )
	 */
	public function get_selected_attribute( $attribute_name, $custom_attribute_name ) {
		global $product;

		// Product attributes
		$attributes = array_filter( $product->get_attributes(), 'wc_attributes_array_filter_visible' );

		// Filter selected
		$attributes = array_filter(
			$attributes,
			function ( $attribute ) use ( $attribute_name, $custom_attribute_name ) {
				/** @var WC_Product_Attribute $attribute */
				if ( $attribute->get_name() === wc_attribute_taxonomy_name( $attribute_name ) ) {
					return true;
				} elseif ( 'custom' === $attribute_name && $attribute->get_name() === $custom_attribute_name ) {
					return true;
				}

				return false;
			}
		);

		// Process attributes
		foreach ( $attributes as $attribute ) {
			$values = [];

			if ( $attribute->is_taxonomy() ) {
				$attribute_taxonomy = $attribute->get_taxonomy_object();
				$attribute_values   = wc_get_product_terms( $product->get_id(), $attribute->get_name(), [ 'fields' => 'all' ] );

				foreach ( $attribute_values as $attribute_value ) {
					$value_name = esc_html( $attribute_value->name );

					if ( $attribute_taxonomy->attribute_public ) {
						$values[] = '<a href="' . esc_url( get_term_link( $attribute_value->term_id, $attribute->get_name() ) ) . '" rel="tag">' . $value_name . '</a>';
					} else {
						$values[] = $value_name;
					}
				}
			} else {
				$values = $attribute->get_options();

				foreach ( $values as &$value ) {
					$value = make_clickable( esc_html( $value ) );
				}
			}

			return [
				'label' => wc_attribute_label( $attribute->get_name() ),
				'value' => apply_filters( 'woocommerce_attribute', wpautop( wptexturize( implode( ', ', $values ) ) ), $attribute, $values ),
			];
		}

		return null;
	}
}

// Register element
\Laborator_Builder::register_element_type( WooCommerce_Product_Attributes::class );
