<?php
/**
 * WooCommerce Product Sale Badge element.
 */

namespace Laborator_Builder\Elements;

use Laborator_Builder\Element;
use Laborator_Builder\Attribute;
use Laborator_Builder\Attribute_Sets\Advanced_Border_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Custom_CSS_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Dimensions_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Layout_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Text_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Visibility_Hover_Set;
use Laborator_Builder\Attribute_Sets\Product_Common_Attributes_Set;

class WooCommerce_Product_Sale_Badge extends Element {

	/**
	 * Name.
	 *
	 * @var string
	 */
	public static $name = 'woocommerce_product_sale_badge';

	/**
	 * Title.
	 *
	 * @var string
	 */
	public static $title = 'Sale Badge';

	/**
	 * Icon.
	 *
	 * @var string
	 */
	public static $icon = 'percent';

	/**
	 * Maximum instances of element.
	 *
	 * @var int
	 * @static
	 */
	public static $max_instances = 1;

	/**
	 * Get attribute sets.
	 *
	 * @return array
	 */
	public function get_attribute_sets() {
		return array_merge(
			parent::get_attribute_sets(),
			[
				Advanced_Layout_Set::class,
				Advanced_Visibility_Hover_Set::class,
				Advanced_Border_Set::class,
				Advanced_Text_Set::class,
				Advanced_Dimensions_Set::class,
				Advanced_Custom_CSS_Set::class,
				Product_Common_Attributes_Set::class,
			]
		);
	}

	/**
	 * Init.
	 */
	public function init() {
		parent::init();

		// Badge style
		$badge_style   = $this->get_attribute( 'badge_style' );
		$badge_outline = $this->get_attribute( 'badge_outline' );

		if ( 'default' === $badge_style->get_value() ) {
			// Badge style
			$badge_style->get_value_object()->set_value( kalium_get_theme_option( 'shop_badges_style' ) );

			// Outline
			$badge_outline->get_value_object()->set_value( kalium_get_theme_option( 'shop_badges_outline' ) );

			// Triangle auto placement
			if ( 'triangle' === $badge_style->get_value() ) {
				$triangle_rotation = $this->get_attribute( 'triangle_rotation' )->get_value_object();

				switch ( $this->get_attribute_value( 'placement' ) ) {
					case 'top-left':
						$triangle_rotation->set_value( '270deg' );
						break;

					case 'bottom-left':
						$triangle_rotation->set_value( '180deg' );
						break;

					case 'bottom-right':
						$triangle_rotation->set_value( '90deg' );
						break;
				}
			}
		}
	}

	/**
	 * Attributes.
	 *
	 * @return Attribute[]
	 */
	public function create_attributes() {
		return array_merge(
			parent::create_attributes(),
			[

				// Style / Style
				Attribute::tab_section(
					Attribute::TAB_CONTENT,
					Attribute::TAB_CONTENT,
					[
						Attribute::create(
							'badge_style',
							[
								'type'    => 'radio',
								'title'   => 'Badge style',
								'choices' => [
									'default'  => [
										'label' => 'Default',
										'icon'  => 'globe',
									],
									'badge'    => [
										'label' => 'Regular',
										'icon'  => 'saleBadgeRegular',
									],
									'circle'   => [
										'label' => 'Circle',
										'icon'  => 'saleBadgeCircle',
									],
									'triangle' => [
										'label' => 'Triangle',
										'icon'  => 'saleBadgeTriangle',
									],
								],
								'buttons' => [
									'icon_size' => 18,
								],
							]
						),
						Attribute::create(
							'badge_outline',
							[
								'type'         => 'toggle',
								'title'        => 'Outline',
								'inline_label' => true,
								'default'      => true,
								'dependency'   => [
									'badge_style' => 'badge',
								],
							]
						),
						Attribute::create(
							'badge_size',
							[
								'type'       => 'range',
								'title'      => 'Size',
								'reset'      => true,
								'min'        => 30,
								'max'        => 200,
								'units'      => kalium_get_default_units( 'box-size' ),
								'dependency' => [
									'badge_style' => [ 'circle', 'triangle' ],
								],
							]
						),
						Attribute::create(
							'badge_size_fixed',
							[
								'type'         => 'toggle',
								'title'        => 'Fixed size',
								'inline_label' => true,
								'dependency'   => [
									'badge_style' => 'circle',
								],
							]
						),
						Attribute::create(
							'circle_radius',
							[
								'type'         => 'range',
								'title'        => 'Border radius',
								'reset'        => true,
								'min'          => 0,
								'max'          => 100,
								'units'        => kalium_get_default_units( 'box-size' ),
								'default_unit' => '%',
								'dependency'   => [
									'badge_style' => 'circle',
								],
							]
						),
						Attribute::create(
							'triangle_rotation',
							[
								'type'         => 'select',
								'title'        => 'Triangle orientation',
								'inline_label' => true,
								'choices'      => [
									'0deg'   => 'Top Right',
									'270deg' => 'Top Left',
									'90deg'  => 'Bottom Right',
									'180deg' => 'Bottom Left',
								],
								'dependency'   => [
									'badge_style' => 'triangle',
								],
							]
						),
					]
				),
			]
		);
	}

	/**
	 * Generate styles.
	 */
	public function generate_styles() {
		parent::generate_styles();

		// Font size and padding is applied to child element
		$this->remove_style(
			'&',
			[
				'padding-top',
				'padding-right',
				'padding-bottom',
				'padding-left',
				'font-size',
			]
		);

		// Main style
		$this->add_style(
			[
				'.product-badge' => [
					kalium_css_var_name( 'product-badge-size' ) => $this->get_attribute( 'badge_size' ),
					'padding'   => [
						'value'  => $this->get_attribute( 'padding' ),
						'parser' => 'length',
					],
					'font-size' => $this->get_attribute( 'font_size' ),
				],
			]
		);

		switch ( $this->get_attribute_value( 'badge_style' ) ) {

			// Circle badge style
			case 'circle':
				$this->add_style(
					[
						'.product-badge' => [
							'border-radius' => $this->get_attribute( 'circle_radius' ),
						],
					]
				);

				// Fixed size
				if ( $this->get_attribute_value( 'badge_size_fixed' ) ) {
					$this->add_style(
						[
							'.product-badge__text' => [
								kalium_css_var_name( 'product-badge-size-fixed' ) => kalium_css_var_reference( 'product-badge-size' ),
							],
						]
					);
				}
				break;

			// Triangle badge style
			case 'triangle':
				// Padding is applied to .product-badge__text
				$this->remove_style(
					'.product-badge',
					[
						'padding-top',
						'padding-right',
						'padding-bottom',
						'padding-left',
					]
				);

				$this->add_style(
					[
						'.product-badge__text' => [
							'padding' => [
								'value'  => $this->get_attribute( 'padding' ),
								'parser' => 'length',
							],
						],
					]
				);

				// When parent is WooCommerce Product Images element and has triangle style
				if ( $this->get_parent() && 'woocommerce_product_images' === $this->get_parent()->get_name() ) {
					$this->add_style(
						[
							'&' => [
								kalium_css_var_name( 'child-offset' ) => '0',
							],
						]
					);
				}
				break;
		}
	}
}

// Register element
\Laborator_Builder::register_element_type( WooCommerce_Product_Sale_Badge::class );
