<?php
/**
 * Laborator Builder.
 *
 * CSS Value object.
 */

namespace Laborator_Builder;

class CSS_Value {

	/**
	 * Prop.
	 *
	 * @var string
	 */
	public $prop;

	/**
	 * Value.
	 *
	 * @var Value
	 */
	public $value;

	/**
	 * Parser.
	 *
	 * @var string
	 */
	public $parser;

	/**
	 * Options.
	 *
	 * @var array
	 */
	public $options = [];

	/**
	 * Constructor.
	 *
	 * @param string $prop
	 * @param mixed  $value
	 */
	public function __construct( $prop, $value ) {
		$this->prop = $prop;

		if ( is_array( $value ) ) {
			$class_vars = array_keys( get_object_vars( $this ) );

			foreach ( $class_vars as $var ) {
				if ( isset( $value[ $var ] ) ) {
					$this->{$var} = $value[ $var ];
				}
			}
		} else {
			$this->value = $value;
		}

		// Convert value to Value container object
		if ( $this->value instanceof Attribute ) {
			$this->value = $this->value->get_value_object();
		} elseif ( is_string( $this->value ) || is_numeric( $this->value ) ) {
			$this->value = new Value( $value );
		} elseif ( ! ( $this->value instanceof Value ) ) {
			$this->value = null;
		}

		// Parse value automatically
		$this->parse();
	}

	/**
	 * Get option.
	 *
	 * @param string $option_name
	 * @param mixed  $default
	 *
	 * @return mixed
	 */
	public function get_option( $option_name, $default = null ) {
		if ( isset( $this->options[ $option_name ] ) ) {
			return $this->options[ $option_name ];
		}

		return $default;
	}

	/**
	 * Parse value.
	 */
	public function parse() {
		$parser = $this->parser;
		$value  = $this->value;

		// Parse length
		if ( 'length' === $parser ) {
			$parsed = $this->parse_length( $value );
		}
		// Parse length-inline
		elseif ( 'length-inline' === $parser ) {
			$parsed = $this->parse_length_inline( $value );
		}
		// Parser function
		elseif ( is_callable( $parser ) ) {
			$parsed = $parser( $value );
		}
		// Legacy unit value handle
		elseif ( $value ) {
			$value->map(
				function ( $value ) {
					if ( is_array( $value ) && array_key_exists( 'size', $value ) && array_key_exists( 'size', $value ) ) {
						return kalium_get_multi_numeric_prop( $value, 'size' );
					}

					return $value;
				},
				true
			);
		}

		if ( isset( $parsed ) ) {

			// Convert values to Value container objects
			if ( is_array( $parsed ) ) {
				foreach ( $parsed as $key => $value ) {
					if ( ! ( $value instanceof Value ) ) {
						$parsed[ $key ] = new Value( $value );
					}
				}
			}

			// Store parsed value
			$this->value = $parsed;
		}
	}

	/**
	 * Get all CSS props.
	 *
	 * @return self[]
	 */
	public function get_all_props() {
		$props = $this->value;

		if ( ! is_array( $props ) ) {
			$props = [
				$this->prop => $props,
			];
		}

		return array_map(
			function ( $prop, $value ) {
				$self = clone $this;

				$self->prop  = $prop;
				$self->value = $value;

				return $self;
			},
			array_keys( $props ),
			$props
		);
	}

	/**
	 * Parse length.
	 *
	 * @param Value $value
	 *
	 * @return array
	 */
	private function parse_length( $value ) {
		$props  = [ 'top', 'right', 'bottom', 'left' ];
		$length = [];

		$props_mapping = [
			'inset'         => array_combine( $props, $props ),
			'border-radius' => [
				'top'    => 'border-top-left-radius',
				'right'  => 'border-top-right-radius',
				'bottom' => 'border-bottom-right-radius',
				'left'   => 'border-bottom-left-radius',
			],
		];

		// Parse props
		foreach ( $props as $prop ) {
			$prop_name = $props_mapping[ $this->prop ][ $prop ] ?? "{$this->prop}-{$prop}";

			$length[ $prop_name ] = $value->map(
				function ( $value ) use ( $prop ) {
					return kalium_get_multi_numeric_prop( $value, $prop );
				}
			);
		}

		return $length;
	}

	/**
	 * Parse multi-numeric value into one line string.
	 *
	 * @param Value $value
	 *
	 * @return Value
	 */
	public function parse_length_inline( $value ) {
		return $value->map(
			function ( $multinum ) {
				if ( is_array( $multinum ) ) {
					$length_arr  = [];
					$unset_props = [ 'unit', 'link' ];
					$unit        = $multinum['unit'] ?? null;

					foreach ( $unset_props as $unset_prop ) {
						unset( $multinum[ $unset_prop ] );
					}

					foreach ( $multinum as $entry_value ) {
						$length_arr[] = $entry_value . ( $unit ?: '' );
					}

					return implode( ' ', $length_arr );
				}

				return null;
			}
		);
	}
}
