<?php
/**
 * Laborator Builder.
 *
 * Value container.
 */

namespace Laborator_Builder;

class Value {

	/**
	 * Default scope.
	 *
	 * @const string
	 */
	const DEFAULT_SCOPE = '_default-scope';

	/**
	 * Scopes.
	 *
	 * @var array
	 */
	protected $scopes = [];

	/**
	 * Current scope.
	 *
	 * @var string
	 */
	protected $current_scope;

	/**
	 * Value container.
	 *
	 * @var array
	 */
	protected $value_container = [];

	/**
	 * Constructor.
	 *
	 * @param mixed $value
	 */
	public function __construct( $value = null ) {
		$this->add_scope( self::DEFAULT_SCOPE );

		if ( ! is_null( $value ) ) {
			$this->set_value( $value );
		}
	}

	/**
	 * Get specific scope or current scope.
	 *
	 * @param string $name
	 *
	 * @return string
	 */
	public function get_scope( $name ) {
		if ( isset( $this->scopes[ $name ] ) ) {
			return $name;
		}

		if ( $this->current_scope ) {
			return $this->current_scope;
		}

		return self::DEFAULT_SCOPE;
	}

	/**
	 * Set current scope.
	 *
	 * @param string $name
	 */
	public function set_scope( $name ) {
		$this->current_scope = $this->get_scope( $name );
	}

	/**
	 * Add/register scope.
	 *
	 * @param string $name
	 * @param array  $options
	 */
	public function add_scope( $name, $options = [] ) {
		$options = wp_parse_args(
			$options,
			[

				/**
				 * Inherit default value from other scope.
				 *
				 * @var string
				 */
				'inherit' => self::DEFAULT_SCOPE,
			]
		);

		// If its default scope
		if ( self::DEFAULT_SCOPE === $name ) {
			$options['inherit'] = null;
		}

		// Add scope
		$this->scopes[ $name ] = $options;
	}

	/**
	 * Get value.
	 *
	 * @param string $scope
	 *
	 * @return mixed
	 */
	public function get_value( $scope ) {
		$scope = $this->get_scope( $scope );

		return $this->retrieve_value( $scope );
	}

	/**
	 * Get all values from value container.
	 *
	 * @return array
	 */
	public function get_all_values() {
		$value = [];

		foreach ( $this->scopes as $scope => $scope_options ) {
			$value[ $scope ] = isset( $this->value_container[ $scope ] ) ? $this->value_container[ $scope ] : null;
		}

		return $value;
	}

	/**
	 * Set value for current or specific scope.
	 *
	 * @param mixed  $value
	 * @param string $scope
	 * @param bool   $set_current_scope
	 */
	public function set_value( $value, $scope = null, $set_current_scope = false ) {
		$scope = $this->get_scope( $scope );

		// Set new value
		$this->value_container[ $scope ] = $value;

		// Set current scope
		if ( $set_current_scope ) {
			$this->set_scope( $scope );
		}
	}

	/**
	 * Get current value.
	 *
	 * @param mixed  $set_value
	 * @param string $scope
	 * @param bool   $set_current_scope
	 *
	 * @return mixed|void
	 */
	public function value( $set_value = null, $scope = null, $set_current_scope = true ) {
		if ( ! is_null( $set_value ) ) {
			$this->set_value( $set_value, $scope, $set_current_scope );

			return;
		}

		return $this->get_value( $scope );
	}

	/**
	 * Assign value.
	 *
	 * @param mixed|array $value
	 */
	public function assign_value( $value ) {
		$is_scoped = false;

		if ( is_array( $value ) ) {
			foreach ( $value as $scope_name => $scope_value ) {
				if ( isset( $this->scopes[ $scope_name ] ) ) {
					$is_scoped = true;
				}
			}
		}

		// Set values by scope
		if ( $is_scoped ) {
			foreach ( $value as $scope_name => $scope_value ) {
				if ( isset( $this->scopes[ $scope_name ] ) ) {
					$this->set_value( $scope_value, $scope_name );
				}
			}
		}  // Set value in the current scope only
		else {
			$this->set_value( $value );
		}
	}

	/**
	 * Retrieve value recursively.
	 *
	 * @param string $scope_name
	 *
	 * @return mixed|null
	 */
	private function retrieve_value( $scope_name ) {
		$scope_name = $this->get_scope( $scope_name );

		// Value exists
		if ( isset( $this->value_container[ $scope_name ] ) ) {
			return $this->value_container[ $scope_name ];
		} else { // Check value inheritance
			$scope = $this->scopes[ $scope_name ] ?? null;

			if ( ! empty( $scope['inherit'] ) && $scope_name !== $scope['inherit'] ) {
				return $this->retrieve_value( $scope['inherit'] );
			}
		}

		return null;
	}

	/**
	 * Map value.
	 *
	 * @param callable $mapper
	 * @param bool     $replace_value
	 *
	 * @return self
	 */
	public function map( $mapper, $replace_value = false ) {
		$value_object = $replace_value ? $this : clone $this;

		if ( is_callable( $mapper ) ) {
			foreach ( $value_object->get_all_values() as $scope => $value ) {
				$value_object->set_value( call_user_func( $mapper, $value, $scope ), $scope );
			}
		}

		return $value_object;
	}
}
