<?php
/**
 * TypoLab - ultimate font management library.
 *
 * Custom selector entry.
 *
 * @author Laborator
 * @link   https://laborator.co
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class TypoLab_Font_Custom_Selector extends TypoLab_Font_Selector {

	/**
	 * Selector type.
	 *
	 * @var string
	 */
	public $type = 'custom-selector';

	/**
	 * Selectors.
	 *
	 * @var array
	 */
	public $selectors = [];

	/**
	 * Text transform.
	 *
	 * @var TypoLab_Responsive_Value
	 */
	public $text_transform;

	/**
	 * Font size.
	 *
	 * @var TypoLab_Responsive_Value
	 */
	public $font_size;

	/**
	 * Line height.
	 *
	 * @var TypoLab_Responsive_Value
	 */
	public $line_height;

	/**
	 * Letter spacing.
	 *
	 * @var TypoLab_Responsive_Value
	 */
	public $letter_spacing;

	/**
	 * Constructor.
	 *
	 * @param array $args
	 */
	public function __construct( $args = [] ) {
		parent::__construct( $args );

		// Selectors
		$this->selectors = kalium_get_array_key( $args, 'selectors', [] );

		// Remove backslashes from selectors
		if ( is_array( $this->selectors ) ) {
			$this->selectors = array_map( 'stripslashes', $this->selectors );
		}

		// Text transform
		$this->text_transform = new TypoLab_Responsive_Value( kalium_get_array_key( $args, 'text_transform' ) );

		// Font size
		$this->font_size = new TypoLab_Responsive_Value( kalium_get_array_key( $args, 'font_size' ) );

		// Line height
		$this->line_height = new TypoLab_Responsive_Value( kalium_get_array_key( $args, 'line_height' ) );

		// Letter spacing
		$this->letter_spacing = new TypoLab_Responsive_Value( kalium_get_array_key( $args, 'letter_spacing' ) );
	}

	/**
	 * Get selectors.
	 *
	 * @return array
	 */
	public function get_selectors() {
		$selectors = [];

		foreach ( $this->selectors as $selector ) {
			if ( preg_match( '/\:(?<id>.*?):/', $selector, $matches ) ) {
				$predefined_selector = TypoLab_Data::get_predefined_selector( $matches['id'] );

				if ( ! empty( $predefined_selector['selectors'] ) ) {
					$selectors = array_merge( $selectors, $predefined_selector['selectors'] );
				}
			} else {
				$selectors[] = $selector;
			}
		}

		return array_unique( apply_filters( 'typolab_font_selectors', $selectors ) );
	}

	/**
	 * Should add CSS vars props.
	 *
	 * @return bool
	 */
	public function should_add_css_vars() {
		foreach ( $this->selectors as $selector ) {
			if ( preg_match( '/\:(?<id>.*?):/', $selector, $matches ) ) {
				$predefined_selector = TypoLab_Data::get_predefined_selector( $matches['id'] );

				if ( $predefined_selector && ! empty( $predefined_selector['css_vars'] ) ) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Get props values.
	 *
	 * @param TypoLab_Font $font
	 *
	 * @return TypoLab_Responsive_Value[]
	 */
	public function get_css_props( $font ) {
		$variant   = $font->get_variant( $this->get_variant() );
		$css_props = $variant ? $variant->get_css_props() : [];

		$css_props = array_merge(
			$css_props,
			[
				'text-transform' => $this->text_transform,
				'font-size'      => $this->font_size,
				'line-height'    => $this->line_height,
				'letter-spacing' => $this->letter_spacing,
			]
		);

		// Adds typography props as CSS vars
		if ( $this->should_add_css_vars() ) {
			$css_vars_props = [];

			foreach ( array_keys( $css_props ) as $prop_name ) {
				$css_vars_props[ kalium_css_var_name( $prop_name ) ] = $css_props[ $prop_name ];
			}

			if ( 1 === count( $this->selectors ) ) {
				$css_props = $css_vars_props;
			} else {
				$css_props = array_merge( $css_props, $css_vars_props );
			}
		}

		return $css_props;
	}
}
