<?php
/**
 * Kalium WordPress Theme
 *
 * Version 4 upgrade tasks.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

return [
	/**
	 * Backup current SMOF options.
	 *
	 * @return bool
	 */
	static function () {
		kalium_backup_smof_options();

		return true;
	},

	/**
	 * Migrate SMOF options.
	 *
	 * @return bool
	 */
	static function () {
		$smof_migrations = [

			/**
			 * Branding.
			 */

			// Site title
			[
				'id'        => 'logo_text',
				'id_new'    => 'option:blogname',
				'overwrite' => true,
			],

			// Brand logo
			'use_uploaded_logo',
			[
				'id'     => 'custom_logo_image',
				'filter' => 'kalium_smof_media_type_option',
			],
			[
				'id'     => 'custom_logo_max_width',
				'filter' => function ( $value, $smof_values ) {
					$responsive = [
						'desktop' => $value,
					];

					if ( isset( $smof_values['custom_logo_mobile_max_width']['value'] ) ) {
						$mobile_max_width = $smof_values['custom_logo_mobile_max_width']['value'];

						if ( is_numeric( $mobile_max_width ) ) {
							$responsive['mobile'] = $mobile_max_width;
						}
					}

					return $responsive;
				},
			],

			// Favicon
			[
				'id'        => 'favicon_image',
				'id_new'    => 'option:site_icon',
				'overwrite' => true,
				'filter'    => function ( $value ) {
					if ( wp_get_attachment_image_src( $value ) ) {
						return $value;
					}
				},
			],

			// Tab Color
			'google_theme_color',

			/**
			 * Performance.
			 */

			'performance_enqueue_style_css',
			'performance_jquery_migrate',
			'performance_wp_emoji',
			'performance_wp_embed',
			'performance_jpeg_quality',

			/**
			 * Theme styling.
			 */

			// Theme colors
			[
				'callback' => function ( $smof_values, $args ) {
					$theme_colors_option = 'theme_colors';

					if ( $smof_values['use_custom_skin']['value'] ) {
						$theme_colors = kalium_get_theme_option( $theme_colors_option );

						if ( $theme_colors === kalium_get_default_theme_option( $theme_colors_option ) ) {
							$color_mapping = [
								'custom_skin_link_color' => 'color-1',
								'custom_skin_paragraph_color' => 'color-3',
								'custom_skin_headings_color' => 'color-4',
								'custom_skin_borders_color' => 'color-5',
								'custom_skin_footer_bg_color' => 'color-6',
								'custom_skin_bg_color'   => 'color-7',
							];

							$changed_colors = [];

							foreach ( $color_mapping as $legacy_color_prop => $theme_color_prop ) {
								$smof_option = kalium_get_array_key( $smof_values, $legacy_color_prop );

								if ( $smof_option ) {
									$theme_colors[ $theme_color_prop ]   = $smof_option['value'];
									$changed_colors[ $theme_color_prop ] = true;
								}
							}

							if ( ! empty( $changed_colors ) ) {
								if ( ! empty( $changed_colors['color-1'] ) ) {
									$theme_colors['color-2'] = kalium()->style->color_shift( $theme_colors['color-1'], 10 );
								}

								if ( ! $args['dry'] ) {
									set_theme_mod( $theme_colors_option, $theme_colors );
								}

								return true;
							}
						}

						return true;
					}

					return false;
				},
			],

			// Theme Borders
			'theme_borders',
			'theme_borders_color',
			[
				'id'     => 'theme_borders_thickness',
				'filter' => 'kalium_smof_filter_add_px_unit',
			],
			'theme_borders_animation',
			'theme_borders_animation_duration',
			'theme_borders_animation_delay',

			// Container width
			'grid_container_fullwidth',
			[
				'id'     => 'grid_container_width',
				'id_new' => 'grid_container_width_custom',
				'filter' => function ( $value ) {
					if ( kalium_can_override_theme_option( 'grid_container_width' ) ) {
						set_theme_mod( 'grid_container_width', is_numeric( $value ) ? 'custom' : 'medium' );
					}

					return $value;
				},
			],

			/**
			 * Social networks.
			 */

			[
				'id'     => 'social_order',
				'id_new' => 'social_icons',
				'filter' => 'kalium_smof_filter_social_networks',
			],

			/**
			 * Coming soon / maintenance mode.
			 */

			[
				'id'             => 'maintenance_mode',
				'ignore_default' => false,
				'filter'         => function ( $value, $smof_values ) {
					// Migrate maintenance mode
					if ( $value ) {
						kalium_migrate_smof_batch(
							[
								'maintenance_mode_title',
								'maintenance_mode_description',
								'maintenance_mode_custom_bg',
								'maintenance_mode_custom_bg_color',
								[
									'id'     => 'maintenance_mode_custom_txt_color',
									'id_new' => 'maintenance_mode_custom_text_color',
								],
								'maintenance_mode_custom_bg_size',
								[
									'id'     => 'maintenance_mode_custom_bg_id',
									'filter' => 'kalium_smof_media_type_option',
								],
							]
						);
					}
					// Migrate coming soon mode
					elseif ( kalium_get_array_key( $smof_values, 'coming_soon_mode/value' ) ) {
						$value = true;

						kalium_migrate_smof_batch(
							[
								[
									'id'     => 'coming_soon_mode_title',
									'id_new' => 'maintenance_mode_title',
								],
								[
									'id'     => 'coming_soon_mode_description',
									'id_new' => 'maintenance_mode_description',
								],
								[
									'id'     => 'coming_soon_mode_custom_bg',
									'id_new' => 'maintenance_mode_custom_bg',
								],
								[
									'id'     => 'coming_soon_mode_custom_bg_id',
									'id_new' => 'maintenance_mode_custom_bg_id',
									'filter' => 'kalium_smof_media_type_option',
								],
								[
									'id'     => 'coming_soon_mode_custom_bg_size',
									'id_new' => 'maintenance_mode_custom_bg_size',
								],
								[
									'id'     => 'coming_soon_mode_custom_bg_color',
									'id_new' => 'maintenance_mode_custom_bg_color',
								],
								[
									'id'     => 'coming_soon_mode_custom_txt_color',
									'id_new' => 'maintenance_mode_custom_text_color',
								],
								[
									'id'     => 'coming_soon_mode_use_uploaded_logo',
									'id_new' => 'maintenance_mode_use_uploaded_logo',
								],
								[
									'id'     => 'coming_soon_mode_custom_logo_image',
									'id_new' => 'maintenance_mode_custom_logo_image',
									'filter' => 'kalium_smof_media_type_option',
								],
								[
									'id'     => 'coming_soon_mode_custom_logo_max_width',
									'id_new' => 'maintenance_mode_custom_logo_max_width',
								],
								[
									'id'     => 'coming_soon_mode_countdown',
									'id_new' => 'maintenance_mode_countdown',
								],
								[
									'id'     => 'coming_soon_mode_social_networks',
									'id_new' => 'maintenance_mode_social_networks',
								],
								[
									'callback' => function ( $smof_values ) {
										$coming_soon_mode_date = kalium_get_array_key( $smof_values, 'coming_soon_mode_date/value' );

										if ( $coming_soon_mode_date ) {
											$coming_soon_mode_date = explode( ' ', $coming_soon_mode_date );

											if ( 2 === count( $coming_soon_mode_date ) ) {
												set_theme_mod( 'maintenance_mode_countdown_date', $coming_soon_mode_date[0] );
												set_theme_mod( 'maintenance_mode_countdown_time', preg_replace( '/(\d+:\d+):\d+/', '${1}', $coming_soon_mode_date[1] ) );
											}
										}
									},
								],
							]
						);
					}

					return $value;
				},
			],

			/**
			 * Footer.
			 */

			[
				'id'     => 'footer_visibility',
				'id_new' => 'footer_enabled',
			],
			[
				'id'     => 'footer_fixed',
				'id_new' => 'footer_type',
				'filter' => function ( $value ) {
					// Fixed type
					if ( in_array( $value, [ 'fixed', 'fixed-fade', 'fixed-slide' ] ) ) {
						$animations = [
							'fixed-fade'  => 'fade',
							'fixed-slide' => 'slide',
						];
						if ( isset( $animations[ $value ] ) ) {
							set_theme_mod( 'footer_fixed_animation', $animations[ $value ] );
						}

						return 'fixed';
					}

					return 'normal';
				},
			],
			[
				'id'     => 'footer_style',
				'id_new' => 'theme_colors_footer',
				'filter' => function ( $value ) {
					if ( 'inverted' === $value ) {
						return [
							'background' => '#333333',
							'text'       => '#bbbbbb',
							'headings'   => '#ffffff',
							'link'       => '#ffffff',
							'link-hover' => '#ffffff',
						];
					}
				},
			],
			[
				'callback' => function ( $smof_values ) {
					$is_centered     = 'vertical' === kalium_get_array_key( $smof_values, 'footer_bottom_style/value' );
					$is_full_width   = kalium_get_array_key( $smof_values, 'footer_fullwidth/value' );
					$sidebar_widgets = wp_get_sidebars_widgets();
					$block_widget    = get_option( 'widget_block' );
					$footer_widgets  = $sidebar_widgets['footer_sidebar'] ?? [];

					$footer_content = [
						'elements' => [],
					];

					$widget_area = [
						'footer_sidebar',
						'footer_sidebar_2',
						'footer_sidebar_3',
						'footer_sidebar_4',
						'footer_sidebar_5',
						'footer_sidebar_6',
					];

					// Copyrights
					if ( kalium_get_array_key( $smof_values, 'footer_bottom_visible/value' ) ) {
						$left_text      = kalium_get_array_key( $smof_values, 'footer_text/value' );
						$right_text     = kalium_get_array_key( $smof_values, 'footer_text_right/value' );
						$has_left_text  = strlen( trim( $left_text ) ) > 0;
						$has_right_text = strlen( trim( $right_text ) ) > 0;

						$copyrights_row = [
							'name'       => 'footer_container_row',
							'attributes' => [
								[
									'name'  => 'fullwidth',
									'value' => $is_full_width,
								],
							],
							'visible'    => true,
							'elements'   => [],
						];

						// Centered layout
						if ( $is_centered ) {
							$copyrights_row['attributes'][] = [
								'name'  => 'gutter_y',
								'value' => [
									'desktop' => '20px',
									'tablet'  => null,
									'mobile'  => null,
								],
							];
						}

						if ( $has_left_text ) {
							$sidebar_id = array_shift( $widget_area );

							// Insert as text widget
							$index     = count( $block_widget ) + 1;
							$widget_id = 'block-' . $index;

							$block_widget[ $index ] = [
								'content' => sprintf( '<!-- wp:paragraph -->%s<!-- /wp:paragraph -->', wpautop( $left_text ) ),
							];

							$sidebar_widgets[ $sidebar_id ] = [
								$widget_id,
							];

							// Add column in the builder
							$left_column = [
								'name'       => 'column',
								'attributes' => [
									[
										'name'  => 'columns',
										'value' => [
											'desktop' => $is_centered ? 12 : ( $has_right_text ? 6 : 12 ),
											'mobile'  => 12,
										],
									],
									[
										'name'  => 'horizontal_align',
										'value' => [
											'desktop' => $is_centered ? 'center' : 'start',
											'tablet'  => null,
											'mobile'  => 'center',
										],
									],
								],
								'visible'    => true,
								'elements'   => [
									[
										'name'       => 'footer_sidebar_' . $sidebar_id,
										'attributes' => [],
										'visible'    => true,
									],
								],
							];

							$copyrights_row['elements'][] = $left_column;
						}

						if ( $has_right_text ) {
							$sidebar_id = array_shift( $widget_area );

							// Insert as text widget
							$index     = count( $block_widget ) + 1;
							$widget_id = 'block-' . $index;

							$block_widget[ $index ] = [
								'content' => sprintf( '<!-- wp:paragraph -->%s<!-- /wp:paragraph -->', wpautop( $right_text ) ),
							];

							$sidebar_widgets[ $sidebar_id ] = [
								$widget_id,
							];

							// Right column
							$right_column = [
								'name'       => 'column',
								'attributes' => [
									[
										'name'  => 'columns',
										'value' => [
											'desktop' => $is_centered ? 12 : ( $has_left_text ? 6 : 12 ),
											'mobile'  => 12,
										],
									],
									[
										'name'  => 'horizontal_align',
										'value' => [
											'desktop' => $is_centered ? 'center' : ( $has_left_text ? 'end' : 'start' ),
											'tablet'  => null,
											'mobile'  => 'center',
										],
									],
								],
								'visible'    => true,
								'elements'   => [
									[
										'name'       => 'footer_sidebar_' . $sidebar_id,
										'attributes' => [],
										'visible'    => true,
									],
								],
							];

							// Set first column on mobile
							if ( $has_left_text ) {
								$right_column['attributes'][] = [
									'name'  => 'flex_order',
									'value' => [
										'desktop' => null,
										'tablet'  => null,
										'mobile'  => -1,
									],
								];
							}

							$copyrights_row['elements'][] = $right_column;
						}

						if ( ! empty( $copyrights_row['elements'] ) ) {
							$footer_content['elements'][] = $copyrights_row;
						}
					}

					// Widgets
					if ( kalium_get_array_key( $smof_values, 'footer_widgets/value' ) ) {
						$sidebar_id      = array_shift( $widget_area );
						$widgets_columns = kalium_get_array_key( $smof_values, 'footer_widgets_columns/value' );

						$widgets_columns_map = [
							'one'   => 1,
							'two'   => 2,
							'three' => 3,
							'four'  => 4,
							'five'  => 5,
							'six'   => 6,
						];

						$widgets_per_row = $widgets_columns_map[ $widgets_columns ] ?? 3;

						array_unshift(
							$footer_content['elements'],
							[
								'name'       => 'footer_container_row',
								'attributes' => [
									[
										'name'  => 'fullwidth',
										'value' => $is_full_width,
									],
									[
										'name'  => 'text_align',
										'value' => [
											'desktop' => $is_centered ? 'center' : null,
											'tablet'  => null,
											'mobile'  => null,
										],
									],
								],
								'visible'    => true,
								'elements'   => [
									[
										'name'       => 'column',
										'attributes' => [],
										'visible'    => true,
										'elements'   => [
											[
												'name'    => 'footer_sidebar_' . $sidebar_id,
												'attributes' => [
													[
														'name' => 'widgets_per_row',
														'value' => [
															'desktop' => $widgets_per_row,
															'tablet'  => ceil( $widgets_per_row / 2 ),
															'mobile'  => 1,
														],
													],
												],
												'visible' => true,
											],
										],
									],
								],
							]
						);

						// Move widgets
						$sidebar_widgets[ $sidebar_id ] = $footer_widgets;
					}

					// Update content builder option
					if ( kalium_can_override_theme_option( 'footer_content' ) ) {
						set_theme_mod( 'footer_content', $footer_content );

						// Update sidebars
						update_option( 'widget_block', $block_widget );
						wp_set_sidebars_widgets( $sidebar_widgets );
					}
				},
			],

			/**
			 * Other settings.
			 */

			// Scroll to Top
			[
				'id'     => 'footer_go_to_top',
				'id_new' => 'scroll_to_top',
			],
			[
				'id'     => 'footer_go_to_top_activate',
				'id_new' => 'scroll_to_top_activate_footer',
				'filter' => function ( $value ) {
					if ( preg_match( '/^\d+(\.\d+)?(\%|px)$/i', $value ) ) {
						set_theme_mod( 'scroll_to_top_activate', strtolower( $value ) );
						return false;
					}

					return 'footer' === $value;
				},
			],
			[
				'id'     => 'footer_go_to_top_type',
				'id_new' => 'scroll_to_top_border_radius',
				'filter' => function ( $value ) {
					if ( 'square' === $value ) {
						return [
							'top'    => 0,
							'right'  => 0,
							'bottom' => 0,
							'left'   => 0,
							'link'   => true,
							'unit'   => 'px',
						];
					}
				},
			],
			[
				'id'     => 'footer_go_to_top_position',
				'id_new' => 'scroll_to_top_position',
			],

			// Sidebar
			[
				'id'     => 'sidebar_skin',
				'id_new' => 'sidebar_type',
				'filter' => function ( $value ) {
					if ( in_array( $value, [ 'bordered', 'background-fill' ] ) ) {
						// Bordered
						if ( 'bordered' === $value ) {
							set_theme_mod(
								'sidebar_border',
								[
									'color' => kalium_color_reference( 'theme_colors_border' ),
									'style' => 'solid',
									'width' => '1px',
								]
							);
						}

						return 'styled';
					}

					return 'plain';
				},
			],

			// Google Maps API key
			'google_maps_api',

			// Custom JavaScript
			'user_custom_js_head',
			'user_custom_js',

			// Search
			[
				'id'     => 'search_thumbnails',
				'filter' => function ( $value ) {
					// Rounded thumbnails
					if ( $value && kalium_can_override_theme_option( 'search_thumbnail_border_radius' ) ) {
						set_theme_mod(
							'search_thumbnail_border_radius',
							[
								'desktop' => [
									'top'    => 50,
									'right'  => 50,
									'bottom' => 50,
									'left'   => 50,
									'link'   => true,
									'unit'   => '%',
								],
							]
						);
					}

					return $value;
				},
			],
			[
				'id'     => 'exclude_search_post_types',
				'filter' => function ( $value ) {
					if ( is_array( $value ) ) {
						$selected = array_keys( array_filter( $value ) );

						return $selected;
					}
				},
			],

			// Breadcrumbs
			'breadcrumbs',
			[
				'id'         => 'breadcrumb_border_type',
				'map_values' => [
					''                  => 'no-border',
					'border'            => 'border',
					'border-horizontal' => 'border-horizontal',
				],
			],
			'breadcrumb_alignment',
			'breadcrumb_visibility_homepage',
			'breadcrumb_visibility_blog',
			'breadcrumb_visibility_portfolio',
			'breadcrumb_visibility_search',
			'breadcrumb_visibility_404',
			[
				'callback' => function ( $smof_values ) {
					if ( kalium_can_override_theme_option( 'breadcrumb_responsive' ) ) {
						$devices = [];

						if ( kalium_get_array_key( $smof_values, 'breadcrumb_support_desktop/value' ) ) {
							$devices[] = 'desktop';
						}

						if ( kalium_get_array_key( $smof_values, 'breadcrumb_support_tablet/value' ) ) {
							$devices[] = 'tablet';
						}

						if ( kalium_get_array_key( $smof_values, 'breadcrumb_support_mobile/value' ) ) {
							$devices[] = 'mobile';
						}

						set_theme_mod( 'breadcrumb_responsive', $devices );

						return true;
					}
				},
			],
			'breadcrumb_background_color',
			'breadcrumb_text_color',
			'breadcrumb_border_color',
			[
				'id'     => 'breadcrumb_border_radius',
				'filter' => function ( $value ) {
					if ( is_numeric( $value ) ) {
						return [
							'top'    => $value,
							'right'  => $value,
							'bottom' => $value,
							'left'   => $value,
							'unit'   => 'px',
							'link'   => true,
						];
					}
				},
			],
			[
				'callback' => function ( $smof_values ) {
					if ( kalium_can_override_theme_option( 'breadcrumb_margin' ) ) {
						set_theme_mod(
							'breadcrumb_margin',
							[
								'top'    => kalium_get_array_key( $smof_values, 'breadcrumb_margin_top/value' ),
								'bottom' => kalium_get_array_key( $smof_values, 'breadcrumb_margin_bottom/value' ),
								'unit'   => 'px',
							]
						);

						return true;
					}
				},
			],

			// Media settings
			'videojs_player_skin',
			[
				'id'     => 'videojs_player_autoplay',
				'id_new' => 'media_autoplay',
			],
			[
				'id'     => 'videojs_player_loop',
				'id_new' => 'media_loop',
				'filter' => 'kalium_validate_boolean',
			],
			[
				'id'         => 'youtube_player',
				'id_new'     => 'videojs_youtube_player',
				'map_values' => [
					'videojs' => 'no',
					'native'  => 'yes',
				],
				'filter'     => 'kalium_validate_boolean',
			],

			// Image placeholder
			[
				'id'         => 'image_loading_placeholder_type',
				'map_values' => [
					'static-color' => 'plain',
					'preselected'  => 'loading-animation',
					'custom'       => 'custom-icon',
				],
			],
			[
				'callback' => function ( $smof_values ) {
					if ( kalium_can_override_theme_option( 'image_loading_placeholder_bg_type' ) ) {
						$type  = 'color';
						$color = kalium_get_array_key( $smof_values, 'image_loading_placeholder_bg/value' );

						// Gradient type
						if ( kalium_get_array_key( $smof_values, 'image_loading_placeholder_use_gradient/value' ) ) {
							$type = 'gradient';
						}

						// Dominant color
						if ( kalium_get_array_key( $smof_values, 'image_loading_placeholder_dominant_color/value' ) ) {
							$type = 'dominant-color';
						}

						// Background type
						switch ( $type ) {
							case 'color':
								if ( $color ) {
									set_theme_mod( 'image_loading_placeholder_bg_color', $color );
								}
								break;

							case 'gradient':
								$color_2 = kalium_get_array_key( $smof_values, 'image_loading_placeholder_gradient_bg/value' );

								if ( $color && $color_2 ) {
									set_theme_mod(
										'image_loading_placeholder_bg_gradient',
										sprintf( 'linear-gradient(to bottom, %s, %s)', esc_attr( $color ), esc_attr( $color_2 ) )
									);
								}
								break;
						}

						// Type options
						switch ( kalium_get_array_key( $smof_values, 'image_loading_placeholder_type/value' ) ) {

							// Loading animation
							case 'preselected':
								// Loader
								$loader = kalium_get_array_key( $smof_values, 'image_loading_placeholder_preselected_loader/value' );

								$available_loaders = [
									'ball-scale'       => [
										'circle-pulse',
										'ball-scale',
										'ball-scale-multiple',
										'ball-scale-ripple',
										'ball-scale-ripple-multiple',
										'ball-scale-random',
										'ball-clip-rotate-pulse',
										'ball-clip-rotate-multiple',
									],
									'line-scale'       => [
										'line-scale',
										'line-scale-party',
										'line-scale-pulse-out',
										'line-scale-pulse-out-rapid',
									],
									'ball-pulse'       => [
										'ball-pulse-sync',
										'ball-pulse',
										'ball-beat',
										'ball-rotate',
									],
									'semi-circle-spin' => [
										'semi-circle-spin',
									],
								];

								foreach ( $available_loaders as $loader_id => $loader_variants ) {
									if ( in_array( $loader, $loader_variants ) ) {
										set_theme_mod( 'image_loading_placeholder_loading_animation', $loader_id );
										break;
									}
								}

								// Spacing
								if ( is_numeric( $loader_spacing = kalium_get_array_key( $smof_values, 'image_loading_placeholder_preselected_spacing/value' ) ) ) {
									set_theme_mod( 'image_loading_placeholder_preselected_spacing', $loader_spacing );
								}

								// Spinner color
								if ( $spinner_color = kalium_get_array_key( $smof_values, 'image_loading_placeholder_preselected_loader_color/value' ) ) {
									set_theme_mod( 'image_loading_placeholder_loader_color', $spinner_color );
								}

								// Position
								$position = str_replace( '-', ' ', kalium_get_array_key( $smof_values, 'image_loading_placeholder_preselected_loader_position/value' ) );

								if ( 'center' === $position ) {
									$position = 'center center';
								}

								set_theme_mod( 'image_loading_placeholder_align', $position );
								break;

							// Custom loading icon
							case 'custom':
								kalium_migrate_smof_batch(
									[
										[
											'id'     => 'image_loading_placeholder_custom_image',
											'id_new' => 'image_loading_placeholder_custom_icon',
											'filter' => 'kalium_smof_media_type_option',
										],
										[
											'id'     => 'image_loading_placeholder_custom_image_width',
											'id_new' => 'image_loading_placeholder_size',
										],
										[
											'id'     => 'image_loading_placeholder_custom_loader_position',
											'id_new' => 'image_loading_placeholder_align',
											'filter' => function ( $value ) {
												$value = str_replace( '-', ' ', $value );

												if ( 'center' === $value ) {
													$value = 'center center';
												}

												return $value;
											},
										],
										[
											'id'     => 'image_loading_placeholder_custom_spacing',
											'id_new' => 'image_loading_placeholder_spacing',
										],
									]
								);

								break;
						}

						// Color type
						set_theme_mod( 'image_loading_placeholder_bg_type', $type );
					}
				},
			],

			/**
			 * Blog settings.
			 */

			// Blog template
			[
				'id'         => 'blog_template',
				'id_new'     => 'blog_layout',
				'map_values' => [
					'blog-squared' => 'list',
					'blog-rounded' => 'list',
					'blog-masonry' => 'grid',
				],
			],

			// Masonry mode
			[
				'id'         => 'blog_masonry_layout_mode',
				'id_new'     => 'blog_grid_masonry_mode',
				'map_values' => [
					'packery'  => true,
					'fit-rows' => false,
				],
			],

			// Columns
			[
				'id'     => 'blog_columns',
				'id_new' => 'blog_grid_columns',
				'filter' => function ( $value, $smof_values ) {
					if ( 'blog-masonry' === kalium_get_array_key( $smof_values, 'blog_template/value' ) ) {
						$columns = [
							'_1' => 1,
							'_2' => 2,
							'_3' => 3,
							'_4' => 4,
						];

						return kalium_parse_responsive_value(
							[
								'desktop' => $columns[ $value ] ?? 3,
								'tablet'  => 2,
								'mobile'  => 1,
							]
						);
					}
				},
			],

			// Gap
			[
				'id'     => 'blog_masonry_columns_gap',
				'id_new' => 'blog_grid_columns_gap',
				'filter' => function ( $value, $smof_values ) {
					if ( is_numeric( $value ) && 'blog-masonry' === kalium_get_array_key( $smof_values, 'blog_template/value' ) ) {
						return $value;
					}
				},
			],

			// Bordered style
			[
				'id'     => 'blog_masonry_borders',
				'filter' => function ( $value, $smof_values ) {
					if ( 'yes' === $value && 'blog-masonry' === kalium_get_array_key( $smof_values, 'blog_template/value' ) && kalium_can_override_theme_option( 'blog_thumbnail_boundless' ) ) {
						set_theme_mod( 'blog_thumbnail_boundless', true );
						set_theme_mod(
							'blog_post_style_border',
							[
								'width' => '1px',
								'style' => 'solid',
								'color' => kalium_color_reference( 'theme_colors.color-6' ),
							]
						);
					}
				},
			],

			// Rounded Thumbnails
			[
				'id'     => 'blog_template',
				'filter' => function ( $value, $smof_values ) {
					if ( kalium_can_override_theme_option( 'blog_thumbnail_width' ) && in_array( $value, [ 'blog-squared', 'blog-rounded' ] ) ) {
						$thumb_width = '25%';

						if ( 'hide' !== kalium_get_array_key( $smof_values, 'blog_sidebar_position/value' ) ) {
							$thumb_width = '30%';
						}

						set_theme_mod(
							'blog_thumbnail_width',
							kalium_parse_responsive_value(
								[
									'desktop' => $thumb_width,
									'mobile'  => '100%',
								]
							)
						);
					}

					if ( 'blog-rounded' === $value && kalium_can_override_theme_option( 'blog_thumbnail_aspect_ratio' ) ) {
						set_theme_mod( 'blog_thumbnail_content_valign', 'center' );
						set_theme_mod( 'blog_thumbnail_aspect_ratio', '1/1' );
						set_theme_mod( 'blog_post_type_icon_position', 'center center' );
						set_theme_mod(
							'blog_thumbnail_border_radius',
							kalium_parse_responsive_value(
								[
									'desktop' => [
										'top'    => 50,
										'right'  => 50,
										'bottom' => 50,
										'left'   => 50,
										'unit'   => '%',
										'link'   => true,
									],
								]
							)
						);
					}
				},
			],

			// Post elements
			'blog_thumbnails',
			'blog_thumbnails_placeholder',
			'blog_post_type_icon',
			'blog_post_formats',
			'blog_post_title',
			'blog_post_excerpt',
			'blog_post_date',
			'blog_category',

			// Proportional thumbnails
			[
				'id'     => 'blog_loop_proportional_thumbnails',
				'id_new' => 'blog_thumbnail_size',
				'filter' => function ( $value ) {
					if ( ! $value ) {
						if ( kalium_can_override_theme_option( 'blog_thumbnail_size_custom' ) ) {
							set_theme_mod(
								'blog_thumbnail_size_custom',
								[
									'width'  => 468,
									'height' => 328,
								]
							);
						}

						return 'custom';
					}
				},
			],

			// Hover layer
			[
				'id'     => 'blog_thumbnail_hover_effect',
				'filter' => function ( $value ) {
					switch ( $value ) {
						case 'distanced':
							set_theme_mod( 'blog_thumbnail_hover_padding', kalium_parse_responsive_value( 15 ) );
							break;

						case 'distanced-no-opacity':
							set_theme_mod( 'blog_thumbnail_hover_padding', kalium_parse_responsive_value( 15 ) );
							set_theme_mod( 'blog_thumbnail_hover_bg', kalium_color_reference( 'theme_colors.color-1' ) );
							break;

						case 'full-cover-no-opacity':
							set_theme_mod( 'blog_thumbnail_hover_bg', kalium_color_reference( 'theme_colors.color-1' ) );
							break;
					}

					return '' !== $value;
				},
			],

			// Hover layer icon
			[
				'id'     => 'blog_post_hover_layer_icon',
				'id_new' => 'blog_thumbnail_hover_icon',
				'filter' => function ( $value ) {
					$mapped = [
						'none'         => 'none',
						'static-eye'   => 'eye',
						'animated-eye' => 'eye',
						'custom'       => 'custom',
					];

					if ( 'static-eye' === $value ) {
						set_theme_mod( 'blog_thumbnail_hover_icon_eye', 'static' );
					} elseif ( 'animated-eye' === $value ) {
						set_theme_mod( 'blog_thumbnail_hover_icon_eye', 'animated' );
					} elseif ( 'custom' === $value ) {
						kalium_migrate_smof_batch(
							[
								[
									'id'     => 'blog_post_hover_layer_icon_custom',
									'id_new' => 'blog_thumbnail_hover_icon_custom',
									'filter' => 'kalium_smof_media_type_option',
								],
								[
									'id'     => 'blog_post_hover_layer_icon_custom_width',
									'id_new' => 'blog_thumbnail_hover_icon_custom_width',
								],
							]
						);
					}

					return $mapped[ $value ] ?? null;
				},
			],

			// Sidebar
			[
				'id'         => 'blog_sidebar_position',
				'id_new'     => 'blog_sidebar',
				'map_values' => [
					'hide'  => false,
					'right' => true,
					'left'  => true,
				],
			],
			[
				'id'         => 'blog_sidebar_position',
				'id_new'     => 'blog_sidebar_align',
				'map_values' => [
					'left'  => 'left',
					'right' => 'right',
				],
			],

			// Pagination
			[
				'id'         => 'blog_pagination_type',
				'map_values' => [
					'normal'         => 'numbers',
					'endless'        => 'load-more',
					'endless-reveal' => 'load-more',
				],
			],
			[
				'id'         => 'blog_pagination_position',
				'id_new'     => 'blog_pagination_align',
				'map_values' => [
					'left'      => 'left',
					'center'    => 'center',
					'right'     => 'right',
					'stretched' => 'space-between',
				],
			],
			[
				'id'         => 'blog_endless_pagination_style',
				'id_new'     => 'blog_pagination_infinite_scroll_loading_effect',
				'map_values' => [
					'_1' => 'spinner',
					'_2' => 'pulsate',
				],
			],
			[
				'id'         => 'blog_pagination_type',
				'id_new'     => 'blog_pagination_infinite_scroll_auto_reveal',
				'map_values' => [
					'endless-reveal' => true,
				],
			],

			// Single post toggles
			'blog_single_thumbnails',
			'blog_single_title',
			'blog_author_info',
			'blog_post_date_single',
			'blog_category_single',
			'blog_tags',
			'blog_post_prev_next',
			'blog_featured_image_placement',
			[
				'id'     => 'blog_featured_image_size_type',
				'id_new' => 'blog_featured_image_size',
				'filter' => function ( $value ) {
					if ( 'full' === $value ) {
						return $value;
					}
				},
			],
			[
				'id'         => 'blog_comments',
				'map_values' => [
					'show' => true,
					'hide' => false,
				],
			],
			'blog_gallery_autoswitch',
			'blog_related_posts',
			'blog_related_posts_columns',
			'blog_related_posts_per_page',
			[
				'id'     => 'blog_related_posts_details',
				'filter' => 'kalium_validate_boolean',
			],
			[
				'id'         => 'blog_single_sidebar_position',
				'id_new'     => 'blog_post_layout',
				'map_values' => [
					'hide'  => 'normal',
					'right' => 'sidebar-right',
					'left'  => 'sidebar-left',
				],
			],

			// Social sharing
			[
				'id'     => 'blog_share_story',
				'id_new' => 'blog_post_social_share',
			],
			[
				'id'     => 'blog_share_story_rounded_icons',
				'id_new' => 'blog_post_social_share_style',
				'filter' => function ( $value ) {
					return $value ? 'icons' : null;
				},
			],
			[
				'id'     => 'blog_share_story_networks',
				'id_new' => 'blog_post_social_share_networks',
				'filter' => 'kalium_smof_filter_share_networks',
			],

			// Blog heading
			'blog_show_header_title',
			'blog_title',
			'blog_description',

			/**
			 * Portfolio settings.
			 */

			// Layout
			[
				'id'         => 'portfolio_type',
				'id_new'     => 'portfolio_layout',
				'map_values' => [
					'type-1' => 'type-1',
					'type-2' => 'type-2',
					'type-3' => 'type-2',
				],
				'filter'     => function ( $value ) {

					/**
					 * Background color value handler for hover layer.
					 */
					$thumbnail_hover_bg_color = function ( $with_transparency, $current_color, $hover_style ) {
						$transparency = 0.9;

						if ( ! $current_color ) {
							switch ( $hover_style ) {
								case 'black':
									$current_color = $with_transparency ? (string) kalium()->style->color( '#000000' )->toRgba( $transparency ) : '#000000';
									break;

								case 'white':
									$current_color = $with_transparency ? (string) kalium()->style->color( '#ffffff' )->toRgba( $transparency ) : '#ffffff';
									break;
							}
						}

						return $current_color;
					};

					// Item type 1
					if ( 'type-1' === $value ) {
						kalium_migrate_smof_batch(
							[
								[
									'id'     => 'portfolio_type_1_dynamic_height',
									'id_new' => 'portfolio_thumbnails_size',
									'filter' => function ( $value, $smof_values ) {
										if ( ! $value ) {
											$custom_size = kalium_get_array_key( $smof_values, 'portfolio_thumbnail_size_1/value' );
											$width       = 655;
											$height      = 545;

											// Custom defined size
											if ( is_string( $custom_size ) && preg_match( '/^[0-9]+x[0-9]+$/i', $custom_size ) ) {
												$custom_size = explode( 'x', $custom_size );
												$width       = absint( $custom_size[0] );
												$height      = absint( $custom_size[1] );
											}

											// Set custom dimensions
											if ( kalium_can_override_theme_option( 'portfolio_thumbnails_size_custom' ) ) {
												set_theme_mod(
													'portfolio_thumbnails_size_custom',
													[
														'width'  => $width,
														'height' => $height,
													]
												);
											}

											return 'custom';
										}
									},
								],
								[
									'id'         => 'portfolio_type_1_columns_count',
									'id_new'     => 'portfolio_grid_columns',
									'map_values' => [
										'1 Item per Row'  => 1,
										'2 Items per Row' => [
											'desktop' => 2,
											'mobile'  => 1,
										],
										'3 Items per Row' => [
											'desktop' => 3,
											'tablet'  => 2,
											'mobile'  => 1,
										],
										'4 Items per Row' => [
											'desktop' => 4,
											'tablet'  => 2,
											'mobile'  => 1,
										],
										'5 Items per Row' => [
											'desktop' => 5,
											'tablet'  => 3,
											'mobile'  => 1,
										],
										'6 Items per Row' => [
											'desktop' => 6,
											'tablet'  => 3,
											'mobile'  => 1,
										],
									],
								],
								[
									'id'     => 'portfolio_type_1_default_spacing',
									'id_new' => 'portfolio_grid_columns_gap',
								],
								[
									'id'       => 'portfolio_type_1_items_per_page',
									'id_new'   => 'portfolio_items_per_page',
									'validate' => function ( $value ) {
										return is_numeric( $value );
									},
								],
								[
									'id'         => 'portfolio_type_1_hover_layer_icon',
									'id_new'     => 'portfolio_thumbnail_hover_icon',
									'map_values' => [
										'static-eye'   => 'eye',
										'animated-eye' => 'eye',
										'custom'       => 'custom',
									],
								],
								[
									'id'         => 'portfolio_type_1_hover_layer_icon',
									'id_new'     => 'portfolio_thumbnail_hover_icon_eye',
									'map_values' => [
										'static-eye'   => 'static',
										'animated-eye' => 'animated',
									],
								],
								[
									'id'     => 'portfolio_type_1_hover_layer_icon_custom',
									'id_new' => 'portfolio_thumbnail_hover_icon_custom',
									'filter' => 'kalium_smof_media_type_option',
								],
								[
									'id'     => 'portfolio_type_1_hover_layer_icon_custom_width',
									'id_new' => 'portfolio_thumbnail_hover_icon_custom_width',
								],
								[
									'id'         => 'portfolio_type_1_hover_effect',
									'id_new'     => 'portfolio_thumbnail_hover_effect',
									'map_values' => [
										'none'      => false,
										'full'      => true,
										'distanced' => true,
									],
								],
								[
									'id'     => 'portfolio_type_1_hover_effect',
									'id_new' => 'portfolio_thumbnail_hover_padding',
									'filter' => function ( $value ) {
										if ( 'distanced' === $value ) {
											return 15;
										}
									},
								],
								[
									'id'         => 'portfolio_type_1_hover_style',
									'id_new'     => 'portfolio_thumbnail_hover_bg_type',
									'map_values' => [
										'dominant-color' => 'dominant-color',
									],
								],
								[
									'id'     => 'portfolio_type_1_hover_transparency',
									'id_new' => 'portfolio_thumbnail_hover_bg',
									'filter' => function ( $value, $smof_values ) use ( &$thumbnail_hover_bg_color ) {
										$with_transparency = 'opacity' === $value;
										$current_color     = kalium_get_array_key( $smof_values, 'portfolio_type_1_hover_color/value' );
										$hover_style       = kalium_get_array_key( $smof_values, 'portfolio_type_1_hover_style/value' );

										return $thumbnail_hover_bg_color( $with_transparency, $current_color, $hover_style );
									},
								],
							]
						);
					}
					// Item type 2
					elseif ( 'type-2' === $value ) {
						kalium_migrate_smof_batch(
							[
								[
									'id'     => 'portfolio_thumbnail_size_2',
									'id_new' => 'portfolio_thumbnails_size',
									'filter' => function ( $value, $smof_values ) {
										$image_sizes = kalium_get_image_sizes();

										if ( isset( $image_sizes[ $value ] ) ) {
											return $value;
										}

										if ( 'type-3' === kalium_get_array_key( $smof_values, 'portfolio_type/value' ) ) {
											return;
										}

										$custom_size = kalium_get_array_key( $smof_values, 'portfolio_thumbnail_size_2/value' );
										$width       = 655;
										$height      = 545;

										// Custom defined size
										if ( is_string( $custom_size ) && preg_match( '/^[0-9]+x[0-9]+$/i', $custom_size ) ) {
											$custom_size = explode( 'x', $custom_size );
											$width       = absint( $custom_size[0] );
											$height      = absint( $custom_size[1] );
										}

										// Set custom dimensions
										if ( kalium_can_override_theme_option( 'portfolio_thumbnails_size_custom' ) ) {
											set_theme_mod(
												'portfolio_thumbnails_size_custom',
												[
													'width'  => $width,
													'height' => $height,
												]
											);
										}

										return 'custom';
									},
								],
								[
									'id'         => 'portfolio_type_2_columns_count',
									'id_new'     => 'portfolio_grid_columns',
									'map_values' => [
										'1 Item per Row'  => 1,
										'2 Items per Row' => [
											'desktop' => 2,
											'mobile'  => 1,
										],
										'3 Items per Row' => [
											'desktop' => 3,
											'tablet'  => 2,
											'mobile'  => 1,
										],
										'4 Items per Row' => [
											'desktop' => 4,
											'tablet'  => 2,
											'mobile'  => 1,
										],
										'5 Items per Row' => [
											'desktop' => 5,
											'tablet'  => 3,
											'mobile'  => 1,
										],
										'6 Items per Row' => [
											'desktop' => 6,
											'tablet'  => 3,
											'mobile'  => 1,
										],
									],
								],
								[
									'id'     => 'portfolio_type_2_default_spacing',
									'id_new' => 'portfolio_grid_columns_gap',
									'filter' => function ( $value, $smof_values ) {
										if ( 'merged' === kalium_get_array_key( $smof_values, 'portfolio_type_2_grid_spacing/value' ) ) {
											$value = 0;
										}

										return $value;
									},
								],
								[
									'id'       => 'portfolio_type_2_items_per_page',
									'id_new'   => 'portfolio_items_per_page',
									'validate' => function ( $value ) {
										return is_numeric( $value );
									},
								],
								[
									'id'         => 'portfolio_type_2_hover_effect',
									'id_new'     => 'portfolio_thumbnail_hover_effect',
									'map_values' => [
										'none'      => false,
										'full'      => true,
										'distanced' => true,
									],
								],
								[
									'id'     => 'portfolio_type_2_hover_effect',
									'id_new' => 'portfolio_thumbnail_hover_padding',
									'filter' => function ( $value ) {
										if ( 'distanced' === $value ) {
											return 15;
										}
									},
								],
								[
									'id'         => 'portfolio_type_2_hover_style',
									'id_new'     => 'portfolio_thumbnail_hover_bg_type',
									'map_values' => [
										'dominant-color' => 'dominant-color',
									],
								],
								[
									'id'     => 'portfolio_type_2_hover_transparency',
									'id_new' => 'portfolio_thumbnail_hover_bg',
									'filter' => function ( $value, $smof_values ) use ( &$thumbnail_hover_bg_color ) {
										$with_transparency = 'opacity' === $value;
										$current_color     = kalium_get_array_key( $smof_values, 'portfolio_type_2_hover_color/value' );
										$hover_style       = kalium_get_array_key( $smof_values, 'portfolio_type_2_hover_style/value' );

										return $thumbnail_hover_bg_color( $with_transparency, $current_color, $hover_style );
									},
								],
								[
									'id'     => 'portfolio_type_2_hover_text_position',
									'id_new' => 'portfolio_layout_2_content_position',
								],
							]
						);
					}

					return $value;
				},
			],
			[
				'id'         => 'portfolio_type',
				'id_new'     => 'portfolio_grid_layout_mode',
				'map_values' => [
					'type-1' => 'fit-rows',
					'type-2' => 'fit-rows',
					'type-3' => 'packery',
				],
				'filter'     => function ( $value, $smof_values ) {

					// When custom image size is set, automatically use packery
					if ( 'type-2' === kalium_get_array_key( $smof_values, 'portfolio_type/value' ) ) {
						$image_size  = kalium_get_array_key( $smof_values, 'portfolio_thumbnail_size_2/value' );
						$image_sizes = kalium_get_image_sizes();

						if ( isset( $image_sizes[ $image_size ] ) ) {
							return 'packery';
						}
					}

					return $value;
				},
			],
			'portfolio_full_width',
			[
				'id'     => 'portfolio_full_width_title_filter_container',
				'id_new' => 'portfolio_full_width_heading',
				'filter' => function ( $value ) {
					if ( kalium_can_override_theme_option( 'portfolio_full_width_stretch_edge' ) ) {
						set_theme_mod( 'portfolio_full_width_stretch_edge', true );
					}

					return ! $value ? 'full-width' : null;
				},
			],

			// URL prefix
			[
				'id'     => 'portfolio_url_add_prefix',
				'id_new' => 'option:kalium_portfolio_permalinks',
				'filter' => function ( $value, $smof_values ) {
					$permalinks = kalium_get_portfolio_permalinks();

					// Portfolio slug
					$slug = kalium_get_array_key( $smof_values, 'portfolio_prefix_url_slug/value' );

					if ( $value && $slug ) {
						$permalinks['portfolio'] = $slug;
					} elseif ( ! $slug ) {
						$permalinks['portfolio'] = '';
					}

					// Category slug
					if ( $category = kalium_get_array_key( $smof_values, 'portfolio_category_prefix_url_slug/value' ) ) {
						$permalinks['portfolio_category'] = $category;
					}

					return $permalinks;
				},
			],

			// Portfolio page toggles
			'portfolio_likes',
			'portfolio_likes_icon',
			'portfolio_reveal_effect',
			[
				'id'     => 'portfolio_category_filter',
				'id_new' => 'portfolio_filters',
			],
			[
				'id'         => 'portfolio_filter_dynamic_heading_title',
				'id_new'     => 'portfolio_filters_dynamic_heading',
				'map_values' => [
					'disable'           => 'disable',
					'title'             => 'title',
					'description'       => 'content',
					'title-description' => 'title-content',
				],
			],
			[
				'id'     => 'portfolio_filter_link_type',
				'id_new' => 'portfolio_filters_history_type',
			],
			[
				'id'     => 'portfolio_filter_enable_subcategories',
				'id_new' => 'portfolio_filters_sub_terms',
			],
			[
				'id'     => 'portfolio_default_filter_category',
				'id_new' => 'portfolio_category_default',
			],
			[
				'id'     => 'portfolio_loop_subtitles',
				'id_new' => 'portfolio_loop_subtitles_content',
			],

			// Pagination
			[
				'id'         => 'portfolio_pagination_type',
				'map_values' => [
					'normal'         => 'numbers',
					'endless'        => 'load-more',
					'endless-reveal' => 'load-more',
				],
			],
			[
				'id'         => 'portfolio_pagination_position',
				'id_new'     => 'portfolio_pagination_align',
				'map_values' => [
					'left'      => 'left',
					'center'    => 'center',
					'right'     => 'right',
					'stretched' => 'space-between',
				],
			],
			[
				'id'         => 'portfolio_endless_pagination_style',
				'id_new'     => 'portfolio_pagination_infinite_scroll_loading_effect',
				'map_values' => [
					'_1' => 'spinner',
					'_2' => 'pulsate',
				],
			],
			[
				'id'         => 'portfolio_pagination_type',
				'id_new'     => 'portfolio_pagination_infinite_scroll_auto_reveal',
				'map_values' => [
					'endless-reveal' => true,
				],
			],

			// Single page
			'portfolio_prev_next',
			'portfolio_prev_next_category',
			'portfolio_archive_url_category',
			'portfolio_archive_url',
			'portfolio_prev_next_show_titles',
			[
				'id'         => 'portfolio_prev_next_type',
				'map_values' => [
					'simple' => 'standard',
					'fixed'  => 'fixed',
				],
			],
			[
				'id'         => 'portfolio_prev_next_position',
				'map_values' => [
					'left-side'  => 'left',
					'centered'   => 'center',
					'right-side' => 'right',
				],
			],

			// Share
			'portfolio_share_item',
			'portfolio_like_share_layout',
			[
				'id'     => 'portfolio_share_item_networks',
				'filter' => 'kalium_smof_filter_share_networks',
			],

			// Gallery
			[
				'id'         => 'portfolio_gallery_caption_position',
				'id_new'     => 'portfolio_gallery_captions',
				'map_values' => [
					'hover' => true,
					'below' => true,
					'hide'  => false,
				],
			],
			[
				'id'         => 'portfolio_gallery_caption_position',
				'map_values' => [
					'hover' => 'hover',
					'below' => 'below',
				],
			],
			[
				'id'     => 'portfolio_disable_lightbox',
				'id_new' => 'portfolio_gallery_lightbox',
				'filter' => function ( $value ) {
					return ! $value;
				},
			],

			// Lightbox
			'portfolio_lb_navigation_mode',

			// General lightbox
			[
				'id'     => 'portfolio_lb_fullscreen',
				'id_new' => 'lightbox_fullscreen',
			],
			[
				'id'     => 'portfolio_lb_captions',
				'id_new' => 'lightbox_captions',
			],
			[
				'id'     => 'portfolio_lb_counter',
				'id_new' => 'lightbox_counter',
			],
			[
				'id'     => 'portfolio_lb_hide_bars_delay',
				'id_new' => 'lightbox_hide_controls',
			],
			[
				'id'     => 'portfolio_lb_hash',
				'id_new' => 'portfolio_lightbox_history_type',
				'filter' => function ( $value ) {
					if ( ! $value ) {
						return 'none';
					}
				},
			],
			[
				'id'         => 'portfolio_lb_skin',
				'id_new'     => 'lightbox_color_scheme',
				'map_values' => [
					'kalium-dark'  => 'dark',
					'kalium-light' => 'light',
				],
			],
			[
				'id'     => 'portfolio_lb_image_size_large',
				'id_new' => 'lightbox_image_size_main',
				'filter' => function ( $value ) {
					$image_sizes = kalium_get_image_sizes();

					if ( isset( $image_sizes[ $value ] ) ) {
						return $value;
					}
				},
			],
			[
				'id'     => 'portfolio_lb_image_size_thumbnail',
				'id_new' => 'lightbox_image_size_thumbnail',
				'filter' => function ( $value ) {
					$image_sizes = kalium_get_image_sizes();

					if ( isset( $image_sizes[ $value ] ) ) {
						return $value;
					}
				},
			],
			[
				'id'     => 'portfolio_lb_thumbnails',
				'id_new' => 'lightbox_thumbnails',
			],
			[
				'id'     => 'portfolio_lb_thumbnails_show',
				'id_new' => 'lightbox_collapse_thumbnails',
				'filter' => function ( $value ) {
					return ! $value;
				},
			],
			[
				'id'     => 'portfolio_lb_autoplay',
				'id_new' => 'lightbox_autoplay',
			],
			[
				'id'     => 'portfolio_lb_autoplay_pause',
				'id_new' => 'lightbox_autoplay_interval',
			],
			[
				'id'     => 'portfolio_lb_zoom',
				'id_new' => 'lightbox_zoom',
			],
			[
				'id'     => 'portfolio_lb_zoom_scale',
				'id_new' => 'lightbox_zoom_scale',
			],

			// Portfolio heading
			'portfolio_show_header_title',
			'portfolio_title',
			'portfolio_description',
			[
				'id'     => 'portfolio_preselected_item_type',
				'id_new' => 'option:portfolio_default_item_type',
			],
			[
				'id'     => 'portfolio_show_header_title',
				'id_new' => 'portfolio_heading_layout',
				'filter' => function ( $value ) {
					return $value ? 'type-3' : null;
				},
			],

			/**
			 * Header and menu.
			 */

			'main_menu_type',
			'submenu_dropdown_indicator',

			// Custom header type
			[
				'id'     => 'main_menu_type',
				'id_new' => 'header_content',
				'filter' => 'kalium_smof_filter_custom_header_content',
			],

			// Menu skin
			[
				'id'     => 'main_menu_type',
				'filter' => function ( $value, $smof_values ) {
					$set_colors = [];

					switch ( $value ) {
						case 'full-bg-menu':
							$header_skin = kalium_get_array_key( $smof_values, 'menu_full_bg_skin/value' );
							break;

						case 'top-menu':
							$header_skin = kalium_get_array_key( $smof_values, 'menu_top_skin/value' );
							break;

						case 'sidebar-menu':
							$header_skin = kalium_get_array_key( $smof_values, 'menu_sidebar_skin/value' );
							break;

						case 'custom-header':
							$header_skin = kalium_get_array_key( $smof_values, 'custom_header_default_skin/value' );
							break;

						default:
							$header_skin = kalium_get_array_key( $smof_values, 'menu_standard_skin/value' );
					}

					switch ( $header_skin ) {
						case 'menu-skin-main':
							$color     = kalium_color_reference( 'theme_colors.color-1' );
							$color_alt = '#ffffff';

							$set_colors = array_merge(
								$set_colors,
								[
									'header_style_links' => [
										'normal' => $color,
										'hover'  => $color,
										'active' => $color,
									],
									'header_style_menu_trigger' => [
										'normal' => $color,
										'hover'  => $color,
										'active' => $color,
									],
									'header_style_submenu_bg' => $color,
									'header_style_submenu_links' => [
										'normal' => $color_alt,
										'hover'  => $color_alt,
										'active' => $color_alt,
									],
								]
							);
							break;

						case 'menu-skin-dark':
							$color = '#333333';

							$set_colors['header_style_links'] = [
								'normal' => $color,
								'hover'  => $color,
								'active' => $color,
							];
							break;

						case 'menu-skin-light':
							$color     = '#ffffff';
							$color_alt = '#333333';

							$set_colors = array_merge(
								$set_colors,
								[
									'header_style_links' => [
										'normal' => $color,
										'hover'  => $color,
										'active' => $color,
									],
									'header_style_menu_trigger' => [
										'normal' => $color,
										'hover'  => $color,
										'active' => $color,
									],
									'header_style_submenu_bg' => $color,
									'header_style_submenu_links' => [
										'normal' => $color_alt,
										'hover'  => $color_alt,
										'active' => $color_alt,
									],
									'header_style_fullscreen_links' => array_merge(
										kalium_get_theme_option( 'header_style_fullscreen_links' ),
										[
											'normal' => $color_alt,
										]
									),
									'header_style_fullscreen_text' => $color_alt,
								]
							);
							break;
					}

					if ( ! empty( $set_colors ) ) {
						foreach ( $set_colors as $option_name => $color ) {
							if ( kalium_can_override_theme_option( $option_name ) ) {
								set_theme_mod( $option_name, $color );
							}
						}
					}
				},
			],

			// Standard menu type
			'menu_standard_menu_bar_visible',
			[
				'id'         => 'menu_standard_menu_bar_effect',
				'map_values' => [
					'reveal-fade'        => 'fade',
					'reveal-from-left'   => 'slide-left',
					'reveal-from-right'  => 'slide-right',
					'reveal-from-top'    => 'slide-top',
					'reveal-from-bottom' => 'slide-bottom',
				],
			],

			// Full screen menu type
			'menu_full_bg_search_field',
			'menu_full_bg_alignment',
			[
				'id'     => 'menu_full_bg_skin',
				'id_new' => 'fullscreen_menu_bg_color',
				'filter' => function ( $value, $smof_values ) {
					$opacity = kalium_get_array_key( $smof_values, 'menu_full_bg_opacity/value' ) ? 0.9 : 1;

					switch ( $value ) {
						case 'menu-skin-main':
							$color = kalium_color_reference( 'theme_colors.color-1', $opacity );
							break;

						case 'menu-skin-dark':
							$color = (string) kalium()->style->color( '#1a1a1a' )->toRgba( $opacity );
							break;

						case 'menu-skin-light':
							$color = (string) kalium()->style->color( '#ffffff' )->toRgba( $opacity );
							break;
					}

					if ( ! empty( $color ) && kalium_can_override_theme_option( 'fullscreen_menu_bg' ) ) {
						set_theme_mod( 'fullscreen_menu_bg', true );

						// Set active color for menu trigger
						if ( in_array( $value, [ 'menu-skin-main', 'menu-skin-dark' ] ) ) {
							set_theme_mod(
								'header_style_menu_trigger',
								array_merge(
									kalium_get_theme_option( 'header_style_menu_trigger' ),
									[
										'active' => '#fff',
									]
								)
							);
						}
					}

					return $color ?? null;
				},
			],

			// Off-canvas top menu type
			'menu_top_show_widgets',
			'menu_top_widgets_container_width',
			'menu_top_force_include',
			[
				'id'     => 'menu_top_menu_id',
				'filter' => 'kalium_smof_filter_menu_id',
			],
			[
				'id'         => 'menu_top_items_per_row',
				'map_values' => [
					'items-1' => 1,
					'items-2' => 2,
					'items-3' => 3,
					'items-4' => 4,
					'items-5' => 5,
					'items-6' => 6,
					'items-7' => 7,
					'items-8' => 8,
				],
			],
			[
				'id'         => 'menu_top_widgets_per_row',
				'map_values' => [
					'six'   => 2,
					'four'  => 3,
					'three' => 4,
				],
			],
			[
				'id'     => 'menu_top_skin',
				'id_new' => 'header_style_off_canvas_top_bg',
				'filter' => function ( $value ) {
					switch ( $value ) {
						case 'menu-skin-dark':
							$color = '#333333';
							break;

						case 'menu-skin-light':
							$color     = '#ffffff';
							$color_alt = '#333333';

							if ( kalium_can_override_theme_option( 'header_style_off_canvas_top_links' ) ) {
								set_theme_mod(
									'header_style_off_canvas_top_links',
									array_merge(
										kalium_get_theme_option( 'header_style_off_canvas_top_links' ),
										[
											'normal' => $color_alt,
										]
									)
								);

								set_theme_mod( 'header_style_off_canvas_top_text', $color_alt );
							}
							break;
					}

					return $color ?? null;
				},
			],

			// Off-canvas side menu
			'menu_sidebar_show_widgets',
			'menu_sidebar_alignment',
			'menu_sidebar_force_include',
			[
				'id'     => 'menu_sidebar_menu_id',
				'filter' => 'kalium_smof_filter_menu_id',
			],
			[
				'id'     => 'menu_sidebar_skin',
				'id_new' => 'header_style_off_canvas_side_bg',
				'filter' => function ( $value ) {
					$ocs_links_option = 'header_style_off_canvas_side_links';

					switch ( $value ) {
						case 'menu-skin-main':
							$color_alt = '#333333';

							if ( kalium_can_override_theme_option( $ocs_links_option ) ) {
								set_theme_mod(
									$ocs_links_option,
									array_merge(
										kalium_get_theme_option( $ocs_links_option ),
										[
											'normal' => $color_alt,
										]
									)
								);
							}
							break;

						case 'menu-skin-dark':
							$color     = '#333333';
							$color_alt = '#ffffff';

							if ( kalium_can_override_theme_option( $ocs_links_option ) ) {
								set_theme_mod(
									$ocs_links_option,
									[
										'normal' => $color_alt,
										'hover'  => $color_alt,
										'active' => $color_alt,
									]
								);
							}

							if ( kalium_can_override_theme_option( 'header_style_off_canvas_side_text' ) ) {
								set_theme_mod( 'header_style_off_canvas_side_text', $color_alt );
							}
							break;

						case 'menu-skin-light':
							$color_alt = '#333333';

							if ( kalium_can_override_theme_option( $ocs_links_option ) ) {
								set_theme_mod(
									$ocs_links_option,
									[
										'normal' => $color_alt,
										'hover'  => $color_alt,
										'active' => $color_alt,
									]
								);
							}
							break;
					}

					return $color ?? null;
				},
			],

			// Top header bar
			'top_header_bar',
			[
				'id'     => 'top_header_bar',
				'id_new' => 'top_bar_content',
				'filter' => 'kalium_smof_filter_top_header_content',
			],
			[
				'id'     => 'top_header_bar_skin',
				'id_new' => 'top_header_bar_container_colors',
				'filter' => function ( $value, $smof_values ) {
					$container_colors = [
						'background' => null,
						'border'     => null,
						'separator'  => null,
					];

					if ( 'dark' === $value ) {
						$container_colors['background'] = '#333333';

						$color_alt = '#ffffff';

						if ( kalium_can_override_theme_option( 'top_header_bar_link_colors' ) ) {
							set_theme_mod(
								'top_header_bar_link_colors',
								[
									'normal' => $color_alt,
									'hover'  => $color_alt,
									'active' => $color_alt,
								]
							);
						}

						if ( kalium_can_override_theme_option( 'top_header_bar_text_color' ) ) {
							set_theme_mod( 'top_header_bar_text_color', $color_alt );
						}
					} elseif ( 'light' === $value ) {
						$container_colors['background'] = '#ffffff';

						$color_alt = '#333333';

						if ( kalium_can_override_theme_option( 'top_header_bar_link_colors' ) ) {
							set_theme_mod(
								'top_header_bar_link_colors',
								[
									'normal' => $color_alt,
									'hover'  => $color_alt,
									'active' => $color_alt,
								]
							);
						}

						if ( kalium_can_override_theme_option( 'top_header_bar_text_color' ) ) {
							set_theme_mod( 'top_header_bar_text_color', $color_alt );
						}
					}

					// Custom colors
					if ( $custom_bg = kalium_get_array_key( $smof_values, 'top_header_bar_background_color/value' ) ) {
						$container_colors['background'] = $custom_bg;
					}

					if ( $custom_border = kalium_get_array_key( $smof_values, 'top_header_bar_border_color/value' ) ) {
						$container_colors['border'] = $custom_border;
					}

					if ( $custom_separator = kalium_get_array_key( $smof_values, 'top_header_bar_separator_color/value' ) ) {
						$container_colors['separator'] = $custom_separator;
					}

					if ( ! empty( array_values( $container_colors ) ) ) {
						return $container_colors;
					}
				},
			],
			[
				'callback' => function ( $smof_values ) {
					if ( kalium_can_override_theme_option( 'top_bar_responsive' ) ) {
						$devices = [];

						if ( kalium_get_array_key( $smof_values, 'top_header_bar_support_desktop/value' ) ) {
							$devices[] = 'desktop';
						}

						if ( kalium_get_array_key( $smof_values, 'top_header_bar_support_tablet/value' ) ) {
							$devices[] = 'tablet';
						}

						if ( kalium_get_array_key( $smof_values, 'top_header_bar_support_mobile/value' ) ) {
							$devices[] = 'mobile';
						}

						set_theme_mod( 'top_bar_responsive', $devices );

						return true;
					}
				},
			],

			// Sticky header
			'sticky_header',
			'sticky_header_background_color',
			'sticky_header_autohide',
			'sticky_header_autohide_duration',
			'sticky_header_autohide_animation_type',
			'sticky_header_tween_changes',
			'sticky_header_logo_width',
			[
				'id'     => 'sticky_header_logo',
				'filter' => 'kalium_smof_media_type_option',
			],
			[
				'id'     => 'sticky_header_animate_duration',
				'filter' => function ( $value ) {
					return ! $value;
				},
			],
			[
				'id'     => 'header_sticky_duration',
				'id_new' => 'sticky_header_duration',
			],
			[
				'id'     => 'header_sticky_initial_offset',
				'id_new' => 'sticky_header_initial_offset',
			],
			[
				'id'     => 'sticky_header_vertical_padding',
				'filter' => function ( $value ) {
					if ( is_numeric( $value ) ) {
						return [
							'top'    => $value,
							'bottom' => $value,
							'unit'   => 'px',
							'link'   => true,
						];
					}
				},
			],
			[
				'id'     => 'sticky_header_skin',
				'id_new' => 'sticky_header_links_color',
				'filter' => function ( $value ) {
					switch ( $value ) {
						case 'menu-skin-dark':
							$color = '#333333';

							$colors = [
								'normal' => $color,
								'hover'  => kalium_color_reference( 'sticky_header_links_color.normal' ),
								'active' => kalium_color_variant_active( 'sticky_header_links_color.hover' ),
							];
							break;

						case 'menu-skin-light':
							$color = '#ffffff';

							$colors = [
								'normal' => $color,
								'hover'  => kalium_color_reference( 'sticky_header_links_color.normal' ),
								'active' => kalium_color_reference( 'sticky_header_links_color.hover' ),
							];
							break;
					}

					if ( ! empty( $color ) && kalium_can_override_theme_option( 'sticky_header_text_color' ) ) {
						set_theme_mod( 'sticky_header_text_color', $color );
					}

					return $colors ?? null;
				},
			],
			[
				'id'     => 'sticky_header_border',
				'id_new' => 'sticky_header_shadow',
				'filter' => function ( $value, $smof_values ) {
					if ( $value ) {
						$shadow_color  = kalium_get_array_key( $smof_values, 'sticky_header_shadow_color/value' );
						$shadow_width  = kalium_get_array_key( $smof_values, 'sticky_header_shadow_width/value' );
						$shadow_radius = kalium_get_array_key( $smof_values, 'sticky_header_shadow_blur/value' );

						if ( $shadow_color ) {
							return [
								'color'         => $shadow_color,
								'offset_x'      => 0,
								'offset_y'      => absint( $shadow_width ),
								'blur_radius'   => absint( $shadow_radius ),
								'spread_radius' => null,
								'unit'          => 'px',
							];
						}
					}
				},
			],
			[
				'id'     => 'sticky_header_border',
				'filter' => function ( $value, $smof_values ) {
					if ( $value ) {
						$border_color = kalium_get_array_key( $smof_values, 'sticky_header_border_color/value' );
						$border_width = kalium_get_array_key( $smof_values, 'sticky_header_border_width/value' );

						if ( $border_color ) {
							return [
								'width' => $border_width,
								'style' => 'solid',
								'color' => $border_color,
							];
						}
					}
				},
			],
			[
				'id'     => 'sticky_header_responsive',
				'filter' => function ( $value, $smof_values ) {
					$devices = [];

					if ( kalium_get_array_key( $smof_values, 'sticky_header_support_desktop/value' ) ) {
						$devices[] = 'desktop';
					}

					if ( kalium_get_array_key( $smof_values, 'sticky_header_support_tablet/value' ) ) {
						$devices[] = 'tablet';
					}

					if ( kalium_get_array_key( $smof_values, 'sticky_header_support_mobile/value' ) ) {
						$devices[] = 'mobile';
					}

					if ( ! empty( $devices ) ) {
						return $devices;
					}
				},
			],

			// Other header settings
			'header_transparent',
			'header_spacing',
			'header_fullwidth',
			'header_bottom_spacing',
			[
				'id'     => 'header_vpadding_top',
				'id_new' => 'header_vpadding',
				'filter' => function ( $value, $smof_values ) {
					$padding_top    = kalium_get_array_key( $smof_values, 'header_vpadding_top/value' );
					$padding_bottom = kalium_get_array_key( $smof_values, 'header_vpadding_bottom/value' );

					if ( is_numeric( $padding_top ) || is_numeric( $padding_bottom ) ) {
						return [
							'top'    => $padding_top,
							'bottom' => $padding_bottom,
							'unit'   => 'px',
							'link'   => true,
						];
					}
				},
			],
			[
				'id'     => 'header_background_color',
				'id_new' => 'header_background',
			],
			[
				'id'     => 'header_bottom_border',
				'id_new' => 'header_border',
				'filter' => function ( $value ) {
					if ( $value ) {
						$null_border = [
							'width' => null,
							'style' => null,
							'color' => null,
						];

						return [
							'top'    => $null_border,
							'right'  => $null_border,
							'bottom' => [
								'width' => '1px',
								'style' => 'solid',
								'color' => $value,
							],
							'left'   => $null_border,
						];
					}
				},
			],

			// Mobile menu
			'menu_mobile_breakpoint',
			[
				'id'     => 'menu_mobile_type',
				'id_new' => 'mobile_menu_content',
				'filter' => function ( $value, $smof_values ) {
					$is_fullscreen = ! $value;
					$has_search    = kalium_get_array_key( $smof_values, 'menu_mobile_search_field/value' );

					$mobile_menu_content = [
						'elements' => [
							[
								'name'       => 'legacy_header_content_top',
								'attributes' => array_filter(
									[
										$is_fullscreen ? [
											'name'  => 'align',
											'value' => 'center',
										] : null,
									]
								),
								'visible'    => true,
								'elements'   => array_filter(
									[
										[
											'name'       => 'legacy_header_menu_mobile',
											'attributes' => [
												[
													'name' => 'menu_id',
													'value' => 6,
												],
											],
											'visible'    => true,
										],
										$has_search ? [
											'name'       => 'legacy_header_search_input',
											'attributes' => [],
											'visible'    => true,
										] : null,
									],
								),
							],
							[
								'name'       => 'legacy_header_content_bottom',
								'attributes' => [],
								'visible'    => true,
								'elements'   => [],
							],
						],
					];

					// Fullscreen options
					if ( $is_fullscreen ) {

						// White links
						if ( kalium_can_override_theme_option( 'mobile_menu_link_colors' ) ) {
							set_theme_mod(
								'mobile_menu_link_colors',
								[
									'normal' => '#ffffff',
									'hover'  => kalium_color_reference( 'mobile_menu_link_colors.normal' ),
									'active' => kalium_color_reference( 'mobile_menu_link_colors.hover' ),
								]
							);
						}

						// Close button
						if ( kalium_can_override_theme_option( 'menu_mobile_close_button' ) ) {
							set_theme_mod( 'menu_mobile_close_button', 'right' );
						}
					}

					return $mobile_menu_content;
				},
			],
			[
				'id'     => 'menu_mobile_type',
				'filter' => function ( $value ) {
					return $value ? 'slide' : 'fullscreen';
				},
			],

			// Other header elements
			'menu_hamburger_custom_label',
			'menu_hamburger_custom_label_text',
			'menu_hamburger_custom_label_close_text',
			'menu_hamburger_custom_icon_position',
			'header_search_field',
			[
				'id'     => 'header_search_field_icon_animation',
				'id_new' => 'header_search_field_animate_icon',
				'filter' => function ( $value ) {
					return 'none' !== $value;
				},
			],
			[
				'id'         => 'header_search_field_icon_animation',
				'map_values' => [
					'shift' => 'shift',
					'scale' => 'scale',
				],
			],
			[
				'id'     => 'fullscreen_menu_bg',
				'filter' => function ( $value, $smof_values ) {
					if ( $value ) {

						// Background
						if ( $background = kalium_get_array_key( $smof_values, 'fullscreen_menu_bg_color/value' ) ) {
							set_theme_mod( 'fullscreen_menu_bg_color', $background );
						}
					}
				},
			],
			[
				'id'     => 'fullscreen_menu_bg_image',
				'filter' => 'kalium_smof_media_type_option',
			],
			[
				'id'     => 'fullscreen_menu_bg_position',
				'filter' => function ( $value ) {
					$value = str_replace( '-', ' ', $value );

					if ( 'center' === $value ) {
						$value = 'center center';
					}

					return $value;
				},
			],
			[
				'id'     => 'fullscreen_menu_bg_repat',
				'id_new' => 'fullscreen_menu_bg_repeat',
			],
			[
				'id'     => 'fullscreen_menu_bg_size',
				'filter' => function ( $value ) {
					$value = strtolower( $value );

					if ( in_array( $value, [ 'cover', 'contain' ] ) ) {
						return $value;
					}

					if ( kalium_can_override_theme_option( 'fullscreen_menu_bg_size_custom' ) ) {
						set_theme_mod( 'fullscreen_menu_bg_size_custom', $value );
					}

					return 'custom';
				},
			],

			// WPML language switcher
			'header_wpml_language_switcher',
			[
				'id'         => 'header_wpml_language_flag_position',
				'id_new'     => 'header_wpml_language_flag',
				'map_values' => [
					'left'  => true,
					'right' => true,
					'hide'  => false,
				],
			],
			[
				'id'         => 'header_wpml_language_flag_position',
				'map_values' => [
					'left'  => 'left',
					'right' => 'right',
				],
			],
			[
				'id'        => 'header_wpml_language_switcher_text_display_type',
				'id_new'    => 'header_wpml_language_switcher_text',
				'map_props' => [
					'translated'      => true,
					'initials'        => true,
					'name-translated' => true,
					'translated-name' => true,
					'hide'            => false,
				],
			],
			[
				'id'         => 'header_wpml_language_switcher_text_display_type',
				'map_values' => [
					'translated'      => 'translated',
					'initials'        => 'initials',
					'name-translated' => 'name-translated',
					'translated-name' => 'translated-name',
				],
			],

			/**
			 * Shop settings.
			 */

			// Product layout
			[
				'id'     => 'shop_catalog_layout',
				'id_new' => 'product_loop_card',
				'filter' => 'kalium_smof_filter_product_card',
			],

			// Product archive
			'shop_catalog_mode',
			'shop_catalog_mode_hide_prices',
			[
				'id'         => 'shop_loop_masonry_layout_mode',
				'id_new'     => 'shop_masonry_mode',
				'map_values' => [
					'masonry' => true,
					'fitRows' => false,
				],
			],
			[
				'id'     => 'option:woocommerce_catalog_rows',
				'id_new' => 'shop_products_per_page',
				'filter' => function ( $rows ) {
					$per_page = 12;
					$columns  = get_option( 'woocommerce_catalog_columns' );

					if ( is_numeric( $rows ) && is_numeric( $columns ) ) {
						$per_page = $columns * $rows;
					}

					return $per_page;
				},
			],
			[
				'id'     => 'option:woocommerce_catalog_columns',
				'id_new' => 'shop_grid_columns',
				'filter' => function ( $columns, $smof_values ) {
					$grid_columns = kalium_get_theme_option( 'shop_grid_columns' );

					return array_merge(
						$grid_columns ?? [],
						[
							'desktop' => $columns,
							'mobile'  => 'two' === kalium_get_array_key( $smof_values, 'shop_product_columns_mobile/value' ) ? 2 : 1,
						]
					);
				},
			],
			[
				'id'     => 'shop_title_show',
				'id_new' => 'shop_title',
			],
			[
				'id'     => 'shop_title_show',
				'id_new' => 'shop_results_count',
			],
			[
				'id'     => 'shop_sorting_show',
				'id_new' => 'shop_catalog_ordering',
			],
			[
				'callback' => function ( $smof_values ) {
					if ( kalium_can_override_theme_option( 'shop_columns_switcher' ) ) {
						set_theme_mod( 'shop_columns_switcher', false );
					}

					if ( kalium_can_override_theme_option( 'shop_category_layout' ) ) {
						set_theme_mod( 'shop_category_layout', 'type-6' );
					}

					if ( kalium_can_override_theme_option( 'shop_category_separate' ) ) {
						set_theme_mod( 'shop_category_separate', true );
					}

					if ( kalium_can_override_theme_option( 'shop_enabled_badges' ) ) {
						$badges = [];

						if ( kalium_get_array_key( $smof_values, 'shop_sale_ribbon_show/value' ) ) {
							$badges[] = 'sale';
						}

						if ( kalium_get_array_key( $smof_values, 'shop_oos_ribbon_show/value' ) ) {
							$badges[] = 'out-of-stock';
						}

						if ( kalium_get_array_key( $smof_values, 'shop_featured_ribbon_show/value' ) ) {
							$badges[] = 'featured';
						}

						set_theme_mod( 'shop_enabled_badges', $badges );
					}
				},
			],

			// Sidebar
			[
				'id'         => 'shop_sidebar',
				'map_values' => [
					'left'  => true,
					'right' => true,
					'hide'  => false,
				],
			],
			[
				'id'         => 'shop_sidebar',
				'id_new'     => 'shop_sidebar_align',
				'map_values' => [
					'left'  => 'left',
					'right' => 'right',
				],
			],

			// Pagination
			[
				'id'         => 'shop_pagination_type',
				'map_values' => [
					'normal'         => 'numbers',
					'endless'        => 'load-more',
					'endless-reveal' => 'load-more',
				],
			],
			[
				'id'         => 'shop_pagination_position',
				'id_new'     => 'shop_pagination_align',
				'map_values' => [
					'left'      => 'left',
					'center'    => 'center',
					'right'     => 'right',
					'stretched' => 'space-between',
				],
			],
			[
				'id'         => 'shop_endless_pagination_style',
				'id_new'     => 'shop_pagination_infinite_scroll_loading_effect',
				'map_values' => [
					'_1' => 'spinner',
					'_2' => 'pulsate',
				],
			],
			[
				'id'         => 'shop_pagination_type',
				'id_new'     => 'shop_pagination_infinite_scroll_auto_reveal',
				'map_values' => [
					'endless-reveal' => true,
				],
			],

			// Social sharing
			[
				'id'     => 'shop_single_share_product',
				'id_new' => 'shop_single_social_share',
			],
			[
				'id'     => 'shop_share_product_rounded_icons',
				'id_new' => 'shop_single_social_share_style',
				'filter' => function ( $value ) {
					return $value ? null : 'plain';
				},
			],
			[
				'id'     => 'shop_share_product_networks',
				'id_new' => 'shop_single_social_share_networks',
				'filter' => 'kalium_smof_filter_share_networks',
			],

			// Single page
			[
				'id'         => 'shop_single_product_images_layout',
				'id_new'     => 'shop_single_gallery_type',
				'map_values' => [
					'default'      => 'type-1',
					'plain'        => 'type-4',
					'plain-sticky' => 'type-4',
				],
			],
			[
				'id'         => 'shop_single_product_images_layout',
				'id_new'     => 'shop_single_sticky_description',
				'map_values' => [
					'plain-sticky' => true,
				],
			],
			[
				'id'         => 'shop_single_image_column_size',
				'id_new'     => 'shop_single_gallery_size',
				'map_values' => [
					'small'  => 'small',
					'medium' => 'medium',
					'large'  => 'medium',
					'xlarge' => 'large',
				],
			],
			[
				'id'         => 'shop_single_image_alignment',
				'id_new'     => 'shop_single_gallery_inverted',
				'map_values' => [
					'left'  => false,
					'right' => true,
				],
			],
			[
				'id'     => 'shop_single_plain_image_stretch',
				'id_new' => 'shop_single_gallery_stretch',
				'filter' => 'kalium_validate_boolean',
			],
			[
				'id'     => 'shop_single_image_carousel_transition_type',
				'id_new' => 'shop_single_gallery_carousel_animation',
			],
			[
				'id'     => 'shop_single_auto_rotate_image',
				'id_new' => 'shop_single_gallery_autoplay',
			],
			[
				'id'         => 'shop_single_rating_style',
				'id_new'     => 'shop_single_reviews_rating_style',
				'map_values' => [
					'stars'      => 'stars',
					'circles'    => 'circles',
					'rectangles' => 'circles',
				],
			],
			[
				'id'     => 'shop_related_products_columns',
				'id_new' => 'shop_single_related_products_columns',
			],
			[
				'id'     => 'shop_related_products_per_page',
				'id_new' => 'shop_single_related_products_limit',
			],
			[
				'id'     => 'shop_related_products_per_page',
				'id_new' => 'shop_single_related_products',
				'filter' => 'kalium_validate_boolean',
			],
			[
				'id'     => 'shop_single_product_sku_visibility',
				'id_new' => 'shop_single_meta_sku',
			],
			[
				'id'     => 'shop_single_product_categories_visibility',
				'id_new' => 'shop_single_meta_categories',
			],
			[
				'id'     => 'shop_single_product_tags_visibility',
				'id_new' => 'shop_single_meta_tags',
			],
			[
				'id'     => 'shop_single_product_image_lightbox',
				'id_new' => 'shop_single_gallery_image_enlarge',
			],
			[
				'id'     => 'shop_single_product_image_zoom',
				'id_new' => 'shop_single_gallery_image_click_action',
				'filter' => function ( $value, $smof_values ) {
					if ( ! ( $value || kalium_get_array_key( $smof_values, 'shop_single_product_image_lightbox/value' ) ) && kalium_can_override_theme_option( 'shop_single_gallery_image_click' ) ) {
						set_theme_mod( 'shop_single_gallery_image_click', false );
					}

					return $value ? 'zoom' : 'lightbox';
				},
			],
			[
				'callback' => function ( $smof_values ) {
					if ( kalium_get_array_key( $smof_values, 'shop_single_product_custom_image_size/value' ) ) {
						$width = kalium_get_array_key( $smof_values, 'shop_single_product_custom_image_size_width/value' );

						if ( is_numeric( $width ) ) {
							update_option( 'woocommerce_single_image_width', $width );
						}
					}
				},
			],

			// Sidebar on single product page
			[
				'id'         => 'shop_single_sidebar_position',
				'id_new'     => 'shop_single_sidebar',
				'map_values' => [
					'hide'  => false,
					'right' => true,
					'left'  => true,
				],
			],
			[
				'id'         => 'shop_single_sidebar_position',
				'id_new'     => 'shop_single_sidebar_align',
				'map_values' => [
					'left'  => 'left',
					'right' => 'right',
				],
			],

			// Header mini cart
			[
				'id'     => 'shop_cart_icon_menu',
				'id_new' => 'header_mini_cart',
			],
			[
				'id'     => 'shop_cart_icon_menu_count',
				'id_new' => 'header_cart_counter_badge',
			],
			[
				'id'     => 'shop_cart_icon_menu_hide_empty',
				'id_new' => 'header_cart_hide_empty',
			],
			[
				'id'         => 'shop_cart_contents',
				'id_new'     => 'header_cart_click_action',
				'map_values' => [
					'hide'          => 'cart-page',
					'show-on-click' => 'mini-cart',
					'show-on-hover' => 'mini-cart',
				],
			],
			[
				'id'         => 'shop_cart_contents',
				'id_new'     => 'header_cart_popup_show_on',
				'map_values' => [
					'show-on-click' => 'click',
					'show-on-hover' => 'hover',
				],
			],
			[
				'id'         => 'shop_cart_icon',
				'id_new'     => 'header_cart_icon',
				'map_values' => [
					'ecommerce-cart-content' => 'icon-2',
					'ecommerce-bag'          => 'icon-1',
					'ecommerce-basket'       => 'icon-3',
				],
			],

			/**
			 * Other options.
			 */

			// Legacy form styles
			[
				'callback' => function () {
					if ( kalium_can_override_theme_option( 'legacy_form_style' ) ) {
						set_theme_mod( 'legacy_form_style', true );
					}
				},
			],
		];

		kalium_migrate_smof_batch( $smof_migrations );

		return true;
	},

	/**
	 * Migrate ACF fields.
	 *
	 * @return bool
	 */
	static function () {
		$before_date = strtotime( '1 September 2024' );

		$acf_migrations = [

			// Header style
			[
				'id'          => 'page_header_styling',
				'filter'      => function ( $value, $post_id ) {
					$bg_color       = kalium_get_field( 'header_background_color', $post_id );
					$border_color   = kalium_get_field( 'header_bottom_border', $post_id );
					$bottom_spacing = kalium_get_field( 'header_bottom_spacing', $post_id );

					if ( $bg_color || $border_color || $bottom_spacing ) {
						$header_style = kalium_get_acf_header_style_value( $post_id );

						if ( $bg_color ) {
							kalium_set_array_key(
								$header_style,
								[
									'container_background' => $bg_color,
								]
							);
						}

						if ( $border_color ) {
							kalium_set_array_key(
								$header_style,
								[
									'container_border_color' => $border_color,
								]
							);
						}

						if ( $bottom_spacing ) {
							kalium_set_array_key(
								$header_style,
								[
									'container_bottom_spacing' => $bottom_spacing,
								]
							);
						}

						// Update field value
						update_field( 'header_style', $header_style, $post_id );
					}
				},
				'before_date' => $before_date,
			],

			// Custom menu skin
			[
				'id'          => 'custom_menu_skin',
				'id_new'      => 'page_header_styling',
				'filter'      => function ( $value, $post_id ) {
					$is_custom_skin = in_array( $value, [ 'menu-skin-main', 'menu-skin-dark', 'menu-skin-light' ] );

					if ( $is_custom_skin ) {
						$header_style  = kalium_get_acf_header_style_value( $post_id );
						$primary_color = kalium_color_reference_value( kalium_color_reference( 'theme_colors.color-1' ) );
						$alt_color     = '#ffffff';

						switch ( $value ) {
							case 'menu-skin-dark':
								$primary_color = '#333333';
								break;

							case 'menu-skin-light':
								$primary_color = '#ffffff';
								$alt_color     = '#333333';
								break;
						}

						kalium_set_array_key(
							$header_style,
							[
								'links'              => [
									'normal' => $primary_color,
									'hover'  => $primary_color,
									'active' => $primary_color,
								],
								'submenu_background' => $primary_color,
								'submenu_links'      => [
									'normal' => $alt_color,
									'hover'  => $alt_color,
									'active' => $alt_color,
								],
								'text'               => $alt_color,
							]
						);

						// Update header style
						update_field( 'header_style', $header_style, $post_id );

						return 'yes';
					}
				},
				'before_date' => $before_date,
			],

			// Custom sticky menu skin
			[
				'id'          => 'sticky_menu_skin',
				'filter'      => function ( $value, $post_id ) {
					$is_custom_skin = in_array( $value, [ 'menu-skin-main', 'menu-skin-dark', 'menu-skin-light' ] );

					if ( $is_custom_skin ) {
						$sticky_header_style = kalium_get_acf_sticky_header_style_value( $post_id );
						$primary_color       = kalium_color_reference_value( kalium_color_reference( 'theme_colors.color-1' ) );

						switch ( $value ) {
							case 'menu-skin-dark':
								$primary_color = '#333333';
								break;

							case 'menu-skin-light':
								$primary_color = '#ffffff';
								break;
						}

						kalium_set_array_key(
							$sticky_header_style,
							[
								'links' => [
									'normal' => $primary_color,
									'hover'  => $primary_color,
									'active' => $primary_color,
								],
								'text'  => $primary_color,
							]
						);

						// Update sticky header style
						update_field( 'sticky_header_style', $sticky_header_style, $post_id );
					}
				},
				'before_date' => $before_date,
			],

			// Container width
			[
				'id'          => 'grid_container_width',
				'id_new'      => 'grid_container_width_custom',
				'filter'      => function ( $value, $post_id ) {
					if ( is_numeric( $value ) ) {
						update_field( 'grid_container_width', 'custom', $post_id );

						return $value;
					}
				},
				'before_date' => $before_date,
			],

			// Masonry mode gallery
			[
				'id'          => 'masonry_mode_gallery',
				'filter'      => function ( $value, $post_id ) {
					if ( kalium_validate_boolean( $value ) ) {
						return;
					}

					$column_variations = array_unique( @wp_list_pluck( kalium_get_field( 'gallery', $post_id ), 'column_width' ) );

					if ( 1 < count( $column_variations ) ) {
						return '1';
					}
				},
				'before_date' => $before_date,
			],
		];

		kalium_migrate_acf_batch( $acf_migrations );

		return true;
	},

	/**
	 * Migrate TypoLab settings.
	 */
	static function () {
		$font_sizes = TypoLab::get_option( 'font_appearance', [] );

		$clean_empty_units = function ( $value ) {
			if ( in_array( $value, [ 'px', 'rem', 'em', 'pt', 'vw', '%' ] ) ) {
				return null;
			}

			return $value;
		};

		$is_empty = function ( $value ) use ( &$clean_empty_units ) {
			if ( is_array( $value ) ) {
				return 0 === count( array_filter( array_map( $clean_empty_units, $value ) ) );
			}

			return true;
		};

		// Base font size
		if ( isset( $font_sizes['paragraphs']['p']['font_size'] ) ) {
			$base_font_size = array_map( $clean_empty_units, $font_sizes['paragraphs']['p']['font_size'] );

			if ( ! $is_empty( $base_font_size ) ) {
				$font_sizes['base-font']['body']['font_size'] = $base_font_size;
			}
		}

		// Revert to default Kalium 3 font size
		if ( $is_empty( $font_sizes['base-font']['body']['font_size'] ?? null ) ) {
			$font_sizes['base-font']['body']['font_size'] = [
				'general' => '15px',
			];
		}

		// Top header bar font size
		$header_top_bar = array_map( $clean_empty_units, $font_sizes['header']['header_top_bar']['font_size'] ?? [] );

		if ( $is_empty( $header_top_bar ) ) {
			$font_sizes['header']['header_top_bar']['font_size'] = [
				'general' => '14px',
				'tablet'  => '14px',
				'mobile'  => '14px',
			];
		}

		// Update Font Sizes
		TypoLab::set_option( 'font_appearance', $font_sizes );

		return true;
	},

	/**
	 * Adjust imported demo content pack from previous version.
	 */
	static function () {
		if ( kalium_is_imported_demo_content() ) {
			$site = kalium_get_imported_demo_content();

			// TMP
			$site = str_replace( 'wpb-', '', $site );

			// Backup current style.css file
			kalium_migrate_stylesheet_file();

			// Extra settings to apply for each demo to make it look alike
			$extras = kalium_get_complementary_demo_content_options( $site );

			// Overwrite theme mods
			foreach ( $extras['theme_mods'] as $theme_mod => $value ) {
				if ( ! is_string( $theme_mod ) ) {
					continue;
				}

				set_theme_mod( $theme_mod, $value );
			}

			// Extra custom CSS
			$custom_css   = wp_get_custom_css();
			$custom_start = '/* Kalium Migrated Custom CSS - START */';
			$custom_end   = '/* Kalium Migrated Custom CSS - END */';

			if ( $extras['custom_css'] && false === strpos( $custom_css, $custom_start ) ) {
				$custom_css .= PHP_EOL . PHP_EOL . $custom_start . PHP_EOL;
				$custom_css .= $extras['custom_css'];
				$custom_css .= PHP_EOL . $custom_end;

				wp_update_custom_css_post( trim( $custom_css ) );
			}

			// Extra custom code
			if ( $custom_code = $extras['custom_code'] ) {
				$functions_file = get_stylesheet_directory() . DIRECTORY_SEPARATOR . 'functions.php';
				$functions_code = file_get_contents( $functions_file );

				if ( false === strpos( $functions_code, $custom_code ) ) {
					$functions_code .= PHP_EOL . PHP_EOL . $custom_code;

					file_put_contents( $functions_file, $functions_code );
				}
			}

			// Font sizes
			if ( ! empty( $extras['font_sizes'] ) ) {
				TypoLab::set_option( 'font_appearance', $extras['font_sizes'] );
			}

			// Fonts
			if ( ! empty( $extras['fonts'] ) ) {
				TypoLab::set_option( 'registered_fonts', $extras['fonts'] );
			}
		}

		return true;
	},

	/**
	 * Miscellaneous stuff.
	 */
	static function () {
		if ( ! function_exists( 'deactivate_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		// Deactivate portfolio post type plugin
		deactivate_plugins( 'portfolio-post-type/portfolio-post-type.php' );

		// Delete update_themes transient
		delete_site_transient( 'update_themes' );

		// Flush rewrite rules
		flush_rewrite_rules();

		return true;
	},
];
