<?php
/**
 * Kalium WordPress Theme
 *
 * Products Carousel widget.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

if ( ! kalium()->is->woocommerce_active() ) {
	return;
}

vc_map(
	[
		'base'        => 'lab_products_carousel',
		'name'        => 'Products Carousel',
		'description' => 'Display shop products with Touch Carousel.',
		'class'       => 'vc_lab_products_carousel',
		'icon'        => kalium()->file_url( 'includes/wpbakery/assets/images/carousel.svg' ),
		'controls'    => 'full',
		'category'    => [ 'Laborator', 'WooCommerce' ],
		'params'      => [
			[
				'type'        => 'loop',
				'heading'     => 'Products Query',
				'param_name'  => 'products_query',
				'settings'    => [
					'size'      => [
						'hidden' => false,
						'value'  => 12,
					],
					'order_by'  => [ 'value' => 'date' ],
					'post_type' => [
						'value'  => 'product',
						'hidden' => false,
					],
				],
				'description' => 'Create WordPress loop, to populate products from your site.',
			],
			[
				'type'        => 'dropdown',
				'heading'     => 'Filter Products by Type',
				'param_name'  => 'product_types_to_show',
				'value'       => [
					'Show all types of products from the above query'  => '',
					'Show only featured products from the above query.'  => 'only_featured',
					'Show only products on sale from the above query.'  => 'only_on_sale',
				],
				'description' => 'Filter products from the above query to show featured or on sale products.',
			],
			[
				'type'        => 'dropdown',
				'heading'     => 'Columns',
				'param_name'  => 'columns',
				'std'         => 4,
				'value'       => [
					'1 column'  => 1,
					'2 columns' => 2,
					'3 columns' => 3,
					'4 columns' => 4,
					'5 columns' => 5,
					'6 columns' => 6,
				],
				'description' => 'Select number of columns to show products.',
			],
			[
				'type'        => 'dropdown',
				'heading'     => 'Columns Tablet',
				'param_name'  => 'columns_tablet',
				'std'         => 3,
				'value'       => [
					'1 column'  => 1,
					'2 columns' => 2,
					'3 columns' => 3,
					'4 columns' => 4,
					'5 columns' => 5,
					'6 columns' => 6,
				],
				'description' => 'Select number of columns to show products in tablet.',
			],
			[
				'type'        => 'dropdown',
				'heading'     => 'Columns Mobile',
				'param_name'  => 'columns_mobile',
				'std'         => 1,
				'value'       => [
					'1 column'  => 1,
					'2 columns' => 2,
					'3 columns' => 3,
					'4 columns' => 4,
					'5 columns' => 5,
					'6 columns' => 6,
				],
				'description' => 'Select number of columns to show products in mobile.',
			],
			[
				'type'        => 'textfield',
				'heading'     => 'Auto Rotate',
				'param_name'  => 'auto_rotate',
				'value'       => '5',
				'description' => 'You can set automatic rotation of carousel, unit is seconds. Enter 0 to disable.',
			],
			[
				'type'        => 'textfield',
				'heading'     => 'Extra class name',
				'param_name'  => 'el_class',
				'value'       => '',
				'description' => 'If you wish to style particular content element differently, then use this field to add a class name and then refer to it in your css file.',
			],

			[
				'type'       => 'css_editor',
				'heading'    => 'Css',
				'param_name' => 'css',
				'group'      => 'Design options',
			],
		],
	]
);

class WPBakeryShortCode_Lab_Products_Carousel extends WPBakeryShortCode {

	/**
	 * Ids to exclude in products query
	 *
	 * @var array
	 */
	private $exclude_ids = [];

	/**
	 * Tax query.
	 *
	 * @var array
	 */
	private $tax_query = [];

	/**
	 * Shortcode content.
	 *
	 * @param array  $atts
	 * @param string $content
	 *
	 * @return string
	 */
	public function content( $atts, $content = '' ) {
		if ( ! kalium()->is->woocommerce_active() ) {
			return '';
		}

		// Attributes
		if ( function_exists( 'vc_map_get_attributes' ) ) {
			$atts = vc_map_get_attributes( $this->getShortcode(), $atts );
		}

		// Set posts_per_page default value
		kalium_vc_loop_param_set_default_value( $atts['products_query'], 'size', '12' );

		// Extract attributes
		extract( $atts );

		// Element class
		$class     = $this->getExtraClass( $el_class );
		$css_class = apply_filters( VC_SHORTCODE_CUSTOM_CSS_FILTER_TAG, $class, $this->settings['base'], $atts ) . vc_shortcode_custom_css_class( $css, ' ' );

		$classes = [
			'woocommerce',
			'lab-products-carousel',
			'products',
			$css_class,
		];

		// Generate query using WC_Shortcode_Products class
		$query_args = kalium_wpb_build_query_args( $products_query );

		$args = [
			'columns' => $columns,
		];

		// Columns
		$columns = [
			'desktop' => $columns,
			'tablet'  => $columns_tablet,
			'mobile'  => $columns_mobile,
		];

		$type = 'products';

		// Items per page
		if ( ! empty( $query_args['posts_per_page'] ) ) {
			$args['limit'] = $query_args['posts_per_page'];
		}

		// Order column
		if ( ! empty( $query_args['orderby'] ) ) {
			$args['orderby'] = $query_args['orderby'];
		}

		// Order direction
		if ( ! empty( $query_args['order'] ) ) {
			$args['order'] = $query_args['order'];
		}

		// Tax Query
		if ( ! empty( $query_args['tax_query'] ) ) {
			$this->tax_query = $categories = [];

			foreach ( $query_args['tax_query'] as $i => $tax ) {

				if ( is_numeric( $i ) && ! empty( $tax['taxonomy'] ) ) {
					// Product Categories
					if ( 'product_cat' == $tax['taxonomy'] ) {
						if ( 'NOT IN' == strtoupper( $tax['operator'] ) ) {
							$this->tax_query[] = $tax;
						} else {
							foreach ( $tax['terms'] as $term_id ) {
								if ( $term = get_term( $term_id, 'product_cat' ) ) {
									$categories[] = $term->slug;
								}
							}
						}
					}
					// Other terms
					else {
						$this->tax_query[] = $tax;
					}
				}
			}

			// Categories
			$args['category'] = implode( ',', $categories );

			// Add tax query to products query
			if ( count( $this->tax_query ) ) {
				add_filter( 'woocommerce_shortcode_products_query', [ $this, 'add_tax_query' ], 100 );
			}
		}

		// Include post ids
		if ( ! empty( $query_args['post__in'] ) ) {
			$args['ids'] = implode( ',', $query_args['post__in'] );
		}

		// Exclude post ids
		if ( ! empty( $query_args['post__not_in'] ) ) {
			$this->exclude_ids = $query_args['post__not_in'];
			add_filter( 'woocommerce_shortcode_products_query', [ $this, 'exclude_ids' ], 100 );
		}

		// Featured items only
		if ( 'only_featured' == $product_types_to_show ) {
			$args['visibility'] = 'featured';
			$type               = 'featured_products';
		}

		// On sale products
		if ( 'only_on_sale' == $product_types_to_show ) {
			$type = 'sale_products';
		}

		// Products
		$shortcode = new WC_Shortcode_Products( $args, $type );
		$products  = $shortcode->get_content();
		$slides    = [];

		if ( preg_match_all( '/<li\b[^>]*>(.*?)<\/li>/is', $products, $matches ) ) {
			$slides = array_map(
				function ( $slide ) {
					return preg_replace( '/(type-product)/', '$1 swiper-slide', $slide, 1 );
				},
				$matches[0]
			);
		}

		return kalium_slider(
			[
				'tag_name'        => 'ul',
				'class'           => $classes,
				'slides'          => $slides,
				'echo'            => false,
				'slides_per_view' => $columns,
				'autoplay'        => $auto_rotate,
				'space_between'   => kalium_parse_responsive_value( 30 ),
			]
		);
	}

	/**
	 * Exclude Ids from query.
	 *
	 * @param array $query
	 *
	 * @return array
	 */
	public function exclude_ids( $query ) {
		if ( empty( $query['post__not_in'] ) ) {
			$query['post__not_in'] = [];
		}

		// Exclude ids
		$query['post__not_in'] = array_merge( $query['post__not_in'], $this->exclude_ids );

		// Remove filter after execution
		remove_filter( 'woocommerce_shortcode_products_query', [ $this, 'exclude_ids' ], 100 );

		return $query;
	}

	/**
	 * Add tax query.
	 *
	 * @param array $query
	 *
	 * @return array
	 */
	public function add_tax_query( $query ) {
		$tax_query_default = [
			'field'    => 'term_id',
			'taxonomy' => '',
			'operator' => 'IN',
			'terms'    => [],
		];

		if ( empty( $query['tax_query'] ) ) {
			$query['tax_query'] = [
				'relation' => 'AND',
			];
		}

		foreach ( $this->tax_query as $tax_query ) {
			$query['tax_query'][] = array_merge( $tax_query_default, $tax_query );
		}

		// Remove filter after execution
		add_filter( 'woocommerce_shortcode_products_query', [ $this, 'add_tax_query' ], 100 );

		return $query;
	}
}
