<?php
/**
 * Kalium WordPress Theme
 *
 * Portfolio items Elementor widget.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Elementor\Widgets;

use Elementor\Plugin;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Kalium\Elementor\Helpers;
use Kalium\Elementor\Control_Sets\Query_Posts;

class Portfolio_Items extends Widget_Base {

	/**
	 * Widget name.
	 *
	 * @return string
	 */
	public function get_name() {
		return 'kalium-portfolio-items-legacy';
	}

	/**
	 * Widget title.
	 *
	 * @return string
	 */
	public function get_title() {
		return 'Portfolio Items';
	}

	/**
	 * Widget icon.
	 *
	 * @return string
	 */
	public function get_icon() {
		return 'eicon-apps';
	}

	/**
	 * Widget category.
	 *
	 * @return array
	 */
	public function get_categories() {
		return [ 'kalium' ];
	}

	/**
	 * Script depends.
	 *
	 * @return array
	 */
	public function get_script_depends() {
		return kalium()->assets->get_scripts_list( 'isotope' );
	}

	/**
	 * Widget controls.
	 *
	 * @return void
	 */
	protected function register_controls() {

		// Inherit from theme options
		$label_inherit_theme_options = 'Inherit from Theme Options';

		/**
		 * Heading section.
		 */

		// Heading section start
		$this->start_controls_section(
			'section_heading',
			[
				'label' => 'Heading',
			]
		);

		// Control: Title
		$this->add_control(
			'show_heading',
			[
				'label'   => 'Show Heading',
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);

		// Control: Title
		$this->add_control(
			'title',
			[
				'label'       => 'Title',
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'default'     => 'Portfolio',
				'condition'   => [
					'show_heading' => 'yes',
				],
			]
		);

		// Control: Description
		$this->add_control(
			'description',
			[
				'label'     => 'Description',
				'type'      => Controls_Manager::WYSIWYG,
				'default'   => 'Our everyday work is presented here, we do what we love,
Case studies, video presentations and photo-shootings below.',
				'condition' => [
					'show_heading' => 'yes',
				],
			]
		);

		// Layout
		$this->add_control(
			'heading_layout',
			[
				'label'   => 'Layout',
				'type'    => Controls_Manager::SELECT,
				'options' => [
					''       => 'Inherit from Theme Options',
					'type-1' => 'Type 1',
					'type-2' => 'Type 2',
					'type-3' => 'Type 3',
				],
			]
		);

		// Heading section end
		$this->end_controls_section();

		// Filters section start
		$this->start_controls_section(
			'section_filters',
			[
				'label' => 'Filters',
			]
		);

		// Control: Category filter
		$this->add_control(
			'filters',
			[
				'label'       => 'Filters',
				'description' => 'Show portfolio items filters.',
				'type'        => Controls_Manager::SWITCHER,
				'label_on'    => 'Yes',
				'label_off'   => 'No',
				'default'     => 'yes',
			]
		);

		foreach ( $this->get_portfolio_taxonomies() as $tax_data ) {
			$taxonomy        = $tax_data['taxonomy'];
			$taxonomy_prefix = $tax_data['prefix'];

			$field_enable  = 'enable_' . $taxonomy_prefix . '_filter';
			$field_default = 'default_' . $taxonomy_prefix . '_filter';

			// Control: Enabled state
			$this->add_control(
				$field_enable,
				[
					'label'     => 'Enable ' . $tax_data['label'],
					'type'      => Controls_Manager::SWITCHER,
					'label_on'  => 'Yes',
					'label_off' => 'No',
					'separator' => 'before',
					'default'   => $tax_data['enabled'] ? 'yes' : false,
					'condition' => [
						'filters' => 'yes',
					],
				]
			);

			// Control: Default selected taxonomy
			$this->add_control(
				$field_default,
				[
					'label'       => 'Default ' . $tax_data['label'] . ' Filter',
					'description' => 'Set default ' . $tax_data['singular'] . ' to filter portfolio items at first page load.',
					'type'        => Controls_Manager::SELECT,
					'label_block' => true,
					'options'     => Helpers::get_terms_options(
						$taxonomy,
						[
							'field'   => 'slug',
							'default' => 'All',
						]
					),
					'condition'   => [
						'filters'     => 'yes',
						$field_enable => 'yes',
					],
				]
			);
		}

		// End of filters section
		$this->end_controls_section();

		/**
		 * Query section.
		 */

		// Query portfolio section
		Query_Posts::add_controls(
			$this,
			[
				'post_type'  => 'portfolio',
				'term_field' => 'slug',
			]
		);

		/**
		 * Pagination section.
		 */

		// Pagination section start
		$this->start_controls_section(
			'section_pagination',
			[
				'label' => 'Pagination',
			]
		);

		// Control: Pagination type
		$this->add_control(
			'pagination_type',
			[
				'label'   => 'Pagination Type',
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'infinite-scroll' => 'Load More',
					'static-link'     => 'Static Link',
					''                => 'No Pagination',
				],
				'default' => 'infinite-scroll',
			]
		);

		// Control: Show More Button Text
		$this->add_control(
			'pagination_static_link_button_text',
			[
				'label'       => 'Show More Button Text',
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__( 'Show more', 'kalium' ),
				'condition'   => [ 'pagination_type' => 'static-link' ],
			]
		);

		// Control Button URL
		$this->add_control(
			'pagination_static_link_button_link',
			[
				'label'       => 'Button Link',
				'type'        => Controls_Manager::URL,
				'options'     => [ 'url', 'is_external' ],
				'label_block' => true,
				'placeholder' => 'Paste URL or type a page title',
				'condition'   => [ 'pagination_type' => 'static-link' ],
			]
		);

		// Control: Auto reveal
		$this->add_control(
			'pagination_infinite_scroll_auto_reveal',
			[
				'label'       => 'Infinite Scroll',
				'description' => 'Load next page as soon as user reaches the end of viewport.',
				'type'        => Controls_Manager::SWITCHER,
				'label_on'    => 'Yes',
				'label_off'   => 'No',
				'condition'   => [ 'pagination_type' => 'infinite-scroll' ],
			]
		);

		// Control: Show More Button Text
		$this->add_control(
			'pagination_infinite_scroll_button_text',
			[
				'label'       => 'Show More Button Text',
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__( 'Show more', 'kalium' ),
				'condition'   => [ 'pagination_type' => 'infinite-scroll' ],
			]
		);

		// Control: Reached End Text
		$this->add_control(
			'pagination_infinite_scroll_reached_end_text',
			[
				'label'       => 'Reached End Text',
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__( 'No more portfolio items to show', 'kalium' ),
				'condition'   => [ 'pagination_type' => 'infinite-scroll' ],
			]
		);

		// Pagination section end
		$this->end_controls_section();

		/**
		 * Layout section.
		 */

		// Layout section start
		$this->start_controls_section(
			'section_layout',
			[
				'label' => 'Layout',
				'tab'   => Controls_Manager::TAB_LAYOUT,
			]
		);

		// Control: Portfolio item layout
		$this->add_control(
			'portfolio_item_layout_type',
			[
				'label'       => 'Item layout type',
				'label_block' => true,
				'type'        => Controls_Manager::SELECT,
				'options'     => [
					''       => $label_inherit_theme_options,
					'type-1' => 'Item thumbnail + details below',
					'type-2' => 'Item thumbnail + details over thumbnail',
				],
			]
		);

		// Control: Layout mode
		$this->add_control(
			'portfolio_layout_mode',
			[
				'label'       => 'Layout mode',
				'label_block' => true,
				'type'        => Controls_Manager::SELECT,
				'options'     => [
					''         => $label_inherit_theme_options,
					'packery'  => 'Masonry',
					'fit-rows' => 'Fit Rows',
				],
			]
		);

		// Control: Portfolio columns
		$this->add_responsive_control(
			'portfolio_columns',
			[
				'label' => 'Columns',
				'type'  => Controls_Manager::NUMBER,
				'min'   => 0,
				'max'   => 6,
			]
		);

		// Control: Portfolio gap
		$this->add_responsive_control(
			'portfolio_gap',
			[
				'type'      => Controls_Manager::SLIDER,
				'label'     => 'Gap',
				'selectors' => [
					'{{WRAPPER}} .portfolio-items' => sprintf( '%s: {{SIZE}}{{UNIT}}', kalium_css_var_name( 'grid-gap' ) ),
				],
			]
		);

		// Control: Reveal effect
		$this->add_control(
			'portfolio_items_reveal_effect',
			[
				'label'       => 'Reveal Effect',
				'label_block' => true,
				'type'        => Controls_Manager::SELECT,
				'options'     => [
					''               => $label_inherit_theme_options,
					'none'           => 'None',
					'fade'           => 'Fade',
					'slidenfade'     => 'Slide and fade',
					'zoom'           => 'Zoom in',
					'fade-one'       => 'Fade (one by one)',
					'slidenfade-one' => 'Slide and fade (one by one)',
					'zoom-one'       => 'Zoom in (one by one)',
				],
			]
		);

		// Layout section end
		$this->end_controls_section();

		/**
		 * Custom masonry section.
		 */

		// Custom masonry section
		$this->start_controls_section(
			'section_custom_masonry',
			[
				'label' => 'Custom Masonry',
				'tab'   => Controls_Manager::TAB_LAYOUT,
			]
		);

		// Custom masonry enabled state
		$this->add_control(
			'custom_masonry',
			[
				'label' => 'Custom Masonry Pattern',
				'type'  => Controls_Manager::SWITCHER,
			]
		);

		// Masonry items
		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'box_size',
			[
				'name'    => 'box_size',
				'label'   => 'Box Size',
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'3x3'  => '25% - small',
					'3x4'  => '25% - medium',
					'3x6'  => '25% - large',

					'4x3'  => '33% - small',
					'4x4'  => '33% - medium',
					'4x6'  => '33% - large',

					'5x3'  => '40% - small',
					'5x4'  => '40% - medium',
					'5x6'  => '40% - large',

					'6x3'  => '50% - small',
					'6x4'  => '50% - medium',
					'6x6'  => '50% - large',

					'8x3'  => '65% - small',
					'8x4'  => '65% - medium',
					'8x6'  => '65% - large',

					'9x3'  => '75% - small',
					'9x4'  => '75% - medium',
					'9x6'  => '75% - large',

					'12x4' => '100% - small',
					'12x5' => '100% - medium',
					'12x6' => '100% - large',
				],
				'default' => '4x3',
			]
		);

		$this->add_control(
			'custom_masonry_items',
			[
				'label'       => 'Pattern',
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $repeater->get_controls(),
				'default'     => [
					[ 'box_size' => '4x3' ],
					[ 'box_size' => '8x3' ],
					[ 'box_size' => '8x3' ],
					[ 'box_size' => '4x3' ],
				],
				'title_field' => <<<EOD
<# 
	var boxWidth = { 
			3: 25, 
			4: 33, 
			5: 40, 
			6: 50, 
			8: 65, 
			9: 75, 
			12: 100 
		},
		boxHeight = { 
			3: 'S', 
			4: 'M', 
			5: 'M', 
			6: 'L',
		},
		boxSize = box_size.split( 'x' );
		
		if ( '12x4' === box_size ) {
			boxSize[1] = 3;
		}
#>
~{{{ boxWidth[ boxSize[0] ] }}}% - {{{ boxHeight[ boxSize[1] ] }}}
EOD,
				'condition'   => [
					'custom_masonry' => 'yes',
				],
			]
		);

		// Layout section end
		$this->end_controls_section();

		/**
		 * Featured image section.
		 */

		// Featured image section
		$this->start_controls_section(
			'section_featured_image',
			[
				'label' => 'Featured Image',
				'tab'   => Controls_Manager::TAB_LAYOUT,
			]
		);

		// Image size
		$this->add_control(
			'thumbnail_image_size',
			[
				'label'   => 'Image Size',
				'type'    => Controls_Manager::SELECT,
				'options' => array_merge(
					[
						'' => $label_inherit_theme_options,
					],
					kalium_get_image_size_names( [ 'custom' => true ] )
				),
			]
		);

		// Custom image size
		$this->add_control(
			'thumbnail_image_size_custom',
			[
				'label'     => 'Custom Image Size',
				'type'      => Controls_Manager::IMAGE_DIMENSIONS,
				'condition' => [
					'thumbnail_image_size' => 'custom',
				],
			]
		);

		$this->add_responsive_control(
			'thumbnail_aspect_ratio',
			[
				'type'    => Controls_Manager::SELECT,
				'label'   => 'Aspect Ratio',
				'options' => array_merge(
					[
						'' => $label_inherit_theme_options,
					],
					kalium_get_aspect_ratio_names(
						[
							'custom' => true,
						]
					)
				),
			]
		);

		$this->add_responsive_control(
			'thumbnail_aspect_ratio_custom',
			[
				'type'        => Controls_Manager::TEXT,
				'label'       => 'Custom Aspect Ratio',
				'placeholder' => 'e.g. 16 / 9',
				'ai'          => [
					'active' => false,
				],
				'condition'   => [
					'thumbnail_aspect_ratio' => 'custom',
				],
			]
		);

		// End featured image section
		$this->end_controls_section();
	}

	/**
	 * Get portfolio categories.
	 *
	 * @return array
	 */
	protected function get_portfolio_taxonomies() {
		return array_map(
			static function ( $taxonomy ) {
				static $tax_data = [
					'portfolio_category' => [
						'prefix'   => 'category',
						'label'    => 'Categories',
						'singular' => 'Category',
						'enabled'  => true,
					],
					'portfolio_tag'      => [
						'prefix'   => 'tag',
						'label'    => 'Tags',
						'singular' => 'Tag',
					],
				];

				return [
					'taxonomy' => $taxonomy,
					'prefix'   => $tax_data[ $taxonomy ]['prefix'],
					'label'    => $tax_data[ $taxonomy ]['label'],
					'singular' => $tax_data[ $taxonomy ]['singular'],
					'enabled'  => $tax_data[ $taxonomy ]['enabled'] ?? false,
				];
			},
			kalium()->portfolio->get_taxonomies()
		);
	}

	/**
	 * Render widget.
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		// Portfolio is not enabled
		if ( ! kalium()->portfolio->is_enabled() ) {
			return null;
		}

		// Custom loop options
		$custom_loop_options = [
			'url' => get_permalink( get_queried_object() ),
		];

		// Query args
		$custom_loop_options['query_vars'] = Query_Posts::to_query_args(
			Query_Posts::get_value(
				$this,
				[
					'post_type' => 'portfolio',
				]
			),
			[
				'term_field' => 'slug',
			]
		);

		// Query
		$custom_loop_options['query'] = new \WP_Query(
			kalium_get_portfolio_loop_query_args(
				$custom_loop_options['query_vars'],
				false
			)
		);

		// Heading
		$custom_loop_options['heading']['title']['visible'] = kalium_validate_boolean( $settings['show_heading'] );
		$custom_loop_options['heading']['title']['text']    = $settings['title'];
		$custom_loop_options['heading']['title']['content'] = kalium_format_content( $settings['description'] );

		if ( $heading_layout = $settings['heading_layout'] ) {
			$custom_loop_options['heading']['layout'] = $heading_layout;
		}

		// Filtering
		$custom_loop_options['filtering']['enabled']    = kalium_validate_boolean( $settings['category_filter'] ?? $settings['filters'] );
		$custom_loop_options['filtering']['taxonomies'] = [];

		foreach ( $this->get_portfolio_taxonomies() as $tax_data ) {
			$taxonomy        = $tax_data['taxonomy'];
			$taxonomy_prefix = $tax_data['prefix'];

			$field_enable  = 'enable_' . $taxonomy_prefix . '_filter';
			$field_default = 'default_' . $taxonomy_prefix . '_filter';

			if ( kalium_validate_boolean( $settings[ $field_enable ] ) ) {
				$custom_loop_options['filtering']['taxonomies'][] = $taxonomy;
			}

			// Default term
			if ( $default_term = $settings[ $field_default ] ) {
				$custom_loop_options['filtering']['current'][ $taxonomy ] = $default_term;
			}
		}

		// Layout
		if ( $layout_type = $settings['portfolio_item_layout_type'] ) {
			$custom_loop_options['layout'] = $layout_type;
		}

		foreach ( kalium_parse_elementor_responsive_value( 'portfolio_columns', $settings ) as $viewport => $columns ) {
			if ( is_numeric( $columns ) ) {
				$custom_loop_options['grid']['columns'][ $viewport ] = $columns;
			}
		}

		foreach ( kalium_parse_elementor_responsive_value( 'portfolio_gap', $settings ) as $viewport => $gap ) {
			if ( is_numeric( $gap['size'] ) ) {
				$custom_loop_options['grid']['gap'][ $viewport ] = $gap['size'];
			}
		}

		// Layout mode
		if ( $layout_type = $settings['portfolio_layout_mode'] ) {
			$custom_loop_options['grid']['masonry']['mode'] = $layout_type;
		}

		// Reveal effect
		if ( $reveal_effect = $settings['portfolio_items_reveal_effect'] ) {
			$custom_loop_options['item']['aov']['enabled']          = 'none' !== $reveal_effect;
			$custom_loop_options['item']['aov']['legacy_animation'] = $reveal_effect;
			$custom_loop_options['item']['aov']['animation']        = null;
		}

		// Custom masonry
		if ( $settings['custom_masonry'] ) {
			$masonry_items = $settings['custom_masonry_items'];

			if ( ! empty( $masonry_items ) ) {
				$custom_loop_options['custom_masonry'] = $masonry_items;
			}
		}

		// Featured image
		if ( $thumbnail_image_size = $settings['thumbnail_image_size'] ) {
			$custom_loop_options['item']['thumbnail']['size'] = $thumbnail_image_size;

			// Custom image size
			if ( 'custom' === $thumbnail_image_size ) {
				$custom_loop_options['item']['thumbnail']['size'] = kalium_list_dimensions( $settings['thumbnail_image_size_custom'] );
			}
		}

		// Aspect ratio
		$aspect_ratio = kalium_parse_elementor_responsive_value( 'thumbnail_aspect_ratio', $settings );

		if ( ! kalium_is_empty_responsive( $aspect_ratio ) ) {
			$custom_aspect_ratio = kalium_parse_elementor_responsive_value( 'thumbnail_aspect_ratio_custom', $settings );

			$custom_loop_options['item']['thumbnail']['aspect_ratio'] = kalium_map_responsive_value(
				$aspect_ratio,
				function ( $value, $viewport ) use ( &$custom_aspect_ratio ) {
					if ( 'custom' === $value ) {
						$custom_ratio = $custom_aspect_ratio;

						return kalium_get_responsive_value( $custom_ratio, $viewport );
					}

					return $value;
				}
			);
		}

		// Pagination
		switch ( $settings['pagination_type'] ) {
			case 'infinite-scroll':
				$custom_loop_options['pagination']['enabled'] = true;
				$custom_loop_options['pagination']['type']    = 'load-more';

				$custom_loop_options['pagination']['load_more']['infinite_scroll'] = kalium_validate_boolean( $settings['pagination_infinite_scroll_auto_reveal'] );
				$custom_loop_options['pagination']['load_more']['view_more_text']  = $settings['pagination_infinite_scroll_button_text'];
				$custom_loop_options['pagination']['load_more']['last_page_text']  = $settings['pagination_infinite_scroll_reached_end_text'];
				break;

			case 'static-link':
				$custom_loop_options['pagination']['enabled'] = true;
				$custom_loop_options['pagination']['type']    = 'more-link';

				$custom_loop_options['pagination']['more_link']['url']    = $settings['pagination_static_link_button_link']['url'];
				$custom_loop_options['pagination']['more_link']['target'] = $settings['pagination_static_link_button_link']['is_external'];
				$custom_loop_options['pagination']['more_link']['text']   = $settings['pagination_static_link_button_text'];
				break;

			case '':
				$custom_loop_options['pagination']['enabled'] = false;
				break;
		}

		// Initialize portfolio loop options
		kalium_init_portfolio_loop_options( $this->get_id(), $custom_loop_options );

		// Heading
		kalium_portfolio_loop_heading();

		// Portfolio items
		kalium_portfolio_items_loop();

		// Pagination
		kalium_portfolio_items_pagination();

		// Reset portfolio loop options
		kalium_reset_portfolio_loop_options();
	}
}

Plugin::instance()->widgets_manager->register( new Portfolio_Items() );
