<?php
/**
 * Kalium WordPress Theme
 *
 * Dashboard page.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Admin;

use Kalium\Utilities\System_Status;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Dashboard {

	/**
	 * Menu slug.
	 *
	 * @const string
	 */
	const MENU_SLUG = 'kalium';

	/**
	 * Menu capability.
	 *
	 * @const string
	 */
	const MENU_CAP = 'edit_theme_options';

	/**
	 * Get link for dashboard page tab.
	 *
	 * @param string $tab
	 *
	 * @return string
	 */
	public static function tab_link( $tab ) {
		return admin_url( 'admin.php?page=' . self::MENU_SLUG . '&tab=' . urlencode( $tab ) );
	}

	/**
	 * Adjustable section visibility.
	 *
	 * @param string $section_name
	 *
	 * @return bool
	 */
	public static function maybe_show_section( $section_name ) {
		return apply_filters( 'kalium_dashboard_maybe_show_section', true, $section_name );
	}

	/**
	 * Construct.
	 */
	public function __construct() {
		if ( ! is_admin() ) {
			return;
		}

		// Hooks
		add_action( 'admin_menu', [ $this, 'menu_item' ], 1 );
		add_action( 'admin_menu', [ $this, 'register_submenu_items' ], 1 );
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueues' ] );
		add_action( 'admin_menu', [ $this, 'current_submenu_item' ], 100 );
		add_action( 'after_setup_theme', [ $this, 'old_urls_redirect' ], 0 );
	}

	/**
	 * Tabs.
	 *
	 * @return array
	 */
	public function get_tabs() {
		return apply_filters(
			'kalium_dashboard_tabs',
			[
				'dashboard'     => [
					'name'       => 'Home',
					'menu_title' => 'Dashboard',
					'main_tab'   => true,
					'content'    => function () {
						$display_name = get_current_user_id() ? wp_get_current_user()->display_name : null;

						// Page heading
						$heading_title       = sprintf( 'Hello, %s 👋', esc_html( $display_name ) );
						$heading_description = 'Easily access site editing sections using the quick links provided. Here, you will also find important information about the theme, including details on your license, system status, and help links to assist you as needed.';

						if ( kalium()->request->has( 'welcome' ) ) {
							$heading_title       = sprintf( 'Welcome, %s 👋', esc_html( $display_name ) );
							$heading_description = 'Welcome aboard! Thanks for choosing ' . kalium_get_name() . ' for your website project. With our extensive range of layouts, rich theme options, and user-friendly drag-and-drop builder, you’re all set to create something truly special. Happy building!';
						}

						$this->page_heading( $heading_title, $heading_description );

						// Dashboard
						kalium()->require_file(
							'includes/admin-templates/dashboard/dashboard.php',
							[
								'welcome'           => kalium()->request->has( 'welcome' ),
								'quick_links'       => $this->get_quick_links(),
								'integrations'      => $this->get_integrations(),
								'latest_version'    => kalium()->get_latest_version(),
								'is_latest_version' => kalium()->is_latest_version(),
								'update_core_link'  => admin_url( 'update-core.php' ),
								'license_data'      => kalium()->license->get_data(),
							]
						);
					},
				],
				'starter-sites' => [
					'name'    => 'Starter Sites',
					'content' => function () {

						// Page heading
						$this->page_heading( 'Starter Sites', 'Choose the starter site to install on your site. You may install one or more starter sites at the same time. You can remove any starter site by simply clicking Uninstall button and choose the content types to remove from your site.' );

						// Starter sites
						kalium()->require_file( 'includes/admin-templates/dashboard/starter-sites.php' );
					},
				],
				'plugins'       => [
					'name'       => 'Plugins',
					'menu_title' => function () {
						$menu_title = 'Plugins';

						if ( $updates_count = kalium_get_plugin_updates_count() ) {
							$menu_title = sprintf( 'Update Plugins <span class="kalium-menu-item-badge">%d</span>', $updates_count );
						}

						return $menu_title;
					},
					'content'    => function () {

						// Page heading
						$this->page_heading( 'Plugins', 'All premium plugins bundled with ' . kalium_get_name() . ' include free updates but require active license. We encourage you to install only the plugins you need, including required ones.' );

						// Plugins
						kalium()->require_file( 'includes/admin-templates/dashboard/plugins.php' );
					},
				],
				'system-status' => [
					'name'    => 'Status',
					'content' => function () {

						// Clipboard
						wp_enqueue_script( 'clipboard' );

						// Init system status vars
						System_Status::init_vars();

						// Page heading
						$this->page_heading( 'System Status', 'Find important information about your server setup. If you see red errors that indicate problems, likely, you\'re not in compliance with <a href="https://docs.kaliumtheme.com/getting-started/introduction/before-getting-started/server-requirements" target="_blank" rel="noopener">Kalium\'s Server Requirements</a>.' );

						// System status
						kalium()->require_file(
							'includes/admin-templates/dashboard/system-status.php',
							[
								'active_plugins' => System_Status::get_active_plugins(),
								'upgrade_tasks'  => array_filter(
									kalium()->version_upgrades->get_upgrade_tasks(),
									function ( $upgrade_task ) {
										return $upgrade_task->is_eligible();
									}
								),
							]
						);
					},
				],
				'settings'      => [
					'name'    => 'Settings',
					'content' => function () {

						// Enqueue Tooltipster
						kalium_enqueue( 'select2' );

						// All sections
						$settings_sections = $this->get_settings_sections();

						// Current/active section
						$section_ids    = wp_list_pluck( $settings_sections, 'id' );
						$active_section = kalium()->request->query( 'section' );

						if ( ! in_array( $active_section, $section_ids ) ) {
							$active_section = reset( $section_ids );
						}

						// Page heading
						$this->page_heading( 'Settings', 'Theme functionality settings, other settings are available in Customizer.' );

						// Settings
						kalium()->require_file(
							'includes/admin-templates/dashboard/settings.php',
							[
								'settings_sections' => $settings_sections,
								'active_section'    => $active_section,
							]
						);

						// Enqueue WP media
						wp_enqueue_media();

						// Register settings in JS
						kalium_admin_js_define_var(
							'dashboard',
							[
								'settings' => kalium()->settings->get_all( true ),
								'nonce'    => kalium()->settings->get_nonce(),
							]
						);
					},
				],
				'changelog'     => [
					'name'      => 'Changelog',
					'menu_item' => false,
					'content'   => function () {

						// Page heading
						$this->page_heading(
							sprintf(
								'New in %s %s',
								esc_html( kalium_get_name() ),
								esc_html( kalium()->get_version() )
							),
							kalium_get_name() . ' is continuously expanding with new features, bug fixes, and adjustments to improve your experience. Scroll down to see what’s new in this version, or <a href="#changelog">check the changelog</a> for a complete list of changes.'
						);

						// Whats new and changelog
						kalium()->require_file(
							'includes/admin-templates/dashboard/changelog.php',
							[
								'version'   => kalium()->get_version(),
								'whats_new' => $this->get_whats_new(),
								'changelog' => $this->get_changelog(),
							]
						);
					},
				],
			]
		);
	}

	/**
	 * Get current tab.
	 *
	 * @return string
	 */
	public function get_current_tab() {
		$current_tab    = kalium()->request->query( 'tab', 'dashboard' );
		$available_tabs = array_keys( $this->get_tabs() );

		if ( in_array( $current_tab, $available_tabs ) ) {
			return $current_tab;
		}

		return reset( $available_tabs );
	}

	/**
	 * Settings sections.
	 *
	 * @return array
	 */
	public function get_settings_sections() {
		return apply_filters(
			'kalium_dashboard_settings_sections',
			[
				[
					'id'       => 'portfolio',
					'title'    => 'Portfolio',
					'icon'     => 'kalium-admin-icon-po-portfolio-item',
					'priority' => 10,
				],
				[
					'id'       => 'backups',
					'title'    => 'Theme Backups',
					'icon'     => 'kalium-admin-icon-refresh',
					'priority' => 100,
				],
			]
		);
	}

	/**
	 * Changelog.
	 *
	 * @return array
	 */
	public function get_changelog() {
		return [

			// Changelog entry
			[
				'expand'  => true,
				'version' => '4.1.3',
				'date'    => '2025-04-03',
				'changes' => [

					// Fix
					'Fix'    => [
						'Product categories count string was not translatable due to missing count parameter in _n() function',
						'Sticky header items clickable through background when not using transparent background',
						'Hamburger menu (toggle bars) not visible when “Hamburger Menu Label” is turned off',
						'Portfolio settings in admin sometimes not showing properly',
						'Portfolio lightbox and filters now properly handle URL hash and push state with unified history updates',
						'Portfolio Load More pagination button not behaving correctly in certain cases',
						'Images with custom aspect ratio appear distorted in certain elements',
						'Various other adjustments and enhancements',
					],

					// Update
					'Update' => [
						'Swiper updated to latest version - 11.2.6',
						'Google Fonts list updated to the latest version',
					],

					// Note
					'Note'   => [
						'Explore our new, better-structured documentation site at: https://docs.kaliumtheme.com for a more organized and user-friendly experience',
						'If you’d like to contribute to translating Kalium into your language, please visit the Kalium Translations site: https://translate.kaliumtheme.com/projects/kalium/',
					],
				],
			],

			// Changelog entry
			[
				'expand'  => false,
				'version' => '4.1.2',
				'date'    => '2025-03-19',
				'changes' => [

					// New
					'New'     => [
						'Product navigation feature – https://d.pr/v/2RUlIO',
						'WooCommerce 9.7.x compatibility added',
					],

					// Improve
					'Improve' => [
						'Post and Portfolio navigation improved with new layout types and greater customization options',
						'Added option to set a default portfolio archive page, updating archive and taxonomy links accordingly – https://d.pr/i/zBDjJg',
						'Improved compatibility with Complianz plugin to prevent JS errors',
						'Added styling controls for the Hamburger Menu in Sticky Header options',
						'Enhanced WPML plugin compatibility',
					],

					// Fix
					'Fix'     => [
						'Portfolio filters on taxonomy pages reload the page instead of filtering via JS',
						'Disabling the Like feature in Portfolio causes JavaScript errors',
						'Product hover block overlaps pagination by preventing clicks',
						'Fullscreen portfolio item title alignment fixed',
						'WPBakery Stretch Row option on nested rows not working properly',
						'Mobile menu closes unexpectedly when clicking anchor (#) links the first time, then works correctly',
						'Active menu color not applying to toggle button in Mobile Menu',
						'Fixed Danish translations',
						'Various minor fixes and refinements',
					],

					// Update
					'Update'  => [
						'Swiper updated to latest version – 11.2.5',
						'Video.js library updated to latest version – 8.22.0',
						'Vimeo Player API updated to latest version – 2.26.0',
						'Google Fonts list updated to the latest version',
					],

					// Note
					'Note'    => [
						'Explore our new, better-structured documentation site at: https://docs.kaliumtheme.com for a more organized and user-friendly experience',
						'If you’d like to contribute to translating Kalium into your language, please visit the Kalium Translations site: https://translate.kaliumtheme.com/projects/kalium/',
					],
				],
			],

			// Changelog entry
			[
				'expand'  => false,
				'version' => '4.1.1',
				'date'    => '2025-02-12',
				'changes' => [

					// Fix
					'Fix'    => [
						'Page heading title causes fatal error with latest WPBakery version',
						'Full-width footer container breaks when content alignment is changed from default',
						'Switching portfolio filters on taxonomy pages (portfolio_category) didn’t load items properly',
						'Portfolio container has irregular gaps when items with featured videos are present',
						'Featured videos add extra space in portfolio grid when using WordPress video player',
						'Unable to delete TypoLab fonts due to fatal PHP errors',
						'Other small fixes and improvements',
					],

					// Update
					'Update' => [
						'Swiper updated to latest version - 11.2.2',
						'Google Fonts list updated to the latest version',
					],
				],
			],

			// Changelog entry
			[
				'expand'  => false,
				'version' => '4.1',
				'date'    => '2025-01-27',
				'changes' => [

					// New
					'New'     => [
						'New Starter Site: Business/SaaS (Elementor) – https://sites.kaliumtheme.com/elementor/business',
						'Portfolio Like feature support in lightbox – https://d.pr/i/oWvn32',
						'WooCommerce 9.6.x compatibility added',
					],

					// Improve
					'Improve' => [
						'Overall performance improvements and decreased execution time',
						'Option to set a custom image size for product category images added – https://d.pr/i/UPwk3K',
					],

					// Fix
					'Fix'     => [
						'Portfolio items not rendered after editing in WPBakery frontend editor',
						'Preselected Portfolio category/tag not working properly (showing all items)',
						'Portfolio Filter: Nested categories/tags not functioning beyond 2 levels deep',
						'Portfolio filters in category/tag not showing items from other categories/tags',
						'Portfolio load-more navigation interfering with layout types when two or more different portfolio sets are included on the same page',
						'Could not update theme plugins via WP-CLI',
						'Distorted videos when viewing in fullscreen mode',
						'Various fixes and improvements',
					],

					// Update
					'Update'  => [
						'Swiper updated to latest version - 11.2.1',
						'Vimeo Player API updated to latest version - 2.25.1',
						'Google Fonts list updated to the latest version',
						'Updated Freemius SDK to the latest version',
					],
				],
			],

			// Changelog entry
			[
				'expand'  => false,
				'version' => '4.0.10',
				'date'    => '2024-12-18',
				'changes' => [

					// New
					'New' => [
						'WooCommerce 9.5.x compatibility added',
					],
				],
			],

			// Changelog entry
			[
				'expand'  => false,
				'version' => '4.0.9',
				'date'    => '2024-12-17',
				'changes' => [

					// New
					'New'     => [
						'New Starter Site: Medical (Elementor) – https://sites.kaliumtheme.com/elementor/medical',
						'Copy/Duplicate Portfolio Item Data – https://d.pr/i/52DRvs https://d.pr/i/HY0keV',
						'Bluesky social icon added – https://d.pr/i/V09WFb',
						'Threads and Bluesky share post social networks added – https://d.pr/i/bWbxXZ',
					],

					// Improve
					'Improve' => [
						'Added support for portfolio tags in single portfolio item navigation as previous/next links – https://d.pr/i/p6ORI2',
						'Option to toggle the portfolio heading container (title and description) – https://d.pr/i/6Bg6pU',
						'Added option to disable autoplay for videos in Lightbox – https://d.pr/i/EHtH67',
						'Improved compatibility with EWWW Image Optimizer plugin',
						'Improved v3-to-v4 upgrade script',
					],

					// Fix
					'Fix'     => [
						'Fixed issue with portfolio filters failing to fetch additional items when a specific taxonomy is selected',
						'Closing Lightbox on portfolio did not restore the URL to its initial path',
						'Broken preview of loading animations for Image Placeholder after WP 6.7 – https://d.pr/i/UxILgE',
						'Fixed PHP warning generated by _load_textdomain_just_in_time function',
						'Reworked fix for portfolio categories linked in the header menu leading to 404 error pages; now fully functional',
						'Fixed issue where WPBakery Heading background color did not apply to the heading element',
						'Infinite Scroll pagination not working with certain configuration options',
						'Cart drawer not closing after changing the quantity of items',
						'Portfolio lightbox item opening twice on a single page',
						'Portfolio item hover overlay icon size incorrect for all items except the first one',
						'Elementor Data Updater failing to complete due to theme-related errors',
						'Deprecation warnings from YITH Wishlisht plugin',
						'WooCommerce quantity buttons deformed due to general button styles',
						'Elementor Portfolio Items widget not visible in edior on Safari',
						'WPBakery Masonry Portfolio not applying 25% item size correctly',
						'Autoplaying featured videos in Portfolio not working with “Infinite Scroll” pagination enabled',
						'General improvements and fixes',
					],

					// Update
					'Update'  => [
						'Video.js library updated to latest version – 8.21.0',
						'Google Fonts list updated to the latest version',
					],

					// Note
					'Note'    => [
						'Merry Christmas and Happy New Year! 🎄🎉',
					],
				],
			],

			// Changelog entry
			[
				'expand'  => false,
				'version' => '4.0.8',
				'date'    => '2024-11-19',
				'changes' => [

					// New
					'New'     => [
						'Lightbox: "Facebook Inspired" backdrop image blur effect added – https://d.pr/i/pSGTP1',
					],

					// Improve
					'Improve' => [
						'Added gap option for Elementor Portfolio Items widget – https://d.pr/i/GyFVlX',
						'Added support for portfolio tags filter in Elementor Portfolio Items widget – https://d.pr/i/UMAdOE',
						'Portfolio filters now support special characters',
					],

					// Fix
					'Fix'     => [
						'Portfolio categories linked in the header menu lead to 404 error page',
						'Improved WPBakery Automotive starter site import',
						'Improved Slider Revolution import on Starter Site',
						'Fixed footer type extends beyond the site frame and spans 100%',
						'Customizer sections not clickable in some areas after WordPress 6.7 update',
						'Other improvements and fixes',
					],

					// Update
					'Update'  => [
						'Swiper updated to latest version - 11.1.15',
					],
				],
			],

			// Changelog entry
			[
				'expand'  => false,
				'version' => '4.0.7',
				'date'    => '2024-11-14',
				'changes' => [

					// Improve
					'Improve' => [
						'Option to control portfolio image sizes in gallery https://d.pr/i/VBT4GA https://d.pr/i/7zI4rh',
						'Option to set columns for tablet and mobile for WPBakery Products Carousel widget',
						'Incorrect WPBakery row width on some browsers when using stretch row option',
					],

					// Fix
					'Fix'     => [
						'Disabling link for individual portfolio item not working',
						'Filtered portfolio items do not maintain order when opened in lightbox',
						'Product Filter plugin not installed during starter site import',
						'Fullscreen portfolio areas incorrectly inherit button styles',
						'Order review content not displaying correctly on checkout page',
						'Blog archive pages do not display the blog page heading',
						'Various other issues and improvements',
					],

					// Update
					'Update'  => [
						'Debounce library updated to latest version – 2.25.0',
						'Google Fonts list updated to the latest version',
					],
				],
			],

			// Changelog entry
			[
				'expand'  => false,
				'version' => '4.0.6',
				'date'    => '2024-11-05',
				'changes' => [

					// New
					'New'     => [
						'Added style options for Backorder badge in WooCommerce',
						'Custom mobile menu colors available in Parameters and Options – https://d.pr/i/x7DHQn',
					],

					// Improve
					'Improve' => [
						'Custom text color for hover overlay for portfolio items – https://d.pr/i/UIhEoT https://d.pr/i/C5w5vX',
					],

					// Fix
					'Fix'     => [
						'Fixed the visited link color for buttons in Elementor, specifically for Safari browsers',
						'Badges using outline variant only and not able to change',
						'Password protected portfolio items exposing the content',
						'Other minor improvements and bug fixes',
					],

					// Update
					'Update'  => [
						'WordPress importer bundled plugin updated to latest version – 0.8.3',
						'Google Fonts list updated to the latest version',
					],

					// Note
					'Note'    => [
						'Explore our new, better-structured documentation site at: https://docs.kaliumtheme.com for a more organized and user-friendly experience',
						'If you’d like to contribute to translating Kalium into your language, please visit the Kalium Translations site: https://translate.kaliumtheme.com/projects/kalium/',
					],
				],
			],

			// Changelog entry
			[
				'expand'  => false,
				'version' => '4.0.5',
				'date'    => '2024-10-30',
				'changes' => [

					// Improve
					'Improve' => [
						'Mobile menu arrow toggles submenu while the link redirects to the assigned URL',
						'Enhanced Starter Site Import for Medical, Travel and Bookstore WPBakery starter sites',
						'Aligned properly WPBakery inner rows with parent rows',
					],

					// Fix
					'Fix'     => [
						'Portfolio items with no featured image showing 404 on single page',
						'Options manager failed to import options file because of JSON errors',
						'Portfolio items not loading in WPBakery Frontend Editor',
						'Like button triggering portfolio lightbox simultaneously',
						'Breadcrumb still showing on blog page even when disabled',
						'Centered fullscreen menu option not working',
						'Portfolio fixed navigation (prev/next) in single page showing incorrectly',
						'Numerous bug fixes and feature refinements',
					],

					// Update
					'Update'  => [
						'Video.js updated to latest version – 8.19.1',
						'Google Fonts list updated to the latest version',
					],
				],
			],

			// Changelog entry
			[
				'expand'  => false,
				'version' => '4.0.4',
				'date'    => '2024-10-23',
				'changes' => [

					// New
					'New'     => [
						'Compatibility with YITH Badge Management plugin added',
					],

					// Improve
					'Improve' => [
						'Other post types can use the WPBakery Blog Posts and Portfolio Items widget',
						'WPBakery Clients element enhancements: Support for aspect ratio height and the box area is now fully clickable',
						'Lazy loading automatically added for portfolio featured images below the fold (viewport)',
						'Enhanced Animate on Viewport handling with on-screen animations',
						'Improved v3-to-v4 upgrade script',
					],

					// Fix
					'Fix'     => [
						'Some portfolio items are missing when using the Load More pagination',
						'Current page in pagination was not selected on the front page when using Numbers pagination',
						'Masonry portfolio images randomly disappearing',
						'Portfolio base slug incompatibility with WPML when set to empty',
						'Videos embeds not previewing in TinyMCE editor',
						'WPML language switcher not showing in the header',
						'Navigation for previous and next portfolio items within the same category not functioning correctly',
						'Incorrect SVG dimensions rendering issue',
						'Redirect to cart page after adding product to cart not working',
						'Cart page emptied when changing the quantity of items',
						'WPBakery Single Image element expanding to 100% of container width',
						'Numerous bug fixes and feature refinements',
					],

					// Update
					'Update'  => [
						'Debounce library updated to latest version – 2.2.0',
						'Google Fonts list updated to the latest version',
					],
				],
			],

			// Changelog entry
			[
				'expand'  => false,
				'version' => '4.0.3',
				'date'    => '2024-10-08',
				'changes' => [

					// Fix
					'Fix'    => [
						'Featured image not opening in lightbox in Portfolio Single page',
						'Fixed an issue where Kalium API fails to function when TinyMCE scripts are loaded',
						'Responsive issue with Side Portfolio item type ',
						'Fullscreen menu items hovering outside the link element',
						'Team members link target not applied on the image',
						'Various enhancements and bug fixes',
					],

					// Update
					'Update' => [
						'Video.js updated to latest version – 8.18.1',
					],
				],
			],

			// Changelog entry
			[
				'expand'  => false,
				'version' => '4.0.2',
				'date'    => '2024-09-26',
				'changes' => [

					// New
					'New' => [
						'Transfer theme options from Child Theme to Parent theme and vice versa – https://d.pr/i/ukJSQV',
					],

					// Fix
					'Fix' => [
						'Theme Options exporter not exporting the Custom CSS',
						'Dominant color causing fatal PHP error on portfolio',
						'WPBakery Image Carousel widget breaking "load-more" pagination functionality',
						'Back to Portfolio Archive link not working in portfolio single page',
						'Product gallery arrows not working and images opening in a new tab',
						'Contact form submission failed due to a JavaScript error',
						'Product gallery thumbnails create a large gap in vertical carousel mode on mobile',
						'Other improvements and fixes',
					],
				],
			],

			// Changelog entry
			[
				'expand'  => false,
				'version' => '4.0.1',
				'date'    => '2024-09-23',
				'changes' => [

					// New
					'New'    => [
						'WooCommerce 9.3.x compatibility added',
						'Compatibility with Classic Widgets plugin added',
						'Improved Starter Sites importing process by generating thumbnails at the end',
					],

					// Fix
					'Fix'    => [
						'Incorrect license expiration date showing on Status page',
						'Footer width creating horizontal scrollbar',
						'JS error on portfolio page with load more pagination',
						'PHP 8.2 warning caused by our theme',
						'Empty gallery item in portfolio page causing Fatal PHP error',
						'Sidebar widget links not clickable on Customizer',
						'Help icon being duplicated on Customizer',
						'Footer visibility on single item not working',
						'Incorrect container width for "Stack" layout type on single product page',
						'Starter Sites autoloading optimized, reducing unnecessary options',
						'Portfolio items incorrect animation showing delays',
						'Removed duplicate heading description or shop tags/category pages',
						'Portfolio filters not showing on archive page',
						'Site frame still applying padding on disabled devices',
						'Sticky header incorrect offset when site frame is enabled',
						'Incorrect quick links on dashboard are fixed',
						'Option to disable the lightbox in Single Portfolio was not working',
						'Sticky description in portfolio displayed incorrectly on mobile',
						'Stretched gallery in Portfolio causing horizontal scroll',
						'Centered gallery mode not working properly in Carousel portfolio item type',
						'Featured videos on portfolio paused unexpectedly',
						'Other improvements and fixes',
					],

					// Update
					'Update' => [
						'Swiper updated to latest version - 11.1.14',
						'Video.js updated to latest version - 8.17.4',
						'Freemius SDK updated to latest version - 2.8.1',
					],
				],
			],

			// Changelog entry
			[
				'expand'  => false,
				'version' => '4.0',
				'date'    => '2024-08-28',
				'changes' => [

					// New
					'New'    => [
						'Refactored the codebase for improved maintainability, performance, and coding standards',
						'Enhanced performance for faster load times and a smoother experience',
						'New theme dashboard for enhanced control and management',
						'Moved theme options to the Customizer for better live preview editing',
						'Fully compatible with Elementor',
						'Enhanced compatibility with Block Editor',
						'Two Elementor widgets included: Blog Posts and Portfolio Items',
						'Integrated Freemius for flexible license management',
						'Theme color palette with options to add custom colors and export palettes',
						'Global recognition of theme color palettes across Block Editor, Elementor, and ACF',
						'11 new Elementor Starter Sites added',
						'Reworked WPBakery Starter Sites',
						'General Settings page in the theme dashboard: Kalium → Settings',
						'Deferred all JavaScript for improved page performance and a higher Core Web Vitals score',
						'Lightbox library is loaded only when gallery appears on viewport',
						'Almost no jQuery dependent - approximately 95% reduction in jQuery usage in the theme',
						'Reduced usage of the GSAP animation library by converting simple animations to CSS for improved performance',
						'Reduced the number of required plugins to just ACF Pro',
						'Latest CSS trends: variables, new properties, and all Baseline 2023 features',
						'Around 25% reduction in overall CSS size',
						'Enhanced overall accessibility to meet WCAG standards, ensuring a better experience for all users',
						'12 new localized theme translations added',
						'RTL support integrated throughout the entire theme',
						'White Label feature for the Agency plan',
						'Reworked Social Networks icons with support for 40+ social networks and custom icons',
						'Reworked Scroll to Top feature',
						'Reworked Maintenance Mode feature',
						'Reworked Site Frame (formerly Theme Border) with border radius support',
						'Supports two media players: Native and Video.js',
						'Fancybox: New general lightbox type implemented',
						'Swiper: New general carousel type implemented',
						'Forms styles: Global forms styling',
						'Buttons styles: Global buttons styling',
						'Link styles with 4 different types',
						'3 predefined container sizes with custom size support',
						'Adjustable block spacing for entry content',
						'Two sidebar styles supported with advanced styling options, including width, spacing, padding and colors',
						'Sticky sidebar support for the entire sidebar or just the last widgets',
						'Featured image support for search results, with options for image size and style',
						'Unified pagination options for post types: Posts, Portfolio, and Products',
						'Footer: Drag-and-drop footer layout builder',
						'Footer: Content now pulled from widgets for greater flexibility',
						'Reworked entire portfolio module for wider support throughout the theme',
						'Portfolio Page: Advanced filtering options with support for two filters (category and tag)',
						'Blog page: Custom styling options for post cards',
						'Blog page: Featured image with size options, aspect ratio and custom style support',
						'Blog page: Enhanced pagination options with greater control over appearance',
						'Post Page: Narrow layout format implemented with width options and wide alignment support',
						'Post Page: Reworked Related Posts section with additional options',
						'Post Page: Reverse navigation option on prev/next links',
						'WooCommerce: Mini Cart (or Card Drawer) implemented with various display options',
						'WooCommerce: Free shipping progress bar implemented',
						'WooCommerce: 3 badge types supported for products',
						'WooCommerce: Sale badge with discount percentage',
						'WooCommerce: List layout type supported on products page',
						'WooCommerce: Columns switcher for products page',
						'WooCommerce: Product card builder, create your own desired card with 12 unique elements',
						'WooCommerce: Option to show/hide sidebar on products page with or without animation',
						'WooCommerce: 6 new category layouts for products page',
						'WooCommerce: Option to add image on login/registration forms on the My Account page for a better appearance',
						'WooCommerce: 2 styles of cart subtotals container on Cart and Checkout page',
						'WooCommerce: Sticky cart subtotals container during the checkout',
						'WooCommerce: Checkout steps implemented',
						'WooCommerce: Full-width products gallery support',
						'WooCommerce: 9 gallery types available for displaying product images',
						'WooCommerce: Fade and slide animations for gallery slider',
						'WooCommerce: Adjustable maximum number of gallery items to display',
						'WooCommerce: Extra tab styles - choose from Tabs, Accordions, or Stacked containers',
						'WooCommerce: Vertical and horizontal tabs supported',
						'WooCommerce: Option to display accordions in product summary container - collapsed or expanded',
						'New theme options export and import interface',
						'Countless additions and improvements in Kalium 4 — so many, we ran out of room on the changelog',
					],

					// Update
					'Update' => [
						'Tested PHP compatibility up to 8.3',
						'WordPress 6.6 compatibility added',
						'WooCommerce 9.2.2 compatibility added',
						'Latest Bootstrap integration (v5.3.3)',
						'Every JavaScript library is now at latest version available',
						'Google Fonts list updated to the latest version',
					],
				],
			],
		];
	}

	/**
	 * Get quick links.
	 *
	 * @return array
	 */
	public function get_quick_links() {
		return [

			// Brand
			[
				'title'       => 'Site Brand',
				'description' => 'Customize your site’s branding elements like logos and taglines.',
				'link'        => kalium()->customize->section_link( 'kalium_styling_brand' ),
			],

			// Header
			[
				'title'       => 'Header',
				'description' => 'Create or tweak your site’s header with different layout options.',
				'link'        => kalium()->customize->section_link( 'kalium_header' ),
			],

			// Colors
			[
				'title'       => 'Colors',
				'description' => 'Adjust colors across your site to match your branding and style.',
				'link'        => kalium()->customize->section_link( 'kalium_styling_colors' ),
			],

			// Footer
			[
				'title'       => 'Footer',
				'description' => 'Design and set up your site’s footer with various customization options.',
				'link'        => kalium()->customize->section_link( 'kalium_footer' ),
			],

			// Performance
			[
				'title'       => 'Performance',
				'description' => 'Optimize your site performance for faster loading times.',
				'link'        => kalium()->customize->section_link( 'kalium_performance' ),
			],

			// Blog
			[
				'title'       => 'Blog Options',
				'description' => 'Configure how your blog page looks with lots of layouts and options.',
				'link'        => kalium()->customize->section_link( 'kalium_blog_archive' ),
			],

			// Post
			[
				'title'       => 'Post Options',
				'description' => 'Manage settings for single posts, such as display and formatting.',
				'link'        => kalium()->customize->section_link( 'kalium_blog_single' ),
			],

			// Portfolio
			[
				'title'       => 'Portfolio',
				'description' => 'Set up and style your portfolio to showcase your work.',
				'link'        => kalium()->customize->section_link( 'kalium_portfolio' ),
				'visible'     => kalium()->is->portfolio_active(),
			],

			[
				'title'       => 'Shop Options',
				'description' => 'Configure your shop\'s layout, design, and display options here.',
				'link'        => kalium()->customize->section_link( 'WooCommerce' ),
				'visible'     => kalium()->is->woocommerce_active(),
			],

			// Social Icons
			[
				'title'       => 'Social Icons',
				'description' => 'Add and style social media icons to fit with the design of your site.',
				'link'        => kalium()->customize->section_link( 'kalium_general_social_icons' ),
			],

			// Sidebars
			[
				'title'       => 'Sidebars',
				'description' => 'Design and arrange sidebar styling and appearance globally in the site.',
				'link'        => kalium()->customize->section_link( 'kalium_sidebars' ),
			],

			// Buttons
			[
				'title'       => 'Button Styling',
				'description' => 'Customize button styles and appearances in global all over the site.',
				'link'        => kalium()->customize->section_link( 'kalium_styling_buttons' ),
			],

			// Forms
			[
				'title'       => 'Form Styling',
				'description' => 'Adjust the forms and inputs appearance for the whole site in one place.',
				'link'        => kalium()->customize->section_link( 'kalium_styling_forms' ),
			],

			// Typography
			[
				'title'       => 'Typography',
				'description' => 'Manage fonts and typography settings across your site.',
				'link'        => admin_url( 'admin.php?page=typolab' ),
			],

			// Custom CSS
			[
				'title'       => 'Custom CSS',
				'description' => 'Add custom CSS to apply specific styles and tweaks to your site.',
				'link'        => kalium()->customize->section_link( 'custom_css' ),
			],

			// Custom JS
			[
				'title'       => 'Custom JavaScript',
				'description' => 'Implement custom JavaScript to add new features and additional functionality.',
				'link'        => kalium()->customize->section_link( 'kalium_general_custom_js' ),
			],
		];
	}

	/**
	 * Get whats new.
	 *
	 * @return array
	 */
	public function get_whats_new() {
		return [
			[
				'image'       => 'new-starter-sites.jpg',
				'title'       => '2 New Starter Sites',
				'description' => 'We’ve added two new Elementor Starter Sites: the brand-new Business starter site, designed for companies and startups, and the popular Medical site, now available in Elementor for healthcare providers.',
				'columns'     => 4,
			],
			[
				'image'       => 'new-sharing-platforms.jpg',
				'title'       => 'Threads and Bluesky Sharing',
				'description' => 'New post sharing platforms added! You can now share your content seamlessly on Threads and Bluesky, expanding your reach to even more social networks. Stay connected and share your updates with ease across these growing platforms.',
			],
			[
				'image'       => 'woocommerce.jpg',
				'title'       => 'WooCommerce Overhaul: Exciting New Features!',
				'description' => 'Added many new features, including a grid/list switcher, card builder, cart drawer, free shipping bar, stepped checkout, and more, to enhance the shopping experience.',
			],
			[
				'image'       => 'colors.jpg',
				'video'       => 'https://kaliumtheme.com/videos/colors.mp4',
				'title'       => 'Global Colors and Palettes',
				'description' => 'Colors and color palettes can be set globally and reused all over the site (Gutenberg, Elementor). Add, remove or export colors.',
			],
			[
				'image'       => 'customizer.jpg',
				'video'       => 'https://kaliumtheme.com/videos/customizer.mp4',
				'title'       => 'Live Theme Options',
				'description' => 'Theme options have moved to the Customizer, edit and preview changes in real-time for a better user experience.',
			],
			[
				'image'       => 'buttons.jpg',
				'video'       => 'https://kaliumtheme.com/videos/forms-buttons-styles.mp4',
				'title'       => 'Buttons and Form Styling',
				'description' => 'Globally control the appearance of buttons and forms to maintain a consistent and professional design throughout your website.',
			],
			[
				'image'       => 'dashboard.jpg',
				'title'       => 'New Theme Dashboard',
				'description' => 'Our new theme dashboard offers improved control and management, with a dedicated general settings.',
			],
			[
				'image'       => 'freemius.jpg',
				'title'       => 'Platform Change',
				'description' => 'We’ve transitioned to Freemius for licensing and purchasing, offering a range of new pricing plans and licensing options to better meet your needs.',
			],
			[
				'image'       => 'block-editor.jpg',
				'title'       => 'Block Editor Integration',
				'description' => 'Enhanced Block Editor integration with front-end matched editing styles, theme.json file adaptation, and additional features for a seamless experience.',
			],
			[
				'image'       => 'white-label.jpg',
				'title'       => 'White Label',
				'description' => 'This feature available in Agency plan allows you to rebrand the theme and offer a branded experience for your clients.',
				'columns'     => 3,
			],
			[
				'image'       => 'elementor-widgets.jpg',
				'video'       => 'https://kaliumtheme.com/videos/elementor-widgets.mp4',
				'title'       => '2 New Elementor Widgets',
				'description' => 'Included in this update are two Elementor widgets: Blog Posts and Portfolio Items, providing you with more tools for Elementor.',
				'columns'     => 3,
			],
			[
				'image'       => 'codebase.jpg',
				'title'       => 'Refactored codebase & better performance',
				'description' => 'Rebuilt the codebase to improve theme performance, resulting in faster load times and compliance with the latest web standards.',
				'columns'     => 3,
			],
			[
				'image'       => 'accessibility.jpg',
				'title'       => 'Accessibility Improvements',
				'description' => 'Designed to meet WCAG accessibility standards, improving keyboard navigation, screen reader support, and color contrast for a better experience for all users',
				'columns'     => 3,
			],
		];
	}

	/**
	 * Get integrations.
	 *
	 * @return array
	 */
	public function get_integrations() {
		return [

			// WPML
			[
				'icon'        => kalium_get_svg_file( 'assets/admin/images/dashboard/integrations-wpml.svg' ),
				'link'        => 'https://kaliumtheme.com/go/wpml',
				'title'       => 'WPML',
				'description' => 'WPML integrates with the Kalium theme for effortless multilingual content management, allowing you to create and maintain a multilingual website with ease.',
			],

			// HubSpot
			[
				'icon'        => kalium_get_svg_file( 'assets/admin/images/dashboard/integrations-hubspot.svg' ),
				'link'        => 'https://kaliumtheme.com/go/hubspot',
				'title'       => 'HubSpot',
				'description' => 'HubSpot’s integration with the Kalium theme provides robust CRM, marketing, and analytics tools to boost your site’s performance and user engagement.',
			],

			// Elementor
			[
				'icon'        => kalium_get_svg_file( 'assets/admin/images/dashboard/integrations-elementor.svg' ),
				'link'        => 'https://kaliumtheme.com/go/elementor',
				'title'       => 'Elementor',
				'description' => 'Elementor works seamlessly with the Kalium theme, offering an intuitive drag-and-drop builder and extensive customization options to enhance your site’s design and functionality.',
			],
		];
	}

	/**
	 * Laborator menu item.
	 */
	public function menu_item() {
		add_menu_page( kalium_get_name(), kalium_get_name(), self::MENU_CAP, self::MENU_SLUG, null, 'none', 2 );
	}

	/**
	 * Register submenu items.
	 */
	public function register_submenu_items() {
		$priority_index = 10;

		foreach ( $this->get_tabs() as $tab_id => $tab ) {
			if ( isset( $tab['tab_item'] ) && false === $tab['tab_item'] ) {
				continue;
			}

			add_action(
				'admin_menu',
				function () use ( $tab_id, $tab ) {
					$title = $tab['menu_title'] ?? $tab['name'];

					// Dynamic title
					if ( is_callable( $title ) ) {
						$title = call_user_func( $title, $tab_id, $tab );
					}

					if ( ! empty( $tab['main_tab'] ) ) {
						add_submenu_page( self::MENU_SLUG, $title, $title, self::MENU_CAP, self::MENU_SLUG, [ $this, 'view_dashboard' ] );
					} elseif ( ! isset( $tab['menu_item'] ) || $tab['menu_item'] ) {
						add_submenu_page( self::MENU_SLUG, $title, $title, self::MENU_CAP, self::tab_link( $tab_id ) );
					}
				},
				$tab['priority'] ?? $priority_index
			);

			$priority_index += 10;
		}
	}

	/**
	 * Set current menu item for Laborator admin menu item.
	 */
	public function current_submenu_item() {
		global $submenu;

		if ( isset( $submenu[ self::MENU_SLUG ] ) ) {
			$laborator_submenu = &$submenu[ self::MENU_SLUG ];

			foreach ( $laborator_submenu as & $submenu_item ) {
				$url = wp_parse_args( $submenu_item[2] );

				if ( $this->get_current_tab() === kalium_get_array_key( $url, 'tab' ) ) {
					$submenu_item[4] = 'current';

					$laborator_submenu[0][2] = self::tab_link( 'dashboard' );
					break;
				}
			}
		}
	}

	/**
	 * Redirect old URLs to their new location.
	 */
	public function old_urls_redirect() {
		global $pagenow;

		if ( ! is_admin() || 'admin.php' !== $pagenow ) {
			return;
		}

		$redirects = [
			'page=laborator_options'             => wp_customize_url(),
			'page=kalium&tab=theme-registration' => kalium_fs()->get_account_url(),
			'page=kalium&tab=whats-new'          => self::tab_link( 'changelog' ),
			'page=kalium&tab=demos'              => self::tab_link( 'starter-sites' ),
			'page=kalium&tab=help'               => self::tab_link( 'dashboard' ),
			'page=kalium&tab=faq'                => self::tab_link( 'dashboard' ),
		];

		foreach ( $redirects as $old_args => $redirect_to ) {
			$old_args = wp_parse_args( $old_args );

			$matched = true;

			foreach ( $old_args as $arg => $value ) {
				if ( kalium()->request->query( $arg ) !== $value ) {
					$matched = false;
					break;
				}
			}

			if ( $matched ) {
				wp_redirect( $redirect_to );
				exit;
			}
		}
	}

	/**
	 * Script enqueues.
	 *
	 * @return void
	 */
	public function enqueues() {

		// Theme dashboard enqueues
		if ( self::MENU_SLUG === kalium()->helpers->get_current_admin_page() ) {
			kalium_enqueue( 'theme-dashboard' );
		}
	}

	/**
	 * About page index.
	 *
	 * @return void
	 */
	public function view_dashboard() {
		$tabs        = $this->get_tabs();
		$current_tab = $this->get_current_tab();
		$tab_entry   = $tabs[ $current_tab ];

		$support_expire_text = 'Support expired!';
		$support_status_code = kalium()->license->get_status_code();

		if ( 1 === $support_status_code ) {
			$support_expire_text = sprintf( 'Support expires in %s!', human_time_diff( 0, kalium()->license->get_time_left() ) );
		} elseif ( 2 === $support_status_code ) {
			$support_expire_text = 'Support active!';
		} elseif ( 3 === $support_status_code ) {
			$support_expire_text = 'License canceled!';
		}

		?>
		<div class="wrap about-wrap kalium-dashboard">

			<?php
			kalium()->require_file(
				'includes/admin-templates/nav.php',
				[
					'page'                => kalium()->helpers->get_current_admin_page(),
					'tabs'                => $tabs,
					'current_tab'         => $current_tab,
					'changelog_link'      => self::tab_link( 'changelog' ),
					'update_core_link'    => admin_url( 'update-core.php' ),

					// Version information
					'latest_version'      => kalium()->get_latest_version(),
					'is_latest_version'   => kalium()->is_latest_version(),

					// License status
					'account_link'        => kalium()->license->get_account_url(),
					'plan_name'           => kalium()->license->get_plan_title(),
					'support_status_code' => $support_status_code,
					'support_expire_text' => $support_expire_text,
				]
			);

			?>
			<div class="kalium-dashboard__content">

				<?php
				if ( is_callable( $tab_entry['content'] ) ) {
					call_user_func( $tab_entry['content'] );
				} elseif ( is_string( $tab_entry['content'] ) ) {
					echo $tab_entry['content']; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				}
				?>

			</div>
			<?php

			// Footer
			$this->footer();
			?>

		</div>
		<?php
	}

	/**
	 * Heading title and description.
	 *
	 * @param string $title
	 * @param string $description
	 */
	public function page_heading( $title, $description = '' ) {
		?>
		<div class="kalium-dashboard__heading">
			<h1><?php echo $title; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?></h1>
			<?php
			if ( $description ) {
				echo wpautop( $description ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			}
			?>
		</div>
		<?php
	}

	/**
	 * About footer.
	 *
	 * @return void
	 */
	public function footer() {
		?>
		<div class="kalium-dashboard__footer wp-clearfix">
			<div class="kalium-dashboard__footer-column">
				<?php
				if ( self::maybe_show_section( 'dashboard-footer-copyrights' ) ) {
					printf( 'Copyright &copy; %s &ndash; %s theme by %s', date_i18n( 'Y' ), esc_html( kalium_get_name() ), kalium_theme_author_link() );
				} else {
					printf( 'Copyright &copy; %s &ndash; %s', date_i18n( 'Y' ), kalium_theme_author_link() );
				}
				?>
			</div>

			<?php if ( self::maybe_show_section( 'dashboard-footer-links' ) ) : ?>
				<div class="kalium-dashboard__footer-column kalium-dashboard__footer-column--right">
					<ul class="kalium-dashboard__footer-links">
						<li>
							<a href="https://kaliumtheme.com/go/kalium-documentation" target="_blank" rel="noreferrer noopener">Documentation</a>
						</li>
						<li>
							<a href="https://www.facebook.com/laboratorcreative" target="_blank" rel="noreferrer noopener">Facebook</a>
						</li>
						<li>
							<a href="https://x.com/thelaborator" target="_blank" rel="noreferrer noopener">X</a>
						</li>
					</ul>
				</div>
			<?php endif; ?>
		</div>
		<?php
	}
}
