<?php
/**
 * Kalium WordPress Theme
 *
 * ACF compatibility.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Compatibility;

use WP_Post;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class ACF {

	/**
	 * Construct.
	 */
	public function __construct() {
		if ( ! kalium()->is->acf_active() ) {
			return;
		}

		// Save and load local fields path
		add_filter( 'acf/settings/save_json', [ $this, 'save_json' ] );
		add_filter( 'acf/settings/load_json', [ $this, 'load_json' ] );

		// ACF custom field types
		kalium()->require_file( 'includes/libraries/acf-revslider-field.php' );

		// Disable revisions in ACF get_field when previewing with Elementor
		add_action( 'wp', [ $this, 'get_field_disable_revisions_workaround' ] );

		// Preselected portfolio item type
		add_action( 'acf/init', [ $this, 'portfolio_preselected_item_type' ] );

		// Location rules
		add_filter( 'acf/location/rule_types', [ $this, 'location_rules' ] );

		// Location rule: portfolio_post_type
		add_filter( 'acf/location/rule_values/portfolio_module', kalium_hook_return_value( [ 'yes' => 'Yes' ] ) );
		add_filter( 'acf/location/rule_match/portfolio_module', [ $this, 'lrm_portfolio_module' ], 10, 3 );

		// Location rule: portfolio_item_type
		add_filter( 'acf/location/rule_values/portfolio_item_type', kalium_hook_return_value( kalium()->portfolio->get_item_types() ) );
		add_filter( 'acf/location/rule_match/portfolio_item_type', [ $this, 'lrm_portfolio_item_type' ], 10, 3 );

		// Theme Color Palette support in Color Picker
		add_action( 'acf/input/admin_footer', [ $this, 'color_picker_theme_palette' ] );

		// Google API key
		add_filter( 'acf/fields/google_map/api', [ $this, 'set_google_api_key' ] );
	}

	/**
	 * ACF fields JSON path.
	 *
	 * @return string
	 * @since 4.1.2
	 */
	public function get_fields_path() {
		return kalium()->locate_file( 'includes/acf-json' );
	}

	/**
	 * Save ACF field groups as JSON in the theme acf-json directory.
	 *
	 * @param string $path
	 *
	 * @return string
	 * @since 4.1.2
	 */
	public function save_json( $path ) {
		if ( ! defined( 'KALIUM_ACF_DEV' ) || ! KALIUM_ACF_DEV ) {
			return $path;
		}

		return $this->get_fields_path();
	}

	/**
	 * Load ACF field groups from the theme acf-json directory.
	 *
	 * @param array $paths
	 *
	 * @return array
	 * @since 4.1.2
	 */
	public function load_json( $paths ) {
		$paths[] = $this->get_fields_path();

		return $paths;
	}

	/**
	 * Get field with fallback function.
	 *
	 * @param string $field_key
	 * @param mixed  $post_id
	 * @param bool   $format_value
	 * @param bool   $escape_html
	 *
	 * @return mixed
	 *
	 * @since 4.1 Added $escape_html parameter
	 */
	public function get_field( $field_key, $post_id = false, $format_value = true, $escape_html = false ) {
		global $post;

		if ( kalium()->is->acf_active() ) {
			return get_field( $field_key, $post_id, $format_value, $escape_html );
		}

		// Get raw field from post
		if ( ! $post_id && $post instanceof WP_Post ) {
			$post_id = $post->ID;
		}

		// Get from post meta
		if ( $post_id ) {
			return get_post_meta( $post_id, $field_key, true );
		}

		return null;
	}

	/**
	 * Preselected portfolio item type.
	 *
	 * @since 3.4.4
	 */
	public function portfolio_preselected_item_type() {
		$item_types        = array_keys( kalium()->portfolio->get_item_types() );
		$default_item_type = kalium()->portfolio->get_default_item_type();

		if ( in_array( $default_item_type, $item_types ) ) {
			$field_key = 'field_54c7b3e324244';
			$field     = acf_get_local_field( $field_key );

			// Set default value for field
			$field['default_value'] = $default_item_type;

			acf_get_local_store( 'fields' )->set( $field_key, $field );
		}
	}

	/**
	 * Disable revisions in ACF get_field when previewing with Elementor.
	 *
	 * This is due to issues with Post Meta fields because get_field function
	 * does not return proper value, but instead it always returns null.
	 *
	 * @since 4.0
	 */
	public function get_field_disable_revisions_workaround() {
		if ( kalium()->is->acf_active() && kalium()->is->elementor_active() && is_preview() ) {
			remove_filter( 'acf/validate_post_id', [ acf()->revisions, 'acf_validate_post_id' ], 10, 2 );
		}
	}

	/**
	 * Custom location rules.
	 *
	 * @param array $rules
	 *
	 * @return array
	 * @since 4.0
	 */
	public function location_rules( $rules ) {
		$rules['Other']['portfolio_module']    = 'Portfolio Module';
		$rules['Other']['portfolio_item_type'] = 'Portfolio Item Type';

		return $rules;
	}

	/**
	 * Portfolio module rule match.
	 *
	 * @param bool  $result
	 * @param array $rule
	 * @param array $screen
	 *
	 * @return bool
	 *
	 * @since 4.0
	 */
	public function lrm_portfolio_module( $result, $rule, $screen ) {
		global $post;

		return in_array( get_post_type( $screen['post_id'] ?? $post ), kalium_supported_portfolio_post_types() );
	}

	/**
	 * Portfolio item type rule match.
	 *
	 * @param bool  $result
	 * @param array $rule
	 * @param array $screen
	 *
	 * @return bool
	 *
	 * @since 4.0
	 */
	public function lrm_portfolio_item_type( $result, $rule, $screen ) {
		$item_type = $screen['item_type'] ?? get_field( 'item_type' );

		return $this->match_by_operator( $rule['value'], $item_type, $rule['operator'] );
	}

	/**
	 * Theme Color Palette in color picker.
	 */
	public function color_picker_theme_palette() {
		$palettes = wp_list_pluck( kalium_get_theme_base_colors(), 'value' );

		?>
		<script type="text/javascript">
			( function () {
				var palettes = <?php echo wp_json_encode( array_slice( $palettes, 0, 10 ) ); ?>;

				acf.add_filter( 'color_picker_args', function ( args ) {
					Object.assign( args, {
						palettes,
					} );

					return args;
				} );
			} )();
		</script>
		<?php
	}

	/**
	 * Set Google Maps API key for ACF.
	 *
	 * @return array
	 */
	public function set_google_api_key() {
		return [
			'libraries' => 'places',
			'key'       => kalium_get_google_api_key(),
		];
	}

	/**
	 * Match by operator.
	 *
	 * @param mixed $a
	 * @param mixed $b
	 * @param bool  $operator
	 *
	 * @return bool
	 * @since 4.0
	 */
	private function match_by_operator( $a, $b, $operator ) {
		if ( '==' === $operator ) {
			return $a == $b; // phpcs:ignore Universal.Operators.StrictComparisons.LooseEqual
		} elseif ( '!=' !== $operator ) {
			return $a != $b; // phpcs:ignore Universal.Operators.StrictComparisons.LooseEqual
		}

		return false;
	}
}
