<?php
/**
 * Kalium WordPress Theme
 *
 * Freemius integration.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Compatibility;

use Freemius as FreemiusMain;
use Kalium\Admin\Dashboard;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * @mixin FreemiusMain
 */
class Freemius {

	/**
	 * Construct.
	 */
	public function __construct() {
		add_action( 'kalium_fs_loaded', [ $this, 'init' ] );
	}

	/**
	 * Access to Freemius API.
	 *
	 * @param string $name
	 * @param array  $args
	 *
	 * @return mixed
	 */
	public function __call( $name, $args ) {
		/** @var FreemiusMain $kalium_fs */
		global $kalium_fs;

		return call_user_func_array( [ $kalium_fs, $name ], $args );
	}

	/**
	 * Init.
	 */
	public function init() {

		// Activate theme sticky message text
		$activate_theme_text = sprintf( 'Thanks for installing %s! <b><a href="%s">Please enter your license key to unlock all features.</a></b>', kalium_get_theme_name(), $this->get_activation_url() );
		$this->add_filter( 'sticky_message_connect_account', kalium_hook_return_value( $activate_theme_text ) );

		// Disable pricing page in admin menu
		$this->add_filter( 'is_submenu_visible', [ $this, 'disable_pricing_page' ], 10, 2 );

		// Pricing and checkout page URLs
		$license_key  = kalium()->license->get_license_key();
		$pricing_url  = add_query_arg( 'license_key', urlencode( $license_key ?? '' ), 'https://kaliumtheme.com/pricing' );
		$checkout_url = add_query_arg(
			[
				'plan'          => kalium()->license->get_plan_name(),
				'billing_cycle' => kalium()->license->is_lifetime() ? WP_FS__PERIOD_LIFETIME : WP_FS__PERIOD_ANNUALLY,
			],
			$pricing_url
		);

		$this->add_filter( 'pricing_url', kalium_hook_return_value( $pricing_url ), 100 );
		$this->add_filter( 'checkout_url', kalium_hook_return_value( $checkout_url ), 100 );

		$this->add_action( 'connect/before_message', [ $this, 'license_dialog_theme_icon' ] );
		$this->add_action( 'connect/before_message', [ $this, 'license_dialog_title' ] );

		// Hide affiliate notice
		$this->add_filter( 'show_affiliate_program_notice', '__return_false' );

		// Renew license link
		add_action( 'admin_menu', [ $this, 'renew_menu_item' ], PHP_INT_MAX );

		// Replace strings
		$this->replace_freemius_strings();
	}

	/**
	 * Disable pricing page.
	 *
	 * @param bool   $visible
	 * @param string $id
	 *
	 * @return bool
	 */
	public function disable_pricing_page( $visible, $id ) {
		return 'pricing' === $id ? false : $visible;
	}

	/**
	 * Connect dialog theme icon.
	 */
	public function license_dialog_theme_icon() {
		?>
		<div class="kalium-fs-icon">
			<i class="kalium-admin-icon-kalium"></i>
		</div>
		<?php
	}

	/**
	 * Connect dialog title.
	 */
	public function license_dialog_title() {
		printf( '<h1>Activate %s</h1>', $this->get_plugin_title() );
	}

	/**
	 * Add renew link in the menu.
	 */
	public function renew_menu_item() {
		$license = kalium()->license->get_license();

		if ( is_object( $license ) && $license->is_expired() ) {
			add_submenu_page( Dashboard::MENU_SLUG, 'Renew License', 'Renew License &rarr;', Dashboard::MENU_CAP, $this->checkout_url() );
			add_action( 'admin_footer', kalium_hook_echo_value( '<style>#toplevel_page_kalium .wp-submenu > li:last-child a { font-weight: bold; color: #7ad90f; } </style>' ) );
		}
	}

	/**
	 * Replace Freemius SDK strings.
	 */
	private function replace_freemius_strings() {
		$this->override_i18n(
			[
				'yee-haw'                => 'Awesome',
				'plan-activated-message' => kalium_get_theme_name() . ' was successfully activated.',
				'plan-upgraded-message'  => kalium_get_theme_name() . ' was successfully upgraded.',
			]
		);
	}
}
