<?php
/**
 * Kalium WordPress Theme
 *
 * Helper functions
 *
 * @link https://kaliumtheme.com
 */
namespace Kalium\Core;

use WP_HTML_Tag_Processor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Helpers {

	/**
	 * Admin notices to show.
	 *
	 * @var array
	 */
	private $admin_notices = [];

	/**
	 * Construct.
	 */
	public function __construct() {
		add_action( 'admin_notices', [ $this, 'show_admin_notices' ], 1000 );
	}

	/**
	 * Show admin notices.
	 */
	public function show_admin_notices() {
		foreach ( $this->admin_notices as $notice ) {
			$classes = [
				'laborator-notice',
				'notice',
				'notice-' . $notice['type'],
			];

			if ( $notice['dismissible'] ) {
				$classes[] = 'is-dismissible';
			}

			if ( ! empty( $notice['class'] ) ) {
				$classes[] = $this->tokenize_list( $notice['class'] );
			}

			// Print notice
			kalium_element(
				[
					'tag_name'   => 'div',
					'attributes' => [
						'id'    => ! empty( $notice['id'] ) ? "kalium-notice-{$notice['id']}" : null,
						'class' => $classes,
					],
					'content'    => wpautop( $notice['message'] ),
					'echo'       => true,
				]
			);
		}
	}

	/**
	 * Add admin notice.
	 *
	 * @param string|array $message
	 * @param string       $type
	 * @param bool         $dismissible
	 * @param string       $id
	 */
	public function add_admin_notice( $message, $type = 'success', $dismissible = true, $id = null ) {
		if ( is_array( $message ) ) {
			$args = wp_parse_args(
				$message,
				[
					'message'     => null,
					'type'        => $type,
					'dismissible' => $dismissible,
					'class'       => null,
					'id'          => $id,
				]
			);

			extract( $args );
		}

		// Allowed notice types
		if ( ! in_array( $type, [ 'success', 'error', 'warning', 'theme' ] ) ) {
			$type = 'info';
		}

		$this->admin_notices[] = [
			'message'     => $message,
			'type'        => $type,
			'dismissible' => (bool) $dismissible,
			'id'          => $id,
			'class'       => $class ?? null,
		];
	}

	/**
	 * Let to Num.
	 *
	 * @param int $size
	 *
	 * @return int
	 */
	public function let_to_num( $size ) {
		$l   = substr( $size, -1 );
		$ret = (int) substr( $size, 0, -1 );

		switch ( strtoupper( $l ) ) {
			case 'P':
				$ret *= 1024;
				// No break.
			case 'T':
				$ret *= 1024;
				// No break.
			case 'G':
				$ret *= 1024;
				// No break.
			case 'M':
				$ret *= 1024;
				// No break.
			case 'K':
				$ret *= 1024;
				// No break.
		}

		return $ret;
	}

	/**
	 * Get SVG dimensions from viewBox.
	 *
	 * @param string|int $file
	 *
	 * @return array
	 */
	public function get_svg_dimensions( $file ) {
		$width = $height = 1;

		// Get attached file
		if ( is_numeric( $file ) ) {
			$file = get_attached_file( $file );
		}

		$svg = @file_get_contents( $file );

		if ( preg_match( '/<svg.*?>/s', $svg, $svg_line ) && preg_match_all( '/(width|height|viewBox)="([^"]+)"/', $svg_line[0], $matches ) ) {
			foreach ( $matches[1] as $i => $attr_name ) {
				$attr_value = $matches[2][ $i ];

				if ( 'width' === $attr_name && $attr_value > 1 ) {
					$width = $attr_value;
				} elseif ( 'height' === $attr_name && $attr_value > 1 ) {
					$height = $attr_value;
				} elseif ( 'viewBox' === $attr_name ) {
					$view_box = array_values( array_filter( array_map( 'absint', explode( ' ', $attr_value ) ) ) );
				}

				if ( $i > 2 ) {
					break;
				}
			}

			if ( 1 === $width && $width === $height && isset( $view_box ) ) {
				list( $width, $height ) = $view_box;
			}
		}

		return [ $width, $height ];
	}

	/**
	 * Add body class.
	 *
	 * @param string|array $classes
	 */
	public function add_body_class( $classes = '' ) {

		// This method has no effect after wp_head is executed
		if ( did_action( 'wp_head' ) && ! doing_action( 'wp_head' ) ) {
			kalium_doing_it_wrong( __FUNCTION__, 'The WP Hook "wp_head" was already executed.', '3.0' );
		}

		add_filter( 'body_class', kalium_hook_add_array_value( $this->tokenize_list( $classes ) ) );
	}

	/**
	 * Space-separated list of sanitized tokens.
	 *
	 * @param string|array $tokens
	 * @param bool         $echo
	 *
	 * @return string|void
	 * @since 4.1.2
	 */
	public function tokenize_list( $tokens, $echo = false ) {
		if ( ! is_array( $tokens ) ) {
			$tokens = (array) $tokens;
		}

		$tokens = trim(
			implode(
				' ',
				array_map(
					'esc_attr',
					array_filter(
						$tokens
					)
				)
			)
		);

		if ( $echo ) {
			echo $tokens;
			return;
		}

		return $tokens;
	}

	/**
	 * Aria expanded boolean parser.
	 *
	 * @param bool|string $state
	 *
	 * @return string
	 */
	public function aria_expanded( $state ) {
		return $state ? 'true' : 'false';
	}

	/**
	 * List element attributes.
	 *
	 * @param array $attributes
	 * @param bool  $echo
	 *
	 * @return string|void
	 */
	public function list_attributes( $attributes, $echo = true ) {
		$attributes_list = [];

		if ( is_array( $attributes ) ) {
			foreach ( $attributes as $attribute_name => $attribute_value ) {
				if ( is_null( $attribute_value ) || false === $attribute_value ) {
					continue;
				}

				// Class attribute
				if ( 'class' === $attribute_name ) {
					$attribute_value = $this->tokenize_list( $attribute_value );
				} elseif ( ! is_scalar( $attribute_value ) ) {
					$attribute_value = wp_json_encode( $attribute_value );
				}

				$attributes_list[] = sprintf( '%s="%s"', $attribute_name, esc_attr( $attribute_value ) );
			}
		}

		$attributes_list = implode( ' ', $attributes_list );

		if ( $echo ) {
			echo $attributes_list;
			return;
		}

		return $attributes_list;
	}

	/**
	 * Build DOM content element.
	 *
	 * @param string|array $tag_name
	 * @param array        $attributes
	 * @param string       $content
	 * @param bool         $close
	 *
	 * @return string|void
	 */
	public function build_dom_element( $tag_name, $attributes = [], $content = null, $close = true ) {
		if ( is_array( $tag_name ) ) {
			$args = wp_parse_args(
				$tag_name,
				[
					'tag_name'   => '',
					'content'    => $content,
					'attributes' => $attributes,
					'close'      => $close,
					'echo'       => false,
				]
			);

			$tag_name   = $args['tag_name'];
			$attributes = $args['attributes'];
			$content    = $args['content'];
			$close      = $args['close'];
		}

		// If no tag is present
		if ( empty( $tag_name ) ) {
			return '';
		}

		// Self closing tags
		$self_closing_tags = [ 'img', 'br', 'input' ];

		// Parse attributes
		$attributes = wp_parse_args( $attributes );

		// Attributes build
		$attributes_str = [];

		foreach ( $attributes as $attribute_name => $attribute_value ) {
			$attr_name = sanitize_title( $attribute_name );

			// Skip null and false attributes
			if ( is_null( $attribute_value ) || ( is_bool( $attribute_value ) && false === $attribute_value ) ) {
				continue;
			}

			// Class attribute
			if ( 'class' === $attribute_name && is_array( $attribute_value ) ) {
				$attr_value = $this->tokenize_list( $attribute_value );
			} else { // Other attributes
				$attr_value = esc_attr( is_scalar( $attribute_value ) ? $attribute_value : wp_json_encode( $attribute_value ) );
			}

			if ( true === $attribute_value ) {
				$attributes_str[] = $attribute_name;
			} else {
				$attributes_str[] = sprintf( '%1$s="%2$s"', $attr_name, $attr_value );
			}
		}

		// Self closing tag
		if ( in_array( strtolower( $tag_name ), $self_closing_tags ) ) {
			$element = sprintf( '<%s %s />', $tag_name, implode( ' ', $attributes_str ) );
		} else {
			$element = sprintf( '<%1$s %2$s>%3$s', $tag_name, implode( ' ', $attributes_str ), $content );

			if ( $close ) {
				$element .= '</' . $tag_name . '>';
			}
		}

		// Echo element
		if ( isset( $args ) && $args['echo'] ) {
			echo $element;

			return;
		}

		return $element;
	}

	/**
	 * Build CSS props.
	 *
	 * @param array $props
	 *
	 * @return string
	 */
	public function build_css_props( $props ) {
		$props_str = [];

		if ( is_array( $props ) ) {
			foreach ( $props as $prop_name => $prop_value ) {
				if ( empty( $prop_value ) ) {
					continue;
				}

				$props_str[] = $prop_name . ':' . $prop_value;
			}
		}

		return implode( ';', $props_str );
	}

	/**
	 * Parse HTML element tag name and attributes.
	 *
	 * @param string $html
	 *
	 * @return array
	 */
	public function parse_html_element( $html ) {
		$results = [
			'tag_name'   => '',
			'attributes' => [],
		];

		// Find nearest match
		if ( preg_match( '#(<)(?<tag_name>[a-zA-Z0-9\-._:]+)((\s)+(?<attributes>.*?))?(>)#ms', $html, $matches ) ) {
			// Tag name and content
			$results['tag_name'] = strtoupper( $matches['tag_name'] );

			// Attributes
			if ( ! empty( $matches['attributes'] ) && preg_match_all( '#(?<attribute_name>[a-z0-9\-_]+)(=("|\')(?<attribute_value>[^"\']+)("|\'))?#im', $matches['attributes'], $matched_attributes ) ) {
				foreach ( $matched_attributes['attribute_name'] as $i => $attribute_name ) {
					$results['attributes'][ $attribute_name ] = $matched_attributes['attribute_value'][ $i ] ?: true;
				}
			}
		}

		return $results;
	}

	/**
	 * Add attribute to matched elements by tag name.
	 *
	 * @param string $html
	 * @param string $tag_name
	 * @param string $attribute_name
	 * @param mixed  $attribute_value
	 *
	 * @return string
	 */
	public function add_element_attribute( $html, $tag_name, $attribute_name, $attribute_value = '' ) {
		$tag_name        = sanitize_title( $tag_name );
		$attribute_name  = sanitize_title( $attribute_name );
		$attribute_value = is_scalar( $attribute_value ) ? $attribute_value : wp_json_encode( $attribute_value );

		if ( class_exists( 'WP_HTML_Tag_Processor' ) ) {
			$processor = new WP_HTML_Tag_Processor( $html );
			while ( $processor->next_tag( $tag_name ) ) {
				$processor->set_attribute( $attribute_name, $attribute_value );
			}

			$html = $processor->get_updated_html();
		} elseif ( ! is_bool( $attribute_value ) || $attribute_value ) {
			$pattern = "/<$tag_name\b[^>]*>/i";

			if ( is_bool( $attribute_value ) ) {
				$replacement = sprintf( '<%s %s', $tag_name, $attribute_name );
			} else {
				$replacement = sprintf( '<%s %s="%s"', $tag_name, $attribute_name, esc_attr( $attribute_value ) );
			}
			$html = preg_replace_callback(
				$pattern,
				function ( $matches ) use ( $tag_name, $replacement ) {
					return str_replace( "<$tag_name", $replacement, $matches[0] );
				},
				$html
			);
		}

		return $html;
	}

	/**
	 * Convert string to camel case.
	 *
	 * @param string $str
	 * @param bool   $capitalize_first_character
	 *
	 * @return string
	 */
	public function to_camelcase( $str, $capitalize_first_character = false ) {
		$str = str_replace( [ '-', '_', ' ' ], '', ucwords( $str, '-_ ' ) );

		if ( ! $capitalize_first_character ) {
			$str = lcfirst( $str );
		}

		return $str;
	}

	/**
	 * Convert underline or camel case string to dashes.
	 *
	 * @param string $string
	 *
	 * @return string
	 */
	public function to_dashes( $string ) {
		return preg_replace_callback(
			'/([A-Z])/',
			function ( $word ) {
				return '-' . strtolower( $word[1] );
			},
			$this->to_camelcase( $string )
		);
	}

	/**
	 * Convert to underscores.
	 *
	 * @param string $string
	 *
	 * @return string
	 * @since 4.0
	 */
	public function to_underscores( $string ) {
		return str_replace( '-', '_', $this->to_dashes( $string ) );
	}

	/**
	 * Starts with.
	 *
	 * @param string $haystack
	 * @param string $needle
	 *
	 * @return bool
	 * @since 4.0
	 */
	public function str_starts_with( $haystack, $needle ) {
		if ( function_exists( 'str_starts_with' ) ) {
			return str_starts_with( $haystack, $needle );
		}

		if ( '' === $needle ) {
			return true;
		}

		return 0 === strpos( $haystack, $needle );
	}

	/**
	 * Get plugin basename from plugin slug.
	 *
	 * @param string $plugin_slug
	 *
	 * @return string|null
	 */
	public function get_plugin_basename( $plugin_slug ) {
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$plugin_basenames = array_keys( get_plugins() );

		foreach ( $plugin_basenames as $plugin_basename ) {
			if ( preg_match( '#^' . $plugin_slug . '/#', $plugin_basename ) ) {
				return $plugin_basename;
			}
		}

		return null;
	}

	/**
	 * Resize dimensions by width.
	 *
	 * @param int $current_width
	 * @param int $current_height
	 * @param int $new_width
	 *
	 * @return array
	 */
	public function resize_by_width( $current_width, $current_height, $new_width ) {
		return [ $new_width, round( $new_width / $current_width * $current_height ) ];
	}

	/**
	 * Resize dimensions by height.
	 *
	 * @param int $current_width
	 * @param int $current_height
	 * @param int $new_height
	 *
	 * @return array
	 */
	public function resize_by_height( $current_width, $current_height, $new_height ) {
		return [ round( $new_height / $current_height * $current_width ), $new_height ];
	}

	/**
	 * Resize the image dimensions to fit the specified aspect ratio and maximum width.
	 *
	 * @param int   $original_width
	 * @param int   $original_height
	 * @param array $aspect_ratio
	 * @param int   $max_width
	 *
	 * @return array An associative array containing the resized width and height.
	 * @since 4.0
	 */
	public function aspect_ratio_fit( $original_width, $original_height, $aspect_ratio, $max_width = null ) {
		$numerator   = $aspect_ratio[0];
		$denominator = $aspect_ratio[1];

		// Calculate the aspect ratio
		$target_aspect_ratio = $numerator / $denominator;

		// Calculate the width and height based on the target aspect ratio
		$target_width  = $max_width ?? $original_width;
		$target_height = $max_width / $target_aspect_ratio;

		// Ensure the resized dimensions do not exceed the original size
		$target_width  = absint( min( $target_width, $original_width ) );
		$target_height = absint( min( $target_height, $original_height ) );

		return [ $target_width, $target_height ];
	}

	/**
	 * Validate boolean value from string, number or bool.
	 *
	 * @param mixed $value
	 *
	 * @return bool
	 * @uses wp_validate_boolean
	 */
	public function validate_boolean( $value ) {
		if ( is_string( $value ) ) {
			if ( 'yes' === $value || bool_from_yn( $value ) ) {
				$value = true;
			} elseif ( 'no' === $value ) {
				$value = false;
			}
		}

		return wp_validate_boolean( $value );
	}

	/**
	 * Escape JSON for use in HTML attributes.
	 *
	 * @param string $json
	 * @param bool   $html
	 *
	 * @return string
	 *
	 * @since 3.0.4
	 */
	public function esc_json( $json, $html = false ) {
		return _wp_specialchars( $json, $html ? ENT_NOQUOTES : ENT_QUOTES, 'UTF-8', true );
	}

	/**
	 * Decode JSON.
	 *
	 * @return array|string
	 */
	public function json_decode() {
		$string = func_get_arg( 0 );

		if ( ! kalium()->is->json( $string ) ) {
			return $string;
		}

		return call_user_func_array( 'json_decode', func_get_args() );
	}

	/**
	 * Get array key that supports path style.
	 *
	 * @param array  $arr
	 * @param string $name
	 * @param mixed  $default
	 *
	 * @return mixed
	 * @since 4.0
	 */
	public function get_array_key( $arr, $name, $default = null ) {
		$path  = explode( '/', $name );
		$value = & $arr;

		while ( $key = array_shift( $path ) ) {
			$value = $arr[ $key ] ?? null;
		}

		if ( is_null( $value ) ) {
			return $default;
		}

		return $value;
	}

	/**
	 * Set array key that supports path style.
	 *
	 * @param array        $arr
	 * @param string|array $name
	 * @param mixed        $value
	 * @param bool         $append_existing
	 *
	 * @since 4.0
	 */
	public function set_array_key( &$arr, $name, $value = null, $append_existing = true ) {

		// Replace multiple keys at once
		if ( is_array( $name ) ) {
			foreach ( $name as $key => $value ) {
				$this->set_array_key( $arr, $key, $value, $append_existing );
			}
			return;
		}

		// Create array
		if ( empty( $arr ) ) {
			$arr = [];
		}

		$name = explode( '/', $name );
		$key  = array_shift( $name );

		// Current value
		$current = & $arr[ $key ];

		while ( $key = array_shift( $name ) ) {
			if ( is_array( $current ) && ! array_key_exists( $key, $current ) ) {
				$current[ $key ] = [];
			}

			$current = & $current[ $key ];
		}

		if ( is_array( $current ) && is_array( $value ) ) {
			$current = $append_existing ? array_replace_recursive( $current, $value ) : $value;
		} else {
			$current = $value;
		}
	}

	/**
	 * Get first array element value (or key).
	 *
	 * @param array $arr
	 * @param bool  $get_key
	 *
	 * @return mixed|null
	 */
	public function get_array_first( $arr, $get_key = false ) {
		if ( is_array( $arr ) && ! empty( $arr ) ) {
			$keys = array_keys( $arr );

			// Get array key
			if ( $get_key ) {
				return $keys[0];
			}

			// Get array value
			return $arr[ $keys[0] ];
		}

		return null;
	}

	/**
	 * Check if an array is associative.
	 *
	 * @param array $arr
	 *
	 * @return bool
	 * @since 4.1.2
	 */
	public function is_array_assoc( $arr ) {
		if ( is_array( $arr ) ) {
			$i = 0;

			foreach ( $arr as $key => $value ) {
				if ( $key !== $i ) {
					return true;
				}

				++$i;
			}
		}

		return false;
	}

	/**
	 * Inserts an array before or after a specified key in the given array.
	 *
	 * If the specified key is not found, the array is inserted at the beginning
	 * or the end of the original array based on the 'before' or 'after' position.
	 *
	 * @param array  $arr
	 * @param string $position
	 * @param string $key
	 * @param array  $insert_array
	 *
	 * @return array
	 * @since  4.0
	 */
	public function insert_at( $arr, $position, $key, $insert_array ) {
		$key_exists = array_key_exists( $key, $arr );

		$result   = [];
		$inserted = false;

		if ( ! $key_exists ) {
			if ( $position === 'before' ) {
				return array_merge( $insert_array, $arr );
			} else {
				return array_merge( $arr, $insert_array );
			}
		}

		foreach ( $arr as $current_key => $value ) {
			if ( $position === 'before' && $current_key === $key ) {
				$result   = array_merge( $result, $insert_array );
				$inserted = true;
			}

			$result[ $current_key ] = $value;

			if ( $position === 'after' && $current_key === $key && ! $inserted ) {
				$result   = array_merge( $result, $insert_array );
				$inserted = true;
			}
		}

		return $result;
	}

	/**
	 * Get current admin page.
	 *
	 * @return string
	 * @since 4.0
	 */
	public function get_current_admin_page() {
		return is_admin() && isset( $_GET['page'] ) ? $_GET['page'] : null;
	}

	/**
	 * Forces decimal point with dot.
	 *
	 * @param float $num
	 *
	 * @return string
	 */
	public function force_decimal_dot( $num ) {
		return str_replace( ',', '.', (string) $num );
	}

	/**
	 * Maybe split dimensions.
	 *
	 * @param string $size
	 *
	 * @return array|string
	 * @since 4.0.7
	 */
	public function maybe_split_dimensions( $size ) {
		if ( is_string( $size ) && preg_match( '/^(\d+)x(\d+)$/', trim( $size ), $matches ) ) {
			return [ $matches[1], $matches[2] ];
		}

		return $size;
	}

	/**
	 * Captures echoed output from a callable (function, closure, etc.) and returns it.
	 *
	 * @param callable $callback
	 * @param mixed    ...$args
	 *
	 * @return string
	 * @since 4.1.1
	 */
	public function capture_output( $callback, ...$args ) {
		ob_start();
		call_user_func_array( $callback, $args );
		return ob_get_clean();
	}

	/**
	 * Validate a value against an accepted list (indexed or associative) with optional lowercasing.
	 *
	 * For an indexed array, the function returns the $current_value if found; otherwise, it returns
	 * the default value (which defaults to the first element of the array if not provided).
	 *
	 * For an associative array, if $current_value is found as a key, the function returns its associated value.
	 * If not found, the default value (defaulting to the first element's value) is returned.
	 *
	 * @param string      $current_value
	 * @param array       $accepted_values
	 * @param string|null $default_value
	 * @param bool        $to_lowercase
	 *
	 * @return string
	 * @since 4.1.2
	 */
	function enum_value( $current_value, $accepted_values, $default_value = null, $to_lowercase = true ) {
		if ( empty( $accepted_values ) ) {
			return $default_value;
		}

		$is_assoc = $this->is_array_assoc( $accepted_values );

		if ( null === $default_value ) {
			$default_value = $is_assoc ? array_keys( $accepted_values ) : $accepted_values;
			$default_value = reset( $default_value );
		}

		if ( $to_lowercase && ! is_null( $current_value ) ) {
			$current_value = strtolower( $current_value );
		}

		if ( $this->is_array_assoc( $accepted_values ) ) {
			return $accepted_values[ $current_value ] ?? $accepted_values[ $default_value ] ?? current( array_values( $accepted_values ) );
		}

		$accepted_lookup = array_flip( $accepted_values );

		return isset( $accepted_lookup[ $current_value ] ) ? $current_value : $default_value;
	}
}
