<?php
/**
 * Kalium WordPress Theme
 *
 * Images handler.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Core;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Images {

	/**
	 * Content before image.
	 *
	 * @var string
	 */
	public $before_image;

	/**
	 * Content after image.
	 *
	 * @var string
	 */
	public $after_image;

	/**
	 * Crop GIFs.
	 *
	 * @var bool
	 */
	public $crop_gifs = false;

	/**
	 * Fluid width.
	 *
	 * @var bool
	 */
	public $fluid_width = true;

	/**
	 * Placeholder dominant color.
	 *
	 * @var bool
	 */
	public $dominant_color = false;

	/**
	 * Get attachment image.
	 *
	 * @param int|string        $attachment_id
	 * @param string            $size
	 * @param array|string|null $atts
	 * @param array|string|null $placeholder_atts
	 *
	 * @return string
	 */
	public function get_image( $attachment_id, $size = 'thumbnail', $atts = null, $placeholder_atts = null ) {

		// Attachment ID
		if ( is_numeric( $attachment_id ) ) {

			// Load original GIF image if enabled
			if ( false === $this->crop_gifs && 'image/gif' === get_post_mime_type( $attachment_id ) ) {
				$size = 'full';
			}

			$image = wp_get_attachment_image( $attachment_id, $size, false, $atts );
		}
		// IMG provided
		elseif ( is_string( $attachment_id ) && false !== strpos( strtolower( $attachment_id ), '<img' ) ) {
			$image = $attachment_id;
		}
		// Get local image
		else {
			$image_file = $this->get_local_image( $attachment_id );

			if ( ! empty( $image_file['image'] ) ) {
				$image = $image_file['image'];
			}
		}

		// No image to show
		if ( empty( $image ) ) {
			return '';
		}

		// Aspect ratio wrapper args
		$wrapper_args = [];

		// Width and height
		if ( preg_match( '/\bwidth=["\']?(\d+)["\']?.*?\bheight=["\']?(\d+)/i', $image, $matches ) ) {
			$wrapper_args['width']  = $matches[1];
			$wrapper_args['height'] = $matches[2];
		}

		// Dominant image background color
		if ( $this->dominant_color ) {
			if ( is_numeric( $attachment_id ) ) {
				$dominant_color = $this->get_dominant_color( $attachment_id );
			}

			if ( ! empty( $dominant_color ) ) {
				$wrapper_args['background'] = $dominant_color;
			}
		}

		return $this->aspect_ratio_wrap( $image, $wrapper_args, $placeholder_atts );
	}

	/**
	 * Aspect ratio wrapper element.
	 *
	 * @param string $element
	 * @param array  $args
	 * @param array  $atts
	 *
	 * @return string
	 */
	public function aspect_ratio_wrap( $element, $args = [], $atts = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'element'    => 'span',
				'width'      => 1,
				'height'     => 1,
				'background' => null,
			]
		);

		// Element type
		$element_type = $args['element'];

		// Placeholder attributes
		$wrapper_attributes = [
			'class' => [
				'image-placeholder',
			],
			'style' => kalium()->helpers->build_css_props(
				[
					// Image aspect ratio
					kalium_css_var_name( 'ratio' ) => number_format( $args['width'] / $args['height'], 6 ),

					// Explicitly defined background
					kalium_css_var_name( 'placeholder-bg' ) => $args['background'],
				]
			),
		];

		// Extend attributes
		foreach ( wp_parse_args( $atts ) as $attr_name => $attr_value ) {
			if ( 'class' === $attr_name ) {
				$extra_class = is_array( $attr_value ) ? $attr_value : explode( ' ', $attr_value );

				$wrapper_attributes[ $attr_name ] = array_merge( $wrapper_attributes[ $attr_name ], $extra_class );
			} elseif ( 'style' !== $attr_name ) {
				$wrapper_attributes[ $attr_name ] = $attr_value;
			}
		}

		// Placeholder classes
		$wrapper_attributes['class'] = kalium_tokenize_list( $wrapper_attributes['class'] );

		// Create placeholder attributes array
		$attributes = [];

		foreach ( $wrapper_attributes as $attr_id => $attr_value ) {
			$attributes[] = sprintf( '%s="%s"', esc_attr( $attr_id ), esc_attr( $attr_value ) );
		}

		// Wrapper start
		$placeholder_wrapper = sprintf( '<%s %s>', $element_type, implode( ' ', $attributes ) );

		// Before
		$placeholder_wrapper .= apply_filters( 'kalium_images_before_image', $this->before_image );

		// Content element
		$placeholder_wrapper .= $element;

		// After
		$placeholder_wrapper .= apply_filters( 'kalium_images_after_image', $this->after_image );

		// Wrapper end
		$placeholder_wrapper .= '</' . $element_type . '>';

		// Maximum width applied
		if ( ! $this->fluid_width ) {
			$placeholder_wrapper = sprintf( '<%1$s class="image-placeholder-realsize" style="max-width:%3$dpx;">%2$s</%1$s>', $element_type, $placeholder_wrapper, $args['width'] );
		}

		return $placeholder_wrapper;
	}

	/**
	 * Set fluid width mode.
	 *
	 * @param bool $fluid
	 */
	public function set_fluid_width( $fluid = true ) {
		$this->fluid_width = (bool) $fluid;
	}

	/**
	 * Use dominant color.
	 *
	 * @param bool $use
	 */
	public function use_dominant_color( $use ) {
		$this->dominant_color = (bool) $use;
	}

	/**
	 * Set before content.
	 *
	 * @param string $str
	 *
	 * @since 4.0
	 */
	public function set_before_content( $str ) {
		$this->before_image = $str;
	}

	/**
	 * Set after content.
	 *
	 * @param string $str
	 *
	 * @since 4.0
	 */
	public function set_after_content( $str ) {
		$this->after_image = $str;
	}

	/**
	 * Get image by URL.
	 *
	 * @param string $image_url
	 *
	 * @return array|null
	 */
	public function get_local_image( $image_url ) {
		if ( ! $image_url ) {
			return null;
		}

		$site_url = site_url();
		$image    = [];

		// Extract from tag
		if ( preg_match( '/<img[^>]+>/', $image_url, $matches ) ) {
			$element            = kalium()->helpers->parse_html_element( $matches[0] );
			$element_attributes = $element['attributes'];

			if ( ! empty( $element_attributes['src'] ) ) {
				$image_url = $element_attributes['src'];

				if ( ! empty( $element_attributes['alt'] ) ) {
					$alt = $element_attributes['alt'];
				}
			}
		}

		// Check for local image
		if ( false !== strpos( $image_url, $site_url ) ) {
			$image_relative_path = ltrim( str_replace( $site_url, '', $image_url ), '\/' );
			$image_full_path     = ABSPATH . $image_relative_path;

			if ( file_exists( $image_full_path ) ) {
				$image_size = @getimagesize( $image_full_path );

				if ( is_array( $image_size ) && isset( $image_size[0] ) && isset( $image_size[1] ) ) {
					$image_element = sprintf( '<img src="%s" width="%d" height="%d" alt="%s" />', $image_url, $image_size[0], $image_size[1], esc_attr( isset( $alt ) ? $alt : basename( $image_url ) ) );

					$image = [
						'src'    => $image_url,
						'path'   => $image_full_path,
						'width'  => $image_size[0],
						'height' => $image_size[1],
						'image'  => $image_element,
					];
				}
			}
		} else {
			// Remote image
			$image_hash  = 'img_' . md5( $image_url );
			$image_sizes = get_option( 'kalium_remote_images_sizes', [] );

			if ( empty( $image_sizes[ $image_hash ] ) ) {
				$image_size = @getimagesize( $image_url );

				if ( is_array( $image_size ) ) {
					$image_size['time']         = time();
					$image_sizes[ $image_hash ] = $image_size;

					update_option( 'kalium_remote_images_sizes', $image_sizes );
				}
			} else {
				$image_size = $image_sizes[ $image_hash ];
			}

			if ( is_array( $image_size ) && isset( $image_size[0] ) && isset( $image_size[1] ) ) {
				$image_element = sprintf( '<img src="%s" width="%d" height="%d" alt="%s" />', $image_url, $image_size[0], $image_size[1], esc_attr( basename( $image_url ) ) );

				$image = [
					'src'    => $image_url,
					'path'   => '',
					'width'  => $image_size[0],
					'height' => $image_size[1],
					'image'  => $image_element,
				];
			}
		}

		return $image;
	}

	/**
	 * Get dominant image color.
	 *
	 * @param int $attachment_id
	 *
	 * @return string
	 */
	public function get_dominant_color( $attachment_id ) {
		$key_name = 'kalium_dominant_color';
		$metadata = wp_get_attachment_metadata( $attachment_id );

		if ( ! isset( $metadata['image_meta'] ) ) {
			return null;
		}

		// Dominant color
		$dominant_color = kalium_get_array_key( $metadata['image_meta'], $key_name );

		// Get dominant color from image once and save it to the attachment
		if ( ! array_key_exists( $key_name, $metadata['image_meta'] ) ) {
			$image_file = get_attached_file( $attachment_id );
			$sizes      = $metadata['sizes'] ?? null;

			// Get the smallest image thumbnail
			if ( is_array( $sizes ) ) {
				usort(
					$sizes,
					function ( $a, $b ) {
						$size_a = $a['width'] * $a['height'];
						$size_b = $b['width'] * $b['height'];
						if ( $size_a == $size_b ) {
							return 0;
						}
						return ( $size_a < $size_b ) ? -1 : 1;
					}
				);

				$thumbnail      = array_shift( $sizes )['file'] ?? null;
				$thumbnail_path = dirname( $image_file ) . DIRECTORY_SEPARATOR . basename( $thumbnail );

				if ( $thumbnail && file_exists( $thumbnail_path ) ) {
					$image_file = $thumbnail_path;
				}
			}

			$dominant_color = kalium_get_image_dominant_colors( $image_file )[0] ?? null;

			// Save as image meta
			$metadata['image_meta'][ $key_name ] = $dominant_color;
			wp_update_attachment_metadata( $attachment_id, $metadata );
		}

		return $dominant_color;
	}
}
