<?php
/**
 * Kalium Request class
 *
 * Laborator.co
 * www.laborator.co
 */
namespace Kalium\Core;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Request {

	/**
	 * Get data parameters based on input source.
	 *
	 * @param string $input_source
	 *
	 * @return array
	 */
	public function get_data( $input_source = 'GET' ) {
		return $GLOBALS[ '_' . strtoupper( $input_source ) ] ?? $_GET;
	}

	/**
	 * Check if URL, or form input exists.
	 *
	 * @param string $var
	 * @param string $input_source
	 *
	 * @return bool
	 */
	public function has( $var, $input_source = 'GET' ) {
		return isset( $this->get_data( $input_source )[ $var ] );
	}

	/**
	 * Get form input (POST).
	 *
	 * @param string $var
	 * @param null   $default
	 *
	 * @return mixed|null
	 */
	public function input( $var, $default = null ) {
		return $this->get( $var, $default, 'POST' );
	}

	/**
	 * Get request parameter with primary check for camelCase var name.
	 *
	 * @param string $var
	 * @param null   $default
	 *
	 * @return mixed|null
	 */
	public function xhr_input( $var, $default = null ) {
		return $this->xhr_get( $var, $default, 'POST' );
	}

	/**
	 * Get query string parameter (GET).
	 *
	 * @param string $var
	 * @param null   $default
	 *
	 * @return mixed|null
	 */
	public function query( $var, $default = null ) {
		return $this->get( $var, $default );
	}

	/**
	 * Get query string parameter with primary check for camelCase var name (GET).
	 *
	 * @param string $var
	 * @param null   $default
	 *
	 * @return mixed|null
	 */
	public function xhr_query( $var, $default = null ) {
		return $this->xhr_get( $var, $default );
	}

	/**
	 * Get request parameter.
	 *
	 * @param string $var
	 * @param null   $default
	 *
	 * @return mixed|null
	 */
	public function request( $var, $default = null ) {
		return $this->get( $var, $default, 'REQUEST' );
	}

	/**
	 * Get request parameter with primary check for camelCase var name.
	 *
	 * @param string $var
	 * @param null   $default
	 *
	 * @return mixed|null
	 */
	public function xhr_request( $var, $default = null ) {
		return $this->xhr_get( $var, $default, 'REQUEST' );
	}

	/**
	 * Get cookie value.
	 *
	 * @param string $var
	 * @param null   $default
	 *
	 * @return mixed|null
	 */
	public function cookie( $var, $default = null ) {
		return $this->get( $var, $default, 'COOKIE' );
	}

	/**
	 * Get cookie value with primary check for camelCase var name.
	 *
	 * @param string $var
	 * @param null   $default
	 *
	 * @return mixed|null
	 */
	public function xhr_cookie( $var, $default = null ) {
		return $this->xhr_get( $var, $default, 'COOKIE' );
	}

	/**
	 * Get server var.
	 *
	 * @param string $var
	 *
	 * @return string|null
	 */
	public function server_var( $var ) {
		return $this->get( strtoupper( $var ), null, 'SERVER' );
	}

	/**
	 * Get request method.
	 *
	 * @return string
	 */
	public function get_method() {
		return strtoupper( $this->server_var( 'REQUEST_METHOD' ) );
	}

	/**
	 * Check if request method matches.
	 *
	 * @param string $type
	 *
	 * @return bool
	 */
	public function is_method( $type ) {
		return $this->get_method() === strtoupper( $type );
	}

	/**
	 * Get var from data input.
	 *
	 * @param string $var
	 * @param null   $default
	 * @param array  $input_source
	 *
	 * @return mixed|null
	 */
	private function get( $var, $default = null, $input_source = 'GET' ) {
		$data = $this->get_data( $input_source );

		if ( isset( $data[ $var ] ) ) {
			return $data[ $var ];
		}

		return $default;
	}

	/**
	 * Get input in XHR style with camelcase primary check then with dashes.
	 *
	 * @param string $var
	 * @param null   $default
	 * @param string $input_source
	 *
	 * @return mixed|null
	 */
	private function xhr_get( $var, $default = null, $input_source = 'GET' ) {
		$value = $this->get( kalium()->helpers->to_camelcase( $var ), $default, $input_source );

		if ( $default !== $value ) {
			return $value;
		}

		return $this->get( $var, $default, $input_source );
	}
}
