<?php
/**
 * Kalium WordPress Theme
 *
 * Theme options class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Core;

use Closure;
use WP_Customize_Manager;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Theme_Options {

	/**
	 * Registered theme options.
	 *
	 * @var array
	 */
	public $registered = [];

	/**
	 * Get all registered options.
	 *
	 * @return array
	 */
	public function get_all() {
		return $this->registered;
	}

	/**
	 * Add option.
	 *
	 * @param string $name
	 * @param array  $args
	 */
	public function add_option( $name, &$args = [] ) {
		$this->process_option_args( $args );

		// Register option
		$this->registered[ $name ] = $args;
	}

	/**
	 * Get option.
	 *
	 * @param string $name
	 * @param bool   $skip_default
	 *
	 * @return mixed|false
	 */
	public function get_option( $name, $skip_default = false ) {
		$value   = get_theme_mod( $name, null );
		$default = $this->get_default( $name );

		if ( is_null( $value ) ) {
			$value = $skip_default ? false : $default;
		} else {
			$value = $this->process_value( $name, $value );

			if ( $skip_default && $this->values_match( $value, $default ) ) {
				$value = false;
			}
		}

		return apply_filters( "kalium_theme_option_{$name}", $value );
	}

	/**
	 * Get modified theme options in key=>value pair.
	 *
	 * @return array
	 */
	public function get_modified_options() {
		$modified_options = [];

		$theme_mods = get_theme_mods();

		foreach ( $this->get_all() as $name => $args ) {
			// Theme mod
			if ( 'theme_mod' === $args['setting_type'] && array_key_exists( $name, $theme_mods ) ) {
				if ( false === $this->values_match( $theme_mods[ $name ], $this->get_default( $name ) ) ) {
					$modified_options[ $name ] = $theme_mods[ $name ];
				}
			}
		}

		return $modified_options;
	}

	/**
	 * Get theme option data.
	 *
	 * @param string $name
	 *
	 * @return array|null
	 */
	public function get_option_args( $name ) {
		$this->maybe_load_theme_options();

		return $this->registered[ $name ] ?? null;
	}

	/**
	 * Get default theme option value.
	 *
	 * @param string $name
	 *
	 * @return mixed|null
	 */
	public function get_default( $name ) {
		$option_data = $this->get_option_args( $name );

		if ( isset( $option_data['default'] ) ) {
			return $this->process_value( $name, $option_data['default'] );
		}

		return null;
	}

	/**
	 * Handle default value of control/setting and other stuff.
	 *
	 * @param array $args
	 */
	public function process_option_args( &$args ) {
		static $single_choice_controls = [
			'kalium-select',
			'kalium-radio-image',
			'kalium-radio-button',
			'kalium-tabs',
			'kalium-custom-select',
			'kalium-radio',
		];

		/** @var WP_Customize_Manager $wp_customize */
		global $wp_customize;

		// Skip default value handling for tabs on frontend
		if ( ! isset( $wp_customize ) && 'kalium-tabs' === $args['type'] ) {
			return;
		}

		// For single choice controls, set first choice as default value
		if ( is_null( $args['default'] ) && in_array( $args['type'], $single_choice_controls ) ) {
			$choices = is_callable( $args['choices'] ) ? call_user_func( $args['choices'] ) : $args['choices'];

			if ( is_array( $choices ) ) {
				$value_key = current( array_keys( $choices ) );
				$value_arr = current( array_values( $choices ) );

				if ( isset( $value_arr['value'] ) ) {
					$args['default'] = $value_arr['value'];
				} else {
					$args['default'] = $value_key;
				}
			}
		}

		// Execute closures on default values on backend
		if ( isset( $wp_customize ) && $args['default'] instanceof Closure ) {
			$args['default'] = call_user_func( $args['default'] );
		}

		// Option type
		$args['setting_type'] = $args['setting']['type'] ?? 'theme_mod';
	}

	/**
	 * Process theme option value.
	 *
	 * @param string $name
	 * @param mixed  $value
	 *
	 * @return mixed
	 */
	public function process_value( $name, $value ) {
		if ( $option_data = $this->get_option_args( $name ) ) {
			$is_responsive = $option_data['responsive'] ?? false;

			// Execute closures
			if ( $value instanceof Closure ) {
				$value = call_user_func( $value );
			}

			// Parse responsive option
			if ( $is_responsive ) {
				$value = kalium_parse_responsive_value( $value );
			}
		}

		return $value;
	}

	/**
	 * Maybe load theme options.
	 */
	public function maybe_load_theme_options() {
		static $loaded;

		if ( ! isset( $GLOBALS['wp_customize'] ) && is_null( $loaded ) ) {
			$loaded = true;
			kalium()->customize->register_theme_options();
		}
	}

	/**
	 * Checks two values if they are are equal.
	 *
	 * @param mixed $a
	 * @param mixed $b
	 *
	 * @return bool
	 */
	private function values_match( $a, $b ) {
		return $a == $b; // phpcs:ignore Universal.Operators.StrictComparisons.LooseEqual
	}
}
