<?php
/**
 * Kalium WordPress Theme
 *
 * Theme version history and upgrade hooks.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Core;

use WP_Error;

class Version_Upgrades {

	/**
	 * Version upgrades option.
	 *
	 * @const string
	 */
	const VERSION_UPGRADES_OPTION = 'kalium_version_upgrades';

	/**
	 * Version upgrades entries.
	 *
	 * @var array
	 */
	private $version_upgrades;

	/**
	 * Upgrade tasks.
	 *
	 * @var array
	 */
	private $upgrade_tasks = [];

	/**
	 * Construct.
	 */
	public function __construct() {
		add_action( 'admin_head', [ $this, 'init_version_upgrades' ] );
		add_action( 'admin_footer', [ $this, 'upgrade_tasks_enqueue' ], 15 );
		add_action( 'rest_api_init', [ $this, 'upgrade_tasks_api_endpoint' ] );
	}

	/**
	 * Init version upgrades.
	 */
	public function init_version_upgrades() {
		$this->process_legacy_version_upgrades();
		$this->run_hooks();
		$this->process_version_upgrades();
	}

	/**
	 * Get version upgrades.
	 *
	 * @return array
	 */
	public function get_version_upgrades() {
		// Init version upgrades
		if ( ! isset( $this->version_upgrades ) ) {
			$this->version_upgrades = get_option( self::VERSION_UPGRADES_OPTION, [] );
		}

		return $this->version_upgrades;
	}

	/**
	 * Get versions list.
	 *
	 * @return string[]
	 */
	public function get_versions_list() {
		return wp_list_pluck( $this->get_version_upgrades(), 'version' );
	}

	/**
	 * Get previous version.
	 *
	 * @return string|null
	 */
	public function get_previous_version() {
		$current_version = kalium()->get_version();
		$versions_list   = $this->get_versions_list();

		// Sort ascending
		sort( $versions_list );

		while ( $previous_version = array_pop( $versions_list ) ) {
			if ( version_compare( $previous_version, $current_version, '<' ) ) {
				return $previous_version;
			}
		}

		return null;
	}

	/**
	 * Get data from latest or specified version.
	 *
	 * @param string $option_name
	 * @param string $version_number
	 *
	 * @return mixed|null
	 */
	public function get_data( $option_name = null, $version_number = null ) {
		if ( ! $version_number ) {
			$version_number = kalium()->get_version();
		}

		foreach ( $this->get_version_upgrades() as $version_upgrade ) {
			if ( $version_number === $version_upgrade['version'] ) {
				if ( is_null( $option_name ) ) {
					return $version_upgrade['data'];
				}

				return kalium_get_array_key( $version_upgrade['data'], $option_name );
			}
		}

		return null;
	}

	/**
	 * Set data on from latest or specified version.
	 *
	 * @param string $option_name
	 * @param mixed  $value
	 * @param string $version_number
	 */
	public function set_data( $option_name, $value, $version_number = null ) {
		if ( ! $version_number ) {
			$version_number = kalium()->get_version();
		}

		foreach ( $this->get_version_upgrades() as $index => $version_upgrade ) {
			if ( $version_number === $version_upgrade['version'] ) {
				$this->version_upgrades[ $index ]['data'][ $option_name ] = $value;
			}
		}

		$this->save_version_upgrades();
	}

	/**
	 * Checks if installation is newer than provided version.
	 *
	 * @param string $compare_version
	 *
	 * @return bool
	 */
	public function is_newer_than( $compare_version ) {
		foreach ( $this->get_versions_list() as $version ) {
			if ( version_compare( $version, $compare_version, '<=' ) ) {
				return false;
			}
		}

		return version_compare( kalium()->get_version(), $compare_version, '>=' );
	}

	/**
	 * Checks if installation is older than provided version.
	 *
	 * @param string $compare_version
	 *
	 * @return bool
	 */
	public function is_older_than( $compare_version ) {
		foreach ( $this->get_versions_list() as $version ) {
			if ( version_compare( $version, $compare_version, '<' ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Save version upgrades entries.
	 */
	private function save_version_upgrades() {
		$version_upgrades = $this->get_version_upgrades();

		// Sort ascending
		usort(
			$version_upgrades,
			function ( $a, $b ) {
				return version_compare( $a['version'], $b['version'], '>' ) ? 1 : -1;
			}
		);

		update_option( self::VERSION_UPGRADES_OPTION, $version_upgrades );
	}

	/**
	 * Process version upgrades.
	 */
	private function process_version_upgrades() {
		$current_version = kalium()->get_version();

		// Register version upgrade
		if ( ! in_array( $current_version, $this->get_versions_list() ) ) {
			$version_upgrade = [
				'version' => $current_version,
				'time'    => time(),
				'data'    => [],
			];

			$this->version_upgrades[] = $version_upgrade;
			$this->save_version_upgrades();
		}
	}

	/**
	 * Process version upgrades.
	 */
	private function process_legacy_version_upgrades() {
		$version_upgrades = $this->get_version_upgrades();

		if ( is_string( kalium_get_array_first( $version_upgrades ) ) ) {
			$new_version_upgrades = [];

			foreach ( $version_upgrades as $previous_version ) {
				$new_version_upgrades[] = [
					'version' => $previous_version,
					'time'    => 0,
					'data'    => [],
				];
			}

			$this->version_upgrades = $new_version_upgrades;
			$this->save_version_upgrades();
		}

		// Remove pre 3.0 version upgrades
		$do_update = false;

		foreach ( $this->version_upgrades as $i => $version_upgrade ) {
			if ( version_compare( $version_upgrade['version'], '3.0', '<' ) ) {
				unset( $this->version_upgrades[ $i ] );
				$do_update = true;
			}
		}

		// Update versions list
		if ( $do_update ) {
			$this->save_version_upgrades();
		}
	}

	/**
	 * Run version upgrade hooks.
	 */
	private function run_hooks() {
		$version_upgrades = $this->get_version_upgrades();
		$current_version  = kalium()->get_version();
		$previous_version = $this->get_previous_version();

		// First installation
		if ( empty( $version_upgrades ) ) {

			/**
			 * First installation hook.
			 *
			 * @param string $current_version
			 *
			 * @since 4.0
			 */
			do_action( 'kalium_first_install', $current_version );
		}

		// Version upgrade
		if ( $previous_version && ! in_array( $current_version, $this->get_versions_list() ) ) {

			/**
			 * Run version upgrade hooks.
			 *
			 * @param string      $current_version
			 * @param string|null $previous_version
			 */
			do_action( 'kalium_version_upgrade', $current_version, $previous_version );
		}
	}

	/**
	 * Get upgrade tasks.
	 *
	 * @return Version_Upgrade_Task[]
	 * @since 4.0
	 */
	public function get_upgrade_tasks() {
		return $this->upgrade_tasks;
	}

	/**
	 * Get single upgrade task.
	 *
	 * @param string $id
	 *
	 * @return Version_Upgrade_Task
	 * @since 4.0
	 */
	public function get_upgrade_task( $id ) {
		return $this->upgrade_tasks[ $id ] ?? null;
	}

	/**
	 * Add an upgrade task.
	 *
	 * @param string $id
	 * @param string $version
	 * @param array  $args
	 *
	 * @since 4.0
	 */
	public function add_upgrade_task( $id, $version, $args = [] ) {
		$this->upgrade_tasks[ $id ] = new Version_Upgrade_Task( $id, $version, $args );
	}

	/**
	 * Register REST API endpoint
	 *
	 * @since 4.1.1
	 */
	public function upgrade_tasks_api_endpoint() {
		register_rest_route(
			kalium_rest_namespace(),
			'/upgrade-task/(?P<task_id>[\w-]+)/(?P<subtask_id>\d+)/',
			[
				'methods'             => [ 'GET' ],
				'callback'            => [ $this, 'run_upgrade_task' ],
				'permission_callback' => kalium_hook_return_value( 'install_themes' ),
				'args'                => [
					'task_id'    => [
						'required' => true,
					],
					'subtask_id' => [
						'required' => false,
					],
				],
			]
		);
	}

	/**
	 * Run DB upgrade on AJAX.
	 *
	 * @return WP_Error|\WP_REST_Response
	 * @since 4.1.1
	 */
	public function run_upgrade_task( $request ) {
		$task_id    = $request->get_param( 'task_id' );
		$subtask_id = $request->get_param( 'subtask_id' );

		$response = new WP_Error( 'kalium_task_not_found', 'Task not found', [ 'status' => 404 ] );

		if ( $task = $this->get_upgrade_task( $task_id ) ) {
			$subtasks = $task->get_sub_tasks();

			// Run sub-task
			if ( is_numeric( $subtask_id ) && isset( $subtasks[ $subtask_id ] ) ) {
				$response = $task->execute( $subtask_id );
			} else {
				$response = $task->execute();
			}
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Display admin task notices.
	 *
	 * @since 4.1.1
	 */
	public function upgrade_tasks_enqueue() {
		$valid_upgrade_tasks = array_values(
			array_filter(
				$this->get_upgrade_tasks(),
				function ( $upgrade_task ) {
					return $upgrade_task->is_eligible() && ! $upgrade_task->is_completed();
				}
			)
		);

		if ( ! empty( $valid_upgrade_tasks ) ) {
			kalium_enqueue( 'theme-version-upgrades' );
			kalium_admin_js_define_var( 'upgradeTasks', $valid_upgrade_tasks );
		}
	}
}
