<?php
/**
 * Kalium WordPress Theme
 *
 * White label.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Core;

use ReflectionClass;
use WP_Customize_Manager;
use WP_Theme;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class White_Label {

	/**
	 * Option name.
	 *
	 * @const string
	 */
	const OPTION_NAME = 'kalium_white_label';

	/**
	 * Section ID.
	 *
	 * @const string
	 */
	const SECTION_ID = 'white-label';

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'kalium_fs_loaded', [ $this, 'init' ] );
	}

	/**
	 * Init.
	 */
	public function init() {
		if ( ! kalium()->license->has_feature( 'white-label' ) ) {
			return;
		}

		if ( $this->enabled() ) {
			add_filter( 'kalium_name', [ $this, 'set_agency_name' ] );
			add_filter( 'kalium_theme_author_link', [ $this, 'set_agency_link' ] );
			add_action( 'admin_print_styles', [ $this, 'replace_agency_icon' ] );
			add_action( 'admin_print_styles', [ $this, 'replace_theme_update_screenshot' ] );
			add_filter( 'kalium_get_theme', [ $this, 'replace_get_theme' ] );
			add_filter( 'all_themes', [ $this, 'replace_network_theme' ] );
			add_filter( 'customize_register', [ $this, 'replace_customizer_theme' ], 0 );
			add_filter( 'wp_prepare_themes_for_js', [ $this, 'replace_appearance_page' ] );
			add_filter( 'kalium_dashboard_maybe_show_section', [ $this, 'hide_extras' ], 10, 2 );
			add_filter( 'kalium_dashboard_tabs', [ $this, 'hide_dashboard_tabs' ] );
			kalium()->freemius->add_filter( 'is_submenu_visible', [ $this, 'hide_account_menu_item' ], 10, 2 );

			if ( $this->hide_white_label_settings() ) {
				return;
			}
		}

		add_action( 'kalium_dashboard_settings_sections', [ $this, 'register_setting_section' ] );
		add_action( 'kalium_settings_init', [ $this, 'register_settings' ] );
		add_action( 'kalium_settings_section_white-label', [ $this, 'settings' ] );
	}

	/**
	 * Register settings section.
	 *
	 * @param array $settings_sections
	 *
	 * @return array
	 */
	public function register_setting_section( $settings_sections ) {
		$settings_sections[] = [
			'id'       => self::SECTION_ID,
			'title'    => 'White Label',
			'icon'     => 'kalium-admin-icon-tag',
			'priority' => 200,
		];

		return $settings_sections;
	}

	/**
	 * Get settings.
	 *
	 * @return array
	 */
	public function get_settings() {
		return wp_parse_args(
			get_option( self::OPTION_NAME, [] ),
			[
				'enabled' => false,
			]
		);
	}

	/**
	 * Check if white label is enabled.
	 *
	 * @return bool
	 */
	public function enabled() {
		return kalium_get_array_key( $this->get_settings(), 'enabled' );
	}

	/**
	 * Register setting.
	 */
	public function register_settings() {
		$is_url_optional = function ( $value ) {
			return empty( $value ) || kalium_is_url( $value );
		};

		kalium()->settings->register(
			[
				'white-label-enabled'            => [
					'input_type'        => 'switch',
					'default'           => false,
					'setting'           => [ self::OPTION_NAME, 'enabled' ],
					'sanitize_callback' => 'kalium_validate_boolean',
				],
				'white-label-agency-name'        => [
					'input_type' => 'text',
					'setting'    => [ self::OPTION_NAME, 'agency_name' ],
				],
				'white-label-agency-url'         => [
					'input_type'        => 'text',
					'setting'           => [ self::OPTION_NAME, 'agency_url' ],
					'input_attributes'  => [
						'placeholder' => 'https://',
					],
					'validate_callback' => $is_url_optional,
				],
				'white-label-agency-icon'        => [
					'input_type'        => 'text',
					'setting'           => [ self::OPTION_NAME, 'agency_icon' ],
					'validate_callback' => $is_url_optional,
				],
				'white-label-theme-name'         => [
					'input_type' => 'text',
					'setting'    => [ self::OPTION_NAME, 'theme_name' ],
				],
				'white-label-theme-description'  => [
					'input_type'       => 'textarea',
					'input_attributes' => [
						'rows' => 5,
					],
					'setting'          => [ self::OPTION_NAME, 'theme_description' ],
				],
				'white-label-theme-screenshot'   => [
					'input_type'        => 'text',
					'setting'           => [ self::OPTION_NAME, 'theme_screenshot' ],
					'validate_callback' => $is_url_optional,
				],
				'white-label-hide-extras'        => [
					'input_type'        => 'switch',
					'setting'           => [ self::OPTION_NAME, 'hide_extras' ],
					'sanitize_callback' => 'kalium_validate_boolean',
				],
				'white-label-hide-plugins'       => [
					'input_type'        => 'switch',
					'setting'           => [ self::OPTION_NAME, 'hide_plugins' ],
					'sanitize_callback' => 'kalium_validate_boolean',
				],
				'white-label-hide-starter-sites' => [
					'input_type'        => 'switch',
					'setting'           => [ self::OPTION_NAME, 'hide_starter_sites' ],
					'sanitize_callback' => 'kalium_validate_boolean',
				],
				'white-label-hide-status'        => [
					'input_type'        => 'switch',
					'setting'           => [ self::OPTION_NAME, 'hide_status' ],
					'sanitize_callback' => 'kalium_validate_boolean',
				],
				'white-label-hide-changelog'     => [
					'input_type'        => 'switch',
					'setting'           => [ self::OPTION_NAME, 'hide_changelog' ],
					'sanitize_callback' => 'kalium_validate_boolean',
				],
				'white-label-hide-account'       => [
					'input_type'        => 'switch',
					'setting'           => [ self::OPTION_NAME, 'hide_account' ],
					'sanitize_callback' => 'kalium_validate_boolean',
				],
				'white-label-hide-white-label'   => [
					'input_type'        => 'switch',
					'setting'           => [ self::OPTION_NAME, 'hide_white_label' ],
					'sanitize_callback' => 'kalium_validate_boolean',
				],
			]
		);
	}

	/**
	 * Settings.
	 */
	public function settings() {
		?>
		<div class="kalium-dashboard__setting">
			<div class="kalium-dashboard__setting__label">
				<h3>White Label</h3>
				<p>Replace Kalium branding throughout the theme and hide licensing info as well.</p>
			</div>
			<div class="kalium-dashboard__setting__control">
				<?php Settings::setting( 'white-label-enabled' ); ?>
			</div>
		</div>

		<div class="kalium-dashboard__setting kalium-dashboard__setting--separator kalium-dashboard__setting--half">
			<div class="kalium-dashboard__setting__label">
				<h4>Agency Details</h4>
				<p>This will replace every word Kalium with your own brand name, link and icon.</p>
			</div>
			<div class="kalium-dashboard__setting__control">
				<div class="kalium-dashboard__setting kalium-dashboard__setting--alt">
					<label for="white-label-agency-name">Agency Name</label>
					<?php Settings::setting( 'white-label-agency-name' ); ?>
				</div>

				<div class="kalium-dashboard__setting kalium-dashboard__setting--alt">
					<label for="white-label-agency-url">Agency URL</label>
					<?php Settings::setting( 'white-label-agency-url' ); ?>
				</div>

				<div class="kalium-dashboard__setting kalium-dashboard__setting--alt">
					<label for="white-label-agency-icon">Agency Icon</label>
					<div class="row">
						<div class="col">
							<?php Settings::setting( 'white-label-agency-icon' ); ?>
						</div>
						<div class="col-auto">
							<button class="button" type="button" data-media-selector="#white-label-agency-icon">Select Image</button>
						</div>
					</div>
					<p class="small">Recommended image size: 64x64</p>
				</div>
			</div>
		</div>

		<div class="kalium-dashboard__setting kalium-dashboard__setting--separator kalium-dashboard__setting--half">
			<div class="kalium-dashboard__setting__label">
				<h3>Theme Details</h3>
				<p>Change the details of the theme to match your agency brand.</p>
			</div>
			<div class="kalium-dashboard__setting__control">
				<div class="kalium-dashboard__setting kalium-dashboard__setting--alt">
					<label for="white-label-theme-name">Theme Name</label>
					<?php Settings::setting( 'white-label-theme-name' ); ?>
				</div>
				<div class="kalium-dashboard__setting kalium-dashboard__setting--alt">
					<label for="white-label-theme-description">Theme Description</label>
					<?php Settings::setting( 'white-label-theme-description' ); ?>
				</div>
				<div class="kalium-dashboard__setting kalium-dashboard__setting--alt">
					<label for="white-label-theme-screenshot">Theme Screenshot</label>
					<div class="row">
						<div class="col">
							<?php Settings::setting( 'white-label-theme-screenshot' ); ?>
						</div>
						<div class="col-auto">
							<button class="button" type="button" data-media-selector="#white-label-theme-screenshot">Select Image</button>
						</div>
					</div>
					<p class="small">Recommended image size: 1200x900</p>
				</div>
			</div>
		</div>

		<div class="kalium-dashboard__setting kalium-dashboard__setting--separator">
			<div class="kalium-dashboard__setting__label">
				<h4>Hide Extras</h4>
				<p>Conceal theme information like version, dashboard sections, and other details.</p>
			</div>
			<div class="kalium-dashboard__setting__control">
				<?php Settings::setting( 'white-label-hide-extras' ); ?>
			</div>
		</div>

		<div class="kalium-dashboard__setting kalium-dashboard__setting--separator">
			<div class="kalium-dashboard__setting__label">
				<h4>Hide Plugins Page</h4>
				<p>This will hide <strong>Plugins</strong> page from theme dashboard and menu.</p>
			</div>
			<div class="kalium-dashboard__setting__control">
				<?php Settings::setting( 'white-label-hide-plugins' ); ?>
			</div>
		</div>

		<div class="kalium-dashboard__setting kalium-dashboard__setting--separator">
			<div class="kalium-dashboard__setting__label">
				<h4>Hide Starter Sites Page</h4>
				<p>This will hide <strong>Starter Sites</strong> page from theme dashboard and menu.</p>
			</div>
			<div class="kalium-dashboard__setting__control">
				<?php Settings::setting( 'white-label-hide-starter-sites' ); ?>
			</div>
		</div>

		<div class="kalium-dashboard__setting kalium-dashboard__setting--separator">
			<div class="kalium-dashboard__setting__label">
				<h4>Hide Status Page</h4>
				<p>This will hide <strong>Status</strong> page from theme dashboard and menu.</p>
			</div>
			<div class="kalium-dashboard__setting__control">
				<?php Settings::setting( 'white-label-hide-status' ); ?>
			</div>
		</div>

		<div class="kalium-dashboard__setting kalium-dashboard__setting--separator">
			<div class="kalium-dashboard__setting__label">
				<h4>Hide Changelog Page</h4>
				<p>This will hide <strong>Changelog</strong> page from theme dashboard and menu.</p>
			</div>
			<div class="kalium-dashboard__setting__control">
				<?php Settings::setting( 'white-label-hide-changelog' ); ?>
			</div>
		</div>

		<div class="kalium-dashboard__setting kalium-dashboard__setting--separator">
			<div class="kalium-dashboard__setting__label">
				<h4>Hide Account Page</h4>
				<p>This will hide <strong>Account</strong> page from menu.</p>
			</div>
			<div class="kalium-dashboard__setting__control">
				<?php Settings::setting( 'white-label-hide-account' ); ?>
			</div>
		</div>

		<div class="kalium-dashboard__setting kalium-dashboard__setting--separator">
			<div class="kalium-dashboard__setting__label">
				<h4>Hide White Label Settings</h4>
				<p>Enabling this option will hide <strong>White Label</strong> settings page. To access this page again, simply save the current URL.</p>
			</div>
			<div class="kalium-dashboard__setting__control">
				<?php Settings::setting( 'white-label-hide-white-label' ); ?>
			</div>
		</div>
		<?php
	}

	/**
	 * Agency name.
	 *
	 * @param string $name
	 *
	 * @return string
	 */
	public function set_agency_name( $name ) {
		if ( $agency_name = kalium_get_array_key( $this->get_settings(), 'agency_name' ) ) {
			return $agency_name;
		}

		return $name;
	}

	/**
	 * Agency link.
	 *
	 * @param string $link
	 *
	 * @return string
	 */
	public function set_agency_link( $link ) {
		$settings    = $this->get_settings();
		$agency_name = kalium_get_array_key( $settings, 'agency_name' );
		$agency_url  = kalium_get_array_key( $settings, 'agency_url' );

		if ( $agency_name ) {
			if ( kalium_is_url( $agency_url ) ) {
				return sprintf( '<a href="%s" target="_blank" rel="noreferrer noopener">%s</a>', esc_url( $agency_url ), $agency_name );
			}

			return $agency_name;
		}

		return $link;
	}

	/**
	 * Replace agency icon.
	 */
	public function replace_agency_icon() {
		if ( ! kalium_is_url( $agency_icon = kalium_get_array_key( $this->get_settings(), 'agency_icon' ) ) ) {
			return;
		}

		?>
		<style>
			#toplevel_page_kalium .wp-menu-image::before,
			.kalium-admin-icon-kalium::before {
				color: transparent !important;
				background-image: url("<?php echo esc_attr( $agency_icon ); ?>");
				background-repeat: no-repeat;
				background-size: 100% auto;
				background-position: center center;
			}

			.toplevel_page_kalium .kalium-dashboard__header-navigation .nav-logo {
				line-height: 1;
			}
		</style>
		<?php
	}

	/**
	 * Replace theme screenshot on updates page.
	 */
	public function replace_theme_update_screenshot() {
		if ( ! kalium_is_url( $theme_screenshot = kalium_get_array_key( $this->get_settings(), 'theme_screenshot' ) ) ) {
			return;
		}

		?>
		<style>
			#update-themes-table .updates-table-screenshot[src*="themes/<?php echo esc_attr( kalium()->get_slug() ); ?>/screenshot"] {
				object-position: -1000px;
				background: url("<?php echo esc_attr( $theme_screenshot ); ?>") no-repeat;
				background-size: calc(100% - 10px) auto;
			}
		</style>
		<?php
	}

	/**
	 * Replaces theme object.
	 *
	 * @param WP_Theme $theme_object
	 *
	 * @return WP_Theme
	 */
	public function replace_get_theme( $theme_object ) {
		$settings          = $this->get_settings();
		$theme_name        = kalium_get_array_key( $settings, 'theme_name' );
		$theme_description = kalium_get_array_key( $settings, 'theme_description' );
		$theme_screenshot  = kalium_get_array_key( $settings, 'theme_screenshot' );

		if ( $theme_name || $theme_description || $theme_screenshot ) {
			$reflection = new ReflectionClass( $theme_object );

			$headers_prop = $reflection->getProperty( 'headers' );
			$headers_prop->setAccessible( true );

			$headers_value = $headers_prop->getValue( $theme_object );

			if ( ! empty( $theme_name ) ) {
				$headers_value['Name'] = $theme_name;
			}

			if ( ! empty( $theme_description ) ) {
				$headers_value['Description'] = $theme_description;
			}

			$headers_prop->setValue( $theme_object, $headers_value );
		}

		return $theme_object;
	}

	/**
	 * Replace network theme.
	 *
	 * @param array $themes
	 *
	 * @return array
	 */
	public function replace_network_theme( $themes ) {
		$slug = kalium()->get_slug();

		if ( ! isset( $themes[ $slug ] ) ) {
			return $themes;
		}

		$themes[ $slug ] = kalium_get_theme();

		return $themes;
	}

	/**
	 * Replaces customizer theme object.
	 *
	 * @param WP_Customize_Manager $wp_customize
	 */
	public function replace_customizer_theme( $wp_customize ) {
		$reflection = new ReflectionClass( $wp_customize );

		$theme_prop = $reflection->getProperty( 'theme' );
		$theme_prop->setAccessible( true );
		$theme_prop->setValue( $wp_customize, kalium_get_theme() );
	}

	/**
	 * Replaces theme data in Theme Appearance page.
	 *
	 * @param array $themes
	 *
	 * @return array
	 */
	public function replace_appearance_page( $themes ) {
		$slug = kalium()->get_slug();

		if ( ! isset( $themes[ $slug ] ) ) {
			return $themes;
		}

		$theme    = & $themes[ $slug ];
		$settings = $this->get_settings();

		// Theme data
		$theme_name        = kalium_get_array_key( $settings, 'theme_name' );
		$theme_description = kalium_get_array_key( $settings, 'theme_description' );
		$theme_screenshot  = kalium_get_array_key( $settings, 'theme_screenshot' );

		// Agency data
		$agency_name = kalium_get_array_key( $settings, 'agency_name' );
		$agency_url  = kalium_get_array_key( $settings, 'agency_url' );

		if ( $theme_name ) {
			$theme['name'] = $theme_name;
		}

		if ( $theme_description ) {
			$theme['description'] = $theme_description;
		}

		if ( kalium_is_url( $theme_screenshot ) ) {
			$theme['screenshot'] = [ $theme_screenshot ];
		}

		if ( $agency_name ) {
			$theme['author']       = $agency_name;
			$theme['authorAndUri'] = kalium_theme_author_link();
		}

		return $themes;
	}

	/**
	 * Hide extras.
	 *
	 * @param bool   $visibility
	 * @param string $section_id
	 *
	 * @return bool
	 */
	public function hide_extras( $visibility, $section_id ) {
		$settings = $this->get_settings();

		if ( ! empty( $settings['hide_extras'] ) ) {
			return false;
		} elseif ( 'dashboard-footer-links' === $section_id ) {
			return false;
		} elseif ( 'dashboard-footer-copyrights' === $section_id ) {
			return false;
		}

		return $visibility;
	}

	/**
	 * Hide dashboard tabs.
	 *
	 * @param array $tabs
	 *
	 * @return array
	 */
	public function hide_dashboard_tabs( $tabs ) {
		$settings = $this->get_settings();

		$hide_tabs = [
			'hide_plugins'       => 'plugins',
			'hide_starter_sites' => 'starter-sites',
			'hide_status'        => 'system-status',
			'hide_changelog'     => 'changelog',
		];

		foreach ( $hide_tabs as $hide_tab => $tab_id ) {
			if ( ! empty( $settings[ $hide_tab ] ) ) {
				unset( $tabs[ $tab_id ] );
			}
		}

		return $tabs;
	}

	/**
	 * Hide account menu item.
	 *
	 * @param bool   $visible
	 * @param string $id
	 *
	 * @return bool
	 */
	public function hide_account_menu_item( $visible, $id ) {
		$settings = $this->get_settings();

		if ( 'account' === $id && ! empty( $settings['hide_account'] ) ) {
			return false;
		}

		return $visible;
	}

	/**
	 * Hide white label settings page.
	 *
	 * @return bool
	 */
	public function hide_white_label_settings() {
		$settings = $this->get_settings();

		if ( ! empty( $settings['hide_white_label'] ) ) {
			return self::SECTION_ID !== kalium()->request->query( 'section' ) && ! wp_doing_ajax();
		}

		return false;
	}
}
