<?php
/**
 * Kalium WordPress Theme
 *
 * Main class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Main class.
 *
 * @property Kalium\Core\Request $request
 * @property Kalium\Core\Helpers $helpers
 * @property Kalium\Core\Is $is
 * @property Kalium\Core\License $license
 * @property Kalium\Core\Filesystem $filesystem
 * @property Kalium\Core\Settings $settings
 * @property Kalium\Core\Assets $assets
 * @property Kalium\Core\Portfolio $portfolio
 * @property Kalium\Core\Page $page
 * @property Kalium\Core\Template_Loader $template_loader
 * @property Kalium\Core\Theme_Plugins $theme_plugins
 * @property Kalium\Core\Theme_Options $theme_options
 * @property Kalium\Core\Images $images
 * @property Kalium\Core\I18n $i18n
 * @property Kalium\Core\Style $style
 * @property Kalium\Core\Version_Upgrades $version_upgrades
 * @property Kalium\Admin\Backups $backups
 * @property Kalium\Compatibility\Freemius $freemius
 * @property Kalium\Compatibility\Elementor $elementor
 * @property Kalium\Compatibility\WooCommerce $woocommerce
 * @property Kalium\Compatibility\Custom_Post_Types $custom_post_types
 * @property Kalium\Compatibility\ACF $acf
 * @property Kalium\Media\Media $media
 * @property Kalium\SEO\Structured_Data $structured_data
 * @property Kalium\Customize\Customize $customize
 */
final class Kalium {

	/**
	 * Theme version.
	 *
	 * @type string
	 */
	const VERSION = '4.1.3';

	/**
	 * Theme build number.
	 *
	 * @type int
	 */
	const BUILD_NUM = 1743684551;

	/**
	 * Kalium API server.
	 *
	 * @type string
	 */
	const API_SERVER = 'https://api.kaliumtheme.com';

	/**
	 * Kalium instance as singleton class.
	 *
	 * @var self $instance
	 */
	private static $instance;

	/**
	 * Loaded sub instances.
	 *
	 * @var array $sub_instances
	 */
	private $sub_instances = [];

	/**
	 * Create instance of this class.
	 *
	 * @return self
	 */
	public static function instance() {
		if ( ! self::$instance ) {

			// Initialize Kalium
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * You cannot clone this class.
	 */
	public function __clone() {
		kalium_doing_it_wrong( __FUNCTION__, 'Cheatin&#8217; huh?', '2.0' );
	}

	/**
	 * You cannot unserialize instance of this class.
	 */
	public function __wakeup() {
		kalium_doing_it_wrong( __FUNCTION__, 'Cheatin&#8217; huh?', '2.0' );
	}

	/**
	 * Getter used to load class instance.
	 *
	 * @param string $name
	 *
	 * @return mixed|null
	 */
	public function __get( $name ) {

		// When sub instance doesn't exists
		if ( false === isset( $this->sub_instances[ $name ] ) ) {
			kalium_doing_it_wrong( "kalium()->{$name}", "Sub class instance <strong>{$name}</strong> doesn't exists.", '3.0' );

			return null;
		}

		return $this->sub_instances[ $name ];
	}

	/**
	 * Get current theme version.
	 *
	 * @return string
	 */
	public function get_version() {
		return self::VERSION;
	}

	/**
	 * Get current build number.
	 *
	 * @return int
	 */
	public function get_build_num() {
		return self::BUILD_NUM;
	}

	/**
	 * Theme framework autoload classes.
	 *
	 * @return array
	 *
	 * @since 4.1
	 */
	public function get_framework_autoload_classes() {
		return apply_filters(
			'kalium_get_framework_classes',
			[
				// Core classes
				[
					'class_name' => 'Kalium\\Core\\Request',
					'instance'   => 'request',
				],
				[
					'class_name' => 'Kalium\\Core\\Helpers',
					'instance'   => 'helpers',
				],
				[
					'class_name' => 'Kalium\\Core\\Is',
					'instance'   => 'is',
				],
				[
					'class_name' => 'Kalium\\Core\\Filesystem',
					'instance'   => 'filesystem',
				],
				[
					'class_name' => 'Kalium\\Core\\Theme_Plugins',
					'instance'   => 'theme_plugins',
				],
				[
					'class_name' => 'Kalium\\Core\\Theme_Options',
					'instance'   => 'theme_options',
				],
				[
					'class_name' => 'Kalium\\Core\\License',
					'instance'   => 'license',
				],
				[
					'class_name' => 'Kalium\\Core\\Settings',
					'instance'   => 'settings',
				],
				[
					'class_name' => 'Kalium\\Core\\Assets',
					'instance'   => 'assets',
				],
				[
					'class_name' => 'Kalium\\Core\\Style',
					'instance'   => 'style',
				],
				[
					'class_name' => 'Kalium\\Core\\Images',
					'instance'   => 'images',
				],
				[
					'class_name' => 'Kalium\\Core\\Portfolio',
					'instance'   => 'portfolio',
				],
				[
					'class_name' => 'Kalium\\Core\\Page',
					'instance'   => 'page',
				],
				[
					'class_name' => 'Kalium\\Core\\I18n',
					'instance'   => 'i18n',
				],
				[
					'class_name' => 'Kalium\\Core\\Template_Loader',
					'instance'   => 'template_loader',
				],
				[
					'class_name' => 'Kalium\\Core\\Version_Upgrades',
					'instance'   => 'version_upgrades',
				],
				[
					'class_name' => 'Kalium\\Core\\Integrations',
				],
				[
					'class_name' => 'Kalium\\Core\\White_Label',
				],

				// Customizer
				[
					'class_name' => 'Kalium\\Customize\\Customize',
					'instance'   => 'customize',
				],

				// Compatibility
				[
					'class_name' => 'Kalium\\Compatibility\\Freemius',
					'instance'   => 'freemius',
				],
				[
					'class_name' => 'Kalium\\Compatibility\\WooCommerce',
					'instance'   => 'woocommerce',
				],
				[
					'class_name' => 'Kalium\\Compatibility\\Custom_Post_Types',
					'instance'   => 'custom_post_types',
				],
				[
					'class_name' => 'Kalium\\Compatibility\\ACF',
					'instance'   => 'acf',
				],
				[
					'class_name' => 'Kalium\\Compatibility\\Elementor',
					'instance'   => 'elementor',
				],
				[
					'class_name' => 'Kalium\\Compatibility\\WPBakery',
				],
				[
					'class_name' => 'Kalium\\Compatibility\\Slider_Revolution',
				],
				[
					'class_name' => 'Kalium\\Compatibility\\LayerSlider',
				],
				[
					'class_name' => 'Kalium\\Compatibility\\WPML',
				],
				[
					'class_name' => 'Kalium\\Compatibility\\Rank_Math',
				],

				// Media
				[
					'class_name' => 'Kalium\\Media\\Media',
					'instance'   => 'media',
				],

				// SEO
				[
					'class_name' => 'Kalium\\SEO\\Structured_Data',
					'instance'   => 'structured_data',
				],

				// Admin
				[
					'class_name' => 'Kalium\\Admin\\Dashboard',
				],
				[
					'class_name' => 'Kalium\\Admin\\Backups',
					'instance'   => 'backups',
				],
				[
					'class_name' => 'Kalium\\Utilities\\System_Status',
				],
			]
		);
	}

	/**
	 * Get latest version of the theme.
	 *
	 * @return string
	 */
	public function get_latest_version() {
		$update_themes = get_site_transient( 'update_themes' );

		if ( is_object( $update_themes ) ) {
			$slug = $this->get_slug();

			if ( isset( $update_themes->response[ $slug ]['new_version'] ) ) {
				$new_version = $update_themes->response[ $slug ]['new_version'];
			} elseif ( isset( $update_themes->no_update[ $slug ]['new_version'] ) ) {
				$new_version = $update_themes->no_update[ $slug ]['new_version'];
			} elseif ( isset( $update_themes->checked[ $slug ] ) ) {
				$new_version = $update_themes->checked[ $slug ];
			}
		}

		return $new_version ?? null;
	}

	/**
	 * Check if current version of the theme is latest one.
	 *
	 * @return bool
	 */
	public function is_latest_version() {
		$latest_version = $this->get_latest_version();

		if ( ! empty( $latest_version ) ) {
			return version_compare( $this->get_version(), $latest_version, '=' );
		}

		return false;
	}

	/**
	 * Get theme slug.
	 *
	 * @return string
	 */
	public function get_slug() {
		static $slug;

		if ( is_null( $slug ) ) {
			$slug = wp_basename( kalium()->locate_file() );
		}

		return $slug;
	}

	/**
	 * Get Theme API endpoints.
	 *
	 * @return array
	 */
	public function get_api_endpoints() {
		return apply_filters(
			'kalium_api_endpoints',
			[
				'sites-list'                 => [
					'method'   => 'GET',
					'endpoint' => 'sites-list.json',
				],
				'theme-plugins'              => [
					'method'   => 'GET',
					'endpoint' => 'theme-plugins.json',
				],
				'starter-site-download-file' => [
					'method'   => 'GET',
					'endpoint' => 'downloads/starter-sites/:type/:id/:file',
					'download' => true,
				],
				'translations'               => [
					'method'   => 'GET',
					'endpoint' => 'translations.json',
				],
				'translation-download'       => [
					'method'   => 'GET',
					'endpoint' => 'downloads/translations/:file',
					'download' => true,
				],
				'notices'                    => [
					'method'   => 'GET',
					'endpoint' => 'notices.json',
				],
				'hello'                      => [
					'method'   => 'GET',
					'endpoint' => 'hello.json',
				],
			]
		);
	}

	/**
	 * Locate a file relative to the theme directory or absolute path.
	 *
	 * @param string $file_path
	 *
	 * @return string|false
	 */
	public function locate_file( $file_path = '' ) {
		static $theme_path;

		if ( is_null( $theme_path ) ) {
			$theme_path = get_parent_theme_file_path();
		}

		$theme_file_path = $theme_path . '/' . ltrim( $file_path, '/' );

		if ( file_exists( $theme_file_path ) ) {
			return $theme_file_path;
		} elseif ( file_exists( $file_path ) ) {
			return $file_path;
		}

		return false;
	}

	/**
	 * File URL in theme directory.
	 *
	 * @param string $relative_path
	 *
	 * @return string
	 */
	public function file_url( $relative_path = '' ) {
		static $theme_uri;

		if ( is_null( $theme_uri ) ) {
			$theme_uri = get_parent_theme_file_uri();
		}

		return $theme_uri . '/' . ltrim( $relative_path, '/' );
	}

	/**
	 * Require file relative to theme base directory.
	 *
	 * @param string $relative_path
	 * @param array  $vars
	 * @param bool   $once
	 *
	 * @since 4.0.9 Added the $once parameter to allow control over whether the file should be included multiple times or only once.
	 */
	public function require_file( $relative_path, $vars = [], $once = true ) {
		if ( ! empty( $vars ) && is_array( $vars ) ) {
			extract( $vars );
		}

		if ( $once ) {
			require_once $this->locate_file( $relative_path );
		} else {
			require $this->locate_file( $relative_path );
		}
	}

	/**
	 * Assets path.
	 *
	 * @param string $relative_path
	 *
	 * @return string
	 */
	public function assets_path( $relative_path = '' ) {
		return $this->locate_file( 'assets/' . ltrim( $relative_path, '/' ) );
	}

	/**
	 * Assets URL.
	 *
	 * @param string $relative_path
	 *
	 * @return string
	 */
	public function assets_url( $relative_path = '' ) {
		return $this->file_url( 'assets/' . $relative_path );
	}

	/**
	 * Get endpoint URL.
	 *
	 * @param string $endpoint
	 * @param array  $params
	 *
	 * @return string|WP_Error
	 */
	public function endpoint_url( $endpoint, $params = [] ) {
		$endpoints = $this->get_api_endpoints();

		// Missing endpoint
		if ( ! isset( $endpoints[ $endpoint ] ) ) {
			return new WP_Error( 'kalium_api_missing_endpoint', 'The endpoint "' . $endpoint . '" is missing.' );
		}

		// Endpoint
		$endpoint = $endpoints[ $endpoint ]['endpoint'];

		// Parse vars
		if ( preg_match_all( '/:(\w+)/', $endpoint, $match ) ) {
			foreach ( $match[1] as $param_name ) {
				if ( ! isset( $params[ $param_name ] ) ) {
					return new WP_Error( 'kalium_api_missing_var', 'Endpoint var "' . $param_name . '" is missing.' );
				}

				$endpoint = preg_replace( '/:' . preg_quote( $param_name ) . '/', $params[ $param_name ], $endpoint, 1 );
			}
		}

		return self::API_SERVER . '/' . $endpoint;
	}

	/**
	 * API communicator.
	 *
	 * @param string $endpoint
	 * @param array  $args
	 *
	 * @return array|string|WP_Error
	 */
	public function api( $endpoint, $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'method'        => null,
				'safe'          => false,
				'params'        => [],
				'fields'        => [],
				'cache'         => false,
				'cache_refresh' => false,
			]
		);

		// Force cache refresh
		if ( isset( $_GET['kalium_api_nocache'] ) && current_user_can( 'activate_plugins' ) ) {
			$args['cache_refresh'] = true;
		}

		// API Endpoints
		$endpoints = $this->get_api_endpoints();

		// Endpoint URL
		$endpoint_url = $this->endpoint_url( $endpoint, $args['params'] );

		if ( is_wp_error( $endpoint_url ) ) {
			return $endpoint_url;
		}

		// Serve cached response
		if ( $args['cache'] ) {
			$hash_value = $endpoint_url;

			if ( ! empty( $args['fields'] ) ) {
				$hash_value .= json_encode( $args['fields'] );
			}

			$cache_key   = '_kalium_api_response_' . $endpoint . '_' . wp_hash( $hash_value, 'nonce' );
			$cache_value = get_transient( $cache_key );

			if ( false !== $cache_value && ! $args['cache_refresh'] ) {
				return $cache_value;
			}
		}

		// Endpoint data
		$endpoint_data = wp_parse_args(
			$endpoints[ $endpoint ],
			[
				'method'   => 'GET',
				'download' => false,
				'timeout'  => null,
			]
		);

		// Request method
		$method = strtoupper( $args['method'] ?? $endpoint_data['method'] );

		// License key
		$endpoint_url = $this->license->wrap_license_key( $endpoint_url );

		// Download file
		if ( $endpoint_data['download'] ) {
			if ( ! function_exists( 'download_url' ) ) {
				require_once ABSPATH . 'wp-admin/includes/file.php';
			}

			return download_url( $endpoint_url );
		}

		// Request args
		$request_args = [
			'method'  => strtoupper( $endpoint_data['method'] ),
			'headers' => [
				'X-Kalium-License-Key' => $this->license->get_license_key(),
				'X-Kalium-License-ID'  => $this->license->get_license_id(),
				'X-Kalium-Version'     => kalium()->get_version(),
			],
		];

		// Timeout
		if ( is_numeric( $endpoint_data['timeout'] ) ) {
			$request_args['timeout'] = $endpoint_data['timeout'];
		}

		// Reject unsafe URLs
		if ( $args['safe'] ) {
			$request_args['reject_unsafe_urls'] = true;
		}

		// Body
		if ( ! empty( $args['fields'] ) ) {
			$request_args['body'] = $args['fields'];
		}

		// Response
		$response = wp_remote_request( $endpoint_url, $request_args );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$response_body = wp_remote_retrieve_body( $response );

		// Content type
		$content_type = explode( ';', wp_remote_retrieve_header( $response, 'content-type' ) )[0];

		// JSON response
		if ( 'application/json' === $content_type ) {
			$response_body = kalium_decode_json( $response_body, true, 512, JSON_OBJECT_AS_ARRAY );
		}

		// Store in cache
		if ( isset( $cache_key ) ) {
			$cache_time = is_numeric( $args['cache'] ) ? absint( $args['cache'] ) : HOUR_IN_SECONDS;

			set_transient( $cache_key, $response_body, $cache_time );
		}

		return $response_body;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		if ( self::$instance ) {
			return;
		}

		// Save reference to the current instance
		self::$instance = $this;

		// Load composer packages
		$this->require_file( 'vendor/autoload.php' );

		// Load functions packages
		$this->load_theme_functions();

		// Register autoloader
		spl_autoload_register(
			kalium_class_autoloader(
				[
					'namespace' => self::class,
					'base_dir'  => __DIR__,
				]
			)
		);

		// Load framework classes
		$this->load_framework_classes();

		// Kalium init hook
		do_action( 'kalium_init' );
	}

	/**
	 * Load theme functions.
	 */
	private function load_theme_functions() {
		$packages = array_keys(
			array_filter(
				[
					'core'        => true,
					'header'      => true,
					'style'       => true,
					'page'        => true,
					'blog'        => true,
					'portfolio'   => true,
					'woocommerce' => class_exists( 'WooCommerce' ),
					'util'        => true,
					'alias'       => true,
					'other'       => true,
					'admin'       => true,
				]
			)
		);

		$function_files = [
			'functions.php',
			'hook-functions.php',
			'template-functions.php',
			'template-hooks.php',
		];

		foreach ( $packages as $package ) {
			foreach ( $function_files as $function_file ) {
				if ( $file_path = $this->locate_file( 'includes/functions/' . $package . '/' . $function_file ) ) {
					require_once $file_path;
				}
			}
		}
	}

	/**
	 * Load Kalium framework classes.
	 */
	private function load_framework_classes() {

		// Autoload framework classes
		foreach ( $this->get_framework_autoload_classes() as $args ) {
			$class_name    = $args['class_name'] ?? null;
			$instance_name = $args['instance'] ?? null;

			// Instantiate class
			if ( false !== $instance_name ) {
				$instance = new $class_name( $this );

				// Accessible sub instance
				if ( $instance_name ) {
					$this->sub_instances[ $instance_name ] = $instance;
				}
			}
		}
	}
}

// Start Kalium
Kalium::instance();
