<?php
/**
 * Kalium WordPress Theme
 *
 * WooCommerce Adjacent Products class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\WooCommerce;

class Adjacent_Products {

	/**
	 * Retrieve an adjacent product, optionally looping back if none found.
	 *
	 * @param array        $args
	 * @param \WP_Post|int $post
	 *
	 * @return \WC_Product|false
	 */
	public static function get_adjacent_product( $args = [], $post = null ) {
		$args = apply_filters(
			'kalium_woocommerce_adjacent_product_args',
			wp_parse_args(
				$args,
				[
					'previous'     => true,
					'in_same_term' => false,
					'taxonomy'     => 'product_cat',
					'loop'         => true,
				]
			)
		);

		$previous     = $args['previous'];
		$in_same_term = $args['in_same_term'];
		$taxonomy     = $args['taxonomy'];
		$loop         = $args['loop'];

		$current_post = get_post( $post );

		// No post found
		if ( ! $current_post ) {
			return false;
		}

		$current_id = $current_post->ID;
		$product    = false;

		// Filter callback that updates the WHERE clause to use the current post's date
		$filter_callback = function ( $where ) use ( &$current_id ) {
			$updated_post = get_post( $current_id );

			return str_replace( $GLOBALS['post']->post_date, $updated_post->post_date, $where );
		};

		// Add the post adjacent filter
		$filter_name = 'get_' . ( $previous ? 'previous' : 'next' ) . '_post_where';

		add_filter( $filter_name, $filter_callback );

		// Repeatedly get the adjacent post until we find a visible product or none left
		while ( true ) {

			// Get adjacent post
			$adjacent = get_adjacent_post( $in_same_term, '', $previous, $taxonomy );

			if ( ! $adjacent ) {
				break;
			}

			$maybe_product = wc_get_product( $adjacent->ID );

			if ( $maybe_product && $maybe_product->is_visible() ) {
				$product = $maybe_product;
				break;
			}

			$current_id = $adjacent->ID;
		}

		// Remove the post adjacent filter
		remove_filter( $filter_name, $filter_callback );

		if ( $product ) {
			return $product;
		}

		// If not looping, stop here
		if ( ! $loop ) {
			return false;
		}

		// Otherwise, loop back: fetch the first or last product in the catalog, optionally within the same term
		$query_args = [
			'limit'      => 2,
			'visibility' => 'catalog',
			'exclude'    => [ $post->ID ],
			'orderby'    => 'date',
			'order'      => $previous ? 'DESC' : 'ASC',
			'status'     => 'publish',
		];

		if ( $in_same_term ) {
			$terms = get_the_terms( $post->ID, $taxonomy );

			if ( ! empty( $terms ) && ! is_wp_error( $terms ) ) {
				$query_args['tax_query'] = [
					[
						'taxonomy' => $taxonomy,
						'field'    => 'slug',
						'terms'    => wp_list_pluck( $terms, 'slug' ),
					],
				];
			}
		}

		$products = wc_get_products( $query_args );

		return ( count( $products ) >= 2 ) ? $products[0] : false;
	}

	/**
	 * Get the previous product.
	 *
	 * @param array        $args
	 * @param \WP_Post\int $post
	 *
	 * @return \WC_Product|false
	 */
	public static function get_previous_product( $args = [], $post = null ) {
		return self::get_adjacent_product(
			array_merge(
				$args,
				[
					'previous' => true,
				]
			),
			$post
		);
	}

	/**
	 * Get the next product.
	 *
	 * @param array        $args
	 * @param \WP_Post\int $post
	 *
	 * @return \WC_Product|false
	 */
	public static function get_next_product( $args = [], $post = null ) {
		return self::get_adjacent_product(
			array_merge(
				$args,
				[
					'previous' => false,
				]
			),
			$post
		);
	}
}
