<?php
/**
 * Kalium WordPress Theme
 *
 * Admin theme functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Count theme bundled plugins that need update.
 *
 * @return int
 */
function kalium_get_plugin_updates_count() {
	global $tgmpa;

	// Plugin updates count
	$plugin_updates = 0;

	if ( $tgmpa ) {
		foreach ( $tgmpa->plugins as $slug => $plugin ) {
			if ( $tgmpa->is_plugin_installed( $slug ) && false !== $tgmpa->does_plugin_have_update( $slug ) && false !== $tgmpa->does_plugin_have_update( $slug ) ) {
				++$plugin_updates;
			}
		}
	}

	return $plugin_updates;
}

/**
 * Get editor color palette.
 *
 * @return array
 * @since 4.0
 */
function kalium_get_editor_color_palette() {
	return array_map(
		function ( $color_entry ) {
			$color_entry['color'] = is_admin() ? $color_entry['color'] : kalium_css_var_reference( $color_entry['slug'] );

			return $color_entry;
		},
		kalium_get_theme_global_colors()
	);
}

/**
 * Checks if current screen is block editor.
 *
 * @return bool
 * @since 4.0
 */
function kalium_is_block_editor() {
	return kalium()->is->block_editor();
}

/**
 * Define JS admin var.
 *
 * @param string $name
 * @param mixed  $value
 *
 * @return string|void
 * @since 4.0
 */
function kalium_admin_js_define_var( $name, $value, $args = [] ) {
	static $admin_var_name = 'kaliumAdmin', $declared = false;

	$args = wp_parse_args(
		$args,
		[
			'sub_prop'    => 'vars',
			'echo'        => true,
			'script_wrap' => true,
		]
	);

	$js_lines = [];

	if ( false === $declared ) {
		$js_lines[] = <<<EOD
if ( 'undefined' === typeof {$admin_var_name} ) {
	var {$admin_var_name} = {};
}
if ( 'undefined' === typeof {$admin_var_name}.{$args['sub_prop']} ) {
	{$admin_var_name}.{$args['sub_prop']} = {};
}
EOD;
		$declared   = true;
	}

	// Handle value
	$value = wp_json_encode( $value, JSON_NUMERIC_CHECK );

	// Add var declaration
	$js_lines[] = sprintf( '%s.%s.%s = %s;', $admin_var_name, $args['sub_prop'], esc_js( $name ), $value );

	$plain_script   = implode( PHP_EOL, $js_lines );
	$wrapped_script = $args['script_wrap'] ? wp_get_inline_script_tag( $plain_script ) : $plain_script;

	// Print JS var
	if ( $args['echo'] ) {
		echo $wrapped_script;
		return;
	}

	return $wrapped_script;
}

/**
 * Define multiple JS admin vars.
 *
 * @param array $vars
 * @param array $args
 *
 * @return string|void
 * @since 4.0
 */
function kalium_admin_js_define_vars( $vars, $args = [] ) {
	$args = wp_parse_args(
		$args,
		[
			'sub_prop'    => 'vars',
			'echo'        => true,
			'script_wrap' => true,
		]
	);

	$js_lines = [];

	foreach ( $vars as $var_name => $var_value ) {
		$js_lines[] = kalium_admin_js_define_var(
			$var_name,
			$var_value,
			[
				'sub_prop'    => $args['sub_prop'],
				'echo'        => false,
				'script_wrap' => false,
			]
		);
	}

	$plain_script   = implode( PHP_EOL, $js_lines );
	$wrapped_script = $args['script_wrap'] ? wp_get_inline_script_tag( $plain_script ) : $plain_script;

	// Print JS var
	if ( $args['echo'] ) {
		echo $wrapped_script;
		return;
	}

	return $wrapped_script;
}

/**
 * Register customize panel.
 *
 * @param string $id
 * @param array  $args
 *
 * @since 4.0
 */
function kalium_customize_register_panel( $id, $args = [] ) {
	kalium()->customize->add_panel( $id, $args );
}

/**
 * Register customize section.
 *
 * @param string $id
 * @param array  $args
 *
 * @since 4.0
 */
function kalium_customize_register_section( $id, $args = [] ) {
	kalium()->customize->add_section( $id, $args );
}

/**
 * End customize section.
 *
 * @param bool $reset
 *
 * @since 4.0
 */
function kalium_customize_end_section( $reset = false ) {
	kalium()->customize->end_section( $reset );
}

/**
 * Register customize control and setting.
 *
 * @param string $id
 * @param array  $args
 *
 * @since 4.0
 */
function kalium_customize_register_control( $id, $args = [] ) {
	kalium()->customize->add_control( $id, $args );
}

/**
 * Start tab.
 *
 * @param string $title
 */
function kalium_customize_start_tab( $title ) {
	kalium()->customize->start_tab( $title );
}

/**
 * End current tab.
 */
function kalium_customize_end_tab() {
	kalium()->customize->end_tab();
}

/**
 * General tab.
 */
function kalium_customize_general_tab() {
	kalium_customize_start_tab( 'General' );
}

/**
 * Style tab.
 */
function kalium_customize_style_tab() {
	kalium_customize_start_tab( 'Style' );
}

/**
 * Register customize preview
 *
 * @param array $args
 *
 * @since 4.0
 */
function kalium_customize_register_preview_update( $args = [] ) {
	kalium()->customize->add_preview_update( $args );
}

/**
 * Default units in Customizer.
 *
 * @param string|array $type
 *
 * @return array
 * @since 4.0
 */
function kalium_get_default_units( $type = 'length-percentage' ) {
	static $units = [
		[
			'value' => 'px',
			'label' => 'PX',
		],
		[
			'value' => '%',
			'label' => '%',
		],
		[
			'value' => 'rem',
			'label' => 'REM',
			'step'  => 0.05,
		],
		[
			'value' => 'em',
			'label' => 'EM',
			'step'  => 0.05,
		],
		[
			'value' => 'vw',
			'label' => 'VW',
		],
		[
			'value' => 'vh',
			'label' => 'VH',
		],
	];

	static $length_types = [
		'length-percentage' => [ 'px', '%', 'rem', 'em', 'vw', 'vh' ],
		'length'            => [ 'px', 'rem', 'em', 'vw', 'vh' ],
		'font-size'         => [ 'px', 'rem', 'em' ],
		'box-size'          => [ 'px', '%', 'rem', 'em', 'vw', 'vh' ],
	];

	$filtered_units = [];

	// Get unit entry
	$get_unit_entry = function ( $unit ) use ( $units ) {
		foreach ( $units as $unit_entry ) {
			if ( $unit === $unit_entry['value'] ) {
				return $unit_entry;
			}
		}

		return null;
	};

	// Provided array of units
	if ( is_array( $type ) ) {
		foreach ( $type as $unit => $unit_options ) {
			if ( is_numeric( $unit ) ) {
				$unit         = $unit_options;
				$unit_options = null;
			}

			if ( $unit_entry = $get_unit_entry( $unit ) ) {
				if ( is_array( $unit_options ) ) {
					$unit_entry = array_merge( $unit_entry, $unit_options );
				}

				$filtered_units[] = $unit_entry;
			}
		}
	} else {
		$type_units = $length_types[ $type ?? $units[0]['value'] ];

		foreach ( $type_units as $unit ) {
			if ( $unit_entry = $get_unit_entry( $unit ) ) {
				$filtered_units[] = $unit_entry;
			}
		}
	}

	return $filtered_units;
}

/**
 * Check if current page is customizer file.
 *
 * @return bool
 * @since 4.0
 */
function kalium_is_customizer_preview_admin() {
	global $pagenow;

	return is_admin() && 'customize.php' === $pagenow;
}

/**
 * Check if given $post's post_type has sidebar on single page.
 *
 * @param int|WP_Post $post_id
 *
 * @return bool
 * @since 4.0
 */
function kalium_has_sidebar_on_singular( $post = null ) {
	/** @var WP_Post $post */
	if ( empty( $post ) && isset( $GLOBALS['post'] ) ) {
		$post = $GLOBALS['post'];
	}

	$has_sidebar = false;

	if ( $post instanceof WP_Post ) {
		switch ( $post->post_type ) {
			case 'post':
				$has_sidebar = kalium_get_blog_option( 'single/sidebar/visible' );
				break;
		}
	}

	return $has_sidebar;
}

/**
 * Get the status for theme backups, true if enabled.
 *
 * @return bool
 */
function kalium_get_theme_backups_status() {
	return false; // TODO: Implement the functionality
}
