<?php
/**
 * Kalium WordPress Theme
 *
 * Admin theme functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Admin enqueues.
 */
function kalium_admin_enqueue_scripts() {

	// Admin styles and scripts
	kalium_enqueue( 'theme-admin' );

	// Admin icons
	kalium_enqueue( 'theme-icons-admin' );
}

add_action( 'admin_enqueue_scripts', 'kalium_admin_enqueue_scripts' );

/**
 * Block editor styles.
 */
function kalium_enqueue_block_assets() {
	if ( ! kalium_is_block_editor() ) {
		return;
	}

	// Block editor style
	kalium_enqueue( 'block-editor' );

	// Gather printed styles
	ob_start();

	// Block Editor CSS vars
	$block_editor_vars = [];

	// Set container width
	kalium_set_container_max_width();

	// Set block spacing
	kalium_set_block_spacing();

	// Theme color vars
	kalium_print_theme_color_vars();

	// Button styles
	kalium_print_button_style();

	// Link styles
	kalium_print_link_styles();

	// TypoLab Fonts
	add_filter( 'typolab_font_selectors', 'kalium_block_editor_typolab_font_selectors' );
	TypoLab_Font_Loader::load_fonts();
	remove_filter( 'typolab_font_selectors', 'kalium_block_editor_typolab_font_selectors' );

	// Widgets spacing
	if ( $sidebar_widgets_spacing = kalium_get_theme_option( 'sidebar_widgets_spacing' ) ) {
		$block_editor_vars['sb-widgets-spacing'] = $sidebar_widgets_spacing . 'px';
	}

	// When sidebar is enabled
	if ( kalium_has_sidebar_on_singular() ) {
		$container_max_width = kalium_get_container_max_width( true );
		$sidebar_width       = kalium_get_sidebar_width( true );
		$sidebar_gap         = kalium_get_theme_option( 'sidebar_gap' ) ?: kalium_css_var_reference( 'sidebar-gap' );

		// Convert container width to percentage value
		if ( false === strpos( $container_max_width, '%' ) && is_numeric( $sidebar_width ) ) {
			$sidebar_width = ( $sidebar_width / 100 );
			$sidebar_width = "$container_max_width * $sidebar_width";
		} else {
			$sidebar_width .= '%';
		}

		// Adjust container width
		kalium_set_css_var(
			'content-width',
			sprintf(
				'calc(%s - (%s + %s))',
				$container_max_width,
				$sidebar_width,
				is_numeric( $sidebar_gap ) ? "{$sidebar_gap}px" : $sidebar_gap,
			)
		);
	}

	// Print Block Editor vars
	kalium_print_inline_style(
		[
			'selector' => kalium_get_root_selector(),
			'vars'     => $block_editor_vars,
		]
	);

	$styles = ob_get_clean();

	// Add inline styles
	if ( preg_match_all( '/<style.*?>(.*?)<\/style>/s', $styles, $matches ) ) {
		$inline_styles_handle = 'kalium-block-editor-' . kalium_generate_unique_string();

		// Replace selectors
		$replace_selectors = [ ':root', 'body', '.link-style' ];

		// CSS output
		$css_output = implode( PHP_EOL, $matches[1] );
		$css_output = preg_replace_callback(
			'/[^{]+\{[^}]+\}/',
			function ( $matches ) use ( &$replace_selectors ) {
				$css_ruleset   = $matches[0];
				$css_selectors = substr( $css_ruleset, 0, strpos( $css_ruleset, '{' ) );
				return str_replace( $css_selectors, str_replace( $replace_selectors, kalium_get_root_selector(), $css_selectors ), $css_ruleset );
			},
			$css_output
		);

		wp_register_style( $inline_styles_handle, false );
		wp_enqueue_style( $inline_styles_handle );
		wp_add_inline_style( $inline_styles_handle, $css_output );
	}
}

add_action( 'enqueue_block_assets', 'kalium_enqueue_block_assets' );

/**
 * Block editor TypoLab selectors.
 *
 * @param array $selectors
 *
 * @return array
 */
function kalium_block_editor_typolab_font_selectors( $selectors ) {
	return array_map(
		function ( $selector ) {
			if ( in_array( $selector, [ ':root', 'body', 'html' ], true ) ) {
				$selector = kalium_get_root_selector();
			} elseif ( kalium()->helpers->str_starts_with( $selector, 'body' ) ) {
				$selector = str_replace( 'body', kalium_get_root_selector(), $selector );
			} else {
				$selector = kalium_get_root_selector() . ' ' . $selector;
			}

			return $selector;
		},
		$selectors
	);
}

/**
 * Tiny MCE editor style.
 *
 * @param array $mce_init
 *
 * @return array
 */
function kalium_tiny_mce_editor_style( $mce_init = [] ) {
	$mce_init = wp_parse_args(
		$mce_init,
		[
			'content_style' => '',
			'content_css'   => '',
		]
	);

	// Editor style
	if ( ! empty( $mce_init['content_css'] ) ) {
		$mce_init['content_css'] .= ',';
	}

	$mce_init['content_css'] .= implode(
		',',
		[
			kalium()->assets_url( 'css/bootstrap.min.css' ),
			kalium()->assets_url( 'admin/css/block-editor.min.css' ),
		]
	);

	// Set content style
	ob_start();
	kalium_print_theme_color_vars();

	$mce_init['content_style'] = preg_replace( '/(<style.*?>|<\/style>|\n*)/', '', ob_get_clean() );

	// CSS vars
	$vars_css = ':root {';

	foreach ( kalium()->style->css_vars as $var_name => $var_value ) {
		$vars_css .= $var_name . ': ' . $var_value . ';';
	}

	$vars_css .= '}';

	$mce_init['content_style'] .= $vars_css;

	return $mce_init;
}

add_filter( 'tiny_mce_before_init', 'kalium_tiny_mce_editor_style' );

/**
 * Deactivate Portfolio Post Type plugin automatically and show a notice.
 *
 * @since 4.0
 */
function kalium_portfolio_post_type_auto_deactivate() {
	if ( ! current_user_can( 'deactivate_plugins' ) ) {
		return;
	}

	$portfolio_post_type = 'portfolio-post-type/portfolio-post-type.php';

	// Automatically deactivate Portfolio Post Type plugin
	if ( kalium()->is->plugin_active( $portfolio_post_type ) && false === get_option( '_kalium_portfolio_post_type_deactivated' ) ) {
		deactivate_plugins( $portfolio_post_type );
		update_option( '_kalium_portfolio_post_type_deactivated', true );
	}

	// Display recommendation notice
	if ( kalium()->is->plugin_active( $portfolio_post_type ) ) {
		$theme_name = kalium_get_theme_name();

		$message = <<<EOD
<p><strong>Action Required:</strong> This version of {$theme_name} theme doesn't require <strong>Portfolio Post Type</strong> plugin anymore because it is a built-in feature inside the theme.</p>
<p>We strongly recommend you to delete the <strong>Portfolio Post Type</strong> plugin.</p>
EOD;

		kalium()->helpers->add_admin_notice( $message, 'warning' );
	}
}

add_action( 'admin_init', 'kalium_portfolio_post_type_auto_deactivate' );

/**
 * Register Laborator Builder control in Kalium customizer.
 *
 * @param array $control_types
 *
 * @return array
 */
function kalium_register_laborator_builder_control_type( $control_types ) {
	$control_types['laborator-builder'] = \Laborator_Builder\Customize_Control::class;

	return $control_types;
}

add_filter( 'kalium_customize_control_types', 'kalium_register_laborator_builder_control_type' );

/**
 * Version upgrade tasks.
 *
 * @since 4.1.1
 */
function kalium_version_upgrade_tasks() {
	if ( false === kalium()->license->is_theme_registered() ) {
		return;
	}

	$danger_color_var = sprintf( 'var(%s, %s)', kalium_css_var_name( 'danger-color' ), '#b42318' );

	/**
	 * Upgrade task for v4.0
	 */
	kalium()->version_upgrades->add_upgrade_task(
		'v4-upgrade',
		'4.0',
		[
			'labels' => [
				'description' => <<<EOD
Thank you for updating to the latest version of Kalium.

As this is a major version upgrade, it involves a data migration process to transition settings from Kalium 3 to the new version, along with other necessary updates.

Please be aware that major updates may introduce some changes and potential issues. To mitigate any risks, it is essential to back up your site before proceeding.

The migration process is irreversible. We strongly recommend reviewing the detailed upgrade instructions on our site prior to starting:

<a href="https://docs.kaliumtheme.com/getting-started/migrating-from-kalium-3-to-4/migrating-manually#id-3.-database-upgrade" target="_blank" rel="noreferrer noopener" style="color: $danger_color_var"><strong>Review Upgrade Instructions &raquo;</strong></a>
EOD,
			],
		]
	);

	/**
	 * Upgrade task for v4.1.2
	 */
	kalium()->version_upgrades->add_upgrade_task(
		'v412-upgrade',
		'4.1.2',
		[
			'auto' => true,
		]
	);
}

add_action( 'admin_head', 'kalium_version_upgrade_tasks' );
add_action( 'rest_api_init', 'kalium_version_upgrade_tasks' );

/**
 * Get license activation details from Kalium 3 upgrade theme process.
 *
 * @since 4.0
 */
function kalium_upgrade_license_activation_details() {
	if ( ! current_user_can( 'update_themes' ) ) {
		return;
	}

	$license_upgrade = get_option( '_kalium_license_upgrade' );

	if ( ! empty( $license_upgrade['success'] ) ) {
		$action           = kalium_fs()->get_ajax_action( 'activate_license' );
		$license_key      = kalium_get_array_key( $license_upgrade, 'license/secret_key' );
		$activate_license = wp_create_nonce( $action );

		// Delete license upgrade option
		delete_option( '_kalium_license_upgrade' );

		// Response
		wp_send_json(
			[
				'success'     => true,
				'ajax_url'    => Freemius::ajax_url( '' ),
				'ajax_action' => $action,
				'security'    => $activate_license,
				'license_key' => $license_key,
				'module_id'   => kalium_fs()->get_id(),
			]
		);
	}
}

add_action( 'wp_ajax_kalium_upgrade_license_activation_details', 'kalium_upgrade_license_activation_details' );

/**
 * Display admin notices.
 */
function kalium_admin_notices_display() {
	$notices = kalium_api(
		'notices',
		[
			'cache' => DAY_IN_SECONDS / 2,
		]
	);

	if ( is_array( $notices ) && ! empty( trim( $notices['content'] ) ) && $notices['timestamp'] < time() && ! get_user_meta( get_current_user_id(), 'kalium_notices_hide-' . $notices['id'], true ) ) {
		kalium()->helpers->build_dom_element(
			[
				'echo'       => true,
				'tag_name'   => 'div',
				'content'    => '<style>' . wp_strip_all_tags( $notices['style'] ) . '</style>' . wp_kses_post( $notices['content'] ),
				'attributes' => [
					'id'    => $notices['id'],
					'class' => [
						'laborator-notice',
						'notice',
						'notice-' . $notices['type'],
						kalium_conditional( $notices['dismissible'], 'is-dismissible' ),
					],
				],
			]
		);
	}
}

add_action( 'admin_notices', 'kalium_admin_notices_display' );

/**
 * Dismiss notice for current user.
 */
function kalium_admin_dismiss_notice() {
	if ( $id = kalium()->request->input( 'id' ) ) {
		update_user_meta( get_current_user_id(), 'kalium_notices_hide-' . $id, true );
	}
}

add_action( 'wp_ajax_kalium_hide_notice', 'kalium_admin_dismiss_notice' );
