<?php
/**
 * Kalium WordPress Theme
 *
 * Blog hook functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Legacy labeled input group class for comment form fields.
 */
function kalium_blog_comment_form_legacy_style( $defaults ) {
	if ( apply_filters( 'kalium_use_legacy_form_style', false ) ) {
		$defaults['comment_field'] = preg_replace( '/(<p.*?)class="(.*?)"/', '\1class="labeled-textarea-row \2"', $defaults['comment_field'] );

		// Comment fields
		foreach ( $defaults['fields'] as $name => & $field ) {
			if ( in_array( $name, [ 'cookies' ] ) ) {
				continue;
			}

			$field = preg_replace( '/(<p.*?)class="(.*?)"/', '\1class="labeled-input-row \2"', $field );
		}
	}

	return $defaults;
}

add_filter( 'comment_form_defaults', 'kalium_blog_comment_form_legacy_style' );

/**
 * External post redirect.
 */
function kalium_blog_external_post_format_redirect() {

	if ( is_single() && 'link' == get_post_format() && apply_filters( 'kalium_blog_external_link_redirect', true ) ) {
		$urls = wp_extract_urls( get_the_content() );

		if ( $urls ) {
			wp_redirect( current( $urls ) );
			exit;
		}
	}
}

add_action( 'template_redirect', 'kalium_blog_external_post_format_redirect' );

/**
 * Blog post content, clear post format if its enabled to be parsed.
 *
 * @param string $content
 *
 * @return string
 */
function kalium_blog_clear_post_format_from_the_content( $content ) {
	global $post;

	if ( in_array( get_post_format(), [ 'video', 'audio', 'quote' ] ) ) {
		$post_format = kalium_extract_post_format_content( $post );

		if ( ! empty( $post_format['content'] ) ) {
			$content = preg_replace( sprintf( '/%s/', preg_quote( $post_format['content'], '/' ) ), '', $content );
		}
	}

	return $content;
}

add_filter( 'kalium_blog_post_content', 'kalium_blog_clear_post_format_from_the_content', 10 );

/**
 * Change "href" for link post formats.
 *
 * @param string  $permalink
 * @param WP_Post $post
 *
 * @return string
 */
function kalium_blog_post_format_link_url( $permalink, $post ) {
	if ( kalium_is_external_url_post( $post ) ) {
		$urls = wp_extract_urls( apply_filters( 'the_content', $post->post_content ) );

		if ( ! empty( $urls ) ) {
			return $urls[0];
		}
	}

	return $permalink;
}

add_action( 'post_link', 'kalium_blog_post_format_link_url', 10, 2 );

/**
 * Single post comments visibility.
 *
 * @param bool $visible
 *
 * @return bool
 */
function kalium_blog_comments_visibility( $visible ) {
	return kalium_get_blog_option( 'single/post_comments' );
}

add_filter( 'kalium_blog_comments', 'kalium_blog_comments_visibility' );

/**
 * Add comment class for each comment entry, this fixes the issue with pingbacks.
 *
 * @param string[] $classes
 *
 * @return array
 *
 * @since 3.2
 */
function kalium_blog_comments_add_comment_class( $classes ) {
	if ( is_array( $classes ) && ! in_array( 'comment', $classes, true ) ) {
		array_unshift( $classes, 'comment' );
	}

	return $classes;
}

add_action( 'comment_class', 'kalium_blog_comments_add_comment_class' );

/**
 * Single post featured image placement.
 */
function kalium_blog_single_handle_featured_image_placement() {
	if ( ! is_single() ) {
		return;
	}

	$image_placement = kalium_get_blog_option( 'single/post_image/placement' );

	// Full width
	if ( 'full-width' === $image_placement ) {
		remove_action( 'kalium_blog_single_post_start', 'kalium_blog_single_post_featured_image' );
		add_action( 'kalium_blog_single', 'kalium_blog_single_post_featured_image', 5 );
	}
	// Sidebar Below
	elseif ( kalium_get_blog_option( 'single/sidebar/visible' ) && kalium_get_theme_option( 'blog_featured_image_sidebar_below' ) ) {
		remove_action( 'kalium_blog_single_post_start', 'kalium_blog_single_post_featured_image' );
		add_action( 'kalium_blog_single', 'kalium_blog_single_post_featured_image', 15 );
	}
}

add_action( 'wp', 'kalium_blog_single_handle_featured_image_placement' );

/**
 * Container options in blog single page.
 */
function kalium_blog_single_container_options() {
	if ( ! ( is_single() || ( kalium_is_block_editor() && 'post' === get_current_screen()->post_type ) ) ) {
		return;
	}

	// Narrow container width
	if ( kalium_get_blog_option( 'single/narrow' ) ) {
		$narrow_width = kalium_get_theme_option( 'blog_post_narrow_container_width' );

		if ( $narrow_width ) {
			kalium_set_css_var( 'content-narrow-width', $narrow_width );
		}

		// Set the width for Block editor
		if ( kalium_is_block_editor() ) {
			kalium_set_css_var( 'container-max-width', kalium_css_var_reference( 'content-narrow-width' ) );
		}
	}

	// Wide align offset
	if ( ! kalium_get_blog_option( 'single/sidebar/visible' ) && ( $wide_align_offset = kalium_get_theme_option( 'blog_post_wide_align_offset' ) ) ) {
		kalium_set_css_var( 'wide-align-offset', $wide_align_offset );
	}
}

add_action( 'wp', 'kalium_blog_single_container_options' );
add_action( 'wp_enqueue_editor', 'kalium_blog_single_container_options' );

/**
 * Excerpt length.
 *
 * @param int $length
 *
 * @return int
 * @since 4.0
 */
function kalium_blog_post_excerpt_length( $length ) {
	global $post;

	if ( 'post' === $post->post_type ) {
		$excerpt_length = kalium_get_blog_option( 'loop/post_excerpt_length' );

		if ( is_numeric( $excerpt_length ) ) {
			return $excerpt_length;
		}
	}

	return $length;
}

add_filter( 'excerpt_length', 'kalium_blog_post_excerpt_length' );

/**
 * Post navigation in single post setup.
 *
 * @since 4.1.2
 */
function kalium_blog_single_post_navigation_setup() {
	if ( ! kalium_get_blog_option( 'single/post_navigation/enabled' ) ) {
		return;
	}

	if ( in_array( kalium_get_blog_option( 'single/post_navigation/options/layout' ), [ 'type-2', 'type-3' ] ) ) {
		remove_action( 'kalium_blog_single_post_footer', 'kalium_blog_single_post_navigation', 40 );
		add_action( 'kalium_blog_single', 'kalium_blog_single_post_navigation', 1000 );
	}
}

add_action( 'wp', 'kalium_blog_single_post_navigation_setup' );
