<?php
/**
 * Kalium WordPress Theme
 *
 * Core theme functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Kalium instance.
 *
 * @return Kalium
 */
function kalium() {
	return Kalium::instance();
}

/**
 * Class autoloader.
 *
 * @param array $args [optional]
 *
 * @return callable
 * @since 4.1
 */
function kalium_class_autoloader( $args ) {
	extract(
		wp_parse_args(
			$args,
			[
				'namespace' => '',
				'base_dir'  => '',
				'separator' => '-',
			]
		)
	);

	$base_dir = trailingslashit( $base_dir );

	if ( is_array( $namespace ) ) {
		$namespace = implode( '\\', $namespace );
	}

	return static function ( $class ) use ( $base_dir, $namespace, $separator ) {
		if ( ! str_starts_with( $class, $namespace ) || ! $namespace ) {
			return;
		}

		// Build full class path
		$class_path = substr( $class, strlen( $namespace ) + 1 );
		$class_path = strtolower( $class_path );
		$class_path = str_replace( '_', $separator, $class_path );
		$class_path = str_replace( '\\', DIRECTORY_SEPARATOR, $class_path );
		$class_path = $base_dir . $class_path . '.php';

		if ( file_exists( $class_path ) ) {
			require $class_path;
		}
	};
}

/**
 * Get template from Kalium theme.
 *
 * @param string $file
 * @param array  $args
 */
function kalium_get_template( $file, $args = [] ) {
	kalium()->template_loader->get_template( $file, $args );
}

/**
 * Replace WooCommerce template.
 *
 * @param string|array    $template_fragment
 * @param string|callable $new_template
 * @param bool|callable   $do_replace
 *
 * @return callable
 * @since 4.0
 */
function kalium_replace_woocommerce_template( $template_fragment, $new_template = null, $do_replace = true ) {
	return kalium()->template_loader->replace_woocommerce_template( $template_fragment, $new_template, $do_replace );
}

/**
 * Replaces template strings with dynamic data.
 *
 * @param string  $template_string
 * @param WP_Post $post
 *
 * @return string
 * @since 4.1.2
 */
function kalium_replace_template_string( $template_string, $post = null ) {
	if ( ! is_string( $template_string ) ) {
		return $template_string;
	}

	return preg_replace_callback(
		'/\{\{\s*([^\s]+)\s*\}\}/',
		function ( $matches ) use ( &$post ) {
			$data_source = $matches[1];

			if ( 'post_title' === $data_source ) {
				return $post ? get_the_title( $post ) : '';
			} elseif ( str_starts_with( $data_source, 'taxonomy:' ) && preg_match( '/taxonomy:([\w-]+)/', $data_source, $taxonomy_matches ) ) {
				$terms = $post ? wp_get_post_terms( $post->ID, $taxonomy_matches[1], [ 'fields' => 'names' ] ) : '';

				if ( is_array( $terms ) ) {
					return implode( ', ', $terms );
				}
			} elseif ( 'product_price' === $data_source && kalium()->is->woocommerce_active() ) {
				return $post ? wc_get_product( $post->ID )->get_price_html() : '';
			}

			return '';
		},
		$template_string
	);
}

/**
 * Doing it wrong, the Kalium way.
 *
 * @param string $function
 * @param string $message
 * @param string $version
 *
 * @return void
 */
function kalium_doing_it_wrong( $function, $message, $version ) {
	$message .= ' Backtrace: ' . wp_debug_backtrace_summary();

	if ( defined( 'DOING_AJAX' ) ) {
		do_action( 'doing_it_wrong_run', $function, $message, $version );
		error_log( "{$function} was called incorrectly. {$message}. This message was added in version {$version}." );
	} else {
		_doing_it_wrong( $function, $message, $version );
	}
}

/**
 * Make Kalium API request and retrieve response.
 *
 * @param string $endpoint
 * @param array  $args
 *
 * @return array|string|WP_Error
 */
function kalium_api( $endpoint, $args = [] ) {
	return kalium()->api( $endpoint, $args );
}

/**
 * Get theme object.
 *
 * @return WP_Theme
 */
function kalium_get_theme() {
	$theme = wp_get_theme();

	return apply_filters( 'kalium_get_theme', $theme );
}

/**
 * Get theme name.
 *
 * @return string
 */
function kalium_get_theme_name() {
	return kalium_get_theme()->display( 'Name', true, false );
}

/**
 * Theme author link.
 *
 * @return string
 * @since 4.0
 */
function kalium_theme_author_link() {
	return apply_filters( 'kalium_theme_author_link', '<a href="https://laborator.co" target="_blank" rel="noreferrer noopener">Laborator</a>' );
}

/**
 * Just Kalium!
 *
 * @return string
 * @since 4.0
 */
function kalium_get_name() {
	return apply_filters( 'kalium_name', 'Kalium' );
}

/**
 * Enqueue script or style (or groups of them) from theme registered assets.
 *
 * @param string $handle
 *
 * @return void
 */
function kalium_enqueue( $handle ) {
	kalium()->assets->enqueue( $handle );
}

/**
 * Dequeue script or style (or groups of them) from theme registered assets.
 *
 * @param string $handle
 *
 * @return void
 */
function kalium_dequeue( $handle ) {
	kalium()->assets->dequeue( $handle );
}

/**
 * Add inline script before or after the script.
 *
 * @param string $handle
 * @param string $data
 * @param string $position
 */
function kalium_add_inline_script( $handle, $data, $position = 'after' ) {
	kalium()->assets->add_inline_script( $handle, $data, $position );
}

/**
 * Add inline style.
 *
 * @param string $handle
 * @param string $data
 */
function kalium_add_inline_style( $handle, $data ) {
	kalium()->assets->add_inline_style( $handle, $data );
}

/**
 * Prefix script/style handle.
 *
 * @param string $handle
 *
 * @return string
 */
function kalium_prefix_script_handle( $handle ) {
	return kalium()->assets->prefix_handle( $handle );
}

/**
 * Gets default assigned value of theme option.
 *
 * @param string $option_name
 *
 * @return mixed
 * @since 4.0
 */
function kalium_get_default_theme_option( $option_name ) {
	return kalium()->theme_options->get_default( $option_name );
}

/**
 * Check if theme option is empty.
 *
 * @param string $option_name
 * @param string $option_type
 *
 * @return bool
 * @since 4.0
 */
function kalium_is_theme_option_empty( $option_name, $option_type = null ) {
	static $has_value_fn;

	if ( is_null( $has_value_fn ) ) {
		$has_value_fn = static function ( $value ) {
			if ( ! is_array( $value ) ) {
				$value = [ $value ];
			}

			foreach ( $value as $entry_value ) {
				if ( is_scalar( $entry_value ) && $entry_value || is_numeric( $entry_value ) ) {
					return true;
				}
			}

			return false;
		};
	}

	if ( $theme_option = kalium_get_theme_option( $option_name ) ) {
		$has_value = false;

		kalium_map_responsive_value(
			$theme_option,
			function ( $value ) use ( &$has_value, &$has_value_fn, $option_type ) {
				if ( $has_value ) {
					return;
				}

				if ( is_array( $value ) ) {
					switch ( $option_type ) {
						// Border box
						case 'kalium-border':
						case 'kalium-border-box':
							foreach ( $value as $border_entry ) {
								if ( is_array( $border_entry ) ) {
									foreach ( $border_entry as $border_side_entry ) {
										if ( $has_value_fn( $border_side_entry ) ) {
											$has_value = true;
											break;
										}
									}
								} elseif ( $has_value_fn( $border_entry ) ) {
									$has_value = true;
									break;
								}
							}
							break;

						// Multi numeric as default
						default:
							unset( $value['unit'] );
							unset( $value['link'] );

							if ( $has_value_fn( $value ) ) {
								$has_value = true;
							}
					}
				} elseif ( $has_value_fn( $value ) ) {
					$has_value = true;
				}
			}
		);

		return ! $has_value;
	}

	return true;
}

/**
 * Get theme option.
 *
 * @param string $option_name
 * @param bool   $skip_default
 *
 * @return mixed|false
 */
function kalium_get_theme_option( $option_name, $skip_default = false ) {
	return kalium()->theme_options->get_option( $option_name, $skip_default );
}

/**
 * Get theme option specified by path.
 *
 * @param string|array $path
 * @param mixed        $default_value
 *
 * @return mixed|false
 * @since 4.0
 */
function kalium_get_theme_option_by_path( $path, $default_value = false ) {
	$path = is_string( $path ) ? explode( '.', $path ) : $path;

	if ( is_array( $path ) ) {
		$value = kalium_get_theme_option( array_shift( $path ) );

		if ( ! is_null( $value ) ) {
			while ( count( $path ) ) {
				$key = array_shift( $path );

				if ( isset( $value[ $key ] ) ) {
					$value = $value[ $key ];
				} else {
					return $default_value;
				}
			}

			return $value;
		}
	}

	return $default_value;
}

/**
 * Get ACF field value with fallback functionality.
 *
 * @param string   $field_key
 * @param int|bool $post_id
 * @param bool     $format_value
 *
 * @return mixed
 */
function kalium_get_field( $field_key, $post_id = false, $format_value = true ) {
	return kalium()->acf->get_field( $field_key, $post_id, $format_value );
}

/**
 * Get element from array by key (fail-safe).
 *
 * @param array  $arr
 * @param string $key
 * @param mixed  $default_value
 *
 * @return mixed|null
 */
function kalium_get_array_key( $arr, $key, $default_value = null ) {
	return kalium()->helpers->get_array_key( $arr, $key, $default_value );
}

/**
 * Get first array element value (or key).
 *
 * @param array $arr
 * @param bool  $get_key
 *
 * @return mixed|null
 */
function kalium_get_array_first( $arr, $get_key = false ) {
	return kalium()->helpers->get_array_first( $arr, $get_key );
}

/**
 * Set array item that supports path style.
 *
 * @param array        $arr
 * @param string|array $name
 * @param mixed        $new_value
 * @param bool         $append_existing
 *
 * @since 4.0
 */
function kalium_set_array_key( &$arr, $name, $new_value = null, $append_existing = true ) {
	kalium()->helpers->set_array_key( $arr, $name, $new_value, $append_existing );
}

/**
 * Filter an array with empty value elements.
 *
 * @param array $arr
 *
 * @return array
 * @since 4.0
 */
function kalium_filter_array_items( $arr ) {
	$filtered = [];

	if ( is_array( $arr ) ) {
		foreach ( $arr as $key => $value ) {
			if ( empty( $value ) && ! is_numeric( $value ) ) {
				continue;
			}

			if ( is_array( $value ) && ! empty( is_array( $value ) ) ) {
				$filtered[ $key ] = kalium_filter_array_items( $value );
			} else {
				$filtered[ $key ] = $value;
			}
		}
	}

	return $filtered;
}

/**
 * Validate a value against an accepted list (indexed or associative) with optional lowercasing.
 *
 * @param string      $current_value
 * @param array       $accepted_values
 * @param string|null $default_value
 * @param bool        $to_lowercase
 *
 * @return string
 * @since 4.1.2
 */
function kalium_enum_value( $current_value, $accepted_values, $default_value = null, $to_lowercase = true ) {
	return kalium()->helpers->enum_value( $current_value, $accepted_values, $default_value, $to_lowercase );
}

/**
 * Translate external gettext strings to ensure theme check warnings are avoided.
 *
 * @param string $func
 * @param mixed  ...$args
 *
 * @return string
 */
function kalium_xtranslate() {
	$args = func_get_args();
	$func = array_shift( $args );

	return call_user_func_array( $func, $args );
}

/**
 * Get SVG file contents from theme directory.
 *
 * @param string         $file_path
 * @param string         $id
 * @param int|int[]|null $size
 *
 * @return string|null
 */
function kalium_get_svg_file( $file_path, $id = null, $size = null ) {
	$file_path = kalium()->locate_file( $file_path );

	if ( ! file_exists( $file_path ) ) {
		return null;
	}

	// File contents
	$file_contents = file_get_contents( $file_path );

	// Remove XML header
	$file_contents = preg_replace( '/<\?xml.*?\?>\s*/i', '', $file_contents );

	// Set or replace ID
	if ( is_string( $id ) && ! empty( $id ) ) {
		$id_attribute  = sprintf( 'id="%s"', esc_attr( $id ) );
		$file_contents = preg_replace( '/(\s)id=".*?"/i', '${1}' . $id_attribute, $file_contents );

		// Assign ID if not exists
		if ( false === strpos( $file_contents, $id_attribute ) ) {
			$file_contents = str_replace( '<svg', "<svg {$id_attribute}", $file_contents );
		}
	}

	// Square size
	if ( is_numeric( $size ) ) {
		$size = [ $size, $size ];
	}

	// Apply new size
	if ( is_array( $size ) && 2 === count( $size ) ) {
		$width  = $size[0];
		$height = $size[1];

		// Set new dimensions replace
		$sizes = [
			'find'    => [
				'/width="\d+"/',
				'/height="\d+"/',
			],
			'replace' => [
				sprintf( 'width="%d"', $width ),
				sprintf( 'height="%d"', $height ),
			],
		];

		$file_contents = preg_replace( $sizes['find'], $sizes['replace'], $file_contents );

		// Assign width if not exists
		if ( false === strpos( $file_contents, $sizes['replace'][0] ) ) {
			$file_contents = str_replace( '<svg', "<svg {$sizes['replace'][0]}", $file_contents );
		}

		// Assign height if not exists
		if ( false === strpos( $file_contents, $sizes['replace'][1] ) ) {
			$file_contents = str_replace( '<svg', "<svg {$sizes['replace'][1]}", $file_contents );
		}
	}

	return $file_contents;
}

/**
 * Get queried object based on Kalium logic.
 *
 * @return int|0
 */
function kalium_get_queried_object_id() {
	static $queried_object_id;

	// Return cached queried object id
	if ( isset( $queried_object_id ) ) {
		return $queried_object_id;
	}

	// Queried object ID
	$queried_object_id = get_queried_object_id();

	// On shop archive pages inherit ID from WooCommerce shop page
	if ( kalium()->is->woocommerce_active() && ( is_shop() || is_product_taxonomy() ) && ( $post = get_post( wc_get_page_id( 'shop' ) ) ) ) {
		$queried_object_id = $post->ID;
	}

	return $queried_object_id;
}

/**
 * Get attachment image, the Kalium way.
 *
 * @param int               $attachment_id
 * @param string            $size
 * @param array|string|null $atts
 * @param array|string|null $placeholder_atts
 *
 * @return string
 */
function kalium_get_attachment_image( $attachment_id, $size = 'thumbnail', $atts = null, $placeholder_atts = null ) {
	return kalium()->images->get_image( $attachment_id, $size, $atts, $placeholder_atts );
}

/**
 * Get attachment image with loading="eager" attribute.
 *
 * @param int               $attachment_id
 * @param string            $size
 * @param array|string|null $atts
 * @param array|string|null $placeholder_atts
 *
 * @return string
 */
function kalium_get_attachment_image_eager( $attachment_id, $size = 'thumbnail', $atts = null, $placeholder_atts = null ) {
	return kalium_get_attachment_image( $attachment_id, $size, wp_parse_args( $atts, [ 'loading' => 'eager' ] ), $placeholder_atts );
}

/**
 * Validate boolean value for given var.
 * Returns true for values 'y' or 'yes'
 * Returns for string value 'false'
 *
 * @param mixed $var
 *
 * @return bool
 * @uses wp_validate_boolean()
 */
function kalium_validate_boolean( $var ) {
	return kalium()->helpers->validate_boolean( $var );
}

/**
 * Space-separated list of sanitized tokens.
 *
 * @param string|array $tokens
 * @param bool         $echo
 *
 * @return string|void
 * @since 4.1.2
 */
function kalium_tokenize_list( $tokens, $echo = false ) {
	return kalium()->helpers->tokenize_list( $tokens, $echo );
}

/**
 * Print classes attribute.
 *
 * @param array $classes
 * @param bool  $echo
 *
 * @return string
 */
function kalium_class_attr( $classes, $echo = true ) {
	$class = sprintf( 'class="%s"', kalium_tokenize_list( $classes ) );

	if ( $echo ) {
		echo $class;

		return '';
	}

	return $class;
}

/**
 * Print element attributes.
 *
 * @param array $attributes
 * @param bool  $echo
 *
 * @return string|void
 * @since 4.0
 */
function kalium_attributes() {
	return call_user_func_array( [ kalium()->helpers, 'list_attributes' ], func_get_args() );
}

/**
 * Print HTML element.
 *
 * @param string|array $tag_name
 * @param array        $attributes
 * @param string       $content
 * @param bool         $close
 *
 * @return string
 * @since 4.0
 */
function kalium_element( $tag_name, $attributes = [], $content = null, $close = true ) {
	return call_user_func_array( [ kalium()->helpers, 'build_dom_element' ], func_get_args() );
}

/**
 * Get nav menu.
 *
 * @param array $args
 *
 * @return string|false
 */
function kalium_nav_menu( $args = [] ) {

	// Load menu by id
	if ( is_numeric( $args ) ) {
		$args = [
			'menu' => $args,
		];
	} // Load menu from theme location
	elseif ( is_string( $args ) ) {
		$args = [
			'theme_location' => $args,
		];
	}

	// Use main menu by default
	if ( empty( $args['menu'] ) && empty( $args['theme_location'] ) ) {
		$args['theme_location'] = 'main-menu';
	}

	// Do not show any menu
	if ( '-' === kalium_get_array_key( $args, 'theme_location' ) ) {
		return false;
	}

	$container_class = kalium_get_array_key( $args, kalium_conditional( isset( $args['theme_location'] ), 'theme_location', 'menu' ), 'none' );

	// Menu args
	$args = wp_parse_args(
		$args,
		[
			'container'       => 'nav',
			'container_class' => kalium_tokenize_list(
				[
					'nav-container-' . $container_class,
					kalium_get_link_plain_class(),
				]
			),
			'echo'            => false,
			'link_before'     => '<span class="link">',
			'link_after'      => '</span>',
			'dropdown_toggle' => ! empty( $args['dropdown_toggle'] ),
			'fallback_cb'     => 'kalium_page_menu',
		]
	);

	// Menu
	$menu = wp_nav_menu( $args );

	return apply_filters( 'kalium_nav_menu', $menu, $args );
}

/**
 * Page menu items fallback function used in kalium_nav_menu.
 *
 * @param array $args
 *
 * @return string|false
 */
function kalium_page_menu( $args = [] ) {
	$page_menu_args = [
		'container'  => kalium_get_array_key( $args, 'container', 'nav' ),
		'menu_class' => kalium_get_array_key( $args, 'container_class', 'none' ),
		'before'     => '<ul class="menu">',
		'echo'       => false,
	];

	// Custom page elements container classes
	$page_css_class_function = function ( $classes ) {
		$classes[] = 'menu-item';

		return $classes;
	};

	// Add filter to "page_css_class" hook
	add_filter( 'page_css_class', $page_css_class_function );

	$menu = wp_page_menu( $page_menu_args );

	// Remove filter from "page_css_class" hook
	remove_filter( 'page_css_class', $page_css_class_function );

	return $menu;
}

/**
 * Theme Data Layer for commands.
 *
 * @return string
 * @since 4.0
 */
function kalium_js_data_layer() {
	$args = [];

	foreach ( func_get_args() as $arg ) {
		if ( is_string( $arg ) ) {
			$arg = "'" . esc_js( $arg ) . "'";
		} elseif ( is_bool( $arg ) ) {
			$arg = $arg ? 'true' : 'false';
		} elseif ( ! is_numeric( $arg ) ) {
			$arg = wp_json_encode( $arg, JSON_NUMERIC_CHECK );
		}

		$args[] = $arg;
	}

	return sprintf( 'kalium( %s );', implode( ', ', $args ) );
}

/**
 * Define theme variables in JS.
 *
 * @param array $vars
 *
 * @since 4.0
 */
function kalium_js_define_vars( $vars ) {
	$entry = $script_attributes = [];

	foreach ( $vars as $var_name => $var_value ) {
		$entry[] = kalium_js_data_layer( 'set', $var_name, $var_value );
	}

	// Compatibility with Complianz GDPR plugin
	if ( class_exists( 'COMPLIANZ' ) ) {
		$script_attributes['data-category'] = 'functional';
	}

	wp_print_inline_script_tag(
		implode(
			PHP_EOL,
			$entry
		),
		$script_attributes
	);
}

/**
 * Return single value in WP Hook.
 *
 * @param mixed $value
 *
 * @return callable():mixed
 */
function kalium_hook_return_value( $value ) {
	return static function () use ( $value ) {
		return $value;
	};
}

/**
 * Return single value in WP Hook.
 *
 * @param mixed $value
 *
 * @return callable
 */
function kalium_hook_echo_value( $value ) {
	return static function () use ( $value ) {
		echo $value;
	};
}

/**
 * Add array value in WP Hook.
 *
 * @param mixed  $value
 * @param string $key
 *
 * @return callable():array
 */
function kalium_hook_add_array_value( $value, $key = '' ) {
	return static function ( $array ) use ( $value, $key ) {
		if ( ! is_array( $array ) ) {
			$array = [];
		}

		if ( isset( $value ) ) {
			if ( $key ) {
				$array[ $key ] = $value;
			} else {
				$array[] = $value;
			}
		}

		return $array;
	};
}

/**
 * Merge arrays in WP Hook.
 *
 * @param array $value
 * @param bool  $prepend
 *
 * @return callable():array
 */
function kalium_hook_array_merge( $value, $prepend = false ) {
	return static function ( $array ) use ( $value, $prepend ) {
		if ( ! is_array( $array ) ) {
			$array = [];
		}

		if ( ! empty( $value ) ) {
			$array = $prepend ? array_merge( $value, $array ) : array_merge( $array, $value );
		}

		return $array;
	};
}

/**
 * Custom style generator.
 *
 * @param string       $selector
 * @param string|array $props
 * @param string       $media
 * @param bool         $footer
 *
 * @return void
 */
function kalium_append_custom_css( $selector, $props = '', $media = '', $footer = false ) {
	global $kalium_append_custom_css;

	if ( ! isset( $kalium_append_custom_css ) ) {
		$kalium_append_custom_css = [];
	}

	$css = '';

	// Selector Start
	$css .= $selector . ' {';

	// Selector Properties
	if ( is_array( $props ) ) {
		$css .= kalium()->helpers->build_css_props( $props );
	} else {
		$css .= $props;
	}

	$css .= '}';
	// Selector End

	// Media Wrap
	if ( trim( $media ) ) {
		if ( strpos( $media, '@' ) == false ) {
			$css = "@media {$media} { {$css} }";
		} else {
			$css = "{$media} { {$css} }";
		}
	}

	if ( ! $footer || defined( 'DOING_AJAX' ) ) {
		printf( '<style data-appended-custom-css="true">%s</style>', $css );

		return;
	}

	$kalium_append_custom_css[] = $css;
}

/**
 * Mobile menu breakpoint.
 *
 * @return int
 */
function kalium_get_mobile_menu_breakpoint() {
	$breakpoint = kalium_get_theme_option( 'menu_mobile_breakpoint' );

	if ( ! $breakpoint || ! is_numeric( $breakpoint ) ) {
		$breakpoint = 768;
	}

	return $breakpoint;
}

/**
 * Get Google API Key.
 *
 * @return string
 */
function kalium_get_google_api_key() {
	return apply_filters( 'kalium_google_api_key', kalium_get_theme_option( 'google_maps_api' ) );
}

/**
 * Format content just like default the_content().
 *
 * @param string $str
 *
 * @return string
 */
function kalium_format_content( $str ) {
	$format_functions = [
		'do_blocks',
		'wptexturize',
		'wpautop',
		'prepend_attachment',
		'convert_smilies',
		'do_shortcode',
	];

	foreach ( $format_functions as $function ) {
		$str = $function( $str );
	}

	return $str;
}

/**
 * Get single prop from multi numeric data.
 *
 * @param array  $value
 * @param string $prop
 * @param mixed  $default
 *
 * @return string|mixed
 * @since 4.0
 */
function kalium_get_multi_numeric_prop( $value, $prop, $default = null ) {
	$unit       = $value['unit'] ?? null;
	$prop_value = $value[ $prop ] ?? null;

	if ( isset( $prop_value ) ) {
		return $prop_value . $unit;
	}

	return $default;
}

/**
 * From unit value.
 *
 * @param array  $value
 * @param string $default
 *
 * @return string
 * @since 4.0
 */
function kalium_to_unit_value( $value, $default = null ) {
	if ( is_string( $value ) ) {
		return $value;
	}

	if ( isset( $value['size'] ) && is_numeric( $value['size'] ) ) {
		return kalium_get_multi_numeric_prop( $value, 'size', $default );
	}

	return $default;
}

/**
 * Convert to CSS length value.
 *
 * @param array        $value
 * @param string|array $default
 *
 * @return string
 * @since 4.0
 */
function kalium_to_length( $value, $default = null ) {
	if ( ! is_array( $value ) ) {
		return null;
	}

	$value = wp_parse_args(
		$value,
		[
			'top'    => null,
			'right'  => null,
			'bottom' => null,
			'left'   => null,
			'unit'   => null,
		]
	);

	// Default value
	if ( is_scalar( $default ) ) {
		$default = explode( ' ', trim( $default ) );

		switch ( count( $default ) ) {
			case 4:
				$default = [
					'top'    => $default[0],
					'right'  => $default[1],
					'bottom' => $default[2],
					'left'   => $default[3],
				];
				break;

			case 3:
				$default = [
					'top'    => $default[0],
					'right'  => $default[1],
					'bottom' => $default[2],
					'left'   => $default[1],
				];
				break;

			case 2:
				$default = [
					'top'    => $default[0],
					'right'  => $default[1],
					'bottom' => $default[0],
					'left'   => $default[1],
				];
				break;

			default:
				$default = [
					'top'    => $default[0],
					'right'  => $default[0],
					'bottom' => $default[0],
					'left'   => $default[0],
				];
		}
	}

	// Vars
	$length    = [];
	$has_value = false;

	foreach ( [ 'top', 'right', 'bottom', 'left' ] as $prop ) {
		$is_empty = empty( $value[ $prop ] ) && ! is_numeric( $value[ $prop ] );

		if ( ! $is_empty ) {
			$has_value = true;
		}

		// Set default value
		if ( $is_empty ) {
			$value[ $prop ] = $default[ $prop ] ?? 0;
		}

		// Add unit
		if ( is_numeric( $value[ $prop ] ) ) {
			$value[ $prop ] .= $value['unit'];
		}
	}

	// No value
	if ( ! $has_value ) {
		return null;
	}

	// Extract props
	extract( $value );

	if ( $top || $right || $bottom || $left ) {
		// All sides are the same
		if ( 1 === count( array_unique( [ $top, $right, $bottom, $left ] ) ) ) {
			$length[] = $top;
		}

		// Vertical and horizontal
		elseif ( $top === $bottom && $left === $right ) {
			$length[] = $top;
			$length[] = $left;
		}

		// All values
		else {
			$length[] = $top;
			$length[] = $right;
			$length[] = $bottom;
			$length[] = $left;
		}
	}

	if ( empty( $length ) ) {
		return null;
	}

	return implode( ' ', $length );
}

/**
 * Convert to border box value.
 *
 * @param array $value
 *
 * @return array
 * @since 4.0
 */
function kalium_to_border_box( $value ) {
	$border_entries    = [];
	$to_border_entry   = function ( $obj ) {
		$width = $obj['width'] ?? null;
		$style = $obj['style'] ?? null;
		$color = $obj['color'] ?? null;

		// Parse color
		if ( $color ) {
			$color = kalium_replace_color_references( $color );
		}

		return [
			'width' => $width,
			'style' => $style,
			'color' => $color,
		];
	};
	$has_border_value  = function ( $border_entry ) {
		return $border_entry['width'] || $border_entry['style'] || $border_entry['color'];
	};
	$list_border_props = function ( $prop_name ) use ( &$border_entries, &$has_border_value ) {
		$list                 = [];
		$value_defaults       = [
			'width' => '.1rem',
			'style' => 'solid',
			'color' => 'currentColor',
		];
		$empty_value_defaults = [
			'width' => '0px',
			'style' => 'none',
			'color' => 'transparent',
		];

		foreach ( $border_entries as $border_entry ) {
			$list[] = $border_entry[ $prop_name ] ?? ( $has_border_value( $border_entry ) ? $value_defaults[ $prop_name ] : $empty_value_defaults[ $prop_name ] );
		}

		if ( count( $list ) ) {
			return implode( ' ', $list );
		}

		return null;
	};

	if ( is_array( $value ) ) {
		// Single entry
		if ( array_key_exists( 'style', $value ) || array_key_exists( 'width', $value ) || array_key_exists( 'color', $value ) ) {
			$border_entries[] = $to_border_entry( $value );
		} else {
			foreach ( $value as $value_entry ) {
				$border_entries[] = $to_border_entry( $value_entry );
			}
		}
	}

	$border = [
		'width' => $list_border_props( 'width' ),
		'style' => $list_border_props( 'style' ),
		'color' => $list_border_props( 'color' ),
	];

	// No border
	if ( '0px' === $border['width'] || 'none' === $border['style'] ) {
		$border['width'] = $border['style'] = $border['color'] = null;
	}

	return $border;
}

/**
 * To pixel unit.
 *
 * @param mixed $value
 *
 * @return string
 * @since 4.1.2
 */
function kalium_to_pixel_unit( $value ) {
	return is_numeric( $value ) ? $value . 'px' : null;
}

/**
 * Has border.
 *
 * @param array $value
 *
 * @return bool
 * @since 4.0
 */
function kalium_has_border( $value ) {
	if ( is_array( $value ) ) {
		foreach ( [ 'width', 'style', 'color' ] as $prop ) {
			if ( ! empty( $value[ $prop ] ) ) {
				return true;
			}
		}
	}

	return false;
}

/**
 * Has multi numeric value.
 *
 * @param array $value
 *
 * @return bool
 * @since 4.0
 */
function kalium_has_multi_numeric_value( $value ) {
	if ( is_array( $value ) ) {
		foreach ( $value as $prop => $prop_value ) {
			if ( ! in_array( $prop, [ 'link', 'unit' ] ) && is_numeric( $prop_value ) ) {
				return true;
			}
		}
	}

	return false;
}

/**
 * Check if box shadow size is provided.
 *
 * @param array $shadow
 *
 * @return bool
 * @return 4.0
 */
function kalium_has_box_shadow_size( $shadow ) {
	if ( is_array( $shadow ) ) {
		$size_props = [ 'offset_x', 'offset_y', 'blur_radius', 'spread_radius' ];

		foreach ( $size_props as $prop ) {
			if ( isset( $shadow[ $prop ] ) && ( ! empty( $shadow[ $prop ] ) || is_numeric( $shadow[ $prop ] ) ) ) {
				return true;
			}
		}
	}
	return false;
}

/**
 * To box shadow color.
 *
 * @param array  $shadow
 * @param string $default
 *
 * @return string
 * @since 4.0
 */
function kalium_to_box_shadow_color( $shadow, $default = 'transparent' ) {
	if ( ! empty( $shadow['color'] ) ) {
		return kalium_replace_color_references( $shadow['color'] );
	}

	return $default;
}

/**
 * To box shadow size.
 *
 * @param array        $size
 * @param string|array $default
 *
 * @return string
 * @since 4.0
 */
function kalium_to_box_shadow_size( $size, $default = null ) {
	if ( ! is_array( $size ) ) {
		return null;
	}

	// Shadow props order
	$props_order = [ 'offset_x', 'offset_y', 'blur_radius', 'spread_radius' ];

	// Default size
	if ( is_array( $default ) ) {
		$size = wp_parse_args( $size, $default );
	} elseif ( is_scalar( $default ) ) {
		$default = explode( ' ', trim( $default ) );

		if ( count( $default ) <= 4 ) {
			foreach ( $default as $i => $value ) {
				$prop = $props_order[ $i ];

				if ( ! isset( $size[ $prop ] ) ) {
					$size[ $prop ] = $value;
				}
			}
		}
	}

	// Size entry
	$size = wp_parse_args(
		$size,
		[
			'offset_x'      => null,
			'offset_y'      => null,
			'blur_radius'   => null,
			'spread_radius' => null,
		]
	);

	// Add units
	if ( ! empty( $size['unit'] ) ) {
		foreach ( $props_order as $prop ) {
			if ( $size[ $prop ] && is_numeric( $size[ $prop ] ) ) {
				$size[ $prop ] .= $size['unit'];
			}
		}
	}

	return trim(
		implode(
			' ',
			array_map(
				function ( $prop ) use ( $size ) {
					return $size[ $prop ] ?: '0';
				},
				$props_order
			)
		)
	);
}

/**
 * To box shadow full value.
 *
 * @param array $shadow
 * @param array $default_size
 *
 * @return string
 * @since 4.0
 */
function kalium_to_box_shadow( $shadow, $default_size = null ) {
	if ( ! is_array( $shadow ) ) {
		return null;
	}

	$shadow = kalium_split_box_shadow( $shadow, $default_size );

	if ( ! empty( $shadow['color'] ) ) {
		return $shadow['size'] . ' ' . $shadow['color'];
	}

	return null;
}

/**
 * Split shadow color and size.
 *
 * @param array  $shadow
 * @param string $default_size
 *
 * @return array
 * @since 4.0
 */
function kalium_split_box_shadow( $shadow, $default_size = null ) {
	$color = $size = null;

	if ( is_array( $shadow ) ) {
		$color = kalium_to_box_shadow_color( $shadow, null );
		$size  = kalium_to_box_shadow_size( $shadow, $default_size );
	}

	return [
		'color' => $color,
		'size'  => $size,
	];
}

/**
 * List dimensions into array.
 *
 * @param array $dimensions
 *
 * @return array<int, int>|null
 * @since 4.0
 */
function kalium_list_dimensions( $dimensions ) {
	if ( is_array( $dimensions ) ) {
		$width = $height = null;

		if ( array_key_exists( 'width', $dimensions ) ) {
			$width = $dimensions['width'];
		}

		if ( array_key_exists( 'height', $dimensions ) ) {
			$height = $dimensions['height'];
		}

		if ( is_numeric( $width ) || is_numeric( $height ) ) {
			return [
				absint( $width ),
				absint( $height ),
			];
		}
	}

	return null;
}

/**
 * Print inline CSS. Replacement for kalium_append_custom_css.
 *
 * @param array $args
 * @param bool  $echo
 *
 * @return void|string
 * @since 4.0
 */
function kalium_print_inline_style( $args = [], $echo = true ) {
	$styles = [];
	$css    = '';

	// Plain CSS
	if ( is_string( $args ) ) {
		$args = [
			'css' => $args,
		];
	}

	// Parse args
	$args = wp_parse_args(
		$args,
		[
			'id'       => null,
			'css'      => '',
			'styles'   => [],

			// Single selector
			'media'    => null,
			'selector' => null,
			'props'    => [],
			'vars'     => null,
		]
	);

	// Plain CSS
	if ( ! empty( $args['css'] ) ) {
		$css .= "\n" . $args['css'];
	}

	// Default selector
	$default_selector = $args['selector'];

	// Single style
	if ( ! empty( $args['selector'] ) ) {
		$args['styles'][] = [
			'media'    => $args['media'],
			'selector' => $args['selector'],
			'props'    => $args['props'],
			'vars'     => $args['vars'],
		];
	}

	// Build styles
	foreach ( $args['styles'] as $style ) {
		$style = wp_parse_args(
			$style,
			[
				'media'    => null,
				'selector' => $default_selector,
				'props'    => null,
				'vars'     => null,
			]
		);

		// Selector
		$selector = is_array( $style['selector'] ) ? implode( ',', array_filter( $style['selector'] ) ) : $style['selector'];

		// No selector
		if ( ! $selector ) {
			continue;
		}

		// Vars
		if ( is_array( $style['vars'] ) ) {
			foreach ( $style['vars'] as $var_name => $var_value ) {
				$var_name = kalium_css_var_name( $var_name );

				if ( is_scalar( $var_value ) ) {
					$styles[ $style['media'] ][ $selector ][ $var_name ] = $var_value;

				} elseif ( is_array( $var_value ) ) {
					foreach ( $var_value as $viewport => $viewport_value ) {
						$styles[ kalium_get_viewport_media( $viewport ) ][ $selector ][ $var_name ] = $viewport_value;
					}
				}
			}
		}

		// Props
		if ( is_array( $style['props'] ) ) {
			foreach ( $style['props'] as $prop_name => $prop_value ) {

				if ( is_scalar( $prop_value ) ) {
					$styles[ $style['media'] ][ $selector ][ $prop_name ] = $prop_value;

				} elseif ( is_array( $prop_value ) ) {
					foreach ( $prop_value as $viewport => $viewport_value ) {
						$styles[ kalium_get_viewport_media( $viewport ) ][ $selector ][ $prop_name ] = $viewport_value;
					}
				}
			}
		}
	}

	// Treat color references and raw colors
	foreach ( $styles as & $selectors ) {
		foreach ( $selectors as & $props ) {
			foreach ( $props as $prop_name => $prop_value ) {
				$is_color = false;

				if ( empty( $prop_value ) && ! is_numeric( $prop_value ) ) {
					continue;
				}

				if ( false !== strpos( $prop_value, 'kalium-color(' ) || false !== strpos( $prop_value, 'kalium-color-variant(' ) ) {
					$is_color = true;
				} elseif ( kalium()->style->color( $prop_value ) ) {
					$is_color = true;
				}

				if ( $is_color ) {
					$props[ $prop_name ] = kalium_replace_color_references( $prop_value );

					if ( ! empty( $props[ $prop_name ] ) && false !== strpos( $prop_name, '--' ) ) {
						$props[ $prop_name . '-rgb' ] = kalium_parse_color_rgb( $prop_value );
					}
				}
			}
		}
	}

	// Clear references
	unset( $selectors, $props );

	// Parse styles
	if ( ! empty( $styles ) ) {
		$new_line = kalium()->is->debugging() ? PHP_EOL : '';

		foreach ( $styles as $media => $selectors ) {
			$has_media = ! empty( $media );

			// Media start
			if ( $has_media ) {
				$css .= sprintf( '@media %s {', $media ) . $new_line;
			}

			foreach ( $selectors as $selector => $props ) {

				// Selector start
				$css .= sprintf( '%s {', $selector ) . $new_line;

				foreach ( $props as $prop_name => $prop_value ) {
					if ( ( empty( $prop_value ) && ! is_numeric( $prop_value ) ) || kalium_is_self_referencing_css_var( $prop_name, $prop_value ) ) {
						continue;
					}

					$css .= sprintf( '%s: %s;', $prop_name, $prop_value ) . $new_line;
				}

				// Selector end
				$css .= '}' . $new_line;
			}

			// Media end
			if ( $has_media ) {
				$css .= '}' . $new_line;
			}
		}
	}

	// No styles to print
	if ( empty( $css ) ) {
		return null;
	}

	// ID attribute
	$id_attr = 'data-inline-style';

	if ( ! empty( $args['id'] ) ) {
		$id_attr .= '="' . esc_attr( $args['id'] ) . '"';
	}

	// Style
	$style_wrapped = sprintf( '<style %s>%s</style>', $id_attr, $css );

	// Echo
	if ( $echo ) {
		echo $style_wrapped . "\n";
		return;
	}

	return $style_wrapped;
}

/**
 * Get predefined container width.
 *
 * @param string $size
 *
 * @return string
 * @since 4.0
 */
function kalium_get_predefined_container_max_width( $size ) {
	$predefined_widths = [
		'small'  => '936px',
		'medium' => '1116px',
	];

	return $predefined_widths[ $size ] ?? '1296px';
}

/**
 * Get container max width.
 *
 * @param bool $include_default
 *
 * @return string|null
 * @since 4.0
 */
function kalium_get_container_max_width( $include_default = false ) {
	$is_fullwidth    = kalium_get_theme_option( 'grid_container_fullwidth' );
	$container_width = kalium_get_theme_option( 'grid_container_width' );
	$custom_width    = kalium_get_theme_option( 'grid_container_width_custom' );
	$max_width       = $include_default ? '1296px' : null;

	// Full width
	if ( $is_fullwidth ) {
		$max_width = '100%';
	} elseif ( 'custom' === $container_width && is_numeric( $custom_width ) ) {
		$max_width = $custom_width . 'px';
	} elseif ( $predefined_width = kalium_get_predefined_container_max_width( $container_width ) ) {
		$max_width = $predefined_width;
	}

	return apply_filters( 'kalium_container_max_width', $max_width );
}

/**
 * Get sidebar width.
 *
 * @param bool $include_default
 *
 * @return string|int
 */
function kalium_get_sidebar_width( $include_default = false ) {
	$sidebar_width = $include_default ? 25 : null;

	if ( $custom_sidebar_width = kalium_get_theme_option( 'sidebar_width' ) ) {
		$sidebar_width = $custom_sidebar_width;
	}

	return $sidebar_width;
}

/**
 * Get responsive viewports.
 *
 * @return array
 * @since 4.0
 */
function kalium_get_responsive_viewports() {
	return [
		'desktop' => [
			'title'   => 'Desktop',
			'icon'    => 'desktop',
			'default' => true,
		],
		'tablet'  => [
			'title'     => 'Tablet',
			'icon'      => 'tablet',
			'inherit'   => 'desktop',
			'min_width' => 768,
			'max_width' => 992,
		],
		'mobile'  => [
			'title'     => 'Mobile',
			'icon'      => 'mobile',
			'inherit'   => 'tablet',
			'min_width' => 576,
			'max_width' => 768,
		],
	];
}

/**
 * Get default viewport.
 *
 * @return string
 * @since 4.0
 */
function kalium_get_default_viewport() {
	static $default_viewport;

	if ( ! is_null( $default_viewport ) ) {
		return $default_viewport;
	}

	foreach ( kalium_get_responsive_viewports() as $viewport => $viewport_options ) {
		if ( ! empty( $viewport_options['default'] ) ) {
			$default_viewport = $viewport;
			return $viewport;
		}
	}

	return null;
}

/**
 * Get responsive viewport.
 *
 * @param string $viewport_id
 *
 * @return array|false
 * @since 4.0
 */
function kalium_get_responsive_viewport( $viewport_id ) {
	$viewports = kalium_get_responsive_viewports();

	return $viewports[ $viewport_id ] ?? false;
}

/**
 * Get viewport min width.
 *
 * @param string $viewport
 *
 * @return int|null
 * @since 4.0
 */
function kalium_get_viewport_min_width( $viewport ) {
	$viewport = kalium_get_responsive_viewport( $viewport );

	if ( $viewport && isset( $viewport['min_width'] ) ) {
		return $viewport['min_width'];
	}

	return null;
}

/**
 * Get viewport max width.
 *
 * @param string $viewport
 *
 * @return int|null
 * @since 4.0
 */
function kalium_get_viewport_max_width( $viewport ) {
	$viewport = kalium_get_responsive_viewport( $viewport );

	if ( $viewport && isset( $viewport['max_width'] ) ) {
		return $viewport['max_width'];
	}

	return null;
}

/**
 * Get largest viewport (excludes default viewport).
 *
 * @return int
 * @since 4.0
 */
function kalium_get_largest_viewport_max_width() {
	$max_width = 0;

	foreach ( kalium_get_responsive_viewports() as $viewport => $viewport_options ) {
		if ( isset( $viewport_options['max_width'] ) ) {
			$max_width = max( $max_width, $viewport_options['max_width'] );
		}
	}

	return $max_width;
}

/**
 * Get viewport media.
 *
 * @param string $viewport
 *
 * @return string
 * @since 4.0
 */
function kalium_get_viewport_media( $viewport ) {
	$max_width = kalium_get_viewport_max_width( $viewport );

	if ( is_numeric( $max_width ) ) {
		return '(max-width: ' . $max_width . 'px)';
	}

	return null;
}

/**
 * Get responsive value.
 *
 * @param array  $value
 * @param string $viewport
 * @param mixed  $default
 *
 * @return mixed
 * @since 4.0
 */
function kalium_get_responsive_value( $value, $viewport = null, $default = null ) {
	if ( ! $viewport ) {
		$viewport = kalium_get_default_viewport();
	}

	// Responsive value is available
	if ( isset( $value[ $viewport ] ) ) {
		return $value[ $viewport ];
	}

	// Find inherited value
	$viewports = kalium_get_responsive_viewports();

	if ( ! empty( $viewports[ $viewport ]['inherit'] ) ) {
		return kalium_get_responsive_value( $value, $viewports[ $viewport ]['inherit'], $default );
	}

	return $default;
}

/**
 * Lowest viewport.
 *
 * @param array $value
 *
 * @return string
 * @since 4.0
 */
function kalium_get_lowest_viewport_id( $value = null ) {
	$viewports_keys = is_array( $value ) ? array_keys( $value ) : array_keys( kalium_get_responsive_viewports() );

	return end( $viewports_keys );
}

/**
 * Highest viewport.
 *
 * @param array $value
 *
 * @return string
 * @since 4.0
 */
function kalium_get_highest_viewport_id( $value = null ) {
	$viewports_keys = is_array( $value ) ? array_keys( $value ) : array_keys( kalium_get_responsive_viewports() );

	return reset( $viewports_keys );
}

/**
 * Get lowest viewport value from responsive value.
 *
 * @param mixed $value
 * @param mixed $default
 *
 * @return mixed
 * @since 4.0
 */
function kalium_get_lowest_viewport_value( $value, $default = null ) {
	$viewports_keys = array_keys( kalium_get_responsive_viewports() );

	return kalium_get_responsive_value( $value, end( $viewports_keys ), $default );
}

/**
 * Get highest viewport value from responsive value.
 *
 * @param mixed $value
 * @param mixed $default
 *
 * @return mixed
 * @since 4.0
 */
function kalium_get_highest_viewport_value( $value, $default = null ) {
	$viewports_keys = array_keys( kalium_get_responsive_viewports() );

	return kalium_get_responsive_value( $value, reset( $viewports_keys ), $default );
}

/**
 * Check if given value is responsive value.
 *
 * @param mixed $value
 *
 * @return bool
 * @since 4.0
 */
function kalium_is_responsive_value( $value ) {
	$viewports = array_keys( kalium_get_responsive_viewports() );

	if ( is_array( $value ) ) {
		$value_keys = array_keys( $value );

		foreach ( $value_keys as $value_key ) {
			if ( ! in_array( $value_key, $viewports ) || is_numeric( $value_key ) ) {
				return false;
			}
		}

		return true;
	}

	return false;
}

/**
 * Parse responsive value.
 *
 * @param mixed $value
 * @param mixed $default
 * @param bool  $unique_filter
 *
 * @return array
 * @since 4.0
 */
function kalium_parse_responsive_value( $value, $default = null, $unique_filter = true ) {
	$viewports = kalium_get_responsive_viewports();
	$parsed    = [];

	// Convert $value to responsive value
	if ( ! kalium_is_responsive_value( $value ) ) {
		$value = [
			kalium_get_default_viewport() => $value,
		];
	}

	// Convert $default to responsive value
	if ( ! kalium_is_responsive_value( $default ) ) {
		$default = [
			kalium_get_default_viewport() => $default,
		];
	}

	// Merge default value with the value
	$value = array_merge(
		$default,
		array_filter(
			$value,
			function ( $value ) {
				return ! is_null( $value );
			}
		)
	);

	// Track previous viewport and value
	$previous_viewport = $previous_value = null;

	foreach ( $viewports as $viewport => $viewport_options ) {
		$viewport_value = kalium_get_responsive_value( $value, $viewport );

		if ( ! $unique_filter || is_null( $previous_value ) || ( $viewport_value !== $previous_value && $viewport_value !== kalium_get_array_key( $parsed, $previous_viewport ) ) ) {
			$parsed[ $viewport ] = $viewport_value;
		}

		$previous_viewport = $viewport;
		$previous_value    = $viewport_value;
	}

	return array_filter(
		$parsed,
		function ( $value ) {
			return ! is_null( $value );
		}
	);
}

/**
 * Parse elementor responsive value.
 *
 * @param string $key
 * @param array  $settings
 * @param mixed  $default
 *
 * @return array
 * @since 4.0
 */
function kalium_parse_elementor_responsive_value( $key, $settings, $default = null ) {
	$value = [];

	foreach ( kalium_get_responsive_viewports() as $viewport => $viewport_options ) {
		if ( kalium_get_viewport_media( $viewport ) ) {
			if ( isset( $settings[ $key . '_' . $viewport ] ) ) {
				$value[ $viewport ] = $settings[ $key . '_' . $viewport ];
			}
		} elseif ( isset( $settings[ $key ] ) ) {
			$value[ $viewport ] = $settings[ $key ];
		}
	}

	return kalium_parse_responsive_value( $value, $default );
}

/**
 * Check if responsive value is empty.
 *
 * @param mixed $value
 *
 * @return bool
 * @since 4.0
 */
function kalium_is_empty_responsive( $value ) {
	$value = kalium_parse_responsive_value( $value );

	foreach ( $value as $viewport_value ) {
		if ( ! is_null( $viewport_value ) && '' !== $viewport_value && false !== $viewport_value ) {
			return false;
		}
	}

	return true;
}

/**
 * Map responsive value.
 *
 * @param array    $responsive_value
 * @param callable $mapper
 * @param bool     $unique_filter
 *
 * @return array
 * @since 4.0
 */
function kalium_map_responsive_value( $responsive_value, $mapper, $unique_filter = true ) {
	$responsive_value = kalium_parse_responsive_value( $responsive_value, null, $unique_filter );

	foreach ( $responsive_value as $viewport => $viewport_value ) {
		if ( is_callable( $mapper ) ) {
			$responsive_value[ $viewport ] = call_user_func( $mapper, $viewport_value, $viewport );
		}
	}

	return $responsive_value;
}

/**
 * Print responsive style.
 *
 * @param array $args
 *
 * @return string|void
 * @since 4.0
 */
function kalium_print_responsive_style( $args, $echo = true ) {
	$args = wp_parse_args(
		$args,
		[
			'id'       => null,
			'value'    => null,
			'selector' => null,
			'mapper'   => null,
		]
	);

	$styles = [];

	// Add styles
	if ( is_callable( $args['mapper'] ) ) {
		$responsive_value = kalium_parse_responsive_value( $args['value'] );

		foreach ( $responsive_value as $viewport => $viewport_value ) {
			$props = call_user_func( $args['mapper'], $viewport_value, $viewport );
			$vars  = [];

			if ( is_array( $props ) ) {
				if ( ! empty( $props['vars'] ) ) {
					$vars = $props['vars'];
					unset( $props['vars'] );
				}

				$styles[] = [
					'selector' => $args['selector'],
					'media'    => kalium_get_viewport_media( $viewport ),
					'props'    => $props,
					'vars'     => $vars,
				];
			}
		}
	}

	return kalium_print_inline_style(
		[
			'id'     => $args['id'],
			'styles' => $styles,
		],
		$echo
	);
}

/**
 * Get cached var.
 *
 * @param string   $key
 * @param callable $value_fn
 * @param int      $expire
 *
 * @return mixed
 * @since 4.0
 */
function kalium_get_cached_var( $key, $value_fn, $expire = 0 ) {
	if ( $value = wp_cache_get( $key, 'kalium' ) ) {
		return $value;
	}

	$value = call_user_func( $value_fn );

	// Add to cache
	wp_cache_add( $key, $value, 'kalium', $expire );

	return $value;
}

/**
 * Create masonry container.
 *
 * @param array $args
 *
 * @since 4.0
 */
function kalium_create_masonry_container( $args = [] ) {
	$args = wp_parse_args(
		$args,
		[
			'show_loading'    => false,
			'loading_text'    => esc_html__( 'Loading...', 'kalium' ),
			'enqueue_masonry' => true,
			'options'         => [],
		]
	);

	// Infinite Scroll Options
	$args['options'] = wp_parse_args(
		$args['options'],
		[
			'container'      => null,
			'item'           => '.post',
			'layout_mode'    => 'packery', // [packery,fitRows]
			'init_animation' => false,
			'stagger'        => null,
			'hidden_style'   => [
				'opacity'   => 0,
				'transform' => 'translate(0px,50px)',
			],
			'visible_style'  => [
				'opacity'   => 1,
				'transform' => 'translate(0px,0px)',
			],
			'loading_hide'   => 0.2,
		]
	);

	// Enqueue library
	if ( $args['enqueue_masonry'] ) {
		kalium_enqueue_isotope_and_packery_library();
	}

	?>
	<div class="masonry-container-loader" data-options="<?php echo esc_attr( wp_json_encode( $args['options'] ) ); ?>">
		<?php
		if ( $args['show_loading'] ) {
			printf( '<span class="masonry-container-loader__loading-text">%s</span>', $args['loading_text'] );
		}
		?>
	</div>
	<?php
}

/**
 * Generate unique string.
 *
 * @param int $length
 *
 * @return string
 * @since 4.0
 */
function kalium_generate_unique_string( $length = 5 ) {
	static $generated = [];

	$characters        = '0123456789abcdefghijklmnopqrstuvwxyz';
	$characters_length = strlen( $characters );
	$random_string     = '';

	for ( $i = 0; $i < $length; $i++ ) {
		$random_string .= $characters[ mt_rand( 0, $characters_length - 1 ) ];
	}

	if ( in_array( $random_string, $generated ) ) {
		return kalium_generate_unique_string( $length );
	}

	$generated[] = $random_string;

	return $random_string;
}

/**
 * Stretch container to the browser edge using JS.
 *
 * @param string $selector
 * @param string $side
 *
 * @since 4.0
 */
function kalium_js_stretch_container( $selector, $args = [] ) {
	$args = wp_parse_args(
		$args,
		[
			'side' => 'auto',
			'top'  => true,
		]
	);

	$side = in_array( $args['side'], [ 'left', 'right', 'auto' ] ) ? $args['side'] : 'auto';
	$top  = $args['top'];

	/*
		( function ( selector, align, stretchTop ) {
			var container = document.querySelector( selector ),
				wrapper = document.querySelector( '.wrapper' );

			if ( ! container ) {
				return;
			}

			function stretchContainer() {
				// Reset offset
				if ( stretchTop ) {
					container.style.marginTop = '';
				}

				container.style.marginLeft = container.style.marginRight = '';

				if ( window.innerWidth < 992 ) {
					return;
				}

				var rect = container.getBoundingClientRect(),
					wrapperOffsetTop = wrapper.offsetTop,
					scrollX = window.scrollX,
					scrollY = window.scrollY,
					viewportWidth = document.documentElement.clientWidth || window.innerWidth;

				if ( 'auto' === align ) {
					align = rect.left < viewportWidth - rect.right ? 'left' : 'right';
				}

				// Stretch to top
				if ( stretchTop ) {
					container.style.marginTop =
						document.documentElement.offsetTop + wrapperOffsetTop - ( rect.y + scrollY ) + 'px';
				}

				// Stick to horizontal edge
				if ( 'left' === align ) {
					container.style.marginLeft = -rect.x + 'px';
				} else {
					container.style.marginRight =
						-( viewportWidth - ( rect.x + scrollX ) - rect.width ) + 'px';
				}
			}

			if ( container ) {
				setTimeout( stretchContainer );

				window.addEventListener( 'resize', stretchContainer, {
					passive: true,
				} );

				// Set to visible
				if ( 'hidden' === getComputedStyle( container ).visibility ) {
					container.style.visibility = 'visible';
				}
			}
		} )( '<?php echo esc_attr( $selector ); ?>', '<?php echo esc_attr( $side ); ?>', <?php echo $top ? 'true' : 'false'; ?> );
	*/
	?>
	<script>
		!function(e,t,i){var n=document.querySelector(e),o=document.querySelector(".wrapper");function l(){if(i&&(n.style.marginTop=""),n.style.marginLeft=n.style.marginRight="",!(window.innerWidth<992)){var e=n.getBoundingClientRect(),l=o.offsetTop,r=window.scrollX,d=window.scrollY,s=document.documentElement.clientWidth||window.innerWidth;"auto"===t&&(t=e.left<s-e.right?"left":"right"),i&&(n.style.marginTop=document.documentElement.offsetTop+l-(e.y+d)+"px"),"left"===t?n.style.marginLeft=-e.x+"px":n.style.marginRight=-(s-(e.x+r)-e.width)+"px"}}n&&n&&(setTimeout(l),window.addEventListener("resize",l,{passive:!0}),"hidden"===getComputedStyle(n).visibility&&(n.style.visibility="visible"))}('<?php echo esc_attr( $selector ); ?>','<?php echo esc_attr( $side ); ?>',<?php echo $top ? 'true' : 'false'; ?>);
	</script>
	<?php
}

/**
 * Lightbox settings.
 *
 * @return array
 * @since 4.0
 */
function kalium_get_lightbox_settings() {
	$hide_controls = kalium_get_theme_option( 'lightbox_hide_controls' );
	$hide_controls = is_numeric( $hide_controls ) ? $hide_controls * 1000 : 3000;

	$autoplay_interval = kalium_get_theme_option( 'lightbox_autoplay_interval' );
	$autoplay_interval = is_numeric( $autoplay_interval ) ? $autoplay_interval * 1000 : 5000;

	return apply_filters(
		'kalium_lightbox_settings',
		[
			'colorScheme'        => kalium_get_theme_option( 'lightbox_color_scheme' ),
			'backdropImage'      => kalium_get_theme_option( 'lightbox_backdrop_image' ),
			'captions'           => kalium_get_theme_option( 'lightbox_captions' ),
			'fullscreen'         => kalium_get_theme_option( 'lightbox_fullscreen' ),
			'download'           => kalium_get_theme_option( 'lightbox_download' ),
			'counter'            => kalium_get_theme_option( 'lightbox_counter' ),
			'hideControls'       => $hide_controls,
			'thumbnails'         => kalium_get_theme_option( 'lightbox_thumbnails' ),
			'collapseThumbnails' => kalium_get_theme_option( 'lightbox_collapse_thumbnails' ),
			'autoplay'           => kalium_get_theme_option( 'lightbox_autoplay' ),
			'autoplayInterval'   => $autoplay_interval,
			'zoom'               => kalium_get_theme_option( 'lightbox_zoom' ),
			'zoomScale'          => kalium_get_theme_option( 'lightbox_zoom_scale' ),
			'videoAutoplay'      => kalium_get_theme_option( 'lightbox_autoplay_videos' ),
		]
	);
}

/**
 * Lightbox image sizes.
 *
 * @return array
 * @since 4.0
 */
function kalium_get_lightbox_image_sizes() {
	return [
		'main'      => kalium_get_theme_option( 'lightbox_image_size_main' ),
		'thumbnail' => kalium_get_theme_option( 'lightbox_image_size_thumbnail' ),
	];
}

/**
 * Video element.
 *
 * @param array $atts
 *
 * @return string
 * @since 4.0
 */
function kalium_video_element( $atts ) {
	return kalium()->media->video( $atts );
}

/**
 * Audio element.
 *
 * @param array $atts
 *
 * @return string
 * @since 4.0
 */
function kalium_audio_element( $atts ) {
	return kalium()->media->audio( $atts );
}

/**
 * Get sidebar options.
 *
 * @return array
 * @since 4.0
 */
function kalium_get_sidebar_options() {
	return [
		'type'                => kalium_get_theme_option( 'sidebar_type' ),
		'width'               => kalium_get_sidebar_width(),
		'gap'                 => kalium_get_theme_option( 'sidebar_gap' ),
		'padding'             => kalium_get_theme_option( 'sidebar_padding' ),
		'widgets_spacing'     => kalium_get_theme_option( 'sidebar_widgets_spacing' ),
		'widgets_separate'    => kalium_get_theme_option( 'sidebar_widgets_separate' ),
		'mobile_position'     => kalium_get_theme_option( 'sidebar_mobile_position' ),
		'sticky'              => kalium_get_theme_option( 'sidebar_sticky' ),
		'sticky_behavior'     => kalium_get_theme_option( 'sidebar_sticky_behavior' ),
		'sticky_last_widgets' => kalium_get_theme_option( 'sidebar_sticky_last_widgets' ),
		'sticky_offset'       => kalium_get_theme_option( 'sidebar_sticky_offset' ),
		'visibility'          => kalium_get_theme_option( 'sidebar_visibility' ),
		'style'               => [
			'text'       => kalium_get_theme_option( 'sidebar_style_text' ),
			'links'      => kalium_get_theme_option( 'sidebar_style_links' ),
			'background' => kalium_get_theme_option( 'sidebar_bg' ),
			'border'     => kalium_get_theme_option( 'sidebar_border' ),
			'shadow'     => kalium_get_theme_option( 'sidebar_shadow' ),
			'radius'     => kalium_get_theme_option( 'sidebar_radius' ),
		],
	];
}

/**
 * Get sidebar CSS vars.
 *
 * @return array
 * @since 4.0
 */
function kalium_get_sidebar_css_vars() {
	$sidebar_options = kalium_get_sidebar_options();
	$sidebar_vars    = [];

	// Padding
	if ( is_numeric( $sidebar_options['padding'] ) ) {
		$sidebar_vars['sidebar-padding'] = $sidebar_options['padding'] . 'px';
	}

	// Widgets spacing
	if ( is_numeric( $sidebar_options['widgets_spacing'] ) ) {
		$sidebar_vars['sb-widgets-spacing'] = $sidebar_options['widgets_spacing'] . 'px';
	}

	// Title
	if ( ! empty( $sidebar_options['style']['text']['title'] ) ) {
		$sidebar_vars['h2-color'] = $sidebar_options['style']['text']['title'];
	}

	// Text
	if ( ! empty( $sidebar_options['style']['text']['text'] ) ) {
		$sidebar_vars['body-color'] = $sidebar_options['style']['text']['text'];
	}

	// Link normal
	if ( ! empty( $sidebar_options['style']['links'] ) ) {
		foreach ( $sidebar_options['style']['links'] as $state => $color ) {
			if ( ! empty( $color ) ) {
				$link_prop                  = 'normal' === $state ? 'link-color' : "link-{$state}-color";
				$sidebar_vars[ $link_prop ] = $color;
			}
		}
	}

	// Background
	if ( false !== $sidebar_options['style']['background'] ) {
		$sidebar_vars['sidebar-bg'] = $sidebar_options['style']['background'];
	}

	// Border
	if ( ! empty( $sidebar_options['style']['border'] ) ) {
		$border = kalium_to_border_box( $sidebar_options['style']['border'] );

		$sidebar_vars['sidebar-border-width'] = $border['width'] ?? null;
		$sidebar_vars['sidebar-border-style'] = $border['style'] ?? null;
		$sidebar_vars['sidebar-border-color'] = $border['color'] ?? null;
	}

	// Shadow
	if ( ! empty( $sidebar_options['style']['shadow'] ) ) {
		$sidebar_vars['sidebar-shadow'] = kalium_to_box_shadow( $sidebar_options['style']['shadow'], '0 0 5px' );
	}

	// Radius
	if ( ! empty( $sidebar_options['style']['radius'] ) ) {
		$radius = $sidebar_options['style']['radius'];

		if ( isset( $radius['top'], $radius['bottom'], $radius['left'], $radius['right'] ) ) {
			$sidebar_vars['sidebar-radius'] = implode(
				' ',
				[
					kalium_get_multi_numeric_prop( $radius, 'top', 0 ),
					kalium_get_multi_numeric_prop( $radius, 'right', 0 ),
					kalium_get_multi_numeric_prop( $radius, 'bottom', 0 ),
					kalium_get_multi_numeric_prop( $radius, 'left', 0 ),
				]
			);
		}
	}

	return $sidebar_vars;
}

/**
 * Get sidebar classes.
 *
 * @param string|string[] $extra_classes
 *
 * @return array
 * @since 4.0
 */
function kalium_get_sidebar_classes( $extra_classes = [] ) {
	$classes = [
		'sidebar',
	];

	$sidebar_options = kalium_get_sidebar_options();

	// Visibility classes
	$classes = array_merge( $classes, kalium_get_device_visibility_classes( $sidebar_options['visibility'] ) );

	// Extra classes
	if ( ! empty( $extra_classes ) ) {
		if ( ! is_array( $extra_classes ) ) {
			$extra_classes = preg_split( '#\s+#', $extra_classes );
		}

		$classes = array_merge( $classes, $extra_classes );
	}

	return apply_filters( 'kalium_sidebar_class', $classes, $sidebar_options );
}

/**
 * Decode JSON.
 *
 * @return array|string
 * @since 4.0
 */
function kalium_decode_json() {
	return call_user_func_array( [ kalium()->helpers, 'json_decode' ], func_get_args() );
}

/**
 * Avatar size on blog comments and author image.
 *
 * @param mixed $id_or_email
 * @param array $args
 *
 * @return string
 * @since 4.0
 */
function kalium_get_avatar( $id_or_email, $args = [] ) {
	$args = wp_parse_args(
		$args,
		[
			'size'             => 60,
			'default'          => '',
			'alt'              => '',
			'loading'          => 'lazy',
			'placeholder_wrap' => true,
		]
	);

	$avatar = get_avatar(
		$id_or_email,
		$args['size'],
		$args['default'],
		$args['alt'],
		[
			'loading' => $args['loading'],
		]
	);

	if ( $args['placeholder_wrap'] ) {
		$avatar = kalium_image_placeholder_wrap_element( $avatar );
	}

	return apply_filters( 'kalium_avatar', $avatar, $id_or_email, $args );
}

/**
 * Get spacing sizes from theme.json
 *
 * @return array
 * @since 4.0
 */
function kalium_get_spacing_sizes() {
	static $sizes;

	if ( is_null( $sizes ) ) {
		$sizes = kalium_get_array_key( WP_Theme_JSON_Resolver::get_theme_data()->get_settings(), 'spacing/spacingSizes/theme' );
	}

	return $sizes;
}

/**
 * Get related posts.
 *
 * @param array       $args
 * @param WP_Post|int $post
 *
 * @return int[]|false
 * @since 4.0
 */
function kalium_get_related_posts( $args, $post = null ) {
	global $wpdb;

	$post = is_null( $post ) ? get_queried_object() : get_post( $post );

	// Parse args
	$args = wp_parse_args(
		$args,
		[
			'posts_per_page' => 2,
			'taxonomies'     => null,
		]
	);

	// Get transient
	$post_type      = $post->post_type;
	$post_id        = $post->ID;
	$limit          = $args['posts_per_page'];
	$taxonomies     = $args['taxonomies'];
	$transient_name = 'kalium_related_' . $post_id;
	$transient      = get_transient( $transient_name );
	$query_args     = http_build_query( $args );
	$related_posts  = $transient && isset( $transient[ $query_args ] ) ? $transient[ $query_args ] : false;

	if ( is_string( $taxonomies ) && ! empty( $taxonomies ) ) {
		$taxonomies = array_map( 'trim', explode( ',', $taxonomies ) );
	}

	// Query related posts and cache them for a day
	if ( false === $related_posts || count( $related_posts ) < $limit ) {
		$include_term_ids = [];

		if ( is_array( $taxonomies ) ) {
			foreach ( $taxonomies as $taxonomy ) {
				if ( is_string( $taxonomy ) && taxonomy_exists( $taxonomy ) ) {
					$terms            =
					$include_term_ids = array_merge(
						$include_term_ids,
						wp_get_object_terms(
							$post_id,
							$taxonomy,
							[
								'fields' => 'ids',
							]
						)
					);
				}
			}
		}

		$query = [
			'fields' => "SELECT DISTINCT ID FROM {$wpdb->posts} p",
			'join'   => '',
			'where'  => "WHERE 1=1 AND p.post_status = 'publish' AND p.post_type = '" . esc_sql( $post_type ) . "' AND p.ID <> '" . esc_sql( $post_id ) . "'",
			'order'  => 'ORDER BY RAND()',
			'limits' => 'LIMIT ' . absint( $limit + 10 ),
		];

		if ( count( $include_term_ids ) ) {
			$query['join'] .= " INNER JOIN ( SELECT object_id FROM {$wpdb->term_relationships} INNER JOIN {$wpdb->term_taxonomy} using( term_taxonomy_id ) WHERE term_id IN ( " . implode( ',', array_map( 'absint', $include_term_ids ) ) . ' ) ) AS include_join ON include_join.object_id = p.ID';
		}

		// Found post ids
		$related_posts = $wpdb->get_col( implode( ' ', $query ) );

		if ( $transient ) {
			$transient[ $query_args ] = $related_posts;
		} else {
			$transient = [ $query_args => $related_posts ];
		}

		set_transient( $transient_name, $transient, DAY_IN_SECONDS );
	}

	if ( apply_filters( 'kalium_related_posts_shuffle', true ) ) {
		shuffle( $related_posts );
	}

	return array_slice( $related_posts, 0, $limit );
}

/**
 * Checks if given string is URL.
 *
 * @param string $string
 *
 * @return bool
 * @since 4.0
 */
function kalium_is_url( $string ) {
	return kalium()->is->url( $string );
}

/**
 * Automatically enables lazy loading for featured images after certain invocations.
 *
 * @return array
 * @since 4.0.4
 */
function kalium_get_featured_image_attrs() {
	static $invoke_count = 0;

	$attrs = [];

	// Auto enable lazy loading after 8th image
	if ( $invoke_count > 8 ) {
		$attrs['loading'] = 'lazy';
	}

	++$invoke_count;

	return apply_filters( 'kalium_featured_image_attrs', $attrs );
}

/**
 * REST API namespace.
 *
 * @param string $version
 *
 * @return string
 * @since 4.0.9
 */
function kalium_rest_namespace( $version = 'v1' ) {
	return apply_filters( 'kalium_rest_namespace', 'kalium' . ( $version ? "/{$version}" : '' ), $version );
}
