<?php
/**
 * Kalium WordPress Theme
 *
 * Core template functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

if ( ! function_exists( 'kalium_site_wrapper_start' ) ) {

	/**
	 * Site wrapper start.
	 *
	 * @since 4.0
	 */
	function kalium_site_wrapper_start() {
		echo '<div class="wrapper" id="main-wrapper">';
	}
}

if ( ! function_exists( 'kalium_site_wrapper_end' ) ) {

	/**
	 * Site wrapper end.
	 *
	 * @since 4.0
	 */
	function kalium_site_wrapper_end() {
		echo '</div>';
	}
}

if ( ! function_exists( 'kalium_logo_element' ) ) {

	/**
	 * Logo element.
	 *
	 * @param int $attachment_id
	 * @param int $max_width
	 */
	function kalium_logo_element( $attachment_id = null, $max_width = null ) {
		$args = [
			'logo_name'  => get_bloginfo( 'name' ),
			'logo_image' => [],
			'link'       => apply_filters( 'kalium_logo_url', home_url() ),
		];

		// Classes
		$classes = [
			'header-logo',
		];

		// Use current logo
		if ( is_null( $attachment_id ) && kalium_get_theme_option( 'use_uploaded_logo' ) ) {
			$attachment_id = kalium_get_attachment_id_from_theme_option( 'custom_logo_image' );
		}

		// Image logo
		if ( $attachment_id && ( $image_logo = wp_get_attachment_image_src( $attachment_id, 'full' ) ) ) {
			$image_url    = $image_logo[0];
			$image_width  = $image_logo[1];
			$image_height = $image_logo[2];

			// Resize image
			if ( is_numeric( $max_width ) ) {
				$new_size = kalium()->helpers->resize_by_width( $image_width, $image_height, $max_width );

				kalium_print_inline_style(
					[
						'selector' => '.header-logo',
						'vars'     => [
							'logo-width'  => $new_size[0] . 'px',
							'logo-height' => $new_size[1] . 'px',
						],
					]
				);
			} else {
				kalium_print_responsive_style(
					[
						'id'       => 'site-logo',
						'value'    => kalium_get_theme_option( 'custom_logo_max_width' ),
						'selector' => '.header-logo',
						'mapper'   => function ( $width ) use ( $image_width, $image_height ) {
							if ( is_numeric( $width ) && $width > 0 ) {
								$new_size = kalium()->helpers->resize_by_width( $image_width, $image_height, $width );

								return [
									'vars' => [
										'logo-width'  => $new_size[0] . 'px',
										'logo-height' => $new_size[1] . 'px',
									],
								];
							}
						},
					]
				);
			}

			// Define logo image
			$args['logo_image'] = [
				'src'    => $image_url,
				'width'  => $image_width,
				'height' => $image_height,
			];

			// Add logo image class
			$classes[] = 'logo-image';
		} else {

			// Add logo text class
			$classes[] = 'logo-text';

		}

		// Pass classes as template argument
		$args['classes'] = $classes;

		// Logo element
		kalium_get_template( 'global/logo.php', $args );
	}
}

if ( ! function_exists( 'kalium_dynamic_sidebar' ) ) {

	/**
	 * Dynamic sidebar implementation for Kalium.
	 *
	 * @param string       $sidebar_id
	 * @param array|string $class
	 *
	 * @return void
	 */
	function kalium_dynamic_sidebar( $sidebar_id, $class = '' ) {
		$classes = [ 'widget-area' ];

		if ( is_array( $class ) ) {
			$classes = array_merge( $classes, $class );
		} elseif ( ! empty( $class ) ) {
			$classes[] = $class;
		}

		/**
		 * Dynamic sidebar before.
		 */
		do_action( 'kalium_dynamic_sidebar_before', $sidebar_id );

		?>
		<div <?php kalium_class_attr( apply_filters( 'kalium_widget_area_class', $classes, $sidebar_id ) ); ?> role="complementary">
			<?php

			// Show sidebar widgets
			dynamic_sidebar( $sidebar_id );
			?>
		</div>
		<?php

		/**
		 * Dynamic sidebar after.
		 */
		do_action( 'kalium_dynamic_sidebar_after', $sidebar_id );
	}
}

if ( ! function_exists( 'kalium_social_networks' ) ) {

	/**
	 * List social networks.
	 *
	 * @param array $args
	 *
	 * @return string|void
	 * @since 4.0
	 */
	function kalium_social_networks( $args = [] ) {
		if ( ! isset( $GLOBALS['kalium_social_networks_counter'] ) ) {
			$GLOBALS['kalium_social_networks_counter'] = 1;
		}

		$args = wp_parse_args(
			$args,
			[
				//
				// General
				//

				// Classes
				'classes'                  => [
					kalium_get_link_plain_class(),
				],

				// Content
				'content'                  => [
					'content_type'   => 'social_icons',
					'content_source' => 'social_icons',
				],

				// Content
				'icon'                     => true,
				'label'                    => false,

				// Icon shape
				'shape_type'               => null,
				'shape_fill'               => null,
				'shape_radius'             => null,
				'shape_padding'            => null,

				// Link
				'new_tab'                  => true,
				'no_follow'                => false,

				// Other
				'spacing'                  => null,
				'size'                     => null,

				// Output
				'echo'                     => true,

				//
				// Style
				//

				// Icon color
				'color'                    => null, // [brand,custom]
				'color_custom'             => null,
				'color_hover'              => null, // [brand,custom]
				'color_hover_custom'       => null,

				// Label color
				'label_color'              => null, // [brand,custom]
				'label_color_custom'       => null,
				'label_hover_color'        => null, // [brand,custom]
				'label_hover_color_custom' => null,

				// Icon Background
				'background'               => null, // [brand,custom]
				'background_custom'        => null,
				'background_hover'         => null, // [brand,custom]
				'background_hover_custom'  => null,
			]
		);

		// Styles
		$style_props = [];

		// Instance class
		$instance_class = 'social-icons--instance-' . $GLOBALS['kalium_social_networks_counter'];

		// Classes
		$classes = [
			'social-icons',
			$instance_class,
		];

		// Extra classes
		if ( is_string( $args['classes'] ) ) {
			$args['classes'] = explode( ' ', $args['classes'] );
		}

		if ( is_array( $args['classes'] ) ) {
			$classes = array_merge( $classes, $args['classes'] );
		}

		// Hide icon
		if ( ! $args['icon'] ) {
			$classes[] = 'social-icons--without-icon';
			$classes[] = kalium_get_link_style_class_by_context( 'social-networks' );
		}

		// Icon with shape
		if ( in_array( $args['shape_type'], [ 'rounded', 'square', 'custom' ] ) ) {
			$classes[] = 'social-icons--with-icon-shape';
			$classes[] = 'social-icons--icon-shape-' . $args['shape_type'];

			if ( 'custom' === $args['shape_type'] ) {
				$radius = is_numeric( $args['shape_radius'] ) ? "{$args['shape_radius']}px" : $args['shape_radius'];

				if ( $radius ) {
					$style_props[ kalium_css_var_name( 'si-radius' ) ] = $radius;
				}
			}

			// Outline
			if ( 'outline' === $args['shape_fill'] ) {
				$classes[] = 'social-icons--icon-shape-outline';
			}

			// Padding
			if ( is_numeric( $args['shape_padding'] ) ) {
				$style_props[ kalium_css_var_name( 'si-padding-scale' ) ] = $args['shape_padding'];
			}
		}

		// Show label
		if ( $args['label'] ) {
			$classes[] = 'social-icons--with-label';
		}

		// Spacing between
		if ( ! is_null( $args['spacing'] ) ) {
			$spacing = is_numeric( $args['spacing'] ) ? "{$args['spacing']}px" : $args['spacing'];

			$style_props[ kalium_css_var_name( 'si-spacing' ) ] = $spacing;
		}

		// Icon size
		if ( $args['size'] || is_numeric( $args['size'] ) ) {
			$size = is_numeric( $args['size'] ) ? "{$args['size']}px" : $args['size'];

			$style_props[ kalium_css_var_name( 'si-size' ) ] = $size;
		}

		// Inherit Hover Background
		if ( ! empty( $args['background'] ) && empty( $args['background_hover'] ) ) {
			$args['background_hover']        = $args['background'];
			$args['background_hover_custom'] = $args['background_custom'];
		}

		// Auto set text color when background is applied
		if ( 'brand' === $args['background'] && empty( $args['color'] ) ) {
			if ( 'outline' !== $args['shape_fill'] ) {
				$args['color']        = 'custom';
				$args['color_custom'] = kalium_color_reference( 'theme_colors_body' );
			} else {
				$args['color'] = $args['background'];
			}

			if ( empty( $args['label_color'] ) ) {
				$args['label_color'] = 'brand';
			}
		}

		/**
		 * Social icon color var.
		 *
		 * @param array $args
		 * @param array $classes
		 * @param array $style_props
		 */
		$set_color = function ( $args, &$classes, &$style_props ) {
			$args = wp_parse_args(
				$args,
				[
					'prefix' => '',
					'type'   => 'custom',
					'color'  => null,
				]
			);

			if ( 'brand' === $args['type'] ) {
				$classes[] = 'social-icons--' . $args['prefix'] . '-' . $args['type'];
			} elseif ( 'custom' === $args['type'] && ! empty( $args['color'] ) ) {
				$style_props[ kalium_css_var_name( 'si-' . $args['prefix'] ) ] = $args['color'];
			}
		};

		// Color
		$set_color(
			[
				'prefix' => 'color',
				'type'   => $args['color'],
				'color'  => $args['color_custom'],
			],
			$classes,
			$style_props
		);

		// Hover Color
		$set_color(
			[
				'prefix' => 'hover-color',
				'type'   => $args['color_hover'],
				'color'  => $args['color_hover_custom'],
			],
			$classes,
			$style_props
		);

		// Label Color
		$set_color(
			[
				'prefix' => 'label-color',
				'type'   => $args['label_color'],
				'color'  => $args['label_color_custom'],
			],
			$classes,
			$style_props
		);

		// Label Hover Color
		$set_color(
			[
				'prefix' => 'label-hover-color',
				'type'   => $args['label_hover_color'],
				'color'  => $args['label_hover_color_custom'],
			],
			$classes,
			$style_props
		);

		// Background Hover Color
		if ( ! is_null( $args['shape_type'] ) ) {

			// Background Color
			$set_color(
				[
					'prefix' => 'background',
					'type'   => $args['background'],
					'color'  => $args['background_custom'],
				],
				$classes,
				$style_props
			);

			$set_color(
				[
					'prefix' => 'background-hover',
					'type'   => $args['background_hover'],
					'color'  => $args['background_hover_custom'],
				],
				$classes,
				$style_props
			);
		}

		ob_start();

		// Print style
		if ( ! empty( $style_props ) ) {
			kalium_print_inline_style(
				[
					'selector' => '.' . $instance_class,
					'props'    => $style_props,
				]
			);
		}

		// Wrapper start
		printf( '<div %s>', kalium_class_attr( $classes, false ) );

		// Render social icons
		laborator_builder_render(
			$args['content'],
			[
				'new_tab'   => $args['new_tab'],
				'no_follow' => $args['no_follow'],
				'has_icon'  => $args['icon'],
				'has_label' => $args['label'],
			]
		);

		// Wrapper end
		echo '</div>';

		// Output
		$social_icons_output = ob_get_clean();

		// Increment instance
		++$GLOBALS['kalium_social_networks_counter'];

		// Print social icons
		if ( $args['echo'] ) {
			echo $social_icons_output;
			return;
		}

		return $social_icons_output;
	}
}

if ( ! function_exists( 'kalium_social_networks_share' ) ) {

	/**
	 * Share on social networks.
	 *
	 * @since 4.0
	 */
	function kalium_social_networks_share( $args ) {
		$args = wp_parse_args(
			$args,
			[
				'post_id'      => null,
				'networks'     => laborator_builder_get_content_type( 'social_share_networks' )->get_default_content(),
				'style'        => 'plain',
				'icon_padding' => 0.65,
			]
		);

		// Post ID
		$post_id = $args['post_id'];

		// Parse social networks
		$networks = array_map(
			function ( $network ) use ( $post_id ) {
				$share_link = kalium_get_social_network_share_post_link( $network, $post_id );
				$element    = [
					'name'       => $share_link['element'],
					'attributes' => [
						[
							'name'  => 'label',
							'value' => $share_link['label'],
						],
						[
							'name'  => 'link',
							'value' => $share_link['url'],
						],
					],
				];

				// Print needs to provide the icon
				if ( 'print' === $network ) {
					$element['attributes'] = array_merge(
						$element['attributes'],
						[
							[
								'name'  => 'icon_type',
								'value' => 'inline-html',
							],
							[
								'name'  => 'icon_inline_html',
								'value' => kalium_get_icon( 'print' ),
							],
						]
					);
				}

				// Open in same window
				if ( in_array( $network, [ 'email', 'print' ] ) ) {
					$element['attributes'][] = [
						'name'  => 'force_self',
						'value' => true,
					];
				}

				return $element;
			},
			kalium_parse_social_share_networks( $args['networks'] )
		);

		// Social networks element args
		$social_networks_args = [
			'classes' => [
				'social-networks-share',
			],
			'content' => [
				'content_type'   => 'social_icons',
				'content_source' => [
					'type'    => 'direct',
					'content' => [
						'elements' => $networks,
					],
				],
			],
		];

		// Style
		switch ( $args['style'] ) {
			// Text
			case 'text':
			case 'plain':
				$social_networks_args = array_merge(
					$social_networks_args,
					[
						'icon'         => false,
						'label'        => true,

						'color'        => 'custom',
						'color_custom' => kalium_css_var_reference( 'body-color' ),

						'color_hover'  => 'brand',
					]
				);
				break;

			// Icons
			case 'icons':
				$social_networks_args = array_merge(
					$social_networks_args,
					[
						'shape_type'         => 'rounded',
						'shape_padding'      => $args['icon_padding'],

						'color'              => 'custom',
						'color_custom'       => kalium_color_reference( 'theme_colors_text.normal' ),

						'color_hover'        => 'custom',
						'color_hover_custom' => kalium_color_reference( 'theme_colors_body' ),

						'background'         => 'custom',
						'background_custom'  => kalium_color_reference( 'theme_colors.color-8' ),

						'background_hover'   => 'brand',
					]
				);
				break;
		}

		kalium_social_networks( $social_networks_args );
	}
}

if ( ! function_exists( 'wp_body_open' ) ) {

	/**
	 * Fire the wp_body_open action, backward compatibility to support pre-5.2.0 WordPress versions.
	 *
	 * @since 3.0
	 */
	function wp_body_open() {
		do_action( 'wp_body_open' );
	}
}

if ( ! function_exists( 'kalium_site_frame_display' ) ) {

	/**
	 * Theme borders (site frame).
	 */
	function kalium_site_frame_display() {

		// Theme borders
		if ( kalium_get_theme_option( 'theme_borders' ) ) {
			$attributes = [
				'class' => [
					'page-border',
				],
			];

			$styles = [
				'id'       => 'site-frame',
				'selector' => '.page-borders',
				'vars'     => [
					'site-frame-color'  => kalium_get_theme_option( 'theme_borders_color' ),
					'site-frame-width'  => kalium_get_theme_option( 'theme_borders_thickness' ),
					'site-frame-radius' => kalium_get_theme_option( 'theme_borders_radius' ),
				],
			];

			if ( $animation = kalium_get_theme_option( 'theme_borders_animation' ) ) {
				$animation_duration = kalium_get_theme_option( 'theme_borders_animation_duration' );
				$animation_delay    = kalium_get_theme_option( 'theme_borders_animation_delay' );

				$styles['vars']['site-frame-animation']          = 'site-frame-' . $animation;
				$styles['vars']['site-frame-animation-duration'] = is_numeric( $animation_duration ) ? sprintf( '%ds', $animation_duration ) : null;
				$styles['vars']['site-frame-animation-delay']    = is_numeric( $animation_delay ) ? sprintf( '%ds', $animation_delay ) : null;

				$attributes['class'][] = 'page-border--animated';
			}

			// Print styles
			kalium_print_inline_style( $styles );

			// Site frame
			echo kalium_element(
				[
					'tag_name'   => 'div',
					'attributes' => $attributes,
				]
			);
		}
	}
}

if ( ! function_exists( 'kalium_footer_display' ) ) {

	/**
	 * Display theme footer.
	 *
	 * @since 4.0
	 */
	function kalium_footer_display() {
		if ( ! apply_filters( 'kalium_show_footer', kalium_get_theme_option( 'footer_enabled' ) ) ) {
			return;
		}

		// Classes
		$classes = [
			'site-footer',
		];

		// Footer type
		$type = kalium_get_theme_option( 'footer_type' );

		if ( 'fixed' === $type ) {
			$animation = kalium_get_theme_option( 'footer_fixed_animation' );
			$classes[] = 'fixed-footer';

			if ( 'fade' === $animation ) {
				$classes[] = 'fixed-footer--fade';
			} elseif ( 'slide' === $animation ) {
				$classes[] = 'fixed-footer--slide';
			}
		}

		// Display footer template
		kalium_get_template(
			'footer.php',
			[
				'classes' => $classes,
			]
		);
	}
}

if ( ! function_exists( 'kalium_footer_content_display' ) ) {

	/**
	 * Display footer content.
	 *
	 * @since 4.0
	 */
	function kalium_footer_content_display() {
		laborator_builder_render(
			[
				'content_type'   => 'footer',
				'content_source' => 'footer_content',
			],
			[
				'parent_selector' => '.site-footer',
			]
		);
	}
}

if ( ! function_exists( 'kalium_breadcrumb' ) ) {

	/**
	 * Breadcrumb display.
	 *
	 * @param array $args
	 *
	 * @return string|void
	 * @since 3.2
	 */
	function kalium_breadcrumb( $args = [] ) {
		if ( ! kalium()->is->breadcrumb_navxt_active() ) {
			return;
		}

		// Breadcrumb instance ID
		static $breadcrumb_instance_id = 1;

		// Breadcrumb args
		$args = wp_parse_args(
			$args,
			[
				'container'        => true,
				'class'            => '',
				'background_color' => '',
				'text_color'       => '',
				'border_color'     => '',
				'border_type'      => '',
				'text_alignment'   => '',
				'echo'             => true,
			]
		);

		// Current Object ID
		$object_id = kalium_get_queried_object_id();

		// Breadcrumb classes
		$classes = [
			'breadcrumb',
		];

		// Container classes
		$container_classes = [
			'breadcrumb__container',
		];

		// Style props
		$style_props = [];

		// Style
		$background_color = kalium_get_theme_option( 'breadcrumb_background_color' );
		$text_color       = kalium_get_theme_option( 'breadcrumb_text_color' );
		$border_color     = kalium_get_theme_option( 'breadcrumb_border_color' );
		$border_type      = kalium_get_theme_option( 'breadcrumb_border_type' );
		$border_radius    = kalium_get_theme_option( 'breadcrumb_border_radius' );
		$text_align       = kalium_get_theme_option( 'breadcrumb_alignment' );
		$margin           = kalium_get_theme_option( 'breadcrumb_margin' );
		$margin_top       = kalium_get_multi_numeric_prop( $margin, 'top' );
		$margin_bottom    = kalium_get_multi_numeric_prop( $margin, 'bottom' );

		// Custom breadcrumb parameters for from post meta fields
		if ( is_singular() && 'enable' === kalium_get_field( 'breadcrumb', $object_id ) ) {

			// Custom background color
			if ( $custom_background_color = kalium_get_field( 'breadcrumb_background_color', $object_id ) ) {
				$background_color = $custom_background_color;
			}

			// Custom text color
			if ( $custom_text_color = kalium_get_field( 'breadcrumb_text_color', $object_id ) ) {
				$text_color = $custom_text_color;
			}

			// Custom border color
			if ( $custom_border_color = kalium_get_field( 'breadcrumb_border_color', $object_id ) ) {
				$border_color = $custom_border_color;
			}

			// Custom border type
			if ( ( $custom_border_type = kalium_get_field( 'breadcrumb_border_type', $object_id ) ) && in_array(
				$custom_border_type,
				[
					'border',
					'border-horizontal',
				]
			) ) {
				$border_type = $custom_border_type;
			}

			// Custom text alignment
			if ( ( $custom_text_align = kalium_get_field( 'breadcrumb_text_alignment', $object_id ) ) && in_array(
				$custom_text_align,
				[
					'left',
					'center',
					'right',
				]
			) ) {
				$text_align = $custom_text_align;
			}

			// Custom border radius
			$custom_border_radius = kalium_get_field( 'breadcrumb_border_radius', $object_id );

			if ( is_numeric( $custom_border_radius ) ) {
				$border_radius = $custom_border_radius;
			}

			// Custom margin top
			$custom_margin_top = kalium_get_field( 'breadcrumb_margin_top', $object_id );

			if ( is_numeric( $custom_margin_top ) ) {
				$margin_top = $custom_margin_top;
			}

			// Custom margin top
			$custom_margin_bottom = kalium_get_field( 'breadcrumb_margin_bottom', $object_id );

			if ( is_numeric( $custom_margin_bottom ) ) {
				$margin_bottom = $custom_margin_bottom;
			}
		}

		// Background color from $args
		if ( ! empty( $args['background_color'] ) ) {
			$background_color = $args['background_color'];
		}

		// Text color from $args
		if ( ! empty( $args['text_color'] ) ) {
			$text_color = $args['text_color'];
		}

		// Border color from $args
		if ( ! empty( $args['border_color'] ) ) {
			$border_color = $args['border_color'];
		}

		// Border type from $args
		if ( ! empty( $args['border_type'] ) && in_array(
			$args['border_type'],
			[
				'border',
				'border-horizontal',
			]
		) ) {
			$border_type = $args['border_type'];
		}

		// Text alignment from $args
		if ( ! empty( $args['text_alignment'] ) && in_array(
			$args['text_alignment'],
			[
				'left',
				'center',
				'right',
			]
		) ) {
			$text_align = $args['text_alignment'];
		}

		// Style: Background
		if ( $background_color ) {
			$container_classes[] = 'breadcrumb__container--has-background';
			$container_classes[] = 'breadcrumb__container--has-padding';

			$style_props['background-color'] = $background_color;
		}

		// Style: Border Radius
		if ( is_numeric( $border_radius ) ) {
			$border_radius               .= 'px';
			$style_props['border-radius'] = $border_radius;
		} elseif ( is_array( $border_radius ) ) {
			if ( $border_radius = kalium_to_length( $border_radius, 0 ) ) {
				$style_props['border-radius'] = $border_radius;
			}
		}

		// Style: Border
		if ( 'border-horizontal' === $border_type ) {
			$container_classes[] = 'breadcrumb__container--border-horizontal';
			$container_classes[] = 'breadcrumb__container--has-padding-horizontal';

			$style_props['border-color'] = $border_color;
			unset( $style_props['border-radius'] );
		} elseif ( 'border' === $border_type ) {
			$container_classes[] = 'breadcrumb__container--border';
			$container_classes[] = 'breadcrumb__container--has-padding';

			$style_props['border-color'] = $border_color;
		}

		// Style: Text
		if ( $text_color ) {
			$container_classes[] = 'breadcrumb__container--has-text-color';

			$style_props['color'] = $text_color;
		}

		// Style: Text alignment
		if ( in_array( $text_align, [ 'left', 'center', 'right' ] ) ) {
			$container_classes[] = 'breadcrumb__container--align-' . $text_align;
		}

		// Style: Margin Top
		if ( $margin_top ) {
			$style_props['margin-top'] = $margin_top . ( is_numeric( $margin_top ) ? 'px' : '' );
		}

		// Style: Margin Bottom
		if ( $margin_bottom ) {
			$classes[] = 'breadcrumb--no-bottom-margin';

			$style_props['margin-bottom'] = $margin_bottom . ( is_numeric( $margin_bottom ) ? 'px' : '' );
		}

		// Responsive
		$responsive = kalium_get_theme_option( 'breadcrumb_responsive' );
		$classes    = array_merge( $classes, kalium_get_device_visibility_classes( $responsive ) );

		// Selector
		$selector = 'breadcrumb-' . $breadcrumb_instance_id;

		// Breadcrumb ID
		$classes[] = $selector;

		// Extra classes
		if ( ! empty( $args['class'] ) ) {
			$extra_classes = $args['class'];

			if ( is_string( $extra_classes ) ) {
				$extra_classes = explode( ' ', $extra_classes );
			}

			$classes = array_merge( $classes, $extra_classes );
		}

		// Breadcrumb trail
		$breadcrumb_html = bcn_display( true );

		if ( ! $breadcrumb_html ) {
			return;
		}

		// Template args
		$template_args = [
			'classes'           => $classes,
			'container'         => $args['container'],
			'container_classes' => array_unique( $container_classes ),
			'breadcrumb_html'   => $breadcrumb_html,
		];

		// Buffer the output
		ob_start();

		// Load template
		kalium_get_template( 'global/breadcrumb.php', $template_args );

		// Style
		if ( ! empty( $style_props ) ) {
			$style_props = array_map( 'kalium_replace_color_references', $style_props );

			kalium_append_custom_css( ".{$selector} .breadcrumb__container", $style_props );
		}

		// Output
		$output = ob_get_clean();

		// Increment instance Id
		++$breadcrumb_instance_id;

		// Print the output
		if ( $args['echo'] ) {
			echo $output;

			return;
		}

		return $output;
	}
}

if ( ! function_exists( 'kalium_page_heading' ) ) {

	/**
	 * Show page heading template.
	 *
	 * @param array $args
	 */
	function kalium_page_heading( $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'classes'     => [],
				'container'   => true,
				'heading_tag' => 'h1',
				'title'       => '',
				'description' => '',
			]
		);

		// Classes of main container
		$classes = array_merge( [ 'page-heading' ], $args['classes'] );

		// Description
		$description = $args['description'];

		if ( false === strpos( $description, '<' ) ) {
			$description = wpautop( $description );
		}

		// Container classes
		$container_classes = [
			'page-heading__container',
		];

		// Has container
		if ( $args['container'] ) {
			$container_classes[] = 'container';
		}

		// Show template
		kalium_get_template(
			'global/page-heading.php',
			[
				'classes'           => $classes,
				'container_classes' => $container_classes,
				'heading_tag'       => $args['heading_tag'],
				'title'             => $args['title'],
				'description'       => $description,
			]
		);
	}
}

if ( ! function_exists( 'kalium_content_wrapper_start' ) ) {

	/**
	 * Content wrapper start.
	 *
	 * @param string $wrapper_id
	 * @param array  $args
	 * @since 4.0
	 */
	function kalium_content_wrapper_start( $wrapper_id, $args = [] ) {
		$args = apply_filters(
			"kalium_{$wrapper_id}_content_wrapper_args",
			wp_parse_args(
				$args,
				[
					'classes'             => null,
					'container_fullwidth' => null,
					'sidebar'             => null,
					'sidebar_align'       => null,
					'sidebar_collapsed'   => null,
				]
			)
		);

		$content_wrapper_vars = [];

		// Container classes
		$container_classes = [
			'content-wrapper',
		];

		// Container
		if ( $args['container_fullwidth'] ) {
			$container_classes[] = 'container-full';
		} else {
			$container_classes[] = 'container';
		}

		// Sidebar
		if ( $args['sidebar'] ) {
			$sidebar_options = kalium_get_sidebar_options();

			// Has sidebar
			$container_classes[] = 'has-sidebar';

			// Is sticky sidebar
			if ( $sidebar_options['sticky'] ) {
				$container_classes[] = 'has-sticky-' . $sidebar_options['sticky_behavior'];
			}

			// Styled sidebar
			if ( 'styled' === $sidebar_options['type'] ) {
				$container_classes[] = 'sidebar-styled';
			}

			// Separated widgets
			if ( $sidebar_options['widgets_separate'] ) {
				$container_classes[] = 'sidebar-widgets-separated';
			}

			// Collapsed sidebar
			$collapsed_sidebar_cookie = 'sidebar_collapsed_' . $wrapper_id;

			if ( kalium()->request->has( $collapsed_sidebar_cookie, 'COOKIE' ) && ! is_customize_preview() ) {
				$args['sidebar_collapsed'] = kalium_validate_boolean( kalium()->request->cookie( $collapsed_sidebar_cookie ) );
			}

			if ( $args['sidebar_collapsed'] ) {
				$container_classes[] = 'sidebar-collapsed';
			}

			// Alignment (larger screens)
			if ( in_array( $args['sidebar_align'], [ 'left', 'right' ] ) ) {
				$container_classes[] = 'sidebar-' . $args['sidebar_align'];
			}

			// Alignment (mobile)
			if ( 'top' === $sidebar_options['mobile_position'] ) {
				$container_classes[] = 'sidebar-mobile-top';
			}

			// Width
			if ( is_numeric( $sidebar_options['width'] ) ) {
				$content_wrapper_vars['sidebar-width'] = $sidebar_options['width'] . '%';
			}

			// Gap
			if ( is_numeric( $sidebar_options['gap'] ) ) {
				$content_wrapper_vars['sidebar-gap'] = $sidebar_options['gap'] . 'px';
			}
		}

		// Extra classes
		if ( is_array( $args['classes'] ) ) {
			$container_classes = array_unique( array_merge( $container_classes, $args['classes'] ) );
		}

		// Print style
		kalium_print_inline_style(
			[
				'id'     => 'content-wrapper',
				'styles' => [
					[
						'selector' => '.content-wrapper',
						'vars'     => $content_wrapper_vars,
					],
					[
						'selector' => '.has-sidebar .sidebar',
						'vars'     => kalium_get_sidebar_css_vars(),
					],
				],
			]
		);

		// Class hooks based on wrapper ID
		$container_classes = apply_filters( "kalium_{$wrapper_id}_content_wrapper_classes", $container_classes );

		?>
		<div <?php kalium_class_attr( $container_classes ); ?> data-wrapper-id="<?php echo esc_attr( $wrapper_id ); ?>">
		<?php

		/**
		 * Hook: kalium_{$wrapper_id}_content_wrapper_before.
		 */
		do_action( "kalium_{$wrapper_id}_content_wrapper_before" );
	}
}

if ( ! function_exists( 'kalium_content_wrapper_end' ) ) {

	/**
	 * Content wrapper end.
	 *
	 * @param string $wrapper_id
	 * @since 4.0
	 */
	function kalium_content_wrapper_end( $wrapper_id ) {
		/**
		 * Hook: kalium_{$wrapper_id}_content_wrapper_after.
		 */
		do_action( "kalium_{$wrapper_id}_content_wrapper_after" );
		?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_content_wrapper_main_content_start' ) ) {

	/**
	 * Main content start.
	 *
	 * @since 4.0
	 */
	function kalium_content_wrapper_main_content_start() {
		echo '<section class="main-content">';
	}
}

if ( ! function_exists( 'kalium_content_wrapper_main_content_end' ) ) {

	/**
	 * Main content end.
	 *
	 * @since 4.0
	 */
	function kalium_content_wrapper_main_content_end() {
		echo '</section>';
	}
}

if ( ! function_exists( 'kalium_the_content' ) ) {

	/**
	 * Prints the current post content wrapped with entry-content class.
	 */
	function kalium_the_content() {
		$classes = [
			'entry-content',
			kalium_get_link_style_class(),
		];

		// Applied on pages only
		if ( is_page() ) {
			$classes[] = 'container-child';
		}

		?>
		<div <?php kalium_class_attr( $classes ); ?>>
			<?php
			// Prints current post content
			the_content();
			?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_search_content_editable_input' ) ) {

	/**
	 * Editable search input box.
	 *
	 * @since 4.0
	 */
	function kalium_search_content_editable_input() {
		?>
		<span class="searchable">
			<span class="searchable__input" contenteditable="plaintext-only"><?php echo esc_html( get_search_query( false ) ); ?></span>
		</span>
		<?php
	}
}

if ( ! function_exists( 'kalium_pagination' ) ) {

	/**
	 * Normal pagination.
	 */
	function kalium_pagination( $args = [] ) {
		$args = apply_filters(
			'kalium_pagination_args',
			wp_parse_args(
				$args,
				[
					'align'           => 'center',
					'display'         => 'numbers-prev-next', // [numbers, numbers-prev-next, prev-next]
					'pagination_args' => [],
				]
			)
		);

		// Pagination args
		$pagination_args = wp_parse_args(
			$args['pagination_args'],
			[
				'end_size'           => 2,
				'prev_text'          => sprintf( '%2$s <span class="page-numbers__label link">%1$s</span>', esc_html__( 'Previous', 'kalium' ), '<i class="kalium-icon-long-arrow-left"></i>' ),
				'next_text'          => sprintf( '<span class="page-numbers__label link">%1$s</span> %2$s', esc_html__( 'Next', 'kalium' ), '<i class="kalium-icon-long-arrow-right"></i>' ),
				'before_page_number' => '<span class="link">',
				'after_page_number'  => '</span>',
			]
		);

		// Display numbers only
		if ( 'numbers' === $args['display'] ) {
			$pagination_args['prev_next'] = false;
		}

		// Fixes the issue when current page is not selected on front-page
		if ( ! empty( $GLOBALS['wp_query']->query['paged'] ) && empty( $GLOBALS['wp_query']->query_vars['paged'] ) ) {
			$GLOBALS['wp_query']->set( 'paged', $GLOBALS['wp_query']->query['paged'] );
		}

		// Generate pagination
		$pagination_links = paginate_links(
			array_merge(
				$pagination_args,
				[
					'type' => 'array',
				]
			)
		);

		// No pagination
		if ( empty( $pagination_links ) ) {
			return;
		}

		// Prev/next
		if ( 'prev-next' === $args['display'] ) {
			$pagination_links = array_filter(
				array_map(
					function ( $pagination_link ) {
						if ( 0 < strpos( $pagination_link, 'prev page-numbers' ) ) {
							return $pagination_link;
						} elseif ( 0 < strpos( $pagination_link, 'next page-numbers' ) ) {
							return $pagination_link;
						}

						return null;
					},
					$pagination_links
				)
			);
		}
		// Wrap numbers
		else {
			$total_links        = count( $pagination_links );
			$first_number_index = false !== strpos( $pagination_links[0], 'prev page-numbers' ) ? 1 : 0;
			$last_number_index  = $total_links - ( false !== strpos( $pagination_links[ $total_links - 1 ], 'next page-numbers' ) ? 2 : 1 );

			$pagination_links[ $first_number_index ] = '<div class="page-numbers-group">' . $pagination_links[ $first_number_index ];
			$pagination_links[ $last_number_index ] .= '</div>';
		}

		// Build pagination
		$page_numbers = sprintf( '<div class="page-numbers">%s</div>', implode( '', $pagination_links ) );

		// Classes
		$classes = [
			'pagination',
			kalium_get_link_plain_class(),
			kalium_get_link_text_class(),
		];

		if ( in_array( $args['align'], [ 'left', 'center', 'right' ] ) ) {
			$classes[] = 'pagination--align-' . $args['align'];
		}

		// Pagination template
		kalium_get_template(
			'global/pagination.php',
			[
				'align'        => $args['align'],
				'classes'      => $classes,
				'page_numbers' => $page_numbers,
			]
		);
	}
}

if ( ! function_exists( 'kalium_load_more_pagination' ) ) {

	/**
	 * Load more pagination.
	 *
	 * @param array $args
	 *
	 * @since 4.0
	 */
	function kalium_load_more_pagination( $args ) {
		$args = wp_parse_args(
			$args,
			[
				'view_more_text' => esc_html__( 'Show more', 'kalium' ),
				'last_page_text' => esc_html__( 'All posts are shown', 'kalium' ),
				'align'          => 'center',
				'loading_style'  => 'spinner',
				'class'          => null,
				'options'        => [],
			]
		);

		// Infinite Scroll Options
		$args['options'] = wp_parse_args(
			$args['options'],
			[
				'path'            => '.next.page-numbers',
				'container'       => null,
				'item'            => '.post',
				'pagination'      => null,
				'history'         => false,
				'infinite_scroll' => false,
				'status'          => '.load-more__status',
				'button'          => '.load-more__view-more',
			]
		);

		// Classes
		$classes = [
			'load-more',
		];

		// Loading style
		if ( $args['loading_style'] ) {
			$classes[] = 'load-more--loading-style-' . $args['loading_style'];
		}

		// Alignment
		if ( in_array( $args['align'], [ 'left', 'center', 'right' ] ) ) {
			$classes[] = 'load-more--align-' . $args['align'];
		}

		// Extra classes
		$classes[] = kalium_tokenize_list( $args['class'] );

		$args['classes'] = $classes;

		// Loading spinner
		$loading_spinner = [
			'load-more__spinner',
		];

		// Pulsating loader
		if ( 'pulsate' === $args['loading_style'] ) {
			$loading_spinner[] = 'load-more__spinner--type-1';
		}
		// Standard spinner
		else {
			$loading_spinner[] = 'load-more__spinner--type-2';
			$loading_spinner[] = 'kalium-icon-refresh';
		}

		$args['loading_spinner'] = kalium_tokenize_list( $loading_spinner );

		// Load more template
		kalium_get_template( 'global/load-more.php', $args );
	}
}

if ( ! function_exists( 'kalium_load_more_link' ) ) {

	/**
	 * Load more link.
	 *
	 * @since 4.0
	 */
	function kalium_load_more_link( $args ) {
		$args = wp_parse_args(
			$args,
			[
				'url'    => null,
				'target' => null,
				'text'   => esc_html__( 'Show more', 'kalium' ),
			]
		);

		if ( empty( $args['url'] ) ) {
			return;
		}

		// Container classes
		$classes = [
			'load-more',
			'load-more--link',
		];

		// Link attributes
		$attributes = [
			'href'  => esc_url( $args['url'] ),
			'class' => 'button',
		];

		if ( in_array( $args['target'], [ '_self', '_blank', '_parent', '_top' ] ) ) {
			$attributes['target'] = $args['target'];
		}

		?>
		<div <?php kalium_class_attr( $classes ); ?> >
			<a <?php kalium_attributes( $attributes ); ?>><?php echo esc_html( $args['text'] ); ?></a>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_comments_pagination' ) ) {

	/**
	 * Comments pagination.
	 */
	function kalium_comments_pagination( $args = [] ) {
		global $wp_rewrite;

		if ( ! isset( $args['pagination_args'] ) ) {
			$args['pagination_args'] = [];
		}

		// Base
		$base = add_query_arg( 'cpage', '%#%' );

		if ( $wp_rewrite->using_permalinks() ) {
			$base = user_trailingslashit( trailingslashit( get_permalink() ) . $wp_rewrite->comments_pagination_base . '-%#%', 'commentpaged' );
		}

		$args['pagination_args'] = array_merge(
			$args['pagination_args'],
			[
				'base'    => $base,
				'total'   => get_comment_pages_count(),
				'current' => get_query_var( 'cpage' ),
			]
		);

		kalium_pagination( $args );
	}
}

if ( ! function_exists( 'kalium_views_switcher' ) ) {

	/**
	 * Views switcher.
	 *
	 * @param array $args
	 *
	 * @since 4.0
	 */
	function kalium_views_switcher( $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'class'     => null,
				'base_url'  => html_entity_decode( get_pagenum_link() ),
				'views'     => [],
				'current'   => null,
				'query_arg' => 'switch_columns',
			]
		);

		// Classes
		$classes = [
			'views-switcher',
		];

		// Hide on mobile
		$classes = array_merge( $classes, kalium_get_device_visibility_classes( [ 'desktop' ] ) );

		if ( ! empty( $args['class'] ) ) {
			if ( is_string( $args['class'] ) ) {
				$args['class'] = explode( ' ', $args['class'] );
			}

			$classes = array_merge( $classes, $args['class'] );
		}

		// Current
		if ( kalium()->request->has( $args['query_arg'] ) ) {
			$args['current'] = kalium()->request->request( $args['query_arg'] );
		}

		// Columns label
		$columns_label = esc_html__( '%d columns', 'kalium' );

		// Defined views
		$views = [
			[
				'value' => 'list',
				'icon'  => 'kalium-icon-layout-list',
				'label' => esc_html__( 'List view', 'kalium' ),
			],
			[
				'value' => 2,
				'icon'  => 'kalium-icon-layout-grid-2',
				'label' => sprintf( $columns_label, 2 ),
			],
			[
				'value' => 3,
				'icon'  => 'kalium-icon-layout-grid-3',
				'label' => sprintf( $columns_label, 3 ),
			],
			[
				'value' => 4,
				'icon'  => 'kalium-icon-layout-grid-4',
				'label' => sprintf( $columns_label, 4 ),
			],
			[
				'value' => 5,
				'icon'  => 'kalium-icon-layout-grid-5',
				'label' => sprintf( $columns_label, 5 ),
			],
			[
				'value' => 6,
				'icon'  => 'kalium-icon-layout-grid-5',
				'label' => sprintf( $columns_label, 6 ),
			],
		];

		// Filter available columns
		$views = array_filter(
			$views,
			function ( $view ) use ( $args ) {
				return in_array( $view['value'], $args['views'] );
			}
		);

		if ( ! empty( $views ) ) {
			?>
			<div <?php kalium_class_attr( $classes ); ?>>
				<?php
				foreach ( $views as $view ) :
					$value = $view['value'];
					$label = $view['label'];
					$icon  = $view['icon'];

					$url = add_query_arg( $args['query_arg'], $value, $args['base_url'] );

					$view_classes = [];

					if ( $value == $args['current'] ) {
						$view_classes[] = 'current';
					}

					echo kalium_element(
						'a',
						[
							'role'      => 'button',
							'href'      => $url,
							'class'     => $view_classes,
							'title'     => $label,
							'data-view' => $value,
						],
						sprintf( '<i class="%s"></i>', esc_attr( $icon ) )
					);
				endforeach;
				?>
			</div>
			<?php
		}
	}
}

if ( ! function_exists( 'kalium_sidebar_toggle' ) ) {

	/**
	 * Sidebar toggle.
	 *
	 * @since 4.0
	 */
	function kalium_sidebar_toggle( $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'expand_text'        => null,
				'collapse_text'      => null,
				'animate'            => false,
				'content_wrapper_id' => null,
				'is_collapsed'       => true,
			]
		);

		$classes = [
			'filters-toggle',
			kalium_get_link_plain_class(),
		];

		// Collapsed state in cookie
		$collapsed_cookie_name = 'sidebar_collapsed_' . $args['content_wrapper_id'];

		if ( $args['content_wrapper_id'] && kalium()->request->has( $collapsed_cookie_name, 'COOKIE' ) && ! is_customize_preview() ) {
			$args['is_collapsed'] = kalium_validate_boolean( kalium()->request->cookie( $collapsed_cookie_name ) );
		}

		// Collapsed
		if ( $args['is_collapsed'] ) {
			$classes[] = 'filters-toggle--expand';
		}
		?>
		
		<div <?php kalium_class_attr( $classes ); ?>>
			<a role="button" href="#" class="sidebar-toggle" data-expand-text="<?php echo esc_attr( $args['expand_text'] ); ?>" data-collapse-text="<?php echo esc_attr( $args['collapse_text'] ); ?>" data-animate="<?php echo esc_attr( $args['animate'] ? 'true' : 'false' ); ?>" aria-label="Show or hide sidebar">
				<i class="kalium-icon-shop-filter"></i>
			</a>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_slider' ) ) {

	/**
	 * Create touch slider.
	 *
	 * @param array|string $args
	 *
	 * @return string|void
	 * @since 4.0
	 */
	function kalium_slider( $args = [] ) {
		$args = apply_filters(
			'kalium_slider_args',
			wp_parse_args(
				$args,
				[
					'tag_name'                   => 'div',
					'id'                         => null,
					'class'                      => [],
					'attributes'                 => null,
					'slides'                     => null,
					'hide_controls'              => false,
					'theme'                      => null,
					'echo'                       => true,

					// Slider options
					'direction'                  => null,
					'navigation'                 => true,
					'navigation_container'       => false,
					'pagination'                 => null,
					'pagination_type'            => 'bullets',
					'pagination_clickable'       => true,
					'pagination_dynamic_bullets' => false,
					'space_between'              => null,
					'css_mode'                   => null,
					'autoplay'                   => null,
					'rewind'                     => null,
					'loop'                       => null,
					'looped_slides'              => null,
					'speed'                      => null,
					'slides_per_view'            => null,
					'slides_max_width'           => null,
					'effect'                     => null,
					'centered'                   => null,
					'auto_height'                => null,
					'cross_fade'                 => false,
					'keyboard'                   => null,
					'mousewheel'                 => null,

					// Thumbnails controller
					'thumbs'                     => null,
					'thumbs_offset'              => null,

					// Other options
					'sync_height'                => null,

					// A temporary prop to fix Swiper bug
					'in_grid'                    => false,
				]
			)
		);

		// Responsive props
		$responsive_props = [
			'direction'       => 'direction',
			'space_between'   => 'spaceBetween',
			'slides_per_view' => 'slidesPerView',
		];

		// Vars
		$slider_unique_id = kalium_generate_unique_string();
		$is_vertical      = 'vertical' === kalium_get_responsive_value( $args['direction'] );

		// Classes
		$classes = [
			'swiper',
			'create-slider',
		];

		// Vertical slider
		if ( $is_vertical ) {
			$classes[] = 'swiper-vertical';
		}

		// Theme preset
		if ( in_array( $args['theme'], [ 'light', 'dark' ] ) ) {
			$classes[] = 'swiper--preset-' . $args['theme'];
		}

		// Hide controls
		if ( $args['hide_controls'] ) {
			$classes[] = 'swiper--hide-controls';
		}

		// In grid
		if ( $args['in_grid'] ) {
			$classes[] = 'swiper--in-grid';
		}

		// Slider per view auto
		if ( 'auto' === $args['slides_per_view'] || 'auto' === kalium_get_responsive_value( $args['slides_per_view'] ) ) {
			$classes[] = 'swiper--auto-slide-' . ( $is_vertical ? 'height' : 'width' );
		}

		// Centered slider is initially invisible
		if ( $args['centered'] ) {
			$classes[] = 'create-slider--invisible';
		}

		if ( is_string( $args['class'] ) ) {
			$classes = array_merge( $classes, explode( ' ', $args['class'] ) );
		} elseif ( is_array( $args['class'] ) ) {
			$classes = array_merge( $classes, $args['class'] );
		}

		// RTL
		if ( is_rtl() ) {
			$classes[] = 'swiper-rtl';
		}

		// Attributes
		$attributes = [
			'class'          => $classes,
			'id'             => $args['id'],
			'data-slider-id' => $slider_unique_id,
		];

		// Extra attributes
		if ( is_array( $args['attributes'] ) ) {
			foreach ( $args['attributes'] as $attr_name => $attr_value ) {
				if ( ! isset( $attributes[ $attr_name ] ) ) {
					$attributes[ $attr_name ] = $attr_value;
				}
			}
		}

		// Slider parts
		$slider_parts = [];

		// Slider options
		$slider_options = $slider_other_options = $slider_breakpoints = [];

		// Direction
		if ( ! is_null( $args['direction'] ) ) {
			$slider_options['direction'] = $args['direction'];
		}

		// Navigation
		if ( ! is_null( $args['navigation'] ) ) {
			$slider_options['navigation'] = [
				'enabled' => (bool) $args['navigation'],
				'prevEl'  => '.swiper-button-prev',
				'nextEl'  => '.swiper-button-next',
			];

			if ( $args['navigation_container'] ) {
				$attributes['class'][] = 'swiper--navigation-container';
			}

			if ( $args['navigation'] ) {
				$slider_parts[] = '<div class="swiper-button-prev"></div>';
				$slider_parts[] = '<div class="swiper-button-next"></div>';
			}
		}

		// Pagination
		if ( ! is_null( $args['pagination'] ) ) {
			$slider_options['pagination'] = [
				'enabled'        => (bool) $args['pagination'],
				'el'             => '.swiper-pagination',
				'type'           => $args['pagination_type'],
				'clickable'      => $args['pagination_clickable'],
				'dynamicBullets' => $args['pagination_dynamic_bullets'],
			];

			$slider_parts[] = '<div class="swiper-pagination"></div>';
		}

		// Assign responsive props to slider options
		foreach ( $responsive_props as $responsive_prop => $slider_prop ) {
			if ( ! empty( $args[ $responsive_prop ] ) ) {
				$responsive_prop_value = kalium_parse_responsive_value( $args[ $responsive_prop ] );

				// Skip empty values
				if ( kalium_is_empty_responsive( $responsive_prop_value ) ) {
					continue;
				}

				// Default
				$slider_options[ $slider_prop ] = kalium_get_responsive_value( $responsive_prop_value, kalium_get_lowest_viewport_id() );

				// Add breakpoint value
				kalium_map_responsive_value(
					$responsive_prop_value,
					function ( $value, $viewport ) use ( $slider_prop, &$slider_options, &$slider_breakpoints ) {
						if ( $viewport === kalium_get_default_viewport() ) {
							$slider_breakpoints[ kalium_get_largest_viewport_max_width() ][ $slider_prop ] = $value;
						} elseif ( $min_width = kalium_get_viewport_min_width( $viewport ) ) {
							$slider_breakpoints[ $min_width ][ $slider_prop ] = $value;
						}
					},
					false
				);
			}
		}

		// CSS mode
		if ( ! is_null( $args['css_mode'] ) ) {
			$slider_options['cssMode'] = (bool) $args['css_mode'];
		}

		// Autoplay
		if ( is_numeric( $args['autoplay'] ) && 0 < $args['autoplay'] ) {
			$slider_options['autoplay'] = [
				'delay' => 100 > $args['autoplay'] ? 1000 * $args['autoplay'] : $args['autoplay'],
			];
		}

		// Rewind
		if ( ! is_null( $args['rewind'] ) ) {
			$slider_options['rewind'] = (bool) $args['rewind'];
		}

		// Loop
		if ( ! is_null( $args['loop'] ) ) {
			$slider_options['loop'] = (bool) $args['loop'];
		}

		// Looped slides
		if ( is_numeric( $args['looped_slides'] ) ) {
			$slider_options['loopAdditionalSlides'] = $args['looped_slides'];
		}

		// Speed
		if ( is_numeric( $args['speed'] ) ) {
			$slider_options['speed'] = $args['speed'];

			// Convert to milliseconds
			if ( 100 > $slider_options['speed'] ) {
				$slider_options['speed'] *= 1000;
			}
		}

		// Fade effect
		if ( 'fade' === $args['effect'] ) {
			$slider_options['effect'] = $args['effect'];

			$slider_options['fadeEffect'] = [
				'crossFade' => $args['cross_fade'],
			];
		}

		// Centered slides
		if ( ! is_null( $args['centered'] ) ) {
			$slider_options['centeredSlides'] = (bool) $args['centered'];
		}

		// Auto height
		if ( ! is_null( $args['auto_height'] ) ) {
			$slider_options['autoHeight'] = (bool) $args['auto_height'];
		}

		// Keyboard control
		if ( ! is_null( $args['keyboard'] ) ) {
			$slider_options['keyboard'] = (bool) $args['keyboard'];
		}

		// Mousewheel control
		if ( ! is_null( $args['mousewheel'] ) ) {
			$slider_options['mousewheel'] = (bool) $args['mousewheel'];
		}

		// Thumbnails controller
		if ( ! empty( $args['thumbs'] ) ) {
			$slider_other_options['thumbs'] = $args['thumbs'];

			if ( is_numeric( $args['thumbs_offset'] ) ) {
				$slider_other_options['thumbsOffset'] = $args['thumbs_offset'];
			}
		}

		// Sync height
		if ( ! empty( $args['sync_height'] ) ) {
			$slider_other_options['syncHeight'] = $args['sync_height'];
		}

		// Other options
		if ( ! empty( $slider_other_options ) ) {
			$slider_options['otherOptions'] = $slider_other_options;
		}

		// Breakpoints
		if ( ! empty( $slider_breakpoints ) ) {
			$slider_options['breakpoints'] = $slider_breakpoints;
		}

		// Add slider options attribute
		$attributes['data-slider-options'] = $slider_options;

		// Maybe wrap slide
		$maybe_wrap_slide = function ( $slide ) {
			if ( false === strpos( $slide, 'swiper-slide' ) ) {
				$slide = kalium_element( 'div', 'class=swiper-slide', $slide );
			}
			return $slide;
		};

		// Slides
		$slides = $args['slides'];

		if ( is_callable( $slides ) ) {
			ob_start();
			$slides( $args );
			$slides = ob_get_clean();
		}

		if ( is_array( $slides ) ) {
			$slides = implode(
				PHP_EOL,
				array_map(
					function ( $slide ) use ( $args, $maybe_wrap_slide ) {
						if ( is_callable( $slide ) ) {
								ob_start();
								$slide_return = $slide( $args );
								$slide        = ob_get_clean();

							if ( is_string( $slide_return ) ) {
								$slide = $slide_return;
							}
						}

						return $maybe_wrap_slide( $slide );
					},
					$slides
				)
			);
		}

		// Inline styles
		kalium_print_inline_style(
			[
				'selector' => sprintf( '[data-slider-id="%s"]', $slider_unique_id ),
				'vars'     => [
					'slider-items'     => $args['slides_per_view'],
					'slider-gap'       => kalium_map_responsive_value( $args['space_between'], 'kalium_map_pixel_unit_values' ),
					'slides-max-width' => kalium_map_responsive_value( $args['slides_max_width'], 'kalium_map_pixel_unit_values' ),
				],
			]
		);

		// Wrap slides
		$slides = kalium_element( 'div', [ 'class' => 'swiper-wrapper' ], $slides );

		// Content
		$content = kalium_element( $args['tag_name'], $attributes, $slides . implode( PHP_EOL, $slider_parts ) );

		if ( $args['echo'] ) {
			echo $content;
		} else {
			return $content;
		}
	}
}

if ( ! function_exists( 'kalium_tabs' ) ) {

	/**
	 * Tabs component.
	 *
	 * @since 4.0
	 *
	 * @since 4.0
	 */
	function kalium_tabs( $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'type'            => null,
				'entries'         => [],
				'classes'         => [],
				'active'          => null,
				'style'           => null,
				'orientation'     => null,

				// Accordion args
				'animate'         => true,
				'collapse_others' => true,
			]
		);

		// Type
		$is_accordion = 'accordion' === $args['type'];
		$is_stacked   = 'stacked' === $args['type'];

		// Class name
		$component_class_name = 'tabs';

		if ( $is_accordion ) {
			$component_class_name = 'accordion';
		} elseif ( $is_stacked ) {
			$component_class_name = 'panels-stack';
		}

		// Entries
		$entries = array_map(
			function ( $entry ) {
				return wp_parse_args(
					$entry,
					[
						'id'      => null,
						'title'   => null,
						'classes' => null,
						'content' => null,
					]
				);
			},
			$args['entries']
		);

		// Entry IDs
		$entry_ids = array_map(
			function ( $entry ) {
				return $entry['id'];
			},
			$entries
		);

		// Current/active tab
		$active_tab = in_array( $args['active'], $entry_ids ) ? $args['active'] : reset( $entry_ids );

		// Wrapper classes
		$classes = is_string( $args['classes'] ) ? explode( ' ', $args['classes'] ) : $args['classes'];

		// Component class
		array_unshift( $classes, $component_class_name );

		// Style
		$style = $args['style'] ?? 1;

		$classes[] = $component_class_name . '--style-' . $style;

		// Orientation
		$orientation = $args['orientation'];

		if ( 'vertical' === $orientation ) {
			$classes[] = $component_class_name . '--vertical';
		}

		// Accordion classes
		if ( $is_accordion ) {
			// Collapse all tabs
			if ( false === $args['active'] ) {
				$active_tab = false;
			}

			// Animate accordion
			if ( $args['animate'] ) {
				$classes[] = $component_class_name . '--animate';
			}

			// Collapse others
			if ( $args['collapse_others'] ) {
				$classes[] = $component_class_name . '--collapse-others';
			}
		}

		// Tabs wrapper start
		printf( '<div class="%s">', esc_attr( kalium_tokenize_list( $classes ) ) );

		// Accordion layout
		if ( $is_accordion ) {
			foreach ( $entries as $entry ) {
				$accordion_item_class_name     = $component_class_name . '__item';
				$accordion_toggle_class_name   = $accordion_item_class_name . '-toggle';
				$accordion_collapse_class_name = $accordion_item_class_name . '-collapse';
				$accordion_content_class_name  = $accordion_item_class_name . '-content';

				$is_active = $active_tab === $entry['id'];

				$accordion_classes = [
					$accordion_item_class_name,
				];

				// Active tab panel
				if ( $is_active ) {
					$accordion_classes[] = $accordion_item_class_name . '--active';
				}

				// Content
				$content = $entry['content'];

				if ( is_callable( $content ) ) {
					ob_start();
					call_user_func( $content );
					$content = ob_get_clean();
				}

				// Entry classes
				$entry_classes = is_string( $entry['classes'] ) ? explode( ' ', $entry['classes'] ) : $entry['classes'];

				array_unshift( $entry_classes, $accordion_content_class_name );

				// Accordion item
				printf(
					'<div class="%1$s" id="accordion-title-%2$s" aria-controls="accordion-%2$s">',
					esc_attr( kalium_tokenize_list( $accordion_classes ) ),
					esc_attr( $entry['id'] )
				);

				// Toggle
				printf(
					'<button class="%1$s" aria-expanded="%2$s"><h3 class="%1$s__title">%3$s</h3></button>',
					esc_attr( $accordion_toggle_class_name ),
					esc_attr( kalium()->helpers->aria_expanded( $is_active ) ),
					wp_kses_post( $entry['title'] )
				);

				// Content entry
				printf(
					'<div class="%4$s"><div class="%1$s" id="accordion-%2$s" aria-labelledby="accordion-title-%2$s">%3$s</div></div>',
					esc_attr( kalium_tokenize_list( $entry_classes ) ),
					esc_attr( $entry['id'] ),
					$content,
					$accordion_collapse_class_name
				);

				// Accordion item end
				echo '</div>';
			}
		}
		// Tabs layout
		else {
			// Tabs list
			if ( ! $is_stacked ) {
				printf( '<ul class="%s" role="tablist">', esc_attr( $component_class_name . '__list' ) );

				foreach ( $entries as $entry ) {
					$tab_class_name = $component_class_name . '__list__tab';

					$tab_classes = [
						$tab_class_name,
					];

					if ( $active_tab === $entry['id'] ) {
						$tab_classes[] = $tab_class_name . '--active';
					}

					// Tab entry
					printf(
						'<li class="%1$s" id="tab-title-%2$s" role="tab" aria-controls="tab-%2$s"><a href="#tab-%3$s">%4$s</a></li>',
						esc_attr( kalium_tokenize_list( $tab_classes ) ),
						esc_attr( $entry['id'] ),
						esc_attr( $entry['id'] ),
						wp_kses_post( $entry['title'] )
					);
				}

				echo '</ul>'; // End of tabs list
			}

			// Tabs content
			foreach ( $entries as $entry ) {
				$tab_panel_class_name = $component_class_name . '__panel';

				$is_active = $active_tab === $entry['id'];

				$panel_classes = [
					$tab_panel_class_name,
				];

				// Active tab panel
				if ( $is_active ) {
					$panel_classes[] = $tab_panel_class_name . '--active';
				}

				// Extra classes
				if ( ! empty( $entry['classes'] ) ) {
					$entry_classes = is_string( $entry['classes'] ) ? explode( ' ', $entry['classes'] ) : $entry['classes'];
					$panel_classes = array_merge( $panel_classes, $entry_classes );
				}

				// Content
				$content = $entry['content'];

				if ( is_callable( $content ) ) {
					ob_start();
					call_user_func( $content );
					$content = ob_get_clean();
				}

				// Panel entry
				printf(
					'<div class="%1$s" id="tab-%2$s" role="tabpanel" aria-labelledby="tab-title-%2$s">%3$s</div>',
					esc_attr( kalium_tokenize_list( $panel_classes ) ),
					esc_attr( $entry['id'] ),
					$content
				);
			}
		}

		echo '</div>'; // End of tabs wrapper
	}
}

if ( ! function_exists( 'kalium_accordion' ) ) {

	/**
	 * Accordion component.
	 *
	 * @param array $args
	 *
	 * @since 4.0
	 */
	function kalium_accordion( $args = [] ) {
		kalium_tabs(
			array_merge(
				$args,
				[
					'type' => 'accordion',
				]
			)
		);
	}
}

if ( ! function_exists( 'kalium_search_page_content_wrapper_start' ) ) {

	/**
	 * Page content wrapper start.
	 */
	function kalium_search_page_content_wrapper_start() {
		kalium_content_wrapper_start( 'search' );
	}
}

if ( ! function_exists( 'kalium_search_page_content_wrapper_end' ) ) {

	/**
	 * Page content wrapper start.
	 */
	function kalium_search_page_content_wrapper_end() {
		kalium_content_wrapper_end( 'page' );
	}
}

if ( ! function_exists( 'kalium_search_results_info' ) ) {

	/**
	 * Search results info.
	 *
	 * @since 4.0
	 */
	function kalium_search_results_info() {
		global $wp_query;

		// Search input
		ob_start();
		kalium_search_content_editable_input();
		$search_input = trim( ob_get_clean() );

		kalium_get_template(
			'search-results-info.php',
			[
				'found_posts'  => $wp_query->found_posts,
				'search_input' => $search_input,
			]
		);
	}
}

if ( ! function_exists( 'kalium_search_results' ) ) {

	/**
	 * Search results.
	 *
	 * @since 4.0
	 */
	function kalium_search_results() {
		global $wp_query;

		if ( have_posts() ) {
			$classes = [
				'search-results',
			];

			$css_vars = [
				'results-thumbnail-border-radius' => kalium_map_responsive_value( kalium_get_theme_option( 'search_thumbnail_border_radius' ), 'kalium_to_length' ),
				'results-thumbnail-width'         => kalium_get_theme_option( 'search_thumbnail_width' ),
				'custom-aspect-ratio'             => kalium()->customize->get_aspect_ratio_value( 'search_thumbnail' ),
			];

			// Print style
			kalium_print_inline_style(
				[
					'selector' => '.search-results',
					'vars'     => $css_vars,
				]
			);

			// Search results wrapper start
			echo '<div ' . kalium_class_attr( $classes, false ) . '>';

			// Search results
			while ( have_posts() ) {
				the_post();

				$post_classes = get_post_class( 'search-results__entry' );

				// Entry
				echo '<div ' . kalium_class_attr( $post_classes, false ) . '>';

				/**
				 * Search result image.
				 *
				 * @hooked kalium_search_result_thumbnail_display - 10
				 */
				do_action( 'kalium_search_result_entry_image' );

				// Entry content
				echo '<div class="search-results__entry-content">';

				/**
				 * Search result entry.
				 *
				 * @hooked kalium_search_result_title - 10
				 * @hooked kalium_search_result_excerpt_display - 20
				 */
				do_action( 'kalium_search_result_entry_content' );

				// Entry actions
				if ( has_filter( 'kalium_search_result_entry_actions' ) ) {
					printf( '<div class="%s">', kalium_tokenize_list( [ 'entry-actions', kalium_get_link_plain_class() ] ) );

					/**
					 * Entry actions.
					 *
					 * @hooked kalium_search_result_read_more_display - 10
					 */
					do_action( 'kalium_search_result_entry_actions' );

					echo '</div>';
				}

				// End of entry content
				echo '</div>';

				// End of entry
				echo '</div>';
			}

			// Search results wrapper end
			echo '</div>';

			// Pagination
			kalium_pagination(
				[
					'pagination_args' => [
						'total' => $wp_query->max_num_pages,
					],
				]
			);
		}
	}
}

if ( ! function_exists( 'kalium_search_result_thumbnail_display' ) ) {

	/**
	 * Search entry thumbnail.
	 *
	 * @since 4.0
	 */
	function kalium_search_result_thumbnail_display() {
		if ( ! kalium_get_theme_option( 'search_thumbnails' ) || ! has_post_thumbnail() ) {
			return;
		}

		$attachment_size = kalium()->customize->get_image_size_value( 'search_thumbnail' );

		?>
		<div class="search-results__entry-image post-thumbnail">
			<a href="<?php the_permalink(); ?>">
				<?php
				echo kalium_get_attachment_image( wp_get_attachment_image( get_post_thumbnail_id(), $attachment_size ) );
				?>
			</a>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_search_result_title_display' ) ) {

	/**
	 * Search entry title.
	 *
	 * @since 4.0
	 */
	function kalium_search_result_title_display() {
		printf(
			'<h2 class="%s"><a href="%s">%s</a></h2>',
			kalium_tokenize_list(
				[
					'entry-title',
					kalium_get_link_plain_class(),
				]
			),
			esc_attr( get_permalink() ),
			get_the_title()
		);
	}
}

if ( ! function_exists( 'kalium_search_result_excerpt_display' ) ) {

	/**
	 * Search entry excerpt.
	 *
	 * @since 4.0
	 */
	function kalium_search_result_excerpt_display() {
		printf(
			'<div class="%s">%s</div>',
			kalium_tokenize_list( [ 'entry-summary', kalium_get_link_style_class() ] ),
			get_the_excerpt()
		);
	}
}

if ( ! function_exists( 'kalium_search_result_read_more_display' ) ) {

	/**
	 * Search entry read more link.
	 *
	 * @since 4.0
	 */
	function kalium_search_result_read_more_display() {
		?>
		<a href="<?php the_permalink(); ?>" class="post-link"><?php printf( esc_html__( 'Continue %s', 'kalium' ), '<i class="kalium-icon-long-arrow-right"></i>' ); ?></a>
		<?php
	}
}

if ( ! function_exists( 'kalium_simple_search_form' ) ) {

	/**
	 * Simple search form.
	 *
	 * @since 4.0
	 */
	function kalium_simple_search_form( $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'placeholder' => __( 'Search site...', 'kalium' ),
			]
		);
		?>
		<form role="search" method="get" class="search-form" action="<?php echo esc_url( kalium_search_url() ); ?>">
			<label>
				<i class="kalium-icon-search"></i>
				<input type="search" class="search-field" placeholder="<?php echo esc_attr( $args['placeholder'] ); ?>" value="<?php echo get_search_query(); ?>" name="s" />
			</label>
		</form>
		<?php
	}
}

if ( ! function_exists( 'kalium_show_post_password_form' ) ) {

	/**
	 * Show post password form.
	 *
	 * @since 4.0.6
	 */
	function kalium_show_post_password_form() {
		echo get_the_password_form();
	}
}

if ( ! function_exists( 'kalium_post_navigation' ) ) {

	/**
	 * Post navigation element.
	 *
	 * @param array $args
	 *
	 * @return string|void
	 *
	 * @since 4.1.2
	 */
	function kalium_post_navigation( $args = [] ) {
		global $post;

		$args = apply_filters(
			'kalium_post_navigation_args',
			wp_parse_args(
				$args,
				[
					'layout'                 => null,
					'align_horizontal'       => null,
					'show_info'              => null,
					'loop'                   => null,
					'reverse'                => null,
					'animate'                => true,
					'archive_url'            => null,
					'image'                  => false,
					'image_align'            => null,
					'image_size'             => null,
					'image_aspect_ratio'     => null,
					'icon'                   => true,
					'icon_type'              => null,
					'title'                  => null,
					'subtitle'               => null,
					'subtitle_position'      => null,
					'subtitle_show_on_hover' => null,
					'taxonomy'               => null,
					'excluded_terms'         => null,
					'visibility'             => null,
					'label_visibility'       => null,
					'image_visibility'       => null,
					'style'                  => null,
					'aria_label'             => 'Post Navigation',
					'echo'                   => true,
				]
			)
		);

		// Layout
		$layout = kalium_enum_value(
			$args['layout'],
			[
				'type-1',
				'type-2',
				'type-3',
				'type-4',
			]
		);

		// Element ID
		$element_id = 'post-navigation-' . kalium_generate_unique_string();

		// In same term
		if ( $args['taxonomy'] && isset( $post ) ) {
			$item_object_terms = wp_get_object_terms( $post->ID, $args['taxonomy'] );

			if ( ! is_array( $item_object_terms ) || ! count( $item_object_terms ) ) {
				$args['taxonomy'] = null;
			}
		}

		// Config
		$element_config = [];

		// CSS vars
		$css_vars = [];

		// Link args
		$prev_title    = is_array( $args['title'] ) ? $args['title'][0] : $args['title'];
		$next_title    = is_array( $args['title'] ) ? $args['title'][1] : $args['title'];
		$prev_subtitle = is_array( $args['subtitle'] ) ? $args['subtitle'][0] : $args['subtitle'];
		$next_subtitle = is_array( $args['subtitle'] ) ? $args['subtitle'][1] : $args['subtitle'];

		$link_args = [
			'icon'              => $args['icon'],
			'icon_type'         => $args['icon_type'],
			'image'             => $args['image'],
			'image_align'       => $args['image_align'],
			'image_size'        => $args['image_size'],
			'subtitle_position' => $args['subtitle_position'],
			'label_visibility'  => $args['label_visibility'],
			'image_visibility'  => $args['image_visibility'],
		];

		if ( ! empty( $args['taxonomy'] ) ) {
			$link_args['loop']           = $args['loop'];
			$link_args['in_same_term']   = true;
			$link_args['taxonomy']       = $args['taxonomy'];
			$link_args['excluded_terms'] = $args['excluded_terms'];
		}

		// Attributes
		$attributes = [
			'id'         => $element_id,
			'class'      => [
				'post-navigation',
				'post-navigation--' . $layout,
			],
			'aria-label' => $args['aria_label'],
		];

		// Classes
		if ( $args['reverse'] ) {
			$attributes['class'][] = 'post-navigation--reverse';
		}

		if ( $args['archive_url'] && 'type-2' !== $args['layout'] ) {
			$attributes['class'][] = 'post-navigation--has-archive-link';
		}

		if ( $args['animate'] ) {
			if ( $args['icon'] && 'type-2' !== $args['layout'] ) {
				$element_config[] = 'animate-icon';
			}

			if ( ! empty( $args['subtitle'] ) && $args['subtitle_show_on_hover'] ) {
				$element_config[] = 'animate-subtitle';
			}

			if ( $args['archive_url'] ) {
				$element_config[] = 'animate-archive';
			}

			$attributes['class'][] = 'post-navigation--animate';
		}

		$css_vars['custom-aspect-ratio'] = $args['image_aspect_ratio'];

		if ( is_array( $args['visibility'] ) ) {
			$attributes['class'] = array_merge(
				$attributes['class'],
				kalium_get_device_visibility_classes(
					$args['visibility']
				)
			);
		}

		// Adjustments based on layout type
		switch ( $layout ) {
			case 'type-2':
				$args['archive_url'] = null;
				break;

			case 'type-3':
				$link_args['show_info'] = false;
				$attributes['class'][]  = 'post-navigation--align-' . kalium_enum_value(
					$args['align_horizontal'],
					[
						'start' ,
						'center',
						'end'   ,
					],
					'end'
				);
				break;

			case 'type-4':
				$link_args['show_info'] = $args['show_info'];
		}

		// Element config
		if ( ! empty( $element_config ) ) {
			$attributes['data-config'] = kalium_tokenize_list( $element_config );
		}

		// Style
		$style = $args['style'];

		if ( ! empty( $style ) ) {
			// Title & arrows
			$css_vars['pn-title-color']       = $style['color']['normal'] ?? null;
			$css_vars['pn-title-hover-color'] = $style['color']['hover'] ?? null;

			// Archive color
			$css_vars['pn-archive-color']       = $style['archive_color']['normal'] ?? null;
			$css_vars['pn-archive-hover-color'] = $style['archive_color']['hover'] ?? null;

			// Subtitle
			$css_vars['pn-subtitle-color']       = $style['subtitle_color']['normal'] ?? null;
			$css_vars['pn-subtitle-hover-color'] = $style['subtitle_color']['hover'] ?? null;

			// Container
			if ( in_array( $layout, [ 'type-2', 'type-4' ], true ) ) {
				// Container background
				$css_vars['pn-bg'] = $style['container_bg'] ?? null;

				// Border
				$css_vars['pn-border-width'] = kalium_map_responsive_value(
					$style['container_border'] ?? null,
					function ( $value ) {
						return $value['width'] ?? null;
					}
				);
				$css_vars['pn-border-style'] = kalium_map_responsive_value(
					$style['container_border'] ?? null,
					function ( $value ) {
						return $value['style'] ?? null;
					}
				);
				$css_vars['pn-border-color'] = kalium_map_responsive_value(
					$style['container_border'] ?? null,
					function ( $value ) {
						return $value['color'] ?? null;
					}
				);

				// Border radius
				$css_vars['pn-border-radius'] = kalium_map_responsive_value( $style['container_border_radius'] ?? null, 'kalium_to_pixel_unit' );

				// Box shadow
				$css_vars['pn-box-shadow'] = kalium_map_responsive_value( $style['container_shadow'] ?? null, 'kalium_to_box_shadow' );

				// Flip the shadow on next link
				if ( 'type-2' === $layout ) {
					$css_vars['pn-next-box-shadow'] = kalium_map_responsive_value(
						$style['container_shadow'] ?? null,
						function ( $value ) {
							$value['offset_x'] *= -1;

							return kalium_to_box_shadow( $value, '0px 0px 0px' );
						}
					);
				}
			}

			// Container height
			if ( 'type-2' === $layout ) {
				$css_vars['pn-height'] = kalium_map_responsive_value( $style['container_height'] ?? [], 'kalium_to_pixel_unit' );
			}

			// Layout type 3 options
			if ( 'type-3' === $layout ) {
				$css_vars['pn-bg']                = $style['alt_container_bg'] ?? null;
				$css_vars['pn-border-color']      = $style['alt_separator_color'] ?? null;
				$css_vars['pn-title-color']       = $style['alt_color']['normal'] ?? null;
				$css_vars['pn-title-hover-color'] = $style['alt_color']['hover'] ?? null;

				if ( ! empty( $css_vars['link-color'] ) ) {
					$css_vars['pn-archive-color'] = $css_vars['link-color'];
				}

				if ( ! empty( $css_vars['link-hover-color'] ) ) {
					$css_vars['pn-archive-hover-color'] = $css_vars['link-hover-color'];
				}
			}

			// Image
			if ( $args['image'] ) {
				$css_vars['pn-image-width']         = kalium_map_responsive_value( $style['image_width'] ?? null, 'kalium_to_pixel_unit' );
				$css_vars['pn-image-border-radius'] = kalium_map_responsive_value( $style['image_border_radius'] ?? null, 'kalium_to_length' ) ?? null;
			}
		}

		// Post navigation template
		ob_start();

		// Print vars
		kalium_print_inline_style(
			[
				'id'       => 'post-navigation',
				'selector' => '#' . $element_id,
				'vars'     => $css_vars,
			]
		);

		?>
		<nav <?php kalium_attributes( $attributes ); ?>>
			<ul class="post-navigation__list">
				<li class="post-navigation__item post-navigation__item--prev">
					<?php
					// Previous post
					kalium_post_navigation_link(
						array_merge(
							$link_args,
							[
								'image'    => $args['image'],
								'title'    => $prev_title,
								'subtitle' => $prev_subtitle,
							]
						)
					);
					?>
				</li>
				<?php if ( $args['archive_url'] ) : ?>
					<li class="post-navigation__item post-navigation__item--back-to-archive">
						<?php
						// Back to archive link
						kalium_back_to_archive_link( $args['archive_url'] );
						?>
					</li>
				<?php endif; ?>
				<li class="post-navigation__item post-navigation__item--next">
					<?php
					// Next post
					kalium_post_navigation_link(
						array_merge(
							$link_args,
							[
								'previous'        => false,
								'image'           => $args['image'],
								'title'           => $next_title,
								'subtitle'        => $next_subtitle,
								'arrow_direction' => 'end',
							]
						)
					);
					?>
				</li>
			</ul>
		</nav>
		<?php

		$output = ob_get_clean();

		// Echo output
		if ( $args['echo'] ) {
			echo $output;

			return;
		}

		return $output;
	}
}

if ( ! function_exists( 'kalium_post_navigation_link' ) ) {

	/**
	 * Post navigation link.
	 *
	 * @param array $args
	 *
	 * @return WP_Post|string|void
	 * @since 4.1.2
	 */
	function kalium_post_navigation_link( $args = [] ) {
		$args = apply_filters(
			'kalium_post_navigation_link_args',
			wp_parse_args(
				$args,
				[
					// Labels
					'show_info'          => true,
					'title'              => '',
					'subtitle'           => '',

					// Direction
					'previous'           => true,
					'loop'               => true,

					// Adjacent post args
					'in_same_term'       => false,
					'excluded_terms'     => '',
					'taxonomy'           => 'category',

					// Image
					'image'              => true,
					'image_align'        => 'start',
					'image_size'         => 'thumbnail',
					'image_aspect_ratio' => null,

					// Arrow
					'icon'               => true,
					'icon_type'          => 'type-1',
					'icon_direction'     => 'start',

					// Subtitle
					'subtitle_position'  => null,

					// Visibility
					'label_visibility'   => null,
					'image_visibility'   => null,

					// Other params
					'return_post'        => false,
					'echo'               => true,
				]
			)
		);

		// Direction
		$is_prev = $args['previous'];

		/**
		 * Filter post object to allow third-party users to modify it.
		 *
		 * @param WP_Post|null $post
		 * @param bool         $is_prev
		 * @param array        $args
		 */
		$post = apply_filters(
			'kalium_adjacent_post_object',
			get_adjacent_post(
				$args['in_same_term'],
				$args['excluded_terms'],
				$is_prev,
				$args['taxonomy']
			),
			$is_prev,
			$args
		);

		// Return post instance
		if ( $args['return_post'] ) {
			return $post;
		}

		// Title and subtitle
		$title    = kalium_replace_template_string( $args['title'], $post );
		$subtitle = kalium_replace_template_string( $args['subtitle'], $post );

		// Link
		$link = $post ? get_permalink( $post ) : false;

		// Default prev/next title
		if ( empty( $args['title'] ) ) {
			$title = kalium_conditional(
				$is_prev,
				__( 'Previous', 'kalium' ),
				__( 'Next', 'kalium' )
			);
		}

		if ( ! $title ) {
			return;
		}

		// Classes
		$classes = [
			'post-navigation__link',
			'post-navigation__link--' . ( $is_prev ? 'prev' : 'next' ),
		];

		// Link disabled
		if ( ! $link ) {
			$classes[] = 'post-navigation__link--disabled';
		}

		// Subtitle position
		if ( $subtitle && $args['subtitle_position'] ) {
			$classes[] = 'post-navigation__link--subtitle-' . kalium_enum_value( $args['subtitle_position'], [ 'above', 'below' ] );
		}

		// Icon
		if ( $args['icon'] ) {
			$classes[] = 'post-navigation__link--icon-' . $args['icon_type'];
		}

		// Label
		$label_classes = [
			'post-navigation__link-label',
		];

		if ( is_array( $args['label_visibility'] ) ) {
			$label_classes = array_merge(
				$label_classes,
				kalium_get_device_visibility_classes(
					$args['label_visibility']
				)
			);
		}

		// Image
		$image_classes = [
			'post-navigation__link-image',
		];

		$args['image'] = ! empty( $args['image'] ) && $post instanceof WP_Post && has_post_thumbnail( $post->ID );

		if ( $args['image'] ) {
			$classes[] = 'post-navigation__link--image';
			$classes[] = 'post-navigation__link--image-' . kalium_enum_value( $args['image_align'], [ 'start', 'end' ] );

			if ( is_array( $args['image_visibility'] ) ) {
				$image_classes = array_merge(
					$image_classes,
					kalium_get_device_visibility_classes(
						$args['image_visibility']
					)
				);
			}
		}

		// Link template
		ob_start();

		?>
		<a href="<?php echo esc_url( $link ); ?>" <?php kalium_class_attr( $classes ); ?>>
			<?php if ( ! empty( $args['icon'] ) ) : ?>
				<span class="post-navigation__link-icon"></span>
			<?php endif; ?>

			<?php if ( $args['show_info'] ) : ?>
			<span class="post-navigation__link-info">
				<?php if ( $args['image'] ) : ?>
					<span <?php kalium_class_attr( $image_classes ); ?>>
						<?php
						echo kalium_get_attachment_image(
							get_post_thumbnail_id( $post->ID ),
							$args['image_size']
						);
						?>
					</span>
				<?php endif; ?>

				<span <?php kalium_class_attr( $label_classes ); ?>>
					<span class="post-navigation__link-label-title">
						<?php echo esc_html( $title ); ?>
					</span>

					<?php if ( ! empty( $subtitle ) ) : ?>
						<span class="post-navigation__link-label-subtitle">
							<?php echo wp_kses_post( $subtitle ); ?>
						</span>
					<?php endif; ?>
				</span>
			</span>
			<?php endif; ?>
		</a>
		<?php

		$output = ob_get_clean();

		// Echo output
		if ( $args['echo'] ) {
			echo $output;

			return;
		}

		return $output;
	}
}

if ( ! function_exists( 'kalium_back_to_archive_link' ) ) {

	/**
	 * Back to archive link.
	 *
	 * @param string $link
	 * @param int    $boxes
	 */
	function kalium_back_to_archive_link( $link, $boxes = 2 ) {
		$classes = [
			'back-to-archive',
			'back-to-archive--boxes-' . $boxes,
		];

		?>
		<a href="<?php echo esc_url( $link ); ?>" <?php kalium_class_attr( $classes ); ?> aria-label="Back to Archive">
			<span class="back-to-archive__boxes" aria-hidden="true">
				<?php echo str_repeat( '<span></span>', pow( $boxes, 2 ) ); ?>
			</span>
		</a>
		<?php
	}
}
