<?php
/**
 * Kalium WordPress Theme
 *
 * Other template functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

if ( ! function_exists( 'kalium_wpml_language_switcher' ) ) {

	/**
	 * Header WPML Language Switcher.
	 *
	 * @param array|string
	 *
	 * @return void
	 */
	function kalium_wpml_language_switcher( $args = [] ) {
		if ( ! kalium()->is->wpml_active() ) {
			return;
		}

		// Args
		$args = wp_parse_args(
			$args,
			[
				'flag_position' => kalium_get_theme_option( 'header_wpml_language_flag_position' ),
				'display_text'  => kalium_get_theme_option( 'header_wpml_language_switcher_text_display_type' ),
				'skip_missing'  => false,
			]
		);

		// Languages list
		$languages = apply_filters(
			'wpml_active_languages',
			null,
			[
				'skip_missing' => $args['skip_missing'],
			]
		);

		// Active language
		$active_language = null;

		foreach ( $languages as $code => $language ) {
			if ( ! empty( $language['active'] ) ) {
				$active_language = $language;
				unset( $languages[ $code ] );
			}
		}

		if ( ! $active_language ) {
			$active_language = array_shift( $languages );
		}

		// Show language switcher when there are available languages
		if ( $active_language ) {
			$classes = [
				'kalium-wpml-language-switcher',
				'standard-menu',
				kalium_get_link_plain_class(),
				kalium_get_link_style_class_by_context( 'header-wpml-switcher' ),
			];
			?>
			<div <?php kalium_class_attr( $classes ); ?>>
				<ul class="menu">
					<li class="menu-item">
						<?php
							// Current language (active)
							kalium_wpml_language_switcher_item( $active_language, $args );
						?>

						<?php if ( count( $languages ) ) { ?>
							<ul class="sub-menu">
								<?php
								// Other languages
								foreach ( $languages as $language ) {
									?>
									<li class="menu-item">
										<?php
											// Language item
											kalium_wpml_language_switcher_item( $language, $args );
										?>
									</li>
									<?php
								}
								?>
							</ul>
						<?php } ?>
					</li>
				</ul>
			</div>
			<?php
		}
	}
}

if ( ! function_exists( 'kalium_wpml_language_switcher_item' ) ) {

	/**
	 * Inner function of kalium_wpml_language_switcher function.
	 *
	 * @see kalium_wpml_language_switcher()
	 */
	function kalium_wpml_language_switcher_item( $language, $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'flag_position' => '',
				'display_text'  => '',
			]
		);

		// Name and flag
		$name = $flag = null;

		// Classes
		$classes = [];

		// Code and URL
		$code = $language['code'] ?? '';
		$url  = $language['url'];

		// Flag
		if ( 'hide' !== $args['flag_position'] ) {
			$flag = $language['country_flag_url'];

			$classes[] = 'has-flag';
			$classes[] = 'flag-' . $args['flag_position'];
		}

		// Name
		$native_name     = $language['native_name'];
		$translated_name = $language['translated_name'];

		switch ( $args['display_text'] ) {
			case 'translated':
				$name = $translated_name;
				break;

			case 'initials':
				$name = strtoupper( $code );
				break;

			case 'name-translated':
				$name = "{$native_name} <span>({$translated_name})</span>";
				break;

			case 'translated-name':
				$name = "{$translated_name} <span>({$native_name})</span>";
				break;

			case 'name':
				$name = $native_name;
				break;
		}

		$allowed_html = [
			'span' => [],
		];

		?>
		<a <?php kalium_class_attr( $classes ); ?> href="<?php echo esc_url( $url ); ?>">
			<?php
			if ( $flag ) {
				printf( '<span class="flag"><img src="%s" alt="%s" /></span>', $flag, $code );
			}

			if ( $name ) {
				printf( '<span class="link">%s</span>', wp_kses( apply_filters( 'kalium_wpml_language_switcher_language_name', $name, $language ), $allowed_html ) );
			}
			?>
		</a>
		<?php
	}
}

if ( ! function_exists( 'kalium_like_button' ) ) {

	/**
	 * Like button.
	 *
	 * @param array $args
	 *
	 * @return string|void
	 *
	 * @since 3.2
	 */
	function kalium_like_button( $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'post_id'       => null,
				'icon'          => '',
				'icon_class'    => '',
				'display_count' => false,
				'class'         => '',
				'echo'          => true,
			]
		);

		$post = get_post( $args['post_id'] );

		// Post does not exist
		if ( ! $post ) {
			return;
		}

		$likes = kalium_get_post_likes( $post->ID );
		$liked = kalium_validate_boolean( $likes['liked'] );
		$count = $likes['count'];

		// Container classes
		$classes = [
			'like-button',
			'like-button--icon-' . $args['icon'],
		];

		// Liked
		if ( $liked ) {
			$classes[] = 'liked';
		}

		// Extra classes
		if ( ! empty( $args['class'] ) ) {
			$classes[] = kalium_tokenize_list( $args['class'] );
		}

		// Like icon
		$like_icon = kalium_element(
			[
				'tag_name'   => 'span',
				'attributes' => [
					'class' => [
						'like-icon',
						$args['icon_class'],
					],
				],
				'content'    => kalium_element(
					[
						'tag_name'   => 'span',
						'attributes' => [
							'class' => [
								'like-icon__icon',
								$liked ? 'like' : '',
							],
						],
						'content'    => implode(
							'',
							array_map(
								function ( $i ) {
									return '<span class="like-icon__bubble like-icon__bubble--index-' . $i . '"></span>';
								},
								range( 1, 8 )
							)
						),
					]
				),
			]
		);

		// Count
		if ( $args['display_count'] ) {
			$like_icon .= '<span class="like-button__count">' . esc_html( $count ) . '</span>';
		}

		// Like button
		$like_button = kalium_element(
			[
				'tag_name'   => 'a',
				'attributes' => [
					'href'         => '#',
					'class'        => $classes,
					'role'         => 'button',
					'aria-label'   => 'Like',
					'data-post-id' => $post->ID,
				],
				'content'    => $like_icon,
			]
		);

		// Echo output
		if ( $args['echo'] ) {
			echo $like_button;
			return;
		}

		return $like_button;
	}
}
