<?php
/**
 * Kalium WordPress Theme
 *
 * Portfolio functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Supported portfolio modules.
 *
 * @return array
 * @since 4.0
 */
function kalium_supported_portfolio_post_types() {
	return kalium()->portfolio->get_post_types();
}

/**
 * Portfolio init loop options.
 *
 * @param string $id
 * @param array  $extend_options
 *
 * @return array
 * @since 4.0
 */
function kalium_init_portfolio_loop_options( $id = null, $extend_options = [] ) {
	global $portfolio_loop_options, $wp_query;

	$pagination_settings = kalium()->customize->get_pagination_settings( 'portfolio_pagination' );

	$portfolio_loop_options = [

		// Instance ID
		'id'             => 'portfolio-items',

		// Layout
		'layout'         => kalium_get_theme_option( 'portfolio_layout' ),

		// Post type
		'post_type'      => null,

		// Archive URL
		'url'            => null,

		// Additional query vars
		'query_vars'     => null,

		// Query
		'query'          => $wp_query,

		// Grid options
		'grid'           => [

			// Items per page
			'items_per_page' => kalium_get_theme_option( 'portfolio_items_per_page' ),

			// Columns
			'columns'        => kalium_get_theme_option( 'portfolio_grid_columns' ),

			// Gap
			'gap'            => kalium_get_theme_option( 'portfolio_grid_columns_gap' ),

			// Masonry
			'masonry'        => [

				// Enabled state
				'enabled' => 'packery' === kalium_get_theme_option( 'portfolio_grid_layout_mode' ),

				// Layout mode
				'mode'    => kalium_get_theme_option( 'portfolio_grid_layout_mode' ),
			],
		],

		// Heading
		'heading'        => [

			// Layout
			'layout' => kalium_get_theme_option( 'portfolio_heading_layout' ),

			// Heading title container
			'title'  => [

				// Visibility
				'visible'  => kalium_get_theme_option( 'portfolio_show_header_title' ),

				// Tag name
				'tag_name' => 'h2',

				// Title text
				'text'     => kalium_get_theme_option( 'portfolio_title' ),

				// Description
				'content'  => kalium_format_content( kalium_get_theme_option( 'portfolio_description' ) ),
			],
		],

		// Filtering
		'filtering'      => [

			// Enabled state
			'enabled'                        => kalium_get_theme_option( 'portfolio_filters' ),

			// Filter handler
			'handler'                        => 'isotope', // [isotope,static]

			// Toggle active filters
			'toggle'                         => true,

			// Reset/any
			'reset'                          => kalium_get_theme_option( 'portfolio_filters_reset' ),

			// Reset text
			'reset_text'                     => kalium_get_theme_option( 'portfolio_filters_reset_text' ),

			// Supported taxonomies
			'taxonomies'                     => kalium_get_theme_option( 'portfolio_filters_taxonomies' ),

			// Item IDs (filled after query is executed)
			'item_ids'                       => null,

			// Terms (filled after query is executed)
			'terms'                          => null,

			// Count
			'count'                          => [

				// Visible state
				'visible' => kalium_get_theme_option( 'portfolio_filters_count' ),

				// Style
				'style'   => kalium_get_theme_option( 'portfolio_filters_count_style' ),

				// Before text
				'before'  => kalium_get_theme_option( 'portfolio_filters_count_before' ),

				// After text
				'after'   => kalium_get_theme_option( 'portfolio_filters_count_after' ),
			],

			// Sub terms
			'sub_terms'                      => kalium_get_theme_option( 'portfolio_filters_sub_terms' ),

			// History type
			'history_type'                   => kalium_get_theme_option( 'portfolio_filters_history_type' ), // [hash,pushState]

			// Dynamic page heading
			'dynamic_heading_content'        => kalium_get_theme_option( 'portfolio_filters_dynamic_heading' ),

			// Dynamic page heading content source
			'dynamic_heading_content_source' => kalium()->helpers->get_array_first( kalium_get_theme_option( 'portfolio_filters_taxonomies' ) ),
		],

		// Lightbox
		'lightbox'       => [

			// Browse mode
			'browse_mode'  => kalium_get_theme_option( 'portfolio_lb_navigation_mode' ),

			// Items (filled after query is executed)
			'items'        => null,

			// History type
			'history_type' => kalium_get_theme_option( 'portfolio_lightbox_history_type' ),

			// Captions
			'captions'     => [

				// Enabled state
				'enabled'     => true,

				// Include description
				'description' => true,
			],
		],

		// Pagination
		'pagination'     => [

			// Enabled state
			'enabled'         => in_array( $pagination_settings['type'], [ 'numbers', 'load-more', 'more-link' ] ),

			// Type
			'type'            => $pagination_settings['type'],

			// Align
			'align'           => $pagination_settings['align'],

			// Numbers display
			'numbers_display' => $pagination_settings['numbers_display'],

			// Load more pagination
			'load_more'       => [

				// Infinite scroll
				'infinite_scroll' => $pagination_settings['infinite_scroll_auto_reveal'],

				// Style
				'loading_style'   => $pagination_settings['infinite_scroll_loading_effect'],

				// Show more
				'view_more_text'  => esc_html__( 'Show more', 'kalium' ),

				// Last page text
				'last_page_text'  => esc_html__( 'No more portfolio items to show', 'kalium' ),
			],

			// More link
			'more_link'       => [

				// URL
				'url'    => null,

				// Target
				'target' => null,

				// Text
				'text'   => esc_html__( 'Show more', 'kalium' ),
			],
		],

		// Likes featured
		'likes'          => [

			// State
			'enabled' => kalium_get_theme_option( 'portfolio_likes' ),

			// Icon
			'icon'    => kalium_get_theme_option( 'portfolio_likes_icon' ),
		],

		// Item
		'item'           => [

			// Thumbnail
			'thumbnail'      => [

				// Visible
				'visible'       => true,

				// Size
				'size'          => kalium()->customize->get_image_size_value( 'portfolio_thumbnails' ),

				// Hover overlay
				'hover_overlay' => [

					// Enabled
					'enabled'         => kalium_get_theme_option( 'portfolio_thumbnail_hover_effect' ),

					// Offset
					'offset'          => kalium_get_theme_option( 'portfolio_thumbnail_hover_padding' ),

					// Color source
					'color_type'      => kalium_get_theme_option( 'portfolio_thumbnail_hover_bg_type' ),

					// Color
					'color'           => kalium_get_theme_option( 'portfolio_thumbnail_hover_bg', true ),

					// Color alpha
					'color_alpha'     => kalium_get_theme_option( 'portfolio_thumbnail_hover_bg_alpha' ),

					// Text color type
					'text_color_type' => kalium_get_theme_option( 'portfolio_thumbnail_hover_text_color_type' ),

					// Custom text color
					'text_color'      => kalium_get_theme_option( 'portfolio_thumbnail_hover_text_color' ),
				],

				// Aspect ratio
				'aspect_ratio'  => kalium()->customize->get_aspect_ratio_value( 'portfolio_thumbnails' ),

				// Border radius
				'border_radius' => kalium_map_responsive_value( kalium_get_theme_option( 'portfolio_thumbnail_border_radius', true ), 'kalium_to_length' ),
			],

			// Featured video
			'featured_video' => [

				// Enabled
				'enabled' => true,
			],

			// Title
			'title'          => [

				// Visible
				'visible'  => true,

				// Tag
				'tag_name' => 'h3',

				// Link
				'link'     => true,
			],

			// Subtitle
			'subtitle'       => [

				// Visible
				'visible'         => kalium_get_theme_option( 'portfolio_loop_subtitles' ),

				// Content type to display
				'content_type'    => kalium_get_theme_option( 'portfolio_loop_subtitles_content' ),

				// Terms separator
				'terms_separator' => kalium_get_theme_option( 'portfolio_loop_subtitles_terms_separator' ),
			],

			// Type 1
			'type-1'         => [

				// Icon
				'icon' => [

					// Visible
					'visible'  => 'none' !== kalium_get_theme_option( 'portfolio_thumbnail_hover_icon' ),

					// Icon type
					'type'     => kalium_get_theme_option( 'portfolio_thumbnail_hover_icon' ),

					// Eye icon type
					'eye_icon' => kalium_get_theme_option( 'portfolio_thumbnail_hover_icon_eye' ),

					// Custom icon
					'custom'   => [

						// Attachment id
						'attachment_id' => kalium_get_array_key( kalium_get_theme_option( 'portfolio_thumbnail_hover_icon_custom' ), 'id' ),

						// Width
						'width'         => kalium_get_theme_option( 'portfolio_thumbnail_hover_icon_custom_width' ),
					],
				],
			],

			// Type 2
			'type-2'         => [

				// Content position
				'content_position' => kalium_get_theme_option( 'portfolio_layout_2_content_position' ),
			],

			// Animate on viewport
			'aov'            => [

				// Enabled state
				'enabled'          => 'none' !== kalium_get_theme_option( 'portfolio_reveal_effect' ),

				// Legacy animation name
				'legacy_animation' => kalium_get_theme_option( 'portfolio_reveal_effect' ),

				// Animation
				'animation'        => null,

				// Duration
				'duration'         => null,

				// Stagger
				'stagger'          => null,

				// Delay
				'delay'            => null,
			],
		],

		// Custom masonry items
		'custom_masonry' => null,
	];

	// Extend/replace portfolio options
	if ( is_array( $extend_options ) ) {
		kalium_set_array_key( $portfolio_loop_options, $extend_options );

		// Filtering taxonomies should be replaced entirely
		if ( isset( $extend_options['filtering']['taxonomies'] ) ) {
			$portfolio_loop_options['filtering']['taxonomies'] = $extend_options['filtering']['taxonomies'];
		}
	}

	// Portfolio instance ID
	if ( $id ) {
		$portfolio_loop_options['id'] = sprintf( 'portfolio-items-%s', esc_attr( $id ) );
	}

	// Selector
	$portfolio_loop_options['selector'] = '.' . $portfolio_loop_options['id'];

	/** @var WP_Query $query */
	$query = & $portfolio_loop_options['query'];

	// Post type setup
	$post_type = $portfolio_loop_options['post_type'];

	if ( $query_post_type = $query->get( 'post_type' ) ) {
		$post_type = $query_post_type;
	}

	// Determine post type
	if ( empty( $post_type ) ) {

		// Determine post type based on post items
		if ( isset( $query->posts[0] ) ) {
			$post_type = $query->posts[0]->post_type;
		}
		// Get post type based on taxonomy
		elseif ( $query->is_tax() ) {
			foreach ( kalium()->portfolio->get_taxonomies() as $taxonomy ) {
				if ( $query->is_tax( $taxonomy ) ) {
					$post_type = kalium_get_array_first( get_taxonomy( $taxonomy )->object_type );
					break;
				}
			}
		}
	}

	// Set post type archive URL
	if ( empty( $portfolio_loop_options['url'] ) ) {
		$portfolio_loop_options['url'] = get_post_type_archive_link( $post_type );
	}

	// Validate post type
	$supported_post_types = kalium_supported_portfolio_post_types();

	if ( ! in_array( $post_type, $supported_post_types ) ) {
		$post_type = current( $supported_post_types );
	}

	// Set post type
	$portfolio_loop_options['post_type'] = $post_type;

	// Custom masonry
	if ( ! empty( $portfolio_loop_options['custom_masonry'] ) ) {
		$height_ratio = apply_filters(
			'kalium_portfolio_masonry_height_ratio',
			[
				's' => 1,
				'm' => 0.835,
				'l' => 0.945,
			]
		);

		$aspect_ratios = [
			'3x3'  => [ 1, 1 * $height_ratio['s'] ],
			'3x4'  => [ 2, 3 * $height_ratio['m'] ],
			'3x6'  => [ 1, 2 * $height_ratio['l'] ],

			'4x3'  => [ 4, 3 * $height_ratio['s'] ],
			'4x4'  => [ 8, 9 * $height_ratio['m'] ],
			'4x6'  => [ 2, 3 * $height_ratio['l'] ],

			'5x3'  => [ 5, 3 * $height_ratio['s'] ],
			'5x4'  => [ 10, 9 * $height_ratio['m'] ],
			'5x6'  => [ 5, 6 * $height_ratio['l'] ],

			'6x3'  => [ 2, 1 * $height_ratio['s'] ],
			'6x4'  => [ 4, 3 * $height_ratio['m'] ],
			'6x6'  => [ 1, 1 * $height_ratio['l'] ],

			'8x3'  => [ 8, 3 * $height_ratio['s'] ],
			'8x4'  => [ 16, 9 * $height_ratio['m'] ],
			'8x6'  => [ 4, 3 * $height_ratio['l'] ],

			'9x3'  => [ 3, 1 * $height_ratio['s'] ],
			'9x4'  => [ 2, 1 * $height_ratio['m'] ],
			'9x6'  => [ 3, 2 * $height_ratio['l'] ],

			'12x4' => [ 4, 1 * $height_ratio['s'] ],
			'12x5' => [ 8, 3 * $height_ratio['m'] ],
			'12x6' => [ 2, 1 * $height_ratio['l'] ],
		];

		// Map masonry entries
		$portfolio_loop_options['custom_masonry'] = array_map(
			function ( $box_entry ) use ( $aspect_ratios ) {
				$box_size     = explode( 'x', $box_entry['box_size'] );
				$post_id      = kalium_get_array_key( $box_entry, 'id' );
				$columns      = kalium_get_array_first( $box_size );
				$aspect_ratio = $aspect_ratios[ $box_entry['box_size'] ] ?? [ 1, 1 ];

				return [
					'id'           => $post_id,
					'columns'      => $columns,
					'aspect_ratio' => $aspect_ratio,
				];
			},
			$portfolio_loop_options['custom_masonry']
		);

		// Force second layout
		$portfolio_loop_options['layout'] = 'type-2';

		// Use packery masonry algorithm
		$portfolio_loop_options['grid']['masonry']['mode'] = 'packery';

		// Set query vars
		$post_ids = array_values( array_filter( wp_list_pluck( $portfolio_loop_options['custom_masonry'], 'id' ) ) );

		if ( ! empty( $post_ids ) ) {
			$portfolio_loop_options['query_vars'] = array_merge(
				$portfolio_loop_options['query_vars'] ?? [],
				[
					'post_type' => $post_type,
					'post__in'  => $post_ids,
					'orderby'   => 'post__in',
				]
			);

			// Execute query
			$portfolio_loop_options['query'] = new WP_Query( $portfolio_loop_options['query_vars'] );
		} else {
			$post_ids = get_posts(
				array_merge(
					$query->query,
					[
						'fields'   => 'ids',
						'nopaging' => true,
					]
				)
			);

			$masonry_variants     = $portfolio_loop_options['custom_masonry'];
			$masonry_variants_max = count( $masonry_variants );

			$portfolio_loop_options['custom_masonry'] = array_map(
				function ( $post ) use ( $post_ids, $masonry_variants, $masonry_variants_max ) {
					$post_id       = $post->ID;
					$masonry_index = array_search( $post_id, $post_ids ) % $masonry_variants_max;

					return array_merge(
						$masonry_variants[ $masonry_index ],
						[
							'id' => $post_id,
						]
					);
				},
				$query->posts
			);
		}
	}

	// Columns
	$columns = kalium_parse_responsive_value( kalium_get_array_key( $portfolio_loop_options, 'grid/columns' ) );

	$portfolio_loop_options['grid']['columns'] = wp_parse_args(
		$columns,
		[
			'desktop' => kalium_get_responsive_value( $columns, 'desktop', 3 ),
			'tablet'  => kalium_get_responsive_value( $columns, 'tablet', 2 ),
			'mobile'  => kalium_get_responsive_value( $columns, 'mobile', 1 ),
		]
	);

	// Taxonomy filtering
	$taxonomies = $portfolio_loop_options['filtering']['taxonomies'];

	if ( $query->is_tax( $taxonomies ) ) {
		$term = get_queried_object();

		if ( empty( $portfolio_loop_options['filtering']['current'] ) ) {
			$portfolio_loop_options['filtering']['current'][ $term->taxonomy ] = $term->slug;
		}

		// Update heading title and content in taxonomy pages
		if ( $term instanceof WP_Term ) {
			$portfolio_loop_options['heading']['title']['text'] = $term->name;

			if ( $content = $term->description ) {
				$portfolio_loop_options['heading']['title']['content'] = kalium_format_content( $content );
			}
		}
	} elseif ( ! empty( $taxonomies ) ) {
		$permalinks = kalium_get_portfolio_permalinks();

		foreach ( $taxonomies as $taxonomy ) {
			$query_var = $permalinks[ $taxonomy ] ?? $taxonomy;

			if ( isset( $wp_query->query[ $query_var ] ) ) {
				$taxonomy_term = $wp_query->get( $query_var );

				$portfolio_loop_options['filtering']['current'][ $taxonomy ] = $taxonomy_term;
			}
			// Inherit from Theme Options
			elseif ( $default_term = kalium_get_theme_option( $taxonomy . '_default' ) ) {
				if ( false === in_array( $default_term, [ 'none', 'default' ], true ) ) {
					$portfolio_loop_options['filtering']['current'][ $taxonomy ] = $default_term;
				}
			}
		}
	}

	// Active filters toggle functionality
	$portfolio_loop_options['filtering']['toggle'] = ! $portfolio_loop_options['filtering']['reset'];

	// Set Animate on Viewport from legacy animation name
	if ( $portfolio_loop_options['item']['aov']['legacy_animation'] && is_null( $portfolio_loop_options['item']['aov']['animation'] ) ) {
		switch ( $portfolio_loop_options['item']['aov']['legacy_animation'] ) {

			// Fade
			case 'fade':
			case 'fade-one':
				$legacy_aov_args = [
					'animation' => 'fade-in',
				];
				break;

			// Slide and fade
			case 'slidenfade':
			case 'slidenfade-one':
				$legacy_aov_args = [
					'animation' => 'fade-in-up',
				];
				break;

			// Zoom in
			case 'zoom':
			case 'zoom-one':
				$legacy_aov_args = [
					'animation' => 'zoom-in',
				];
				break;
		}

		if ( ! empty( $legacy_aov_args ) ) {
			$legacy_aov_args['duration'] = 3;

			if ( str_ends_with( $portfolio_loop_options['item']['aov']['legacy_animation'], '-one' ) ) {
				$legacy_aov_args['stagger'] = 2;
			}

			$portfolio_loop_options['item']['aov'] = array_merge( $portfolio_loop_options['item']['aov'], $legacy_aov_args );
		}
	}

	// Force default pagination on "load-more" pagination type
	if ( 'load-more' === $portfolio_loop_options['pagination']['type'] ) {
		$portfolio_loop_options['pagination']['numbers_display'] = null;
	}

	/**
	 * Portfolio loop options.
	 *
	 * @param array $portfolio_loop_options
	 */
	$portfolio_loop_options = apply_filters( 'kalium_portfolio_loop_options', $portfolio_loop_options );

	return $portfolio_loop_options;
}

/**
 * Single portfolio init options.
 *
 * @param int   $id
 * @param array $extend_options
 *
 * @return array
 * @since 4.0
 */
function kalium_init_portfolio_single_options( $id = null, $extend_options = [] ) {
	global $portfolio_single_options;

	if ( is_null( $id ) ) {
		$id = get_queried_object_id();
	}

	$portfolio_single_options = [

		// Instance ID
		'id'         => 'single-portfolio-item-' . $id,

		// Item id
		'item_id'    => $id,

		// Selector
		'selector'   => null,

		// Item type
		'type'       => kalium_get_field( 'item_type', $id ),

		// Title
		'title'      => [

			// Tag name
			'tag_name' => 'h1',
		],

		// Subtitle
		'subtitle'   => kalium_get_field( 'sub_title', $id ),

		// Navigation
		'navigation' => [

			// Enabled state
			'enabled' => kalium_get_theme_option( 'portfolio_navigation' ),

			// Options
			'options' => kalium()->customize->get_post_navigation_settings(
				'portfolio_navigation',
				[
					'post_type' => 'portfolio',
				]
			),
		],

		// Link
		'link'       => [

			// Title
			'title'         => kalium_get_field( 'launch_link_title', $id ),

			// URL
			'url'           => kalium_get_field( 'launch_link_href', $id ),

			// New window
			'new_window'    => kalium_get_field( 'new_window', $id ),

			// Redirect type
			'redirect_type' => kalium_get_field( 'item_linking', $id ),
		],

		// Checklist
		'checklists' => kalium_get_field( 'checklists', $id ),

		// Share
		'share'      => [

			// Enabled state
			'enabled'         => kalium_get_theme_option( 'portfolio_share_item' ),

			// Layout
			'layout'          => kalium_get_theme_option( 'portfolio_like_share_layout' ),

			// Social networks
			'social_networks' => kalium_get_theme_option( 'portfolio_share_item_networks' ),
		],

		// General gallery options
		'gallery'    => [

			// Items
			'items'      => [],

			// Image size
			'image_size' => kalium()->customize->get_image_size_value( 'portfolio_gallery' ),

			// Gap
			'gap'        => true,

			// Lightbox
			'lightbox'   => kalium_get_theme_option( 'portfolio_gallery_lightbox' ),

			// Captions
			'captions'   => [

				// Enabled state
				'enabled'  => kalium_get_theme_option( 'portfolio_gallery_captions' ),

				// Position
				'position' => kalium_get_theme_option( 'portfolio_gallery_caption_position' ),
			],
		],

		// Item details
		'item'       => [

			// Side
			'type-1' => [

				// Content
				'content' => [

					// Align
					'align'  => kalium_get_field( 'item_description_alignment', $id ),

					// Width
					'width'  => kalium_get_field( 'item_description_width', $id ),

					// Sticky
					'sticky' => kalium_get_field( 'sticky_description', $id ),
				],

				// Gallery
				'gallery' => [

					// Type
					'type'         => kalium_get_field( 'gallery_type', $id ),

					// Stick to top
					'stick_to_top' => kalium_get_field( 'gallery_stick_to_top', $id ),

					// Masonry
					'masonry'      => kalium_get_field( 'masonry_mode_gallery', $id ),
				],
			],

			// Columned
			'type-2' => [

				// Layout
				'layout'              => kalium_get_field( 'layout_type', $id ),

				// Featured image
				'featured_image'      => [

					// Enabled state
					'enabled'    => kalium_get_field( 'show_featured_image', $id ),

					// Full width
					'full_width' => kalium_get_field( 'fullwidth_featured_image', $id ),
				],

				// Gallery
				'gallery'             => [

					// Position
					'position'   => kalium_get_field( 'gallery_position', $id ),

					// Full width
					'full_width' => kalium_get_field( 'full_width_gallery', $id ),

					// Masonry
					'masonry'    => kalium_get_field( 'masonry_mode_gallery', $id ),
				],

				// Title position
				'title_position'      => kalium_get_field( 'title_position', $id ),

				// Share and like position
				'share_like_position' => kalium_get_field( 'share_and_like_position', $id ),
			],

			// Carousel
			'type-3' => [

				// Layout
				'layout' => kalium_get_field( 'layout_type', $id ),
			],

			// Zig Zag
			'type-4' => [

				// Layout
				'layout'              => kalium_get_field( 'layout_type', $id ),

				// Share and like position
				'share_like_position' => kalium_get_field( 'share_and_like_position', $id ),
			],

			// Full screen
			'type-5' => [

				// Content
				'content' => [

					// Enabled state
					'enabled'                => ! kalium_get_field( 'hide_project_description', $id ),

					// Toggle state
					'toggle_state'           => kalium_get_field( 'item_description_visibility', $id ),

					// Toggle text
					'show_project_info_text' => kalium_get_field( 'show_project_info_text', $id ),
				],
			],

			// Lightbox
			'type-6' => [

				// Featured image
				'featured_image' => [

					// Enabled state
					'enabled' => true,

					// Overlay text
					'overlay' => [

						// Enabled state
						'enabled' => true,

						// Text
						'text'    => __( 'Open in Lightbox', 'kalium' ),

						// Icon
						'icon'    => 'kalium-icon-zoom',
					],
				],
			],
		],
	];

	// Post navigation adjustments
	if ( 'type-5' === $portfolio_single_options['type'] && $portfolio_single_options['navigation']['enabled'] ) {
		if ( ! in_array( $portfolio_single_options['navigation']['options']['layout'], [ 'type-2', 'type-3' ] ) ) {
			$portfolio_single_options['navigation']['options']['layout'] = 'type-3';
		}

		if ( 'type-3' === $portfolio_single_options['navigation']['options']['layout'] ) {
			$portfolio_single_options['navigation']['options']['align_horizontal'] = 'end';
		}
	}

	// Custom archive URL
	if ( kalium_get_field( 'custom_prevnext_links', $id ) && ( $custom_archive_url = kalium_get_field( 'custom_archive_url', $id ) ) ) {
		$portfolio_single_options['navigation']['options']['archive_url'] = $custom_archive_url;
	}

	// Gallery
	$gallery = & $portfolio_single_options['gallery'];

	// Common options
	if ( in_array( $portfolio_single_options['type'], [ 'type-1', 'type-2', 'type-3', 'type-4' ] ) ) {
		// Reveal effect
		switch ( kalium_get_field( 'images_reveal_effect', $id ) ) {
			case 'fade':
				$aov_animation = 'fade-in';
				break;

			case 'slidenfade':
				$aov_animation = 'fade-in-up';
				break;
		}

		if ( isset( $aov_animation ) ) {
			$gallery['reveal_effect'] = kalium_animate_on_viewport(
				[
					'animation' => $aov_animation,
					'duration'  => 3,
					'stagger'   => 2,
				]
			);
		}

		// Gap
		$gallery['gap'] = kalium_get_field( 'gallery_columns_gap', $id );

		if ( 'nospacing' === kalium_get_field( 'image_spacing', $id ) ) {
			$gallery['gap'] = 0;
		}

		// Captions position
		if ( in_array( $captions_position = kalium_get_field( 'image_captions_position', $id ), [ 'hover', 'below' ] ) ) {
			$gallery['captions']['position'] = $captions_position;
		}
	}

	switch ( $portfolio_single_options['type'] ) {

		// Side, Columned and Zig Zag type options
		case 'type-1':
		case 'type-2':
		case 'type-4':
			// Gallery items
			if ( is_array( $gallery_items = kalium_get_field( 'gallery', $id ) ) ) {
				$gallery['items'] = array_values(
					array_filter(
						array_map(
							function ( $item ) use ( &$portfolio_single_options ) {
								$fc_layout  = $item['acf_fc_layout'];
								$image_size = kalium_get_array_key( $item, 'image_size' );

								switch ( $fc_layout ) {

									// Image
									case 'image':
										$item_entry = [
											'type'       => 'image',
											'image_id'   => kalium_get_array_key( $item, 'image/id' ),
											'image_size' => $image_size,
										];
										break;

									// Images slider
									case 'images_slider':
										$item_entry = [
											'type'        => 'images',
											'images'      => wp_list_pluck( $item['images'], 'id' ),
											'image_size'  => $image_size,
											'auto_switch' => is_numeric( $item['auto_switch'] ) ? absint( $item['auto_switch'] ) : 0,
										];
										break;

									// Before/after
									case 'comparison_images':
										$item_entry = [
											'type'       => 'comparison',
											'left_image_id' => kalium_get_array_key( $item, 'image_1/id' ),
											'right_image_id' => kalium_get_array_key( $item, 'image_2/id' ),
											'image_size' => $image_size,
										];
										break;

									// YouTube video
									case 'selfhosted_video':
									case 'youtube_video':
									case 'vimeo_video':
										$video_types = [
											'selfhosted_video' => 'self-hosted',
											'youtube_video' => 'youtube',
											'vimeo_video' => 'vimeo',
										];

										$item_entry = [
											'type'         => 'video',
											'video_type'   => $video_types[ $fc_layout ] ?? null,
											'source'       => kalium_get_array_key( $item, 'video_url', kalium_get_array_key( $item, 'video_file/url' ) ),
											'aspect_ratio' => kalium_get_array_key( $item, 'video_resolution' ),
											'poster'       => kalium_get_array_key( $item, 'video_poster/id' ),
											'autoplay'     => kalium_get_array_key( $item, 'auto_play' ),
											'loop'         => kalium_get_array_key( $item, 'loop' ),
										];
										break;

									// Quote
									case 'text_quote':
										$item_entry = [
											'type'         => 'quote',
											'quote'        => kalium_get_array_key( $item, 'quote_text' ),
											'quote_author' => kalium_get_array_key( $item, 'quote_author' ),
										];
										break;

									// HTML
									case 'html':
										$item_entry = [
											'type'    => 'html',
											'content' => trim( kalium_get_array_key( $item, 'content' ) ),
										];
										break;
								}

								if ( isset( $item_entry ) ) {

									// Description
									if ( isset( $item['description'] ) ) {
										$item_entry = array_merge(
											$item_entry,
											[
												'description' => kalium_get_array_key( $item, 'description' ),
												'description_width' => kalium_get_array_key( $item, 'description_width' ),
												'description_align' => kalium_get_array_key( $item, 'description_alignment' ),
											]
										);
									}

									// Column width
									if ( ! empty( $item['column_width'] ) ) {
										$item_entry['column_width'] = $item['column_width'];
									}

									return $item_entry;
								}

								return false;
							},
							$gallery_items
						)
					)
				);
			}
			break;

		// Carousel
		case 'type-3':
			// Max height
			$gallery['max_height'] = kalium_get_field( 'maximum_image_height', $id );

			if ( is_numeric( $gallery['max_height'] ) ) {
				$gallery['max_height'] .= 'px';
			}

			// Centered slider
			$gallery['slider_centered'] = 'center' === kalium_get_field( 'carousel_start_position', $id );

			// Loop slider
			$gallery['slider_loop'] = kalium_validate_boolean( kalium_get_field( 'infinite_loop_slides', $id ) );

			// Loop slider
			$gallery['slider_autoplay'] = kalium_get_field( 'auto_play', $id );

			// Gallery items
			if ( is_array( $gallery_items = kalium_get_field( 'gallery', $id ) ) ) {
				$gallery['items'] = array_values(
					array_filter(
						array_map(
							function ( $item ) use ( &$portfolio_single_options ) {
								$fc_layout = $item['acf_fc_layout'];

								switch ( $fc_layout ) {

									// Image
									case 'image':
										$item_entry = [
											'type'       => 'image',
											'image_id'   => kalium_get_array_key( $item, 'image/id' ),
											'image_size' => kalium_get_array_key( $item, 'image_size' ),
											'link'       => kalium_get_array_key( $item, 'link_url' ) ?: false,
											'new_tab'    => kalium_get_array_key( $item, 'link_target' ),
										];
										break;
								}

								if ( isset( $item_entry ) ) {
									return $item_entry;
								}

								return false;
							},
							$gallery_items
						)
					)
				);
			}
			break;

		// Full screen
		case 'type-5':
			// Loop slider
			$gallery['slider_autoplay'] = kalium_get_field( 'auto_play', $id );

			// Gallery items
			if ( is_array( $gallery_items = kalium_get_field( 'gallery', $id ) ) ) {
				$gallery['items'] = array_values(
					array_filter(
						array_map(
							function ( $item ) use ( &$portfolio_single_options ) {
								$fc_layout = $item['acf_fc_layout'];

								switch ( $fc_layout ) {

									// Image
									case 'image':
										$item_entry = [
											'type'       => 'image',
											'image_id'   => kalium_get_array_key( $item, 'image/id' ),
											'image_size' => kalium_get_array_key( $item, 'image_size' ),
											'link'       => false,
										];
										break;

									// Video
									case 'video':
										$item_entry = [
											'type'       => 'video',
											'video_type' => null,
											'source'     => null,
											'poster'     => kalium_get_array_key( $item, 'video_poster' ),
											'autoplay'   => kalium_get_array_key( $item, 'autoplay' ),
											'muted'      => kalium_get_array_key( $item, 'mute' ),
											'controls'   => kalium_get_array_key( $item, 'controls' ),
											'loop'       => kalium_get_array_key( $item, 'loop' ),
											'object_fit' => kalium_get_array_key( $item, 'viewport_fit' ) ? 'cover' : 'auto',
										];

										if ( $item_entry['autoplay'] ) {
											$item_entry['muted'] = true;
										}

										if ( $source = kalium_get_array_key( $item, 'video_src/url' ) ) {
											$item_entry['video_type'] = 'self-hosted';
										} elseif ( $source = kalium_get_array_key( $item, 'youtube_url' ) ) {
											$item_entry['video_type'] = 'youtube';
										}

										if ( $item_entry['video_type'] && ! empty( $source ) ) {
											$item_entry['source'] = $source;
										}
										break;
								}

								if ( isset( $item_entry ) ) {
									return $item_entry;
								}

								return false;
							},
							$gallery_items
						)
					)
				);
			}
			break;

		// Lightbox
		case 'type-6':
			$gallery['lightbox'] = kalium_prepare_portfolio_lightbox_entry( get_post( $id ) );
			break;
	}

	// Extend/replace portfolio options
	if ( is_array( $extend_options ) ) {
		kalium_set_array_key( $portfolio_single_options, $extend_options );
	}

	// Selector
	$portfolio_single_options['selector'] = '.' . $portfolio_single_options['id'];

	// Execute hooks
	$portfolio_single_options = apply_filters( 'kalium_portfolio_single_options', $portfolio_single_options );

	return $portfolio_single_options;
}

/**
 * Reset portfolio options.
 *
 * @since 4.0
 */
function kalium_reset_portfolio_loop_options() {
	global $portfolio_loop_options;

	$portfolio_loop_options = [];
}

/**
 * Reset portfolio single options.
 *
 * @since 4.0
 */
function kalium_reset_portfolio_single_options() {
	global $portfolio_single_options;

	$portfolio_single_options = [];
}

/**
 * Get portfolio loop option.
 *
 * @param string $name
 *
 * @return mixed
 * @since 4.0
 */
function kalium_get_portfolio_loop_option( $name ) {
	global $portfolio_loop_options;

	if ( empty( $portfolio_loop_options ) ) {
		kalium_init_portfolio_loop_options();
	}

	// Single use value
	if ( isset( kalium()->portfolio->_loop_once_values[ $name ] ) ) {
		$single_use_value = kalium()->portfolio->_loop_once_values[ $name ];
		unset( kalium()->portfolio->_loop_once_values[ $name ] );

		return $single_use_value;
	}

	return kalium_get_array_key( $portfolio_loop_options, $name );
}

/**
 * Update portfolio loop option.
 *
 * @param string|array $name
 * @param mixed        $new_value
 * @param bool         $once
 *
 * @since 4.0
 */
function kalium_set_portfolio_loop_option( $name, $new_value = null, $once = false ) {
	global $portfolio_loop_options;

	// Single use value
	if ( $once ) {
		kalium()->portfolio->_loop_once_values[ $name ] = $new_value;
	} else {
		kalium()->helpers->set_array_key( $portfolio_loop_options, $name, $new_value );
	}
}

/**
 * Get portfolio single option.
 *
 * @param string $name
 *
 * @return mixed
 * @since 4.0
 */
function kalium_get_portfolio_single_option( $name ) {
	global $portfolio_single_options;

	if ( empty( $portfolio_single_options ) ) {
		kalium_init_portfolio_single_options();
	}

	return kalium_get_array_key( $portfolio_single_options, $name );
}

/**
 * Update portfolio single option.
 *
 * @param string|array $name
 * @param mixed        $new_value
 *
 * @since 4.0.7
 */
function kalium_set_portfolio_single_option( $name, $new_value ) {
	global $portfolio_single_options;

	kalium()->helpers->set_array_key( $portfolio_single_options, $name, $new_value );
}

/**
 * Get portfolio loop query args.
 *
 * @param array $extend_query_args
 * @param bool  $inherit_query_vars
 *
 * @return array
 * @since 4.0
 */
function kalium_get_portfolio_loop_query_args( $extend_query_args = [], $inherit_query_vars = true ) {

	/** @var WP_Query $query */
	$query = kalium_get_portfolio_loop_option( 'query' );

	// Query args
	$query_args = $inherit_query_vars ? $query->query : [];

	// Taxonomies
	$taxonomies = kalium_get_portfolio_loop_option( 'filtering/taxonomies' );

	// Unset "pagename" and "name"
	unset( $query_args['pagename'], $query_args['name'] );

	// Unset any taxonomy var
	foreach ( $taxonomies as $taxonomy ) {
		if ( isset( $query_args[ $taxonomy ] ) ) {
			$has_taxonomies = true;

			unset( $query_args[ $taxonomy ] );
		}
	}

	// Pagination
	$paged = $GLOBALS['wp_query']->query['paged'] ?? null;

	if ( is_numeric( $paged ) ) {
		$query_args['paged'] = absint( $paged );
	}

	// Use defined query args in the loop options
	if ( is_array( $loop_query_args = kalium_get_portfolio_loop_option( 'query_vars' ) ) ) {
		foreach ( $loop_query_args as $key => $value ) {
			if ( isset( $query_args[ $key ] ) && is_array( $query_args[ $key ] ) ) {
				$query_args[ $key ] = array_merge( $query_args[ $key ], $value );
			} else {
				$query_args[ $key ] = $value;
			}
		}
	}

	// Add post type query arg
	if ( empty( $query_args['post_type'] ) ) {
		$query_args['post_type'] = kalium_get_portfolio_loop_option( 'post_type' );
	}

	// Extend query args
	if ( ! empty( $extend_query_args ) ) {
		foreach ( $extend_query_args as $key => $value ) {
			if ( isset( $query_args[ $key ] ) && is_array( $query_args[ $key ] ) && is_array( $value ) ) {
				$query_args[ $key ] = array_merge( $query_args[ $key ], $value );
			} else {
				$query_args[ $key ] = $value;
			}
		}
	}

	return $query_args;
}

/**
 * Get portfolio loop terms.
 *
 * @return array
 * @since 4.0
 */
function kalium_get_portfolio_loop_terms() {
	global $portfolio_loop_options, $wpdb;

	// Use already initialized/filled terms
	if ( isset( $portfolio_loop_options['filtering']['terms'] ) ) {
		return $portfolio_loop_options['filtering']['terms'];
	}

	/**
	 * Get child posts from terms tree.
	 *
	 * @param WP_Term $parent_term
	 * @param WP_Term[] $terms
	 *
	 * @return int[]
	 */
	$get_child_posts = static function ( $parent_term, &$terms ) use ( &$get_child_posts ) {
		$child_posts = $parent_term->posts;

		foreach ( $terms as $term ) {
			if ( $parent_term->term_id === $term->parent ) {
				$child_posts = array_merge( $child_posts, $get_child_posts( $term, $terms ) );
			}
		}

		return array_values( array_unique( $child_posts ) );
	};

	/** @var WP_Query $query */
	$query = kalium_get_portfolio_loop_option( 'query' );

	// Post IDs
	$post_ids = wp_list_pluck( $query->posts, 'ID' );

	// For load-more pagination, we should fetch all posts of the query
	if ( kalium_get_portfolio_loop_option( 'pagination/enabled' ) && 'load-more' === kalium_get_portfolio_loop_option( 'pagination/type' ) ) {
		$query = new WP_Query();

		$post_ids = $query->query(
			kalium_get_portfolio_loop_query_args(
				[
					'fields'         => 'ids',
					'posts_per_page' => -1,
				]
			)
		);
	}

	// Fill term IDs
	$portfolio_loop_options['filtering']['item_ids'] = $post_ids;

	// Taxonomies
	$taxonomies = kalium_get_portfolio_loop_option( 'filtering/taxonomies' );

	// Taxonomy terms array
	$portfolio_loop_options['filtering']['terms'] = [];

	foreach ( $taxonomies as $taxonomy ) {
		$terms = get_terms(
			[
				'taxonomy'   => $taxonomy,
				'hide_empty' => false,
			]
		);

		if ( is_wp_error( $terms ) ) {
			continue;
		}

		$term_relationships = $wpdb->get_results(
			sprintf(
				'SELECT t1.* FROM %s t1 INNER JOIN %s t2 USING(term_taxonomy_id) WHERE t1.object_id IN (%s) AND t2.term_id IN (%s)',
				$wpdb->term_relationships,
				$wpdb->term_taxonomy,
				implode( ',', array_merge( [ 0 ], $post_ids ) ),
				implode( ',', array_merge( [ 0 ], wp_list_pluck( $terms, 'term_id' ) ) )
			)
		);

		// Assign term posts
		foreach ( $terms as $term ) {
			$term->posts = [];

			foreach ( $term_relationships as $term_relationship ) {
				if ( $term->term_taxonomy_id === (int) $term_relationship->term_taxonomy_id ) {
					$term->posts[] = (int) $term_relationship->object_id;
				}
			}
		}

		// Merge child posts and update counts
		foreach ( $terms as $term ) {
			$term->posts = $get_child_posts( $term, $terms );
			$term->count = count( $term->posts );
		}

		// Filter empty terms
		$terms = array_values(
			array_filter(
				$terms,
				function ( $term ) {
					return 0 !== $term->count;
				}
			)
		);

		$portfolio_loop_options['filtering']['terms'][ $taxonomy ] = array_map( 'kalium_prepare_portfolio_term_entry', $terms );
	}

	return $portfolio_loop_options['filtering']['terms'];
}

/**
 * Get portfolio loop lightbox items.
 *
 * @return array
 * @since 4.0
 */
function kalium_get_portfolio_loop_lightbox_items() {
	global $portfolio_loop_options;

	// Use already initialized lightbox items
	if ( isset( $portfolio_loop_options['lightbox']['items'] ) ) {
		return $portfolio_loop_options['lightbox']['items'];
	}

	// Get lightbox items
	$query = new WP_Query();

	$items = $query->query(
		kalium_get_portfolio_loop_query_args(
			[
				'posts_per_page'   => -1,
				'suppress_filters' => false, // WPML fix to prevent other languages to include
				'meta_query'       => [
					[
						'key'      => 'item_type',
						'operator' => '=',
						'value'    => 'type-6',
					],
				],
			]
		)
	);

	// Fill lightbox items
	$portfolio_loop_options['lightbox']['items'] = array_values(
		array_filter(
			array_map(
				'kalium_prepare_portfolio_lightbox_entry',
				$items
			)
		)
	);

	return $portfolio_loop_options['lightbox']['items'];
}

/**
 * Prepare WP term object to array.
 *
 * @param WP_Term $term
 *
 * @return array
 * @since 4.0
 */
function kalium_prepare_portfolio_term_entry( $term ) {
	return [
		'id'          => $term->term_id,
		'parent'      => $term->parent,
		'slug'        => $term->slug,
		'name'        => $term->name,
		'description' => kalium_format_content( term_description( $term ) ),
		'count'       => $term->count,
		'posts'       => $term->posts,
		'url'         => get_term_link( $term, $term->taxonomy ),
	];
}

/**
 * Prepare lightbox item to array.
 *
 * @param WP_Post $post
 *
 * @return array
 * @since 4.0
 */
function kalium_prepare_portfolio_lightbox_entry( $post ) {
	static $sizes, $get_image, $get_video;

	// Protected items
	if ( post_password_required( $post ) ) {
		return null;
	}

	$item = [
		'id'          => $post->ID,
		'slug'        => $post->post_name,
		'title'       => get_the_title( $post ),
		'description' => kalium_format_content( $post->post_content ),
		'url'         => get_permalink( $post ),
		'entries'     => [],
	];

	if ( is_null( $sizes ) || is_null( $get_image ) || is_null( $get_video ) ) {

		// Image sizes
		$sizes = kalium_get_lightbox_image_sizes();

		/**
		 * Get image.
		 *
		 * @param int $attachment_id
		 *
		 * @return array|false
		 */
		$get_image = static function ( $attachment_id ) use ( &$sizes ) {
			$main      = wp_get_attachment_image_src( $attachment_id, $sizes['main'] );
			$thumbnail = wp_get_attachment_image_src( $attachment_id, $sizes['thumbnail'] );

			if ( $main && $thumbnail ) {
				return [
					'type'      => 'image',
					'id'        => $attachment_id,
					'src'       => $main[0],
					'thumbnail' => $thumbnail[0],
				];
			}

			return false;
		};

		/**
		 * Get video.
		 *
		 * @param string $type
		 * @param string $url
		 *
		 * @return array|false
		 */
		$get_video = static function ( $type, $url ) use ( &$post, &$sizes, &$get_image, &$featured_image ) {
			$src = null;

			switch ( $type ) {
				case 'youtube':
					if ( kalium()->media->is_youtube( $url ) ) {
						$src = $url;
					}
					break;

				case 'vimeo':
					if ( kalium()->media->is_vimeo( $url ) ) {
						$src = $url;
					}
					break;

				case 'video':
					if ( kalium()->media->is_video( $url ) ) {
						$src = $url;
					}
					break;
			}

			// Video poster
			if ( $video_poster = kalium_get_field( 'video_poster', $post->ID ) ) {
				$image  = $get_image( $video_poster );
				$poster = $image['thumbnail'] ?? null;
			} else {
				$poster = $featured_image['thumbnail'] ?? null;
			}

			if ( $src ) {
				return [
					'type'   => 'video',
					'src'    => $src,
					'poster' => $poster,
				];
			}

			return false;
		};
	}

	// Featured image
	$featured_image = $get_image( get_post_thumbnail_id( $post ) );

	switch ( kalium_get_field( 'content_to_show', $post->ID ) ) {

		// Other Image
		case 'other-image':
			if ( $image = $get_image( kalium_get_field( 'custom_image', $post->ID ) ) ) {
				$item['entries'][] = $image;
			}
			break;

		// YouTube video
		case 'youtube':
			$video_url = kalium_get_field( 'youtube_video_url', $post->ID );

			if ( $video = $get_video( 'youtube', $video_url ) ) {
				$item['entries'][] = $video;
			}
			break;

		// Vimeo video
		case 'vimeo':
			$video_url = kalium_get_field( 'vimeo_video_url', $post->ID );

			if ( $video = $get_video( 'vimeo', $video_url ) ) {
				$item['entries'][] = $video;
			}
			break;

		// Self-hosted video
		case 'self-hosted-video':
			$video_url = kalium_get_field( 'self_hosted_video', $post->ID );

			if ( $video = $get_video( 'video', $video_url['url'] ?? null ) ) {
				$item['entries'][] = $video;
			}
			break;

		// Mix of images and videos
		case 'gallery':
			$gallery = kalium_get_field( 'image_and_video_gallery', $post->ID );

			if ( is_array( $gallery ) ) {
				foreach ( $gallery as $gallery_item ) {
					$id = $gallery_item['ID'] ?? $gallery_item['id'];

					// Image
					if ( 'image' === $gallery_item['type'] ) {
						if ( $image = $get_image( $id ) ) {
							$item['entries'][] = $image;
						}
					}
					// Video
					elseif ( 'video' === $gallery_item['type'] ) {
						if ( $video = $get_video( 'video', $gallery_item['url'] ) ) {
							$item['entries'][] = $video;
						}
					}
				}
			}
			break;
	}

	// Add featured image as lightbox item entry
	if ( empty( $item['entries'] ) && $featured_image ) {
		$item['entries'][] = $featured_image;
	}

	// Likes
	if ( kalium_get_portfolio_loop_option( 'likes/enabled' ) ) {
		$item['likes'] = kalium_get_post_likes( $post->ID );
	}

	return apply_filters( 'kalium_portfolio_lightbox_item', $item, $post );
}

/**
 * Gets portfolio container options to use in Portfolio_Items_Container JS instance.
 *
 * @return array
 * @since 4.0
 */
function kalium_get_portfolio_container_js_options() {

	// Initialize filtering terms
	kalium_get_portfolio_loop_terms();

	// Initialize lightbox items
	kalium_get_portfolio_loop_lightbox_items();

	return [
		'url'        => kalium_get_portfolio_loop_option( 'url' ),
		'heading'    => kalium_get_portfolio_loop_option( 'heading' ),
		'filtering'  => kalium_get_portfolio_loop_option( 'filtering' ),
		'pagination' => kalium_get_portfolio_loop_option( 'pagination' ),
		'lightbox'   => kalium_get_portfolio_loop_option( 'lightbox' ),
		'likes'      => kalium_get_portfolio_loop_option( 'likes' ),
	];
}

/**
 * Gets portfolio single item options to use in
 *
 * @return array
 * @since 4.0
 */
function kalium_get_portfolio_single_js_options() {
	return [
		'type'     => kalium_get_portfolio_single_option( 'type' ),
		'lightbox' => kalium_get_portfolio_single_option( 'gallery/lightbox' ),
	];
}

/**
 * Get portfolio permalinks structure.
 *
 * @return array
 * @since 4.0
 */
function kalium_get_portfolio_permalinks() {
	return wp_parse_args(
		get_option( 'kalium_portfolio_permalinks' ),
		[
			'portfolio_category' => 'portfolio-category',
			'portfolio_tag'      => 'portfolio-tag',
		]
	);
}

/**
 * Parse portfolio filters.
 *
 * @param array $filters
 *
 * @return array
 * @since 4.0
 */
function kalium_parse_portfolio_filters( $filters ) {
	$parsed = [];

	if ( is_array( $filters ) ) {
		$taxonomies = kalium_get_portfolio_loop_option( 'filtering/taxonomies' );

		// Parse allowed taxonomies
		foreach ( $taxonomies as $taxonomy ) {
			if ( isset( $filters[ $taxonomy ] ) ) {
				$taxonomy_filters = $filters[ $taxonomy ];

				// Split OR terms
				if ( is_string( $taxonomy_filters ) ) {
					$taxonomy_filters = array_map( 'trim', explode( ',', $taxonomy_filters ) );
				}

				// Inner split AND terms
				if ( ! empty( $taxonomy_filters ) ) {
					$taxonomy_filters = array_map(
						function ( $terms ) {
							if ( is_string( $terms ) ) {
								$terms = array_map( 'trim', explode( '+', $terms ) );
							}

							return $terms;
						},
						$taxonomy_filters
					);
				}

				// Remove non-string terms
				$taxonomy_filters = array_map(
					function ( $terms ) {
						// String only
						$terms = is_array( $terms ) ? array_filter( $terms, 'is_string' ) : [];

						return array_values( array_filter( $terms ) );
					},
					$taxonomy_filters
				);

				// Add to parsed list
				$parsed[ $taxonomy ] = $taxonomy_filters;
			}
		}
	}

	return $parsed;
}

/**
 * Get portfolio item type based option.
 *
 * @param string $name
 *
 * @return mixed
 * @since 4.0
 */
function kalium_get_portfolio_single_item_type_option( $name ) {
	return kalium_get_portfolio_single_option( 'item/' . kalium_get_portfolio_single_option( 'type' ) . '/' . $name );
}

/**
 * Get total number of portfolio items in the loop.
 *
 * @return int
 * @since 4.1.2
 */
function kalium_get_portfolio_loop_total_posts() {
	$terms = kalium_get_portfolio_loop_terms();
	$total = wp_list_pluck( kalium_get_portfolio_loop_option( 'query' )->posts, 'ID' );

	foreach ( $terms as $taxonomy_terms ) {
		$total = array_merge(
			$total,
			call_user_func_array(
				'array_merge',
				wp_list_pluck(
					$taxonomy_terms,
					'posts'
				)
			)
		);
	}

	return count( array_unique( $total ) );
}
