<?php
/**
 * Kalium WordPress Theme
 *
 * Starter Sites Type - WordPress Widgets class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Importer;

use WP_Term;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Import_WordPress_Widgets extends Import_Type {

	/**
	 * Get content pack name.
	 *
	 * @return string
	 */
	public function get_name() {
		return 'Widgets';
	}

	/**
	 * Backup widgets.
	 *
	 * @return void
	 */
	public function do_backup() {
		$backup_manager = $this->get_content_pack()->backup_manager();

		// Backup current widgets
		$backup_manager->update_backup_option( 'widgets', wp_get_sidebars_widgets() );
	}

	/**
	 * Import WordPress widgets.
	 *
	 * @return void
	 */
	public function do_import() {
		parent::do_import();

		// Do not run if there are errors reported or option is unchecked
		if ( $this->errors->has_errors() || ! $this->is_checked() ) {
			return;
		}

		// Widget importer functions
		if ( ! function_exists( '_wie_available_widgets' ) ) {
			require_once __DIR__ . '/plugins/widget-importer-exporter.php';
		}

		// Vars
		$content_pack    = $this->get_content_pack();
		$import_manager  = $content_pack->import_manager();
		$import_instance = $content_pack->get_import_instance();

		// Delete widgets
		$sidebars_widgets = wp_get_sidebars_widgets();

		foreach ( $sidebars_widgets as $sidebar => $widgets ) {
			if ( in_array( $sidebar, [ 'wp_inactive_widgets', 'array_version' ] ) ) {
				continue;
			}

			unset( $sidebars_widgets[ $sidebar ] );
		}

		wp_set_sidebars_widgets( $sidebars_widgets );

		// Loop through each source
		foreach ( $this->get_sources() as $source ) {

			// WordPress Widgets
			$widgets_file = $import_manager->get_content_pack_import_source_path( $source['name'] );

			// Check if widgets file exists
			if ( kalium()->filesystem->exists( $widgets_file ) ) {
				$widgets = kalium()->helpers->json_decode(
					file_get_contents( $widgets_file ),
					true,
					512,
					JSON_NUMERIC_CHECK
				);

				// Import WordPress Widgets
				_wie_import_data( (object) $widgets );

				// Mark as successful import
				$import_instance->set_import_success();
			} else {

				// Widgets file doesn't exists
				$this->errors->add( 'kalium_importer_wordpress_widgets_not_exists', 'Widgets file doesn\'t exists!' );
			}
		}

		// Add errors to import instance
		if ( $this->errors->has_errors() ) {
			$import_instance->add_error( $this->errors );
		}
	}

	/**
	 * Adjust widgets settings and map menu ids.
	 *
	 * @return void
	 */
	public function do_complete() {
		$options = $this->get_options();

		// Nav menus
		if ( $widget_nav_menu_data  = kalium_get_array_key( $options, 'widget_nav_menu', [] ) ) {
			$widget_nav_menu       = get_option( 'widget_nav_menu' );
			$widget_nav_menu_terms = [];

			// Get matching term ids
			foreach ( $widget_nav_menu_data as $widget_instance ) {
				$term = Helpers::get_term_by( null, 'name', $widget_instance['name'], 'nav_menu' );

				if ( $term instanceof WP_Term ) {
					$widget_nav_menu_terms[ $widget_instance['term_id'] ] = $term->term_id;
				}
			}

			// Update nav menu values
			if ( ! empty( $widget_nav_menu_terms ) ) {
				foreach ( $widget_nav_menu as & $widget_instance ) {
					$nav_menu = kalium_get_array_key( $widget_instance, 'nav_menu' );

					if ( isset( $widget_nav_menu_terms[ $nav_menu ] ) ) {
						$widget_instance['nav_menu'] = $widget_nav_menu_terms[ $nav_menu ];
					}
				}

				// Save widget nav menu
				update_option( 'widget_nav_menu', $widget_nav_menu );
			}
		}

		// Mark as successful task
		parent::do_complete();
	}

	/**
	 * Remove imported widgets.
	 *
	 * @return void
	 */
	public function do_remove() {
		$backup_manager = $this->get_content_pack()->backup_manager();

		// Restore widgets
		if ( $widgets = $backup_manager->get_backup_option( 'widgets' ) ) {
			wp_set_sidebars_widgets( $widgets );

			$backup_manager->delete_backup_option( 'widgets' );
		}

		// Mark as removed
		parent::do_remove();
	}
}
