<?php
/**
 * Legacy Header Button element.
 */

namespace Laborator_Builder\Elements;

use Laborator_Builder\Attribute;
use Laborator_Builder\Predefined_Attributes;

class Legacy_Header_Button extends Legacy_Header_Base_Element {

	/**
	 * Element name.
	 *
	 * @var string
	 * @static
	 */
	public static $name = 'legacy_header_button';

	/**
	 * Title.
	 *
	 * @var string
	 * @static
	 */
	public static $title = 'Button';

	/**
	 * Dynamic title.
	 *
	 * @var string
	 * @static
	 */
	public static $dynamic_title = '{{ data.attributes.label }}';

	/**
	 * Icon.
	 *
	 * @var string
	 * @static
	 */
	public static $icon = 'button';

	/**
	 * Create attributes.
	 *
	 * @return Attribute[]
	 */
	public function create_attributes() {
		return array_merge(
			parent::create_attributes(),
			[
				// Content / Content
				Attribute::tab_section(
					Attribute::TAB_CONTENT,
					Attribute::SECTION_CONTENT,
					[
						Attribute::create(
							'label',
							[
								'type'         => 'text',
								'title'        => 'Label',
								'inline_label' => true,
								'default'      => 'Button',
							]
						),
						Attribute::create(
							'link',
							[
								'type'         => 'text',
								'title'        => 'Link',
								'inline_label' => true,
								'placeholder'  => 'https://',
							]
						),
						Attribute::create(
							'new_tab',
							[
								'type'         => 'toggle',
								'title'        => 'Open in New Tab',
								'inline_label' => true,
							]
						),
					]
				),

				// Style / Button
				Attribute::tab_section(
					Attribute::TAB_STYLE,
					'Color',
					[

						Attribute::create_tabbed(
							[
								'Normal' => [
									Attribute::create(
										'button_bg_color',
										[
											'type'         => 'color',
											'title'        => 'Background',
											'inline_label' => true,
											'separator'    => false,
											'reset'        => true,
											'default'      => kalium_color_reference( 'button_bg.normal' ),
										]
									),
									Attribute::create(
										'button_text_color',
										[
											'type'         => 'color',
											'title'        => 'Text',
											'inline_label' => true,
											'separator'    => false,
											'reset'        => true,
											'default'      => kalium_color_reference( 'button_color.normal' ),
										]
									),
								],
								'Hover'  => [
									Attribute::create(
										'button_hover_bg_color',
										[
											'type'         => 'color',
											'title'        => 'Background',
											'inline_label' => true,
											'separator'    => false,
											'reset'        => true,
											'default'      => kalium_color_reference( 'button_bg.hover' ),
										]
									),
									Attribute::create(
										'button_hover_text_color',
										[
											'type'         => 'color',
											'title'        => 'Text',
											'inline_label' => true,
											'separator'    => false,
											'reset'        => true,
											'default'      => kalium_color_reference( 'button_color.hover' ),
										]
									),
								],
								'Active' => [
									Attribute::create(
										'button_active_bg_color',
										[
											'type'         => 'color',
											'title'        => 'Background',
											'inline_label' => true,
											'separator'    => false,
											'reset'        => true,
											'default'      => kalium_color_reference( 'button_bg.active' ),
										]
									),
									Attribute::create(
										'button_active_text_color',
										[
											'type'         => 'color',
											'title'        => 'Text',
											'inline_label' => true,
											'separator'    => false,
											'reset'        => true,
											'default'      => kalium_color_reference( 'button_color.active' ),
										]
									),
								],
							]
						),
					]
				),

				// Style / Button
				Attribute::tab_section(
					Attribute::TAB_STYLE,
					'Fill',
					[
						Attribute::create(
							'fill',
							[
								'type'         => 'radio',
								'title'        => 'Fill Type',
								'inline_label' => true,
								'choices'      => [
									'solid'   => 'Solid',
									'outline' => 'Outline',
								],
								'buttons'      => true,
							]
						),
					],
				),

				// Advanced / Layout
				Attribute::tab_section(
					Attribute::TAB_ADVANCED,
					'Layout',
					[
						Attribute::create(
							'padding',
							[
								'type'       => 'multi-numeric',
								'title'      => 'Padding',
								'responsive' => true,
								'reset'      => true,
								'units'      => kalium_get_default_units( 'box-size' ),
							]
						),
					]
				),

				// Advanced / Border
				Attribute::tab_section(
					Attribute::TAB_ADVANCED,
					'Border',
					[
						Attribute::create( Predefined_Attributes::$BORDER_RADIUS ),
					]
				),
			]
		);
	}

	/**
	 * Element content.
	 */
	public function element_content() {
		$attributes = [
			'class' => [
				'button',
			],
		];

		// Rel
		$rel = [];

		// Vars
		$label   = $this->get_attribute_value( 'label' );
		$link    = $this->get_attribute_value( 'link' );
		$new_tab = $this->get_attribute_value( 'new_tab' );
		$fill    = $this->get_attribute_value( 'fill' );

		// Label
		if ( ! trim( $label ) ) {
			$label = 'Button';
		}

		// Link
		$attributes['href'] = $link ?? '#';

		// New tab
		if ( $new_tab ) {
			$attributes['target'] = '_blank';

			$rel[] = 'noopener';
		}

		// Rel attribute
		if ( ! empty( $rel ) ) {
			$attributes['rel'] = implode( ' ', $rel );
		}

		// Fill type
		if ( 'outline' === $fill ) {
			$attributes['class'][] = 'button-outline';
		}

		// Show button
		echo kalium_element( 'a', $attributes, $label );
	}

	/**
	 * Generate styles.
	 */
	public function generate_styles() {
		parent::generate_styles();

		// Color
		$this->add_style(
			[
				// Button element
				'.button' => [

					// Background
					kalium_css_var_name( 'button-bg' )    => $this->get_attribute_value( 'button_bg_color' ),
					kalium_css_var_name( 'button-hover-bg' ) => $this->get_attribute_value( 'button_hover_bg_color' ),
					kalium_css_var_name( 'button-active-bg' ) => $this->get_attribute_value( 'button_active_bg_color' ),

					// Text
					kalium_css_var_name( 'button-color' ) => $this->get_attribute_value( 'button_text_color' ),
					kalium_css_var_name( 'button-hover-color' ) => $this->get_attribute_value( 'button_hover_text_color' ),
					kalium_css_var_name( 'button-active-color' ) => $this->get_attribute_value( 'button_active_text_color' ),

					// Border radius
					kalium_css_var_name( 'button-border-radius' ) => [
						'value'  => $this->get_attribute( 'border_radius' ),
						'parser' => 'length-inline',
					],

					// Padding
					kalium_css_var_name( 'button-padding' ) => [
						'value'  => $this->get_attribute( 'padding' ),
						'parser' => 'length-inline',
					],
				],
			]
		);
	}
}

// Register element
\Laborator_Builder::register_element_type( Legacy_Header_Button::class );
