<?php
/**
 * Menu trigger.
 */
namespace Laborator_Builder\Elements;

use Laborator_Builder\Attribute;

class Legacy_Header_Menu_Trigger extends Legacy_Header_Base_Element {

	/**
	 * Element name.
	 *
	 * @var string
	 * @static
	 */
	public static $name = 'legacy_header_menu_trigger';

	/**
	 * Title.
	 *
	 * @var string
	 * @static
	 */
	public static $title = 'Menu Trigger';

	/**
	 * Icon.
	 *
	 * @var string
	 * @static
	 */
	public static $icon = 'menuTrigger';

	/**
	 * Create attributes.
	 *
	 * @return Attribute[]
	 */
	public function create_attributes() {
		return array_merge(
			parent::create_attributes(),
			[
				Attribute::tab_section(
					Attribute::TAB_CONTENT,
					Attribute::SECTION_CONTENT,
					[
						Attribute::create(
							'trigger_type',
							[
								'type'         => 'select',
								'title'        => 'Menu Type',
								'inline_label' => true,
								'choices'      => [
									'standard'        => 'Standard',
									'fullscreen'      => 'Fullscreen',
									'off-canvas-side' => 'Off-Canvas Side',
									'off-canvas-top'  => 'Off-Canvas Top',
								],
							]
						),
						Attribute::create(
							'menu_id',
							[
								'type'         => 'select',
								'title'        => 'Menu',
								'inline_label' => true,
								'choices'      => array_merge(
									[
										[
											'value' => 'main-menu',
											'label' => '- Main Menu -',
										],
									],
									kalium_header_get_menu_choices(),
								),
							]
						),
					]
				),
				Attribute::tab_section(
					Attribute::TAB_CONTENT,
					'Menu Options',
					[
						// Standard Menu
						Attribute::create(
							'standard_menu_position',
							[
								'type'         => 'select',
								'title'        => 'Position',
								'inline_label' => true,
								'choices'      => [
									'left'  => 'Left',
									'right' => 'Right',
									'start' => 'Beginning of column',
									'end'   => 'End of column',
								],
								'default'      => 'start',
								'dependency'   => [
									'trigger_type' => 'standard',
								],
							]
						),
						Attribute::create(
							'standard_menu_animation',
							[
								'type'         => 'select',
								'title'        => 'Animation',
								'inline_label' => true,
								'choices'      => [
									'fade'         => 'Fade',
									'slide-left'   => 'Slide from left',
									'slide-right'  => 'Slide from right',
									'slide-top'    => 'Slide from top',
									'slide-bottom' => 'Slide from bottom',
									'scale'        => 'Scale',
								],
								'dependency'   => [
									'trigger_type' => 'standard',
								],
							]
						),
						Attribute::create(
							'standard_menu_animation_direction',
							[
								'type'         => 'select',
								'title'        => 'Animate Direction',
								'inline_label' => true,
								'choices'      => [
									'all'   => 'All at once',
									'left'  => 'From left',
									'right' => 'From right',
								],
								'default'      => 'right',
								'dependency'   => [
									'trigger_type' => 'standard',
								],
							]
						),
						Attribute::create(
							'standard_menu_dropdown_caret',
							[
								'type'         => 'toggle',
								'title'        => 'Submenu Arrow',
								'inline_label' => true,
								'dependency'   => [
									'trigger_type' => 'standard',
								],
							]
						),
						Attribute::create(
							'standard_menu_item_spacing',
							[
								'type'             => 'range',
								'title'            => 'Menu Item Spacing',
								'reset'            => true,
								'units'            => kalium_get_default_units( 'box-size' ),
								'initial_position' => 30,
								'placeholder'      => 30,
								'min'              => 0,
								'max'              => 100,
								'dependency'       => [
									'trigger_type' => 'standard',
								],
							]
						),

						// Fullscreen Menu
						Attribute::create(
							'fullscreen_alignment',
							[
								'type'         => 'select',
								'title'        => 'Content Align',
								'inline_label' => true,
								'choices'      => [
									'left'                => 'Left',
									'centered'            => 'Centered',
									'centered-horizontal' => 'Horizontally Centered',
								],
								'dependency'   => [
									'trigger_type' => 'fullscreen',
								],
							]
						),
						Attribute::create(
							'fullscreen_search',
							[
								'type'         => 'toggle',
								'title'        => 'Search Field',
								'inline_label' => true,
								'default'      => true,
								'dependency'   => [
									'trigger_type' => 'fullscreen',
								],
							]
						),

						// Off-Canvas Side Menu
						Attribute::create(
							'off_canvas_side_show_widgets',
							[
								'type'         => 'toggle',
								'title'        => 'Widgets',
								'inline_label' => true,
								'default'      => true,
								'dependency'   => [
									'trigger_type' => 'off-canvas-side',
								],
							]
						),
						Attribute::create(
							'off_canvas_side_alignment',
							[
								'type'         => 'radio',
								'title'        => 'Alignment',
								'inline_label' => true,
								'choices'      => [
									'left'  => [
										'icon' => 'justifyLeft',
									],
									'right' => [
										'icon' => 'justifyRight',
									],
								],
								'buttons'      => [
									'icon_size' => 18,
								],
								'default'      => 'right',
								'dependency'   => [
									'trigger_type' => 'off-canvas-side',
								],
							]
						),

						// Off-Canvas Top Menu
						Attribute::create(
							'off_canvas_top_items_per_row',
							[
								'type'         => 'quantity',
								'title'        => 'Items Per Row',
								'inline_label' => true,
								'min'          => 1,
								'max'          => 10,
								'default'      => 3,
								'dependency'   => [
									'trigger_type' => 'off-canvas-top',
								],
							]
						),
						Attribute::create(
							'off_canvas_top_widgets',
							[
								'type'         => 'toggle',
								'title'        => 'Widgets',
								'inline_label' => true,
								'default'      => true,
								'dependency'   => [
									'trigger_type' => 'off-canvas-top',
								],
							]
						),
						Attribute::create(
							'off_canvas_top_widgets_per_row',
							[
								'type'         => 'quantity',
								'title'        => 'Columns',
								'inline_label' => true,
								'min'          => 1,
								'max'          => 4,
								'default'      => 3,
								'dependency'   => [
									'trigger_type' => 'off-canvas-top',
									'off_canvas_top_widgets' => true,
								],
							]
						),
						Attribute::create(
							'off_canvas_top_widgets_container_width',
							[
								'type'         => 'select',
								'title'        => 'Container',
								'inline_label' => true,
								'choices'      => [
									'col-3' => '25% of row width',
									'col-4' => '33% of row width',
									'col-5' => '40% of row width',
									'col-6' => '50% of row width',
									'col-7' => '60% of row width',
									'col-8' => '65% of row width',
								],
								'default'      => 'col-6',
								'dependency'   => [
									'trigger_type' => 'off-canvas-top',
									'off_canvas_top_widgets' => true,
								],
							]
						),
					]
				),
			]
		);
	}

	/**
	 * Get element class list.
	 *
	 * @return array
	 */
	public function get_dom_class() {
		$classes = [];

		// Unregistered attributes
		$unregistered_attributes = $this->get_option( 'unregistered_attributes' );

		if ( ! empty( $unregistered_attributes['mobile_menu_toggle'] ) ) {
			$classes[] = 'header-block__item--mobile-menu-toggle';
		}

		return array_merge( parent::get_dom_class(), $classes );
	}

	/**
	 * Element content.
	 */
	public function element_content() {
		switch ( $this->get_attribute_value( 'trigger_type' ) ) {

			// Fullscreen menu
			case 'fullscreen':
				kalium_toggle_bars(
					[
						'action' => 'fullscreen-menu',
					]
				);
				break;

			// Off-Canvas side menu
			case 'off-canvas-side':
				kalium_toggle_bars(
					[
						'action' => 'sidebar-menu',
					]
				);
				break;

			// Off-Canvas top menu
			case 'off-canvas-top':
				kalium_toggle_bars(
					[
						'action' => 'top-menu',
					]
				);
				break;

			// Standard menu
			default:
				$this->standard_menu_toggle();
		}
	}

	/**
	 * Standard menu toggle.
	 */
	public function standard_menu_toggle() {
		$menu_id             = $this->get_attribute_value( 'menu_id' );
		$menu_position       = $this->get_attribute_value( 'standard_menu_position' );
		$animation           = $this->get_attribute_value( 'standard_menu_animation' );
		$animation_direction = $this->get_attribute_value( 'standard_menu_animation_direction' );
		$submenu_arrow       = $this->get_attribute_value( 'standard_menu_dropdown_caret' );

		// Unregistered attributes
		$unregistered_attributes = $this->get_option( 'unregistered_attributes' );

		$classes = [
			'standard-toggle-menu',
			'standard-toggle-menu--position-' . $menu_position,
		];

		// Menu
		$menu_classes = [
			'standard-toggle-menu--col',
			'standard-toggle-menu--menu',
			'standard-menu',
			kalium_get_link_style_class_by_context( 'standard-menu' ),
		];

		// CSS vars
		$menu_css_vars = [];

		// Toggle ID
		$toggle_id        = $unregistered_attributes['toggle_id'] ?? null;
		$is_mobile_toggle = $unregistered_attributes['mobile_menu_toggle'] ?? null;

		// Item spacing
		$menu_item_spacing = kalium_to_unit_value( $this->get_attribute_value( 'standard_menu_item_spacing' ) );

		if ( ! empty( $menu_item_spacing ) ) {
			$menu_css_vars['menu-item-spacing'] = $menu_css_vars['pill-menu-item-spacing'] = $menu_item_spacing;
		}

		// Style vars
		kalium_print_inline_style(
			[
				'selector' => '.' . $this->get_common_class() . ' .standard-menu',
				'vars'     => $menu_css_vars,
			]
		);

		// Wrapper start
		printf( '<div %1$s data-toggle-effect="%2$s" data-stagger-direction="%3$s" data-toggle-id="%4$s">', kalium_class_attr( $classes, false ), $animation, $animation_direction, $toggle_id );

		// Menu items
		if ( empty( $is_mobile_toggle ) && ( in_array( $menu_position, [ 'left', 'right' ] ) || $toggle_id ) ) {
			printf( '<div %s>', kalium_class_attr( $menu_classes, false ) );
			echo kalium_nav_menu(
				[
					'menu'            => $menu_id,
					'dropdown_toggle' => $submenu_arrow,
				]
			);
			echo '</div>';
		}

		// Toggle menu button
		if ( empty( $toggle_id ) ) {
			echo '<div class="standard-toggle-menu--col standard-toggle-menu--button">';

			kalium_toggle_bars(
				[
					'action'     => 'standard-menu',
					'attributes' => [
						'data-toggle-id' => $unregistered_attributes['trigger_menu_id'] ?? null,
					],
				]
			);

			echo '</div>';
		}

		// Wrapper end
		echo '</div>';
	}
}

// Register element
\Laborator_Builder::register_element_type( Legacy_Header_Menu_Trigger::class );
