<?php
/**
 * WooCommerce Product Add to cart element.
 */

namespace Laborator_Builder\Elements;

use Laborator_Builder\Element;
use Laborator_Builder\Attribute;
use Laborator_Builder\Attribute_Sets\Advanced_Border_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Custom_CSS_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Dimensions_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Layout_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Text_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Visibility_Hover_Set;
use Laborator_Builder\Attribute_Sets\Product_Common_Attributes_Set;

class WooCommerce_Product_Add_To_Cart extends Element {

	/**
	 * Name.
	 *
	 * @var string
	 */
	public static $name = 'woocommerce_product_add_to_cart';

	/**
	 * Title.
	 *
	 * @var string
	 */
	public static $title = 'Add to Cart';

	/**
	 * Icon.
	 *
	 * @var string
	 */
	public static $icon = 'addToCart';

	/**
	 * Maximum instances of element.
	 *
	 * @var int
	 * @static
	 */
	public static $max_instances = 1;

	/**
	 * Check element visibility.
	 *
	 * @return bool
	 */
	public function is_visible() {
		// Catalog mode
		if ( kalium_woocommerce_is_catalog_mode() ) {
			return false;
		}

		return parent::is_visible();
	}

	/**
	 * Get attribute sets.
	 *
	 * @return array
	 */
	public function get_attribute_sets() {
		return array_merge(
			parent::get_attribute_sets(),
			[
				Advanced_Layout_Set::class,
				Advanced_Visibility_Hover_Set::class,
				Advanced_Border_Set::class,
				Advanced_Text_Set::class,
				Advanced_Dimensions_Set::class,
				Advanced_Custom_CSS_Set::class,
				Product_Common_Attributes_Set::class,
			]
		);
	}

	/**
	 * Attributes.
	 *
	 * @return Attribute[]
	 */
	public function create_attributes() {
		$attributes = array_merge(
			parent::create_attributes(),
			[
				// Content / Structure
				Attribute::tab_section(
					Attribute::TAB_CONTENT,
					'Structure',
					[
						Attribute::create(
							'display_type',
							[
								'type'         => 'radio',
								'title'        => 'Display type',
								'inline_label' => true,
								'choices'      => [
									'button' => 'Button',
									'link'   => 'Link',
								],
								'buttons'      => true,
							]
						),
						Attribute::create(
							'structure',
							[
								'type'         => 'select',
								'title'        => 'Structure',
								'inline_label' => true,
								'separator'    => false,
								'choices'      => [
									'icon-text' => 'Icon and text',
									'icon'      => 'Icon only',
									'text'      => 'Text only',
								],
							]
						),
						Attribute::create(
							'tooltip_hover',
							[
								'type'         => 'toggle',
								'title'        => 'Tooltip on hover',
								'inline_label' => true,
								'dependency'   => [
									'structure' => 'icon',
								],
							]
						),
						Attribute::create(
							'tooltip_added',
							[
								'type'              => 'toggle',
								'title'             => 'Tooltip on add to ' . ( $this->is_wishlist_element() ? 'wishlist' : 'cart' ),
								'inline_label'      => true,
								'separator'         => false,
								'tooltip_placement' => 'top-end',
								'dependency'        => [
									'structure' => 'icon',
								],
							]
						),
						Attribute::create(
							'tooltip_placement',
							[
								'type'         => 'select',
								'title'        => 'Tooltip placement',
								'inline_label' => true,
								'separator'    => false,
								'choices'      => [
									'top'    => 'Top',
									'bottom' => 'Bottom',
									'left'   => 'Left',
									'right'  => 'Right',
								],
								'dependency'   => [
									[
										'name'  => 'structure',
										'value' => 'icon',
									],
									[
										'name'     => 'tooltip_hover',
										'value'    => true,
										'relation' => 'OR',
									],
									[
										'name'  => 'tooltip_added',
										'value' => true,
									],
								],
							]
						),
					]
				),

				// Style / Button
				Attribute::tab_section(
					Attribute::TAB_STYLE,
					'Button',
					[
						Attribute::create_tabbed(
							[
								'Normal' => [
									Attribute::create(
										'button_background_color',
										[
											'type'         => 'color',
											'title'        => 'Background',
											'inline_label' => true,
											'separator'    => false,
											'enable_clear' => true,
											'dependency'   => [
												'display_type' => 'button',
											],
										]
									),
									Attribute::create(
										'button_text_color',
										[
											'type'         => 'color',
											'title'        => 'Text',
											'inline_label' => true,
											'enable_clear' => true,
											'separator'    => false,
										]
									),
								],
								'Hover'  => [
									Attribute::create(
										'button_background_color_hover',
										[
											'type'         => 'color',
											'title'        => 'Background',
											'inline_label' => true,
											'enable_clear' => true,
											'separator'    => false,
											'dependency'   => [
												'display_type' => 'button',
											],
										]
									),
									Attribute::create(
										'button_text_color_hover',
										[
											'type'         => 'color',
											'title'        => 'Text',
											'inline_label' => true,
											'enable_clear' => true,
											'separator'    => false,
										]
									),
								],
							]
						),
						Attribute::create(
							'button_border_radius',
							[
								'name'       => 'margin',
								'type'       => 'multi-numeric',
								'title'      => 'Border radius',
								'units'      => kalium_get_default_units( 'box-size' ),
								'default'    => [
									'link' => true,
								],
								'dependency' => [
									'display_type' => 'button',
								],
							]
						),
						Attribute::create(
							'button_block',
							[
								'type'         => 'toggle',
								'title'        => 'Full width button',
								'inline_label' => true,
								'dependency'   => [
									'display_type' => 'button',
								],
							]
						),
					]
				),
			]
		);

		// Supported by add to cart button
		if ( ! $this->is_wishlist_element() ) {
			$attributes = array_merge(
				$attributes,
				[

					// Content / Icon
					Attribute::tab_section(
						Attribute::TAB_CONTENT,
						'Icon',
						[
							Attribute::create(
								'button_icon',
								[
									'type'       => 'radio',
									'title'      => 'Add to cart icon',
									'choices'    => [
										'icon-1' => [
											'icon' => 'kalium-icon-plus',
										],
										'icon-2' => [
											'icon' => 'kalium-icon-shopping-bag',
										],
										'icon-3' => [
											'icon' => 'kalium-icon-shopping-cart',
										],
										'icon-4' => [
											'icon' => 'kalium-icon-shopping-basket',
										],
										'custom' => [
											'label' => 'Custom',
										],
									],
									'buttons'    => [
										'icon_size' => 15,
									],
									'dependency' => [
										'structure' => [
											'value'   => [ 'icon-text', 'icon' ],
											'compare' => 'IN',
										],
									],
								]
							),
							Attribute::create(
								'button_icon_svg',
								[
									'type'            => 'media',
									'title'           => 'Upload SVG icon',
									'select_label'    => 'Upload SVG',
									'library'         => [
										'type' => 'image/svg+xml',
									],
									'background_size' => '25%',
									'tooltip'         => 'Preview / media gallery',
									'dependency'      => [
										'button_icon' => 'custom',
									],
								]
							),
							Attribute::create(
								'button_icon_svg_size',
								[
									'type'       => 'range',
									'title'      => 'Icon size',
									'separator'  => false,
									'min'        => 8,
									'max'        => 100,
									'dependency' => [
										'button_icon'     => 'custom',
										'button_icon_svg' => [
											'value'   => null,
											'compare' => '!=',
										],
									],
								]
							),
						]
					),
				]
			);
		}

		return $attributes;
	}

	/**
	 * Generate styles.
	 */
	public function generate_styles() {
		parent::generate_styles();

		$element_class = $this->is_wishlist_element() ? '.add-to-wishlist' : '.add-to-cart';
		$is_button     = 'button' === $this->get_attribute_value( 'display_type' );
		$has_icon      = in_array( $this->get_attribute_value( 'structure' ), [ 'icon-text', 'icon' ] );
		$has_svg_icon  = $has_icon && 'custom' === $this->get_attribute_value( 'button_icon' ) && ! empty( $this->get_attribute_value( 'button_icon_svg' ) );

		// Font size and padding is applied to child element
		$this->remove_style(
			'&',
			[
				'padding-top',
				'padding-right',
				'padding-bottom',
				'padding-left',
				'font-size',
			]
		);

		// Element style
		$this->add_style(
			[
				// Link/button
				$element_class            => [
					'color'         => $this->get_attribute( 'button_text_color' ),
					'padding'       => [
						'value'  => $this->get_attribute( 'padding' ),
						'parser' => 'length',
					],
					'border-radius' => [
						'value'  => $this->get_attribute( 'button_border_radius' ),
						'parser' => 'length',
					],
					'font-size'     => $this->get_attribute( 'font_size' ),
				],

				// Link/button on hover
				$element_class . ':hover' => [
					'color' => $this->get_attribute( 'button_text_color_hover' ),
				],
			]
		);

		// Background applied only when its button
		if ( $is_button ) {
			$this->add_style(
				[
					$element_class            => [
						'background-color' => $this->get_attribute( 'button_background_color' ),
					],
					$element_class . ':hover' => [
						'background-color' => $this->get_attribute( 'button_background_color_hover' ),
					],
				]
			);

			// Text align
			$text_align = $this->get_attribute_value( 'text_align' );

			if ( in_array( $text_align, [ 'start', 'center', 'end' ] ) ) {
				$this->add_style(
					[
						$element_class => [
							'justify-content' => $text_align,
						],
					]
				);
			}
		}

		// SVG icon fill
		if ( $has_svg_icon ) {
			$this->add_style(
				[
					'.cart-icon svg'     => [
						'fill' => $this->get_attribute( 'button_text_color' ),
					],
					'.add-to-cart:hover' => [
						'fill' => $this->get_attribute( 'button_text_color_hover' ),
					],
				]
			);
		}
	}

	/**
	 * Check if its wishlist element.
	 *
	 * @return bool
	 */
	public function is_wishlist_element() {
		return $this instanceof WooCommerce_Product_Wishlist;
	}
}

// Register element
\Laborator_Builder::register_element_type( WooCommerce_Product_Add_To_Cart::class );
