<?php
/**
 * Product container.
 */

namespace Laborator_Builder\Elements;

use Laborator_Builder\Element;
use Laborator_Builder\Attribute;
use Laborator_Builder\Attribute_Sets\Advanced_Border_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Custom_CSS_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Dimensions_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Layout_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Text_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Visibility_Hover_Set;
use Laborator_Builder\Attribute_Sets\Product_Common_Attributes_Set;
use Laborator_Builder\Predefined_Attributes;

class WooCommerce_Product_Hover_Block extends Element {

	/**
	 * Name.
	 *
	 * @var string
	 */
	public static $name = 'woocommerce_product_hover_block';

	/**
	 * Title.
	 *
	 * @var string
	 */
	public static $title = 'Hover Block';

	/**
	 * Icon.
	 *
	 * @var string
	 */
	public static $icon = 'hoverBlock';

	/**
	 * Maximum nesting level.
	 *
	 * @var int
	 * @static
	 */
	public static $max_nesting_level = 0;

	/**
	 * Maximum instances of element.
	 *
	 * @var int
	 * @static
	 */
	public static $max_instances = 1;

	/**
	 * Child container.
	 *
	 * @var bool
	 * @static
	 */
	public static $child_container = true;

	/**
	 * Allowed child elements.
	 *
	 * @var bool
	 * @static
	 */
	public static $allowed_child_elements = [
		'!woocommerce_product_column',
	];

	/**
	 * Get attribute sets.
	 *
	 * @return array
	 */
	public function get_attribute_sets() {
		return array_merge(
			parent::get_attribute_sets(),
			[
				Advanced_Layout_Set::class,
				Advanced_Visibility_Hover_Set::class,
				Advanced_Border_Set::class,
				Advanced_Text_Set::class,
				Advanced_Dimensions_Set::class,
				Advanced_Custom_CSS_Set::class,
				Product_Common_Attributes_Set::class,
			]
		);
	}

	/**
	 * Attributes.
	 *
	 * @return Attribute[]
	 */
	public function create_attributes() {
		return [
			// Style / Layer Container
			Attribute::tab_section(
				Attribute::TAB_STYLE,
				'Layer Container',
				[
					Attribute::create(
						'background_color',
						[
							'type'         => 'color',
							'title'        => 'Background color',
							'inline_label' => true,
							'default'      => '#ffffff',
						]
					),
					Attribute::create(
						'box_shadow_color',
						[
							'type'         => 'color',
							'title'        => 'Box shadow',
							'inline_label' => true,
							'default'      => '#0000001a',
						]
					),
					Attribute::create(
						'box_shadow_offset_x',
						[
							'type'       => 'range',
							'title'      => 'Offset X',
							'separator'  => false,
							'min'        => - 50,
							'max'        => 50,
							'default'    => 0,
							'dependency' => [
								'box_shadow_color' => [
									'validate' => 'not_empty',
								],
							],
						]
					),
					Attribute::create(
						'box_shadow_offset_y',
						[
							'type'       => 'range',
							'title'      => 'Offset Y',
							'separator'  => false,
							'min'        => -50,
							'max'        => 50,
							'default'    => 2,
							'dependency' => [
								'box_shadow_color' => [
									'validate' => 'not_empty',
								],
							],
						]
					),
					Attribute::create(
						'box_shadow_blur_radius',
						[
							'type'       => 'range',
							'title'      => 'Blur radius',
							'separator'  => false,
							'min'        => 0,
							'max'        => 100,
							'default'    => 15,
							'dependency' => [
								'box_shadow_color' => [
									'validate' => 'not_empty',
								],
							],
						]
					),
					Attribute::create(
						'box_shadow_spread_radius',
						[
							'type'       => 'range',
							'title'      => 'Spread radius',
							'separator'  => false,
							'min'        => -50,
							'max'        => 50,
							'default'    => 0,
							'dependency' => [
								'box_shadow_color' => [
									'validate' => 'not_empty',
								],
							],
						]
					),
				]
			),

			// Advanced / Layout
			Attribute::tab_section(
				Attribute::TAB_ADVANCED,
				'Layout',
				[
					Attribute::create(
						'padding',
						[
							'type'  => 'range',
							'title' => 'Spacing',
							'min'   => 0,
							'max'   => 50,
							'units' => kalium_get_default_units( 'box-size' ),
						]
					),
					Attribute::create( Predefined_Attributes::$BORDER_RADIUS ),
					Attribute::create(
						'animation',
						[
							'type'         => 'select',
							'title'        => 'Animation',
							'inline_label' => true,
							'choices'      => [
								'none'        => 'None',
								'fade'        => 'Fade',
								'slide-scale' => 'Slide and scale',
							],
							'default'      => 'fade',
						]
					),
				]
			),

			// Advanced / Custom CSS
			Attribute::tab_section(
				Attribute::TAB_ADVANCED,
				Attribute::SECTION_CUSTOM_CSS,
				[
					Attribute::create( Predefined_Attributes::$CUSTOM_CSS ),
				]
			),
		];
	}

	/**
	 * Get element class list.
	 *
	 * @return array
	 */
	public function get_dom_class() {
		$classes = [];

		// Animation
		$animation = $this->get_attribute_value( 'animation' );

		if ( in_array( $animation, [ 'fade', 'slide-scale' ] ) ) {
			$classes[] = $this->get_base_class() . '--animation-' . $animation;
		}

		return array_merge( parent::get_dom_class(), $classes );
	}

	/**
	 * Generate styles.
	 */
	public function generate_styles() {
		$hover_layer_class = '.' . $this->get_hover_layer_class();

		// Values
		$box_shadow_color         = $this->get_attribute_value( 'box_shadow_color' );
		$box_shadow_offset_x      = $this->get_attribute_value( 'box_shadow_offset_x' );
		$box_shadow_offset_y      = $this->get_attribute_value( 'box_shadow_offset_y' );
		$box_shadow_blur_radius   = $this->get_attribute_value( 'box_shadow_blur_radius' );
		$box_shadow_spread_radius = $this->get_attribute_value( 'box_shadow_spread_radius' );

		// Border radius
		$border_radius = $this->get_attribute( 'border_radius' );

		// Box shadow
		$box_shadow = [];

		if ( $box_shadow_color ) {
			$box_shadow[] = sprintf( '%dpx', $box_shadow_offset_x );
			$box_shadow[] = sprintf( '%dpx', $box_shadow_offset_y );
			$box_shadow[] = sprintf( '%dpx', $box_shadow_blur_radius );

			if ( $box_shadow_spread_radius ) {
				$box_shadow[] = sprintf( '%dpx', $box_shadow_spread_radius );
			}

			$box_shadow[] = $box_shadow_color;
		}

		// Styles
		$this->add_style(
			[
				// Container
				'&'                => [
					kalium_css_var_name( 'product-hover-block-padding' ) => $this->get_attribute( 'padding' ),
				],

				// Hover layer
				$hover_layer_class => [
					'background'    => $this->get_attribute( 'background_color' ),
					'box-shadow'    => trim( implode( ' ', $box_shadow ) ),
					'border-radius' => [
						'value'  => $border_radius,
						'parser' => 'length',
					],
				],
			]
		);

		// Custom CSS
		$this->add_custom_css( 'custom_css', $this->get_attribute( 'custom_css' ) );
	}

	/**
	 * Hover layer class.
	 *
	 * @return string
	 */
	public function get_hover_layer_class() {
		return $this->get_base_class() . '__hover-layer';
	}
}

// Register element
\Laborator_Builder::register_element_type( WooCommerce_Product_Hover_Block::class );
