<?php
/**
 * WooCommerce Product Images element.
 */

namespace Laborator_Builder\Elements;

use Laborator_Builder\Element;
use Laborator_Builder\Attribute;
use Laborator_Builder\Attribute_Sets\Advanced_Border_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Custom_CSS_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Dimensions_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Layout_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Text_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Visibility_Hover_Set;
use Laborator_Builder\Attribute_Sets\Product_Common_Attributes_Set;

class WooCommerce_Product_Images extends Element {

	/**
	 * Name.
	 *
	 * @var string
	 */
	public static $name = 'woocommerce_product_images';

	/**
	 * Title.
	 *
	 * @var string
	 */
	public static $title = 'Image';

	/**
	 * Icon.
	 *
	 * @var string
	 */
	public static $icon = 'image';

	/**
	 * Child container.
	 *
	 * @var bool
	 */
	public static $child_container = true;

	/**
	 * Allowed child elements.
	 *
	 * @var bool
	 */
	public static $allowed_child_elements = [
		'!woocommerce_product_column',
	];

	/**
	 * Maximum instances of element.
	 *
	 * @var int
	 * @static
	 */
	public static $max_instances = 1;

	/**
	 * Collapse status.
	 *
	 * @var bool
	 * @static
	 */
	public static $default_collapse_state = true;

	/**
	 * Get attribute sets.
	 *
	 * @return array
	 */
	public function get_attribute_sets() {
		return array_merge(
			parent::get_attribute_sets(),
			[
				Advanced_Layout_Set::class,
				Advanced_Visibility_Hover_Set::class,
				Advanced_Border_Set::class,
				Advanced_Text_Set::class,
				Advanced_Dimensions_Set::class,
				Advanced_Custom_CSS_Set::class,
				Product_Common_Attributes_Set::class,
			]
		);
	}

	/**
	 * Attributes.
	 *
	 * @return Attribute[]
	 */
	public function create_attributes() {
		return array_merge(
			parent::create_attributes(),
			[
				// Content / Content
				Attribute::tab_section(
					Attribute::TAB_CONTENT,
					Attribute::SECTION_CONTENT,
					[
						Attribute::create(
							'gallery_type',
							[
								'type'    => 'select',
								'title'   => 'Gallery type',
								'choices' => [
									'featured-image'     => 'Featured image',
									'second-image-hover' => 'Second image on hover',
									'product-gallery'    => 'Product gallery',
								],
							]
						),
						Attribute::create(
							'gallery_max_images',
							[
								'type'         => 'quantity',
								'title'        => 'Max. images',
								'inline_label' => true,
								'default'      => 5,
								'min'          => 1,
								'max'          => 10,
								'dependency'   => [
									'gallery_type' => 'product-gallery',
								],
							]
						),
					]
				),

				// Content / Gallery Settings
				Attribute::tab_section(
					Attribute::TAB_CONTENT,
					'Gallery Settings',
					[
						Attribute::create(
							'gallery_navigation_type',
							[
								'type'         => 'radio',
								'title'        => 'Navigation type',
								'inline_label' => true,
								'choices'      => [
									'slider' => 'Slider',
									'hover'  => 'Hover',
								],
								'buttons'      => true,
								'dependency'   => [
									'gallery_type' => 'product-gallery',
								],
								'section'      => 'Gallery Settings',
								'tab'          => Attribute::TAB_CONTENT,
							]
						),
						Attribute::create(
							'gallery_navigation_arrows',
							[
								'type'         => 'toggle',
								'title'        => 'Navigation arrows',
								'inline_label' => true,
								'default'      => false,
								'dependency'   => [
									'gallery_type' => 'product-gallery',
									'gallery_navigation_type' => 'slider',
								],
								'section'      => 'Gallery Settings',
								'tab'          => Attribute::TAB_CONTENT,
							]
						),
						Attribute::create(
							'gallery_navigation_arrows_visible_on_hover',
							[
								'type'         => 'toggle',
								'title'        => 'Visible on hover',
								'inline_label' => true,
								'separator'    => false,
								'dependency'   => [
									'gallery_type' => 'product-gallery',
									'gallery_navigation_type' => 'slider',
									'gallery_navigation_arrows' => true,
								],
								'section'      => 'Gallery Settings',
								'tab'          => Attribute::TAB_CONTENT,
							]
						),
						Attribute::create(
							'gallery_navigation_arrows_size',
							[
								'type'         => 'radio',
								'title'        => 'Arrows size',
								'inline_label' => true,
								'separator'    => false,
								'choices'      => [
									'small'  => [
										'label'   => 'S',
										'tooltip' => 'Small',
									],
									'medium' => [
										'label'   => 'M',
										'tooltip' => 'Medium',
									],
									'large'  => [
										'label'   => 'L',
										'tooltip' => 'Large',
									],
								],
								'default'      => 'medium',
								'buttons'      => true,
								'dependency'   => [
									'gallery_type' => 'product-gallery',
									'gallery_navigation_type' => 'slider',
									'gallery_navigation_arrows' => true,
								],
							]
						),
						Attribute::create(
							'gallery_navigation_arrows_type',
							[
								'type'         => 'radio',
								'title'        => 'Arrows type',
								'inline_label' => true,
								'separator'    => false,
								'choices'      => [
									'type-1' => [
										'icon' => 'kalium-icon-arrow-right',
									],
									'type-2' => [
										'icon' => 'kalium-icon-long-arrow-right',
									],
								],
								'buttons'      => true,
								'dependency'   => [
									'gallery_type' => 'product-gallery',
									'gallery_navigation_type' => 'slider',
									'gallery_navigation_arrows' => true,
								],
							]
						),
						Attribute::create_tabbed(
							[

								// Normal
								'Normal' => [

									// Navigation arrows style
									Attribute::create(
										'gallery_navigation_arrows_style',
										[
											'type'         => 'color',
											'title'        => 'Arrows style',
											'inline_label' => true,
											'reset'        => true,
											'separator'    => false,
											'choices'      => [
												'background' => 'Background color',
												'text' => 'Arrow color',
											],
											'tooltip'      => true,
											'enableAlpha'  => true,
										]
									),

									// Navigation arrows border radius
									Attribute::create(
										'gallery_navigation_arrows_border_radius',
										[
											'type'      => 'range',
											'title'     => 'Border radius',
											'separator' => false,
											'min'       => 0,
											'max'       => 100,
											'units'     => kalium_get_default_units( 'box-size' ),
											'default'   => '50%',
										]
									),
								],

								// Hover
								'Hover'  => [

									// Navigation arrows style on hover
									Attribute::create(
										'gallery_navigation_arrows_style_hover',
										[
											'type'         => 'color',
											'title'        => 'Arrows style',
											'inline_label' => true,
											'reset'        => true,
											'separator'    => false,
											'choices'      => [
												'background' => 'Background color',
												'text' => 'Arrow color',
											],
											'tooltip'      => true,
											'allowClear'   => true,
										]
									),
								],
							],
							[
								'dependency' => [
									'gallery_type' => 'product-gallery',
									'gallery_navigation_type' => 'slider',
									'gallery_navigation_arrows' => true,
								],
							]
						),
						Attribute::create(
							'gallery_pagination',
							[
								'type'         => 'toggle',
								'title'        => 'Gallery pagination',
								'inline_label' => true,
								'default'      => true,
								'dependency'   => [
									'gallery_type' => 'product-gallery',
								],
							]
						),
						Attribute::create(
							'gallery_pagination_style',
							[
								'type'         => 'select',
								'title'        => 'Pagination style',
								'inline_label' => true,
								'separator'    => false,
								'choices'      => [
									'bars' => 'Bars',
									'dots' => 'Dots',
								],
								'dependency'   => [
									'gallery_type'       => 'product-gallery',
									'gallery_pagination' => true,
								],
							]
						),
						Attribute::create(
							'gallery_pagination_visible_on_hover',
							[
								'type'         => 'toggle',
								'title'        => 'Visible on hover',
								'inline_label' => true,
								'separator'    => false,
								'dependency'   => [
									'gallery_type'       => 'product-gallery',
									'gallery_pagination' => true,
								],
							]
						),
						Attribute::create(
							'gallery_pagination_hover_animation',
							[
								'type'         => 'select',
								'title'        => 'Hover animation',
								'inline_label' => true,
								'separator'    => false,
								'choices'      => [
									'fade'       => 'Fade',
									'slide-fade' => 'Slide and fade',
								],
								'dependency'   => [
									'gallery_type'       => 'product-gallery',
									'gallery_pagination' => true,
									'gallery_pagination_visible_on_hover' => true,
								],
							]
						),
						Attribute::create(
							'gallery_navigation_effect',
							[
								'type'         => 'select',
								'title'        => 'Effect',
								'inline_label' => true,
								'choices'      => [
									'slide' => 'Slide',
									'fade'  => 'Fade',
								],
								'dependency'   => [
									'gallery_type' => 'product-gallery',
									'gallery_navigation_type' => 'slider',
								],
							]
						),
						Attribute::create(
							'gallery_navigation_loop',
							[
								'type'              => 'toggle',
								'title'             => 'Loop',
								'inline_label'      => true,
								'separator'         => false,
								'default'           => false,
								'tooltip'           => 'Continuous loop mode',
								'tooltip_placement' => 'top-end',
								'dependency'        => [
									'gallery_type' => 'product-gallery',
									'gallery_navigation_type' => 'slider',
								],
							]
						),
					]
				),

				// Content / Link
				Attribute::tab_section(
					Attribute::TAB_CONTENT,
					'Link',
					[
						Attribute::create(
							'link_to_product',
							[
								'type'         => 'toggle',
								'title'        => 'Link to product',
								'inline_label' => true,
								'default'      => true,
							]
						),
					]
				),
			]
		);
	}

	/**
	 * Generate styles.
	 */
	public function generate_styles() {
		parent::generate_styles();

		// Apply border radius to gallery images
		$this->add_style(
			[
				'.image-placeholder' => [
					'border-radius' => [
						'value'   => $this->get_attribute( 'border_radius' ),
						'parser'  => 'length',
						'options' => [
							'append' => [
								'overflow' => 'hidden',
							],
						],
					],
				],
			]
		);

		// Style navigation arrows if enabled
		if ( 'slider' === $this->get_attribute_value( 'gallery_navigation_type' ) && true === $this->get_attribute_value( 'gallery_navigation_arrows' ) ) {
			$nav_arrows_color       = $this->get_attribute_value( 'gallery_navigation_arrows_style' );
			$nav_arrows_hover_color = $this->get_attribute_value( 'gallery_navigation_arrows_style_hover' );

			$this->add_style(
				[
					// Navigation arrows style and border radius
					'.image-set__navigation-button'       => [
						'border-radius'    => $this->get_attribute( 'gallery_navigation_arrows_border_radius' ),
						'background-color' => $nav_arrows_color['background'] ?? null,
						'color'            => $nav_arrows_color['text'] ?? null,
					],

					// Navigation arrows style on hover
					'.image-set__navigation-button:hover' => [
						'background-color' => $nav_arrows_hover_color['background'] ?? null,
						'color'            => $nav_arrows_hover_color['text'] ?? null,
					],
				]
			);
		}
	}
}

// Register element
\Laborator_Builder::register_element_type( WooCommerce_Product_Images::class );
