<?php
/**
 * Layout attributes.
 */

namespace Laborator_Builder\Attribute_Sets;

use Laborator_Builder\Attribute;
use Laborator_Builder\Attribute_Set;

class Advanced_Layout_Set extends Attribute_Set {

	/**
	 * Get attributes.
	 *
	 * @return array
	 */
	public function get_attributes() {
		return Attribute::tab_section(
			Attribute::TAB_ADVANCED,
			Attribute::SECTION_LAYOUT,
			[
				Attribute::create(
					'margin',
					[
						'type'       => 'multi-numeric',
						'title'      => 'Margin',
						'responsive' => true,
						'reset'      => true,
						'units'      => kalium_get_default_units( 'box-size' ),
					]
				),
				Attribute::create(
					'padding',
					[
						'type'       => 'multi-numeric',
						'title'      => 'Padding',
						'responsive' => true,
						'reset'      => true,
						'separator'  => false,
						'units'      => kalium_get_default_units( 'box-size' ),
					]
				),
				Attribute::create(
					'css_position',
					[
						'type'         => 'select',
						'title'        => 'Position',
						'inline_label' => true,
						'responsive'   => true,
						'choices'      => [
							'default'  => 'Default',
							'absolute' => 'Absolute',
							'relative' => 'Relative',
						],
					]
				),
				Attribute::create(
					'position_offset',
					[
						'type'       => 'multi-numeric',
						'title'      => 'Position Offset',
						'responsive' => true,
						'reset'      => true,
						'units'      => kalium_get_default_units( 'box-size' ),
						'separator'  => false,
						'dependency' => [
							'css_position' => [ 'absolute' ],
						],
					]
				),
				Attribute::create(
					'z_index',
					[
						'type'         => 'quantity',
						'title'        => 'Z-Index',
						'inline_label' => true,
						'dependency'   => [
							'css_position' => [ 'absolute', 'fixed' ],
						],
					]
				),
				Attribute::create(
					'overflow',
					[
						'type'         => 'select',
						'title'        => 'Overflow',
						'inline_label' => true,
						'choices'      => [
							'default'  => 'Default',
							'auto'     => 'Auto',
							'visible'  => 'Visible',
							'scroll'   => 'Scroll',
							'hidden'   => 'Hidden',
							'x-hidden' => 'Hidden X',
							'y-hidden' => 'Hidden Y',
						],
					]
				),
				Attribute::create(
					'css_class',
					[
						'type'         => 'text',
						'title'        => 'CSS Classes',
						'inline_label' => true,
					]
				),
			]
		);
	}

	/**
	 * Get DOM classes.
	 *
	 * @return array
	 */
	public function get_dom_class() {
		$classes = [];
		$element = $this->get_element();

		// CSS position
		$css_position = $element->get_attribute_value( 'css_position' );

		$classes = array_merge(
			$classes,
			kalium_responsive_classname_infix(
				$css_position,
				function ( $value, $bs_viewport ) {
					if ( in_array( $value, [ 'absolute', 'relative' ] ) ) {
						return 'position' . kalium_conditional( $bs_viewport, "-{$bs_viewport}" ) . '-' . $value;
					}

					return false;
				}
			)
		);

		// Overflow
		$overflow = $element->get_attribute_value( 'overflow' );

		if ( in_array( $overflow, [ 'visible', 'hidden', 'hidden-x', 'hidden-y' ] ) ) {
			$classes[] = 'overflow-' . $overflow;
		}

		// Custom CSS class
		if ( $custom_css_classes = $element->get_attribute_value( 'css_class' ) ) {
			$classes = array_merge( $classes, array_filter( explode( ' ', $custom_css_classes ) ) );
		}

		return $classes;
	}

	/**
	 * Generate styles.
	 */
	public function generate_styles() {
		$element = $this->get_element();

		$element->add_style(
			[
				'&' => [
					'margin'  => [
						'value'  => $element->get_attribute( 'margin' ),
						'parser' => 'length',
					],
					'padding' => [
						'value'  => $element->get_attribute( 'padding' ),
						'parser' => 'length',
					],
					'inset'   => [
						'value'  => $element->get_attribute( 'position_offset' ),
						'parser' => 'length',
					],
					'z-index' => $element->get_attribute( 'z_index' ),
				],
			]
		);
	}
}
