<?php
/**
 * Flex alignment attributes.
 */
namespace Laborator_Builder\Attribute_Sets;

use Laborator_Builder\Attribute;
use Laborator_Builder\Attribute_Set;

class Content_Flex_Alignment_Set extends Attribute_Set {

	/**
	 * Attributes.
	 *
	 * @return Attribute[]
	 */
	public function get_attributes() {
		return Attribute::tab_section(
			Attribute::TAB_CONTENT,
			Attribute::SECTION_CONTENT,
			[
				Attribute::create(
					'flex_direction',
					[
						'type'       => 'radio',
						'title'      => 'Direction',
						'responsive' => true,
						'reset'      => true,
						'choices'    => [
							'default' => [
								'label' => 'Default',
								'icon'  => 'globe',
							],
							'stack'   => [
								'label' => 'Stack',
								'icon'  => 'arrowDown',
							],
							'row'     => [
								'label' => 'Row',
								'icon'  => 'arrowRight',
							],
						],
						'buttons'    => [
							'icon_size' => 18,
						],
					]
				),
				Attribute::create(
					'horizontal_align',
					[
						'type'       => 'radio',
						'title'      => 'Horizontal alignment',
						'responsive' => true,
						'reset'      => true,
						'choices'    => [
							'default' => [
								'label' => 'Default',
								'icon'  => 'globe',
							],
							'start'   => [
								'label' => 'Left',
								'icon'  => 'justifyLeft',
							],
							'center'  => [
								'label' => 'Center',
								'icon'  => 'justifyCenter',
							],
							'end'     => [
								'label' => 'Right',
								'icon'  => 'justifyRight',
							],
						],
						'buttons'    => [
							'icon_size' => 18,
						],
					]
				),
				Attribute::create(
					'vertical_align',
					[
						'type'       => 'radio',
						'title'      => 'Vertical alignment',
						'responsive' => true,
						'reset'      => true,
						'choices'    => [
							'default' => [
								'label' => 'Default',
								'icon'  => 'globe',
							],
							'start'   => [
								'label' => 'Top',
								'icon'  => 'verticalAlignTop',
							],
							'center'  => [
								'label' => 'Center',
								'icon'  => 'verticalAlignCenter',
							],
							'end'     => [
								'label' => 'Bottom',
								'icon'  => 'verticalAlignBottom',
							],
						],
						'buttons'    => [
							'icon_size' => 18,
						],
					]
				),
				Attribute::create(
					'flex_order',
					[
						'type'         => 'quantity',
						'title'        => 'Order',
						'inline_label' => true,
						'numeric'      => true,
						'responsive'   => true,
						'min'          => -1,
						'max'          => 10,
					]
				),
			]
		);
	}

	/**
	 * Get element class list.
	 *
	 * @return array
	 */
	public function get_dom_class() {
		$element = $this->get_element();
		$classes = [];

		// Content alignment
		$skip_default     = function ( $value ) {
			return 'default' !== $value;
		};
		$flex_direction   = array_filter( $element->get_attribute_value( 'flex_direction' ), $skip_default );
		$vertical_align   = array_filter( $element->get_attribute_value( 'vertical_align' ), $skip_default );
		$horizontal_align = array_filter( $element->get_attribute_value( 'horizontal_align' ), $skip_default );

		// Flex direction
		if ( false === kalium_is_empty_responsive( $vertical_align ) ) {
			$classes = array_merge(
				$classes,
				[ 'd-flex' ],
				kalium_responsive_classname_infix(
					$flex_direction,
					kalium_responsive_classname_infix(
						$element->get_attribute_value( 'flex_direction' ),
						function ( $value, $bs_viewport ) {
							if ( 'default' === $value ) {
								return;
							}

							$direction = 'row' === $value ? 'row' : 'column';

							return 'flex' . kalium_conditional( $bs_viewport, '-' . $bs_viewport ) . '-' . $direction;
						}
					)
				)
			);
		}

		// Vertical align
		if ( false === kalium_is_empty_responsive( $vertical_align ) ) {
			$classes = array_merge( $classes, [ 'd-flex' ], kalium_responsive_classname_infix( $vertical_align, 'align-items' ) );
		}

		// Horizontal align
		if ( false === kalium_is_empty_responsive( $horizontal_align ) ) {
			$classes = array_merge( $classes, [ 'd-flex' ], kalium_responsive_classname_infix( $horizontal_align, 'justify-content' ) );
		}

		// Flex order
		$has_flex_order = false;

		$flex_order = array_map(
			function ( $value ) use ( &$has_flex_order ) {
				if ( is_numeric( $value ) ) {
					$has_flex_order = true;

					return -1 >= $value ? 'first' : ( 10 <= $value ? 'last' : $value );
				}

				return 0;
			},
			$element->get_attribute_value( 'flex_order' )
		);

		if ( $has_flex_order ) {
			$classes = array_merge( $classes, kalium_responsive_classname_infix( $flex_order, 'order' ) );
		}

		return $classes;
	}
}
