<?php
/**
 * Laborator Builder.
 *
 * Content type definition class.
 */

namespace Laborator_Builder;

class Content_Type {

	/**
	 * Content type ID.
	 *
	 * @var string
	 */
	public $id;

	/**
	 * Mode.
	 *
	 * @var string
	 */
	public $mode;

	/**
	 * Registered elements.
	 *
	 * @var array
	 */
	public $elements = [];

	/**
	 * Content regions based on templates.
	 *
	 * @var array
	 */
	public $regions = [];

	/**
	 * Templates.
	 *
	 * @var Content_Type_Template[]
	 */
	public $templates = [];

	/**
	 * Template columns.
	 *
	 * @var int
	 */
	public $template_columns;

	/**
	 * Default content.
	 *
	 * @var array
	 */
	public $default_content;

	/**
	 * Default template.
	 *
	 * @var string
	 */
	public $default_template;

	/**
	 * Content source.
	 *
	 * @var array
	 */
	public $content_source;

	/**
	 * Tabs.
	 *
	 * @var array
	 */
	public $tabs;

	/**
	 * Preview URL.
	 *
	 * @var string
	 */
	public $preview_url;

	/**
	 * Element variations.
	 *
	 * @var string
	 */
	public $element_variations;

	/**
	 * Visibility feature.
	 *
	 * @var bool
	 */
	public $visibility;

	/**
	 * Display element icon.
	 *
	 * @var bool
	 */
	public $element_icon;

	/**
	 * Empty label.
	 *
	 * @var string
	 */
	public $empty_label;

	/**
	 * Add label.
	 *
	 * @var string
	 */
	public $add_label;

	/**
	 * Search field.
	 *
	 * @var int|false
	 */
	public $search;

	/**
	 * Show attributes on add.
	 *
	 * @var bool
	 */
	public $show_attributes_on_add;

	/**
	 * Debug mode.
	 *
	 * @var bool
	 */
	public $debug_mode;

	/**
	 * Constructor.
	 *
	 * @param string $id
	 * @param array  $args
	 */
	public function __construct( $id, $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'mode'               => null, // [basic,advanced]
				'elements'           => null,
				'regions'            => null,
				'templates'          => null,
				'element_variations' => null,
				'template_columns'   => null,
				'default_content'    => null,
				'default_template'   => null,
				'content_source'     => null,
				'tabs'               => null,
				'preview_url'        => null,
				'visibility'         => true,
				'element_icon'       => false,
				'empty_label'        => null,
				'add_label'          => null,
				'search'             => false,
				'show_attributes'    => false,
				'debug_mode'         => false,
			]
		);

		// Set ID
		$this->id = $id;

		// Mode
		$this->mode = 'advanced' === $args['mode'] ? $args['mode'] : 'basic';

		// Elements
		if ( is_array( $args['elements'] ) ) {
			foreach ( $args['elements'] as $element_name => $element_args ) {
				if ( is_numeric( $element_name ) ) {
					$element_name = $element_args;
					$element_args = [];
				}

				$this->add_element( $element_name, $element_args );
			}
		}

		// Regions
		if ( is_array( $args['regions'] ) ) {
			$this->regions = $args['regions'];
		}

		// Templates
		if ( is_array( $args['templates'] ) ) {
			foreach ( $args['templates'] as $template_args ) {
				$this->add_template( $template_args );
			}
		}

		// Element variations
		$this->element_variations = $args['element_variations'];

		if ( ! empty( $this->element_variations ) ) {
			$this->element_variations = array_map(
				function ( $element_variation ) {
					if ( is_array( $element_variation ) ) {
						$element_variation['id'] = '_ev' . mt_rand( 99999, 999999 );
					}

					return $element_variation;
				},
				$this->element_variations
			);
		}

		// Template columns
		if ( is_numeric( $args['template_columns'] ) ) {
			$this->template_columns = $args['template_columns'];
		}

		// Default content
		$this->default_content = $args['default_content'];

		// Default template
		if ( ! empty( $args['default_template'] ) ) {
			$this->default_template = $args['default_template'];
		} elseif ( ! empty( $this->get_templates() ) ) {
			$this->default_template = current( $this->get_templates() )->get_id();
		}

		// Set content source
		$this->set_content_source( $args['content_source'] );

		// Tabs
		if ( is_array( $args['tabs'] ) ) {
			$this->tabs = $args['tabs'];
		}

		// Preview URL
		if ( ! empty( $args['preview_url'] ) ) {
			$this->preview_url = $args['preview_url'];
		}

		// Visibility feature
		$this->visibility = $args['visibility'];

		// Element icon
		$this->element_icon = $args['element_icon'];

		// Empty label
		$this->empty_label = $args['empty_label'];

		// Add label
		$this->add_label = $args['add_label'];

		// Search
		$this->search = $args['search'];

		// Show attributes on add
		$this->show_attributes_on_add = $args['show_attributes'];

		// Debug mode
		$this->debug_mode = $args['debug_mode'] || array_key_exists( 'laborator-builder-debug', $_GET );
	}

	/**
	 * Get content type ID.
	 *
	 * @return string
	 */
	public function get_id() {
		return $this->id;
	}

	/**
	 * Get mode.
	 *
	 * @return string
	 */
	public function get_mode() {
		return $this->mode;
	}

	/**
	 * Get elements.
	 *
	 * @return array
	 */
	public function get_elements() {
		return $this->elements;
	}

	/**
	 * Get element.
	 *
	 * @return array
	 */
	public function get_element( $name ) {
		return $this->elements[ $name ] ?? null;
	}

	/**
	 * Add element.
	 *
	 * @param string $name
	 * @param array  $args
	 */
	public function add_element( $name, $options = [] ) {
		$element_type = wp_parse_args(
			$options,
			[
				'global' => true,
				'extend' => [],
			]
		);

		// Get registered element
		if ( $element = \Laborator_Builder::get_element( $name ) ) {
			$element_type['constructor'] = $element['class'];
		}
		// Extend existing element
		elseif ( ! empty( $element_type['extend']['element'] ) ) {
			$extend  = wp_parse_args(
				$element_type['extend'],
				[
					// New element name
					'name' => $name,
				]
			);
			$element = $extend['element'];

			// Remove extend from $args, element from $extend
			unset( $element_type['extend'], $extend['element'] );

			// Extend existing element
			if ( $extend_element = \Laborator_Builder::get_element( $element ) ) {

				// Set extend prop
				$element_type['extend'] = $extend;

				// Assign constructor
				$element_type['constructor'] = $extend_element['class'];
			}
		}

		// Register element
		if ( ! empty( $element_type['constructor'] ) ) {
			$element_type['instance'] = new $element_type['constructor']( $element_type['extend'] );
			$element_type['instance']->init();

			// Add to registered elements
			$this->elements[ $name ] = $element_type;
		}
	}

	/**
	 * Remove element.
	 *
	 * @param string $name
	 */
	public function remove_element( $name ) {
		unset( $this->elements[ $name ] );
	}

	/**
	 * Get content regions.
	 *
	 * @return array
	 */
	public function get_regions() {
		return $this->regions;
	}

	/**
	 * Get templates.
	 *
	 * @return Content_Type_Template[]
	 */
	public function get_templates() {
		return $this->templates;
	}

	/**
	 * Get template by ID.
	 *
	 * @return Content_Type_Template|null
	 */
	public function get_template( $id ) {
		$templates = $this->get_templates();

		foreach ( $templates as $template ) {
			if ( $template->get_id() === $id ) {
				return $template;
			}
		}

		return null;
	}

	/**
	 * Has templates.
	 *
	 * @return bool
	 */
	public function has_templates() {
		return ! empty( $this->get_templates() );
	}

	/**
	 * Add template.
	 *
	 * @param array $args
	 *
	 * @return bool
	 */
	public function add_template( $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'id'       => null,
				'title'    => null,
				'icon'     => null,
				'template' => null,
			]
		);

		// Add template if not exists
		if ( ! $this->get_template( $args['id'] ) ) {
			$content_type = new Content_Type_Template( $args );

			if ( $content_type->is_valid() ) {
				$this->templates[] = $content_type;

				return true;
			}
		}

		return false;
	}

	/**
	 * Remove template.
	 *
	 * @param string $id
	 *
	 * @return bool
	 */
	public function remove_template( $id ) {
		foreach ( $this->get_templates() as $i => $template ) {
			if ( $id === $template->get_id() ) {
				unset( $this->templates[ $i ] );

				return true;
			}
		}

		return false;
	}

	/**
	 * Get default content.
	 *
	 * @return array|null
	 */
	public function get_default_content() {
		$default_content = $this->default_content;

		// Provided default content
		if ( is_array( $default_content ) ) {

			// Wrap elements
			if ( ! isset( $default_content['elements'] ) ) {
				$default_content = [
					'elements' => $default_content,
				];
			}

			return $default_content;
		}

		// Default template
		if ( $this->has_templates() ) {
			$default_template = $this->get_template( $this->get_default_template() );

			if ( $default_template instanceof Content_Type_Template ) {
				return $default_template->get_template_content();
			}
		}

		return null;
	}

	/**
	 * Get content.
	 *
	 * @return array|null
	 */
	public function get_content() {
		$content = null;

		// Mode
		$mode = $this->get_mode();

		// Content source
		$content_source = $this->get_content_source();

		// Get content from provided content source
		switch ( $content_source['type'] ) {

			// Theme mod
			case 'theme_mod':
				$content_data = get_theme_mod( $content_source['id'] );
				break;

			// Direct (elements provided with array)
			case 'direct':
				$content_data = $content_source['content'];
				break;
		}

		// Get default value
		if ( empty( $content_data ) ) {

			// Use default template for advanced mode
			if ( 'advanced' === $mode ) {
				$content_data = [
					'content_source' => 'templates',
					'template_id'    => $this->get_default_template(),
				];
			} else {
				$content_data = $this->get_default_content();
			}
		}

		// Process content data
		if ( ! empty( $content_data ) ) {

			// Advanced mode
			if ( 'advanced' === $mode ) {
				$content_data = wp_parse_args(
					$content_data,
					[
						'content_source'  => null,
						'template_id'     => null,
						'preview_content' => null,
						'user_templates'  => null,
					]
				);

				// Get content from predefined template
				if ( 'templates' === $content_data['content_source'] ) {
					$template = $this->get_template( $content_data['template_id'] );

					if ( $template instanceof Content_Type_Template ) {
						$content = $template->get_template_content();
					}
				} // Get content from user templates
				elseif ( 'user-templates' === $content_data['content_source'] ) {
					$user_template_content = $this->get_user_template_content( $content_data );

					if ( ! empty( $user_template_content ) ) {
						$content = $user_template_content;
					}
				}

				// If content is empty, load from default template
				if ( empty( $content ) && ( $template = $this->get_template( $this->get_default_template() ) ) ) {
					$content = $template->get_template_content();
				}

				// Preview content from WP Customizer
				if ( is_customize_preview() && ! empty( $content_data['preview_content'] ) ) {
					$content = $content_data['preview_content'];
				}
			}
			// Basic content mode
			else {
				$content = $content_data;
			}
		}

		return $content;
	}

	/**
	 * Get user template content from content data.
	 *
	 * @param array $content_data
	 *
	 * @return array|null
	 */
	public function get_user_template_content( $content_data ) {
		$user_templates = $content_data['user_templates'] ?? [];
		$template_id    = $content_data['template_id'] ?? null;

		foreach ( $user_templates as $user_template ) {
			if ( $template_id === $user_template['id'] ) {
				return $user_template['template_content'];
			}
		}

		return null;
	}

	/**
	 * Get template columns.
	 *
	 * @return int
	 */
	public function get_template_columns() {
		return $this->template_columns;
	}

	/**
	 * Get default template.
	 *
	 * @return string
	 */
	public function get_default_template() {
		return $this->default_template;
	}

	/**
	 * Get element variations.
	 *
	 * @return bool
	 */
	public function get_element_variations() {
		return $this->element_variations;
	}

	/**
	 * Get content source.
	 *
	 * @return array|null
	 */
	public function get_content_source() {
		return $this->content_source;
	}

	/**
	 * Set content source.
	 *
	 * @param array|string $content_source
	 */
	public function set_content_source( $content_source ) {
		$valid_sources       = [ 'theme_mod', 'direct' ];
		$default_source_type = 'theme_mod';

		// Default source as theme mod
		if ( is_string( $content_source ) ) {
			$content_source = [
				'type' => $default_source_type,
				'id'   => $content_source,
			];
		}

		// Content source
		$content_source = wp_parse_args(
			$content_source,
			[
				'type'    => $default_source_type,
				'id'      => null,
				'content' => null,
			]
		);

		// Accept valid content types
		if ( ! in_array( $content_source['type'], $valid_sources ) ) {
			return;
		}

		$this->content_source = $content_source;
	}

	/**
	 * Get tabs.
	 *
	 * @return array
	 */
	public function get_tabs() {
		return $this->tabs;
	}

	/**
	 * Get preview url.
	 *
	 * @return string
	 */
	public function get_preview_url() {
		return $this->preview_url;
	}

	/**
	 * Supports visibility feature.
	 *
	 * @return bool
	 */
	public function supports_visibility() {
		return $this->visibility;
	}

	/**
	 * Display element icon.
	 *
	 * @return bool
	 */
	public function display_element_icon() {
		return $this->element_icon;
	}

	/**
	 * Empty label.
	 *
	 * @return string
	 */
	public function get_empty_label() {
		return $this->empty_label;
	}

	/**
	 * Add label.
	 *
	 * @return string
	 */
	public function get_add_label() {
		return $this->add_label;
	}

	/**
	 * Get minimum number of elements required to show search field.
	 *
	 * @return bool|int
	 */
	public function get_search() {
		return $this->search;
	}

	/**
	 * Show attributes on add.
	 *
	 * @return bool
	 */
	public function show_attributes_on_add() {
		return $this->show_attributes_on_add;
	}

	/**
	 * Get debug mode.
	 *
	 * @return bool
	 */
	public function get_debug_mode() {
		return $this->debug_mode;
	}

	/**
	 * Export content type to JS.
	 *
	 * @return array
	 */
	public function export() {

		/**
		 * Map element.
		 *
		 * @param Element $element
		 *
		 * @return array
		 */
		$map_element = function ( $element ) {
			return [
				'elementConstructor' => $element['instance']->export(),
				'elementOptions'     => [
					'global' => $element['global'],
				],
			];
		};

		/**
		 * Map template.
		 *
		 * @param Content_Type_Template $template
		 *
		 * @return array
		 */
		$map_template = function ( $template ) {
			return $template->export();
		};

		return [
			'id'                  => $this->get_id(),
			'mode'                => $this->get_mode(),
			'elements'            => array_map( $map_element, $this->get_elements() ),
			'regions'             => $this->get_regions(),
			'elementVariations'   => $this->get_element_variations(),
			'templates'           => array_map( $map_template, $this->get_templates() ),
			'templateColumns'     => $this->get_template_columns(),
			'defaultTemplate'     => $this->get_default_template(),
			'tabs'                => $this->get_tabs(),
			'previewUrl'          => $this->get_preview_url(),
			'supportsVisibility'  => $this->supports_visibility(),
			'emptyLabel'          => $this->get_empty_label(),
			'addLabel'            => $this->get_add_label(),
			'search'              => $this->get_search(),
			'showAttributesOnAdd' => $this->show_attributes_on_add(),
			'displayElementIcon'  => $this->display_element_icon(),
			'debugMode'           => $this->get_debug_mode(),
		];
	}
}
