<?php
/**
 * Laborator Builder.
 *
 * Base element for the builder.
 */

namespace Laborator_Builder;

class Element {

	/**
	 * Element components.
	 */
	use Styles;
	use Exporter;

	/**
	 * Instance counter.
	 *
	 * @var int
	 * @static
	 */
	private static $instance_index = 1;

	/**
	 * Element name.
	 *
	 * @var string
	 * @static
	 */
	public static $name;

	/**
	 * Title.
	 *
	 * @var string
	 * @static
	 */
	public static $title;

	/**
	 * Dynamic title.
	 *
	 * @var string
	 * @static
	 */
	public static $dynamic_title;

	/**
	 * Icon.
	 *
	 * @var string
	 * @static
	 */
	public static $icon;

	/**
	 * Category.
	 *
	 * @var string
	 * @static
	 */
	public static $category;

	/**
	 * Child container.
	 *
	 * @var bool
	 * @static
	 */
	public static $child_container = false;

	/**
	 * Allowed child elements.
	 *
	 * @var bool
	 * @static
	 */
	public static $allowed_child_elements = [];

	/**
	 * Maximum nesting level.
	 *
	 * @var int
	 * @static
	 */
	public static $max_nesting_level = -1;

	/**
	 * Maximum elements that can be added.
	 *
	 * @var int
	 * @static
	 */
	public static $max_elements = -1;

	/**
	 * Maximum instances of element.
	 *
	 * @var int
	 * @static
	 */
	public static $max_instances = -1;

	/**
	 * Unique elements only.
	 *
	 * @var bool
	 * @static
	 */
	public static $unique_elements = false;

	/**
	 * Attribute sets.
	 *
	 * @var array
	 * @static
	 */
	public static $attribute_sets;

	/**
	 * Empty elements label.
	 *
	 * @var string
	 * @static
	 */
	public static $empty_label = '';

	/**
	 * Add elements label.
	 *
	 * @var string
	 * @static
	 */
	public static $add_label = '';

	/**
	 * Enable status.
	 *
	 * @var bool
	 * @static
	 */
	public static $enabled = true;

	/**
	 * Disabled message.
	 *
	 * @var string
	 * @static
	 */
	public static $disabled_message = 'Element is disabled!';

	/**
	 * Collapse status.
	 *
	 * @var bool
	 * @static
	 */
	public static $default_collapse_state = false;

	/**
	 * Dynamic element static props.
	 *
	 * @var array
	 */
	public $dynamic_element_props = [];

	/**
	 * Instance ID.
	 *
	 * @var int
	 */
	public $instance_id;

	/**
	 * Parent element.
	 *
	 * @var Element
	 */
	public $parent;

	/**
	 * Child elements.
	 *
	 * @var Elements
	 */
	public $children = null;

	/**
	 * Element attributes.
	 *
	 * @var Attribute[]
	 */
	public $attributes = [];

	/**
	 * Options.
	 *
	 * @var array
	 */
	public $options = [];

	/**
	 * Visibility status.
	 *
	 * @var bool
	 */
	public $visible = true;

	/**
	 * Tag name.
	 *
	 * @var string
	 */
	public $tag_name = 'div';

	/**
	 * DOM attributes of HTML element.
	 *
	 * @var array
	 */
	public $dom_attributes = [];

	/**
	 * Index.
	 *
	 * @var string
	 */
	public $index;

	/**
	 * Constructor.
	 *
	 * @param array $args
	 */
	public function __construct( $args = [] ) {

		// Set instance ID
		$this->instance_id = self::$instance_index;

		// Dynamic props
		foreach ( [
			'name',
			'dynamic_title',
			'title',
			'icon',
			'category',
			'child_container',
			'allowed_child_elements',
			'max_nesting_level',
			'max_elements',
			'max_instances',
			'unique_elements',
			'attribute_sets',
			'empty_label',
			'add_label',
			'enabled',
			'disabled_message',
			'default_collapse_state',
		] as $element_prop ) {
			if ( isset( $args[ $element_prop ] ) ) {
				$this->dynamic_element_props[ $element_prop ] = $args[ $element_prop ];
			}
		}

		// Initialize attributes
		$this->attributes = Attribute::flatten_attributes( $this->create_attributes() );

		// Remove attributes
		if ( ! empty( $args['remove_attributes'] ) && is_array( $args['remove_attributes'] ) ) {
			foreach ( $args['remove_attributes'] as $attribute_name ) {
				$this->remove_attribute( $attribute_name );
			}
		}

		// Dynamic attributes
		if ( ! empty( $args['extend_attributes'] ) && is_array( $args['extend_attributes'] ) ) {
			foreach ( $args['extend_attributes'] as $attribute_name => $attribute_args ) {
				$this->add_dynamic_attribute( $attribute_name, $attribute_args );
			}
		}

		// Visible
		if ( isset( $args['visible'] ) ) {
			$this->visible = boolval( $args['visible'] );
		}

		// Assign default value to attributes
		foreach ( $this->get_attributes() as $attribute ) {
			$attribute->reset_value();
		}

		// Options
		if ( ! empty( $args['options'] ) ) {
			$this->options = $args['options'];
		}

		// Assign attribute values
		if ( ! empty( $args['attributes'] ) ) {
			$this->assign_attribute_values( $args['attributes'] );
		}

		// Assign parent
		if ( ! empty( $args['parent'] ) && $args['parent'] instanceof Element ) {
			$this->parent = $args['parent'];
		}

		// Index
		if ( isset( $args['index'] ) ) {
			$this->index = $args['index'];
		}

		/**
		 * After element is created.
		 *
		 * Hook: laborator_builder_element
		 */
		do_action( 'laborator_builder_element', $this );

		// Increase instance index counter
		++self::$instance_index;
	}

	/**
	 * Get instance ID.
	 *
	 * @return int
	 */
	public function get_instance_id() {
		return $this->instance_id;
	}

	/**
	 * Conditionally gets element static prop or if its defined in $dynamic_element_props will return it.
	 *
	 * @param string $prop
	 *
	 * @return mixed
	 */
	public function get_element_prop( $prop ) {
		$prop_value = static::$$prop;

		if ( isset( $this->dynamic_element_props[ $prop ] ) ) {
			$dynamic_prop_value = $this->dynamic_element_props[ $prop ];

			if ( is_callable( $dynamic_prop_value ) ) {
				return $dynamic_prop_value( $prop_value );
			}

			return $dynamic_prop_value;
		}

		return $prop_value;
	}

	/**
	 * Get name.
	 *
	 * @return string
	 */
	public function get_name() {
		return $this->get_element_prop( 'name' );
	}

	/**
	 * Dynamic title.
	 *
	 * @return string|null
	 */
	public function get_dynamic_title() {
		return $this->get_element_prop( 'dynamic_title' );
	}

	/**
	 * Get title.
	 *
	 * @return string
	 */
	public function get_title() {
		return $this->get_element_prop( 'title' );
	}

	/**
	 * Get icon.
	 *
	 * @return string
	 */
	public function get_icon() {
		return $this->get_element_prop( 'icon' );
	}

	/**
	 * Get category.
	 *
	 * @return string
	 */
	public function get_category() {
		return $this->get_element_prop( 'category' );
	}

	/**
	 * Is child container element?
	 *
	 * @return bool
	 */
	public function is_child_container() {
		return $this->get_element_prop( 'child_container' );
	}

	/**
	 * Get allowed child elements.
	 *
	 * @return array
	 */
	public function get_allowed_child_elements() {
		return $this->get_element_prop( 'allowed_child_elements' );
	}

	/**
	 * Get maximum nesting level.
	 *
	 * @return int
	 */
	public function get_max_nesting_level() {
		return $this->get_element_prop( 'max_nesting_level' );
	}

	/**
	 * Get maximum elements.
	 *
	 * @return int
	 */
	public function get_max_elements() {
		return $this->get_element_prop( 'max_elements' );
	}

	/**
	 * Get maximum instances of element.
	 *
	 * @return int
	 */
	public function get_max_instances() {
		return $this->get_element_prop( 'max_instances' );
	}

	/**
	 * Is unique elements container only.
	 *
	 * @return bool
	 */
	public function is_unique_elements() {
		return $this->get_element_prop( 'unique_elements' );
	}

	/**
	 * Get current nesting level.
	 *
	 * @return int
	 */
	public function get_nesting_level() {
		$nesting_level = function ( $element ) use ( &$nesting_level ) {
			if ( $element instanceof Element ) {
				$parent = $element->get_parent();

				if ( $parent ) {
					return 1 + $nesting_level( $parent );
				}
			}

			return 0;
		};

		return $nesting_level( $this );
	}

	/**
	 * Get index.
	 *
	 * @return int
	 */
	public function get_index() {
		return $this->index;
	}

	/**
	 * Get empty elements label.
	 *
	 * @return string
	 */
	public function get_empty_label() {
		return $this->get_element_prop( 'empty_label' );
	}

	/**
	 * Get add elements label.
	 *
	 * @return string
	 */
	public function get_add_label() {
		return $this->get_element_prop( 'add_label' );
	}

	/**
	 * Check if element is enabled.
	 *
	 * @return bool
	 */
	public function is_enabled() {
		return $this->get_element_prop( 'enabled' );
	}

	/**
	 * Get disabled message.
	 *
	 * @return string
	 */
	public function get_disabled_message() {
		return $this->get_element_prop( 'disabled_message' );
	}

	/**
	 * Collapse status.
	 *
	 * @return bool
	 */
	public function get_default_collapse_state() {
		return $this->get_element_prop( 'default_collapse_state' );
	}

	/**
	 * Get attribute sets.
	 *
	 * @return array
	 */
	public function get_attribute_sets() {
		return [];
	}

	/**
	 * Get attribute sets with initialized instances.
	 *
	 * @return array<int, array{instance: Attribute_Set, args: array}>
	 */
	public function get_all_attribute_sets() {
		$attribute_sets_instances = [];

		// Element defined attribute sets
		$attribute_sets = $this->get_attribute_sets();

		// Explicitly defined attribute sets
		if ( is_array( $this->get_element_prop( 'attribute_sets' ) ) ) {
			$attribute_sets = array_merge( $attribute_sets, $this->get_element_prop( 'attribute_sets' ) );
		}

		// Create attribute sets instance
		foreach ( $attribute_sets as $attribute_set => $args ) {
			if ( is_string( $args ) && is_numeric( $attribute_set ) ) {
				$attribute_set = $args;
			}

			// Reset args to array
			if ( ! is_array( $args ) ) {
				$args = [];
			}

			if ( class_exists( $attribute_set ) ) {
				/** @var Attribute_Set $instance */
				$instance = new $attribute_set( $this );

				$attribute_sets_instances[] = [
					'instance' => $instance,
					'args'     => $args,
				];
			}
		}

		return $attribute_sets_instances;
	}

	/**
	 * Create attributes.
	 *
	 * @return Attribute[]
	 */
	public function create_attributes() {
		$attributes = [];

		// Get attributes from attribute sets
		foreach ( $this->get_all_attribute_sets() as $attribute_set ) {
			$instance   = $attribute_set['instance'];
			$attributes = array_merge( $attributes, $instance->get_attributes() );
		}

		return $attributes;
	}

	/**
	 * Get attributes.
	 *
	 * @return Attribute[]
	 */
	public function get_attributes() {
		return $this->attributes;
	}

	/**
	 * Get single attribute.
	 *
	 * @param string $name
	 *
	 * @return Attribute|null
	 */
	public function get_attribute( $name ) {
		foreach ( $this->attributes as $attribute ) {
			if ( $attribute->get_name() === $name ) {
				return $attribute;
			}
		}

		return null;
	}

	/**
	 * Get attribute value.
	 *
	 * @param string $name
	 *
	 * @return mixed|null
	 */
	public function get_attribute_value( $name ) {
		if ( $attribute = $this->get_attribute( $name ) ) {
			return $attribute->get_value();
		}

		return null;
	}

	/**
	 * Remove attribute.
	 *
	 * @param string|array $name
	 *
	 * @return bool
	 */
	public function remove_attribute( $name ) {

		// Remove array attributes
		if ( is_array( $name ) ) {
			foreach ( $name as $attr_name ) {
				$this->remove_attribute( $attr_name );
			}

			return true;
		}

		// Attributes list
		$attributes = $this->get_attributes();

		// Filter attributes and remove selected
		$this->attributes = array_filter(
			$attributes,
			function ( $attribute ) use ( $name ) {
				return $name !== $attribute->get_name();
			}
		);

		return count( $this->get_attributes() ) !== count( $attributes );
	}

	/**
	 * Assign attribute values.
	 *
	 * @param array $attributes
	 */
	public function assign_attribute_values( $attributes ) {
		if ( is_array( $attributes ) ) {
			foreach ( $attributes as $attribute_data ) {
				if ( empty( $attribute_data['name'] ) ) {
					continue;
				}

				if ( $attribute = $this->get_attribute( $attribute_data['name'] ) ) {
					$attribute->get_value_object()->assign_value( $attribute_data['value'] ?? null );
				} else {
					if ( ! isset( $this->options['unregistered_attributes'] ) ) {
						$this->options['unregistered_attributes'] = [];
					}

					$this->options['unregistered_attributes'][ $attribute_data['name'] ] = $attribute_data['value'];
				}
			}
		}
	}

	/**
	 * Add dynamic attribute. If it already exists, applies the new configuration.
	 *
	 * @param string|int $attribute_name
	 * @param array      $attribute_args
	 */
	public function add_dynamic_attribute( $attribute_name, $attribute_args ) {
		$attribute_name = $attribute_args['name'] ?? $attribute_name;

		if ( is_numeric( $attribute_name ) ) {
			return;
		}

		// If attribute already exists modify it
		if ( $attribute = $this->get_attribute( $attribute_name ) ) {
			foreach ( $attribute_args as $attr_prop => $attr_val ) {
				$attribute->set_prop( $attr_prop, $attr_val );
			}
		} else {
			$this->attributes[] = Attribute::create( $attribute_name, $attribute_args );
		}
	}

	/**
	 * Get parent element.
	 *
	 * @return Element|null
	 */
	public function get_parent() {
		return $this->parent;
	}

	/**
	 * Get children.
	 *
	 * @return Elements|null
	 */
	public function get_children() {
		return $this->children;
	}

	/**
	 * Get options.
	 *
	 * @return array
	 */
	public function get_options() {
		return $this->options;
	}

	/**
	 * Get single option.
	 *
	 * @param string $name
	 * @param mixed  $default
	 *
	 * @return mixed
	 */
	public function get_option( $name, $default = null ) {
		if ( isset( $this->options[ $name ] ) ) {
			return $this->options[ $name ];
		}

		return $default;
	}

	/**
	 * Check element visibility.
	 *
	 * @return bool
	 */
	public function is_visible() {
		return $this->visible;
	}

	/**
	 * Get template name.
	 *
	 * @return string
	 */
	public function get_template_name() {
		return $this->get_name();
	}

	/**
	 * Get element template file.
	 *
	 * @return string
	 */
	public function get_template_file() {
		$file = sprintf( 'templates/lb-elements/%s.php', $this->get_template_name() );

		return locate_template( $file );
	}

	/**
	 * Init.
	 */
	public function init() {

		// Sort attribute tabs by default order
		$this->sort_attributes_default();
	}

	/**
	 * Get base class.
	 *
	 * @return string
	 */
	public function get_base_class() {
		return 'lb-element-' . str_replace( '_', '-', $this->get_name() );
	}

	/**
	 * Common class for all iterations depending on {element-base-class}-{index}.
	 *
	 * @return string
	 */
	public function get_common_class() {
		return $this->get_base_class() . '-' . implode( '', laborator_builder_traverse_index( $this ) );
	}

	/**
	 * DOM tag name of element.
	 *
	 * @return string
	 */
	public function get_dom_tag_name() {
		return apply_filters( 'laborator_builder_element_dom_tag_name', $this->tag_name, $this );
	}

	/**
	 * Get DOM attributes for the element.
	 *
	 * @return array
	 */
	public function get_dom_attributes() {
		return apply_filters( 'laborator_builder_element_dom_attributes', $this->dom_attributes, $this );
	}

	/**
	 * Get element class list.
	 *
	 * @return array
	 */
	public function get_dom_class() {
		$classes = [
			'lb-element',
			$this->get_base_class(),
			$this->get_common_class(),
		];

		// Add DOM classes from attribute sets
		foreach ( $this->get_all_attribute_sets() as $attribute_set ) {
			$instance = $attribute_set['instance'];

			$classes = array_merge( $classes, $instance->get_dom_class() );
		}

		return $classes;
	}

	/**
	 * Generate element styles (before rendering).
	 */
	public function generate_styles() {

		// Generate styles from attribute sets
		foreach ( $this->get_all_attribute_sets() as $attribute_set ) {
			$attribute_set['instance']->generate_styles();
		}
	}

	/**
	 * Element start (render function).
	 */
	public function element_start() {
		$attributes_list = [];
		$dom_attributes  = $this->get_dom_attributes();

		// Classes
		$class_list = $this->get_dom_class();

		if ( empty( $dom_attributes['class'] ) ) {
			$dom_attributes['class'] = [];
		} else {
			$dom_attributes['class'] = is_array( $dom_attributes['class'] ) ? $dom_attributes['class'] : explode( ' ', $dom_attributes['class'] );
		}

		$dom_attributes['class'] = array_filter( array_merge( $class_list, $dom_attributes['class'] ) );

		// Create attributes
		foreach ( $dom_attributes as $name => $value ) {
			if ( is_array( $value ) ) {
				$value = implode( ' ', $value );
			}

			// Escape attribute value
			$value = esc_attr( $value );

			$attributes_list[] = sprintf( ' %1$s="%2$s"', sanitize_title( $name ), $value );
		}

		printf( '<%1$s%2$s>', $this->get_dom_tag_name(), implode( ' ', $attributes_list ) );
	}

	/**
	 * Element content (render function).
	 */
	public function element_content() {
	}

	/**
	 * Element end (render function).
	 */
	public function element_end() {
		printf( '</%s>', $this->get_dom_tag_name() );
	}

	/**
	 * Export to JSON.
	 *
	 * @return array
	 */
	public function export() {
		return [
			'elementName'          => $this->get_name(),
			'title'                => $this->get_title(),
			'dynamicTitle'         => $this->get_dynamic_title(),
			'icon'                 => $this->get_icon(),
			'category'             => $this->get_category(),
			'childContainer'       => $this->is_child_container(),
			'allowedChildElements' => $this->get_allowed_child_elements(),
			'maxNestingLevel'      => $this->get_max_nesting_level(),
			'maxElements'          => $this->get_max_elements(),
			'maxInstances'         => $this->get_max_instances(),
			'uniqueElements'       => $this->is_unique_elements(),
			'emptyLabel'           => $this->get_empty_label(),
			'addLabel'             => $this->get_add_label(),
			'defaultCollapseState' => $this->get_default_collapse_state(),
			'enabled'              => $this->is_enabled(),
			'disabledMessage'      => $this->get_disabled_message(),
			'attributes'           => array_map( [ $this, 'attribute_export' ], $this->get_attributes() ),
			'options'              => $this->get_options(),
		];
	}

	/**
	 * Sort attributes by default order of tabs.
	 *
	 * This function is intended to be used only internally.
	 */
	private function sort_attributes_default() {
		$tabs_order = [
			Attribute::TAB_CONTENT,
			Attribute::TAB_STYLE,
			Attribute::TAB_ADVANCED,
		];

		// Set order indexes
		$tabs_order_index = [];

		foreach ( $this->attributes as $attribute ) {
			$attribute_tab = $attribute->get_tab();

			if ( ! isset( $tabs_order_index[ $attribute_tab ] ) ) {
				$tabs_order_index[ $attribute_tab ] = 0;
			}

			// Increase order index by ten
			if ( ! isset( $attribute->order ) ) {
				$attribute->order = ++$tabs_order_index[ $attribute_tab ] * 10;
			}
		}

		// Sort by tabs order index
		usort(
			$this->attributes,
			function ( $a, $b ) use ( $tabs_order ) {
				return array_search( $a->get_tab(), $tabs_order ) - array_search( $b->get_tab(), $tabs_order );
			}
		);
	}
}
