<?php
/**
 * Kalium WordPress Theme
 *
 * Masonry Portfolio widget.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class WPBakeryShortCode_Lab_Masonry_Portfolio extends WPBakeryShortCodesContainer {

	/**
	 * Register widget.
	 */
	public static function register_widget() {
		if ( ! kalium()->portfolio->is_enabled() ) {
			return;
		}

		if ( kalium()->is->ajax_action( 'vc_edit_form' ) && in_array( kalium()->request->request( 'tag' ), [ 'lab_masonry_portfolio','lab_masonry_portfolio_item' ], true ) ) {

			// Portfolio categories
			$portfolio_categories = wp_list_pluck(
				get_terms(
					[
						'taxonomy'   => 'portfolio_category',
						'hide_empty' => true,
					]
				),
				'slug',
				'name'
			);

			// Portfolio items
			$portfolio_items = wp_list_pluck(
				get_posts(
					[
						'post_type'      => kalium_supported_portfolio_post_types(),
						'posts_per_page' => -1,
					]
				),
				'ID',
				'post_title'
			);
		}

		$icon = kalium()->file_url( 'includes/wpbakery/assets/images/masonry.svg' );

		// Masonry Portfolio
		vc_map(
			[
				'base'                    => 'lab_masonry_portfolio',
				'name'                    => 'Masonry Portfolio',
				'description'             => 'Custom portfolio boxes',
				'content_element'         => true,
				'show_settings_on_create' => true,
				'as_parent'               => [ 'only' => 'lab_masonry_portfolio_item' ],
				'category'                => [ 'Laborator', 'Portfolio' ],
				'icon'                    => $icon,
				'params'                  => [
					[
						'type'        => 'textfield',
						'heading'     => 'Title',
						'param_name'  => 'title',
						'value'       => '',
						'description' => 'Main title of this widget. (Optional)',
					],
					[
						'type'        => 'dropdown',
						'heading'     => 'Title tag',
						'param_name'  => 'title_tag',
						'std'         => 'h2',
						'value'       => [
							'H1' => 'h1',
							'H2' => 'h2',
							'H3' => 'h3',
							'H4' => 'h4',
						],
						'description' => 'Select title tag for widget title.',
					],
					[
						'type'        => 'textarea',
						'heading'     => 'Description',
						'param_name'  => 'description',
						'value'       => '',
						'description' => 'Description under main portfolio title. (Optional)',
					],
					[
						'type'        => 'dropdown',
						'heading'     => 'Category Filter',
						'param_name'  => 'category_filter',
						'value'       => [
							'Yes' => 'yes',
							'No'  => 'no',
						],
						'description' => 'Show category filter above the portfolio items.',
					],
					[
						'type'        => 'dropdown',
						'heading'     => 'Default filter category',
						'param_name'  => 'default_filter_category',
						'value'       => array_merge(
							[
								'Default (All)' => 'default',
							],
							$portfolio_categories ?? []
						),
						'description' => 'Set default category to filter portfolio items at first page load.',
						'dependency'  => [
							'element' => 'category_filter',
							'value'   => [ 'yes' ],
						],
					],
					[
						'type'       => 'checkbox',
						'heading'    => 'Hide "All" filter link from portfolio',
						'param_name' => 'filter_category_hide_all',
						'value'      => [
							'Yes' => 'yes',
						],
						'dependency' => [
							'element'            => 'default_filter_category',
							'value_not_equal_to' => [ 'default' ],
						],
					],
					[
						'group'       => 'Layout',
						'type'        => 'dropdown',
						'heading'     => 'Reveal Effect',
						'param_name'  => 'reveal_effect',
						'std'         => 'inherit',
						'value'       => [
							'Inherit from Theme Options'  => 'inherit',
							'None'                        => 'none',
							'Fade'                        => 'fade',
							'Slide and Fade'              => 'slidenfade',
							'Zoom In'                     => 'zoom',
							'Fade (one by one)'           => 'fade-one',
							'Slide and Fade (one by one)' => 'slidenfade-one',
							'Zoom In (one by one)'        => 'zoom-one',
						],
						'description' => 'Reveal effect for portfolio items.',
					],

					[
						'group'       => 'Layout',
						'type'        => 'dropdown',
						'param_name'  => 'portfolio_spacing',
						'value'       => [
							'Inherit from Theme Options' => 'inherit',
							'Yes'                        => 'yes',
							'No'                         => 'no',
						],
						'heading'     => 'Item Spacing',
						'description' => 'Spacing between portfolio items.',
					],
					[
						'group'       => 'Pagination',
						'type'        => 'dropdown',
						'heading'     => 'Pagination Type',
						'param_name'  => 'pagination_type',
						'description' => 'Select pagination type to use with this widget.',
						'std'         => 'static',
						'value'       => [
							'No "Show More" button'     => 'hide',
							'Static "Show More" button' => 'static',
							'Endless Pagination'        => 'endless',
						],
					],
					[
						'group'       => 'Pagination',
						'type'        => 'vc_link',
						'heading'     => 'More Link',
						'param_name'  => 'more_link',
						'value'       => '',
						'description' => 'This will show "More" button in the end of portfolio items.',
						'dependency'  => [
							'element' => 'pagination_type',
							'value'   => [ 'static' ],
						],
					],
					[
						'group'      => 'Pagination',
						'type'       => 'checkbox',
						'heading'    => 'Auto Reveal',
						'param_name' => 'endless_auto_reveal',
						'value'      => [
							'Yes' => 'yes',
						],
						'dependency' => [
							'element' => 'pagination_type',
							'value'   => [ 'endless' ],
						],
					],
					[
						'group'      => 'Pagination',
						'type'       => 'textfield',
						'heading'    => 'Show more button text',
						'param_name' => 'endless_show_more_button_text',
						'value'      => 'Show More',
						'dependency' => [
							'element' => 'pagination_type',
							'value'   => [ 'endless' ],
						],
					],
					[
						'group'      => 'Pagination',
						'type'       => 'textfield',
						'heading'    => 'No more items to show text',
						'param_name' => 'endless_no_more_items_button_text',
						'value'      => 'No more portfolio items to show',
						'dependency' => [
							'element' => 'pagination_type',
							'value'   => [ 'endless' ],
						],
					],
					[
						'group'       => 'Pagination',
						'type'        => 'textfield',
						'heading'     => 'Items per Page',
						'param_name'  => 'per_page',
						'value'       => '',
						'description' => 'Set the initial number of items you want to show (leave empty to show all added items)',
						'dependency'  => [
							'element' => 'pagination_type',
							'value'   => [ 'endless' ],
						],
					],
					[
						'type'        => 'textfield',
						'heading'     => 'Extra class name',
						'param_name'  => 'el_class',
						'description' => 'If you wish to style particular content element differently, then use this field to add a class name and then refer to it in your css file.',
					],
					[
						'type'       => 'css_editor',
						'heading'    => 'Css',
						'param_name' => 'css',
						'group'      => 'Design options',
					],
				],
				'js_view'                 => 'VcColumnView',
			]
		);

		// Portfolio Item (child of Masonry Portfolio)
		vc_map(
			[
				'base'             => 'lab_masonry_portfolio_item',
				'name'             => 'Portfolio Item',
				'description'      => 'Insert single item',
				'category'         => 'Laborator',
				'content_element'  => true,
				'icon'             => $icon,
				'as_child'         => [ 'only' => 'lab_masonry_portfolio' ],
				'admin_enqueue_js' => kalium()->file_url( 'includes/wpbakery/assets/js/init-lab-masonry.js' ),
				'params'           => [
					[
						'type'        => 'dropdown',
						'heading'     => 'Box Size',
						'admin_label' => true,
						'param_name'  => 'box_size',
						'value'       => [
							'25% - small'   => '3x3',
							'25% - medium'  => '3x4',
							'25% - large'   => '3x6',

							'33% - small'   => '4x3',
							'33% - medium'  => '4x4',
							'33% - large'   => '4x6',

							'40% - small'   => '5x3',
							'40% - medium'  => '5x4',
							'40% - large'   => '5x6',

							'50% - small'   => '6x3',
							'50% - medium'  => '6x4',
							'50% - large'   => '6x6',

							'65% - small'   => '8x3',
							'65% - medium'  => '8x4',
							'65% - large'   => '8x6',

							'75% - small'   => '9x3',
							'75% - medium'  => '9x4',
							'75% - large'   => '9x6',

							'100% - small'  => '12x4',
							'100% - medium' => '12x5',
							'100% - large'  => '12x6',
						],
						'description' => 'Select portfolio type to show items.',
					],
					[
						'type'        => 'dropdown',
						'heading'     => 'Portfolio Item',
						'admin_label' => true,
						'param_name'  => 'portfolio_id',
						'value'       => $portfolio_items ?? [],
						'description' => 'Select an item from portfolio to show in masonry grid. Duplicate Items will be removed.',
					],
				],
			]
		);
	}

	/**
	 * Shortcode content.
	 *
	 * @param array  $atts
	 * @param string $content
	 *
	 * @return string
	 */
	public function content( $atts, $content = '' ) {
		static $instance_id = 0;

		// Portfolio is not enabled
		if ( ! kalium()->portfolio->is_enabled() ) {
			return null;
		}

		// Attributes
		$atts = vc_map_get_attributes( $this->getShortcode(), $atts );

		// Set loop options
		$custom_loop_options = [
			'url'            => get_permalink( get_queried_object() ),
			'custom_masonry' => [],
		];

		// Masonry items
		if ( preg_match_all( '/' . get_shortcode_regex() . '/', $content, $items ) ) {
			foreach ( $items[0] as $item ) {
				$portfolio_item = preg_replace( '/^\[[^\s]+/i', '', substr( $item, 0, -1 ) );
				$portfolio_item = $this->prepareAtts( shortcode_parse_atts( $portfolio_item ) );

				if ( ! isset( $portfolio_item['portfolio_id'] ) ) {
					continue;
				}

				// Portfolio item ID
				$id = $portfolio_item['portfolio_id'];

				// WPML translate object IDs
				$id = apply_filters( 'wpml_object_id', $id, 'portfolio', true );

				$custom_loop_options['custom_masonry'][] = [
					'id'       => $id,
					'box_size' => $portfolio_item['box_size'] ?? '3x3',
				];
			}
		}

		// No portfolio items
		if ( empty( $custom_loop_options['custom_masonry'] ) ) {
			return null;
		}

		// Heading options
		$custom_loop_options['heading']['title']['tag_name'] = $atts['title_tag'];
		$custom_loop_options['heading']['title']['text']     = $atts['title'];
		$custom_loop_options['heading']['title']['content']  = $atts['description'];

		if ( empty( $atts['title'] ) && empty( $atts['description'] ) ) {
			$custom_loop_options['heading']['title']['visible'] = false;
		}

		// Category filter
		$custom_loop_options['filtering']['enabled'] = kalium_validate_boolean( $atts['category_filter'] );

		// Default category
		$default_category = $atts['default_filter_category'];
		$hide_all         = $atts['filter_category_hide_all'];

		if ( 'default' !== $default_category ) {
			$custom_loop_options['filtering']['current']['portfolio_category'] = $default_category;

			if ( $hide_all ) {
				$custom_loop_options['filtering']['reset'] = false;
			}
		}

		// Reveal effect
		if ( 'inherit' !== $atts['reveal_effect'] ) {
			$custom_loop_options['item']['aov']['enabled']          = 'none' !== $atts['reveal_effect'];
			$custom_loop_options['item']['aov']['legacy_animation'] = $atts['reveal_effect'];
			$custom_loop_options['item']['aov']['animation']        = null;
		}

		// Grid spacing
		$grid_spacing = $atts['portfolio_spacing'];

		if ( 'yes' === $grid_spacing ) {
			$custom_loop_options['grid']['gap'] = 30;
		} elseif ( 'no' === $grid_spacing ) {
			$custom_loop_options['grid']['gap'] = 0;
		}

		// Pagination
		switch ( $atts['pagination_type'] ) {

			// Load more
			case 'endless':
				$custom_loop_options['pagination']['enabled'] = true;
				$custom_loop_options['pagination']['type']    = 'load-more';

				$custom_loop_options['pagination']['load_more']['infinite_scroll'] = kalium_validate_boolean( $atts['endless_auto_reveal'] );
				$custom_loop_options['pagination']['load_more']['view_more_text']  = $atts['endless_show_more_button_text'];
				$custom_loop_options['pagination']['load_more']['last_page_text']  = $atts['endless_no_more_items_button_text'];

				$custom_loop_options['grid']['items_per_page'] = $atts['per_page'];

				break;

			// Static more link
			case 'static':
				$more_link = vc_build_link( $atts['more_link'] );

				$custom_loop_options['pagination']['type'] = 'more-link';

				$custom_loop_options['pagination']['enabled']             = true;
				$custom_loop_options['pagination']['more_link']['url']    = $more_link['url'];
				$custom_loop_options['pagination']['more_link']['target'] = $more_link['target'];
				$custom_loop_options['pagination']['more_link']['text']   = $more_link['title'];
				break;

			// Hide
			case 'hide':
				$custom_loop_options['pagination']['enabled'] = false;
				break;
		}

		// Element class
		$css       = $atts['css'];
		$el_class  = $atts['el_class'];
		$class     = $this->getExtraClass( $el_class );
		$css_class = apply_filters( VC_SHORTCODE_CUSTOM_CSS_FILTER_TAG, $class, $this->settings['base'], $atts ) . vc_shortcode_custom_css_class( $css, ' ' );

		$classes = [
			'lab-masonry-portfolio-items',
			'portfolio',
			$css_class,
		];

		// Output template
		ob_start();

		// Init portfolio options
		kalium_init_portfolio_loop_options( 'masonry-wpb-' . ++$instance_id, $custom_loop_options );

		// Heading
		kalium_portfolio_loop_heading();

		// Portfolio loop
		kalium_portfolio_items_loop();

		// Pagination
		kalium_portfolio_items_pagination();

		// Reset portfolio loop options
		kalium_reset_portfolio_loop_options();

		return kalium_element(
			[
				'tag_name'   => 'div',
				'attributes' => [
					'class' => $classes,
				],
				'content'    => ob_get_clean(),
			]
		);
	}
}

// Register widget
WPBakeryShortCode_Lab_Masonry_Portfolio::register_widget();
