<?php if ( ! defined('KERA_THEME_DIR')) exit('No direct script access allowed');
/**
 * Custom template tags for kera
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 * @package WordPress
 * @subpackage Kera
 * @since Kera 1.0
 */

if ( ! function_exists( 'kera_tbay_comment_nav' ) ) :
/**
 * Display navigation to next/previous comments when applicable.
 *
 * @since Kera 1.0
 */
function kera_tbay_comment_nav() {
	// Are there comments to navigate through?
	if ( get_comment_pages_count() > 1 && get_option( 'page_comments' ) ) :
	?>
	<nav class="navigation comment-navigation">
		<h2 class="screen-reader-text"><?php esc_html_e( 'Comment navigation', 'kera' ); ?></h2>
		<div class="nav-links">
			<?php
				if ( $prev_link = get_previous_comments_link( esc_html__( 'Older Comments', 'kera' ) ) ) :
					printf( '<div class="nav-previous">%s</div>', $prev_link );
				endif;

				if ( $next_link = get_next_comments_link( esc_html__( 'Newer Comments', 'kera' ) ) ) :
					printf( '<div class="nav-next">%s</div>', $next_link );
				endif;
			?>
		</div><!-- .nav-links -->
	</nav><!-- .comment-navigation -->
	<?php
	endif;
}
endif;

if ( ! function_exists( 'kera_tbay_entry_meta' ) ) :
/**
 * Prints HTML with meta information for the categories, tags.
 *
 * @since Kera 1.0
 */
function kera_tbay_entry_meta() {
	if ( is_sticky() && is_home() && ! is_paged() ) {
		printf( '<span class="sticky-post">%s</span>', esc_html__( 'Featured', 'kera' ) );
	}

	$format = get_post_format();
	if ( current_theme_supports( 'post-formats', $format ) ) {
		printf( '<span class="entry-format">%1$s<a href="%2$s">%3$s</a></span>',
			sprintf( '<span class="screen-reader-text">%s </span>', _x( 'Format', 'Used before post format.', 'kera' ) ),
			esc_url( get_post_format_link( $format ) ),
			get_post_format_string( $format )
		);
	}

	if ( in_array( get_post_type(), array( 'post', 'attachment' ) ) ) {
		$time_string = '<time class="published updated" datetime="%1$s">%2$s</time>';

		if ( get_the_time( 'U' ) !== get_the_modified_time( 'U' ) ) {
			$time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
		}

		$time_string = sprintf( $time_string,
			esc_attr( get_the_date( 'c' ) ),
			get_the_date(),
			esc_attr( get_the_modified_date( 'c' ) ),
			get_the_modified_date()
		);

		printf( '<span class="posted-on"><span class="screen-reader-text">%1$s </span><a href="%2$s" rel="bookmark">%3$s</a></span>',
			_x( 'Posted on', 'Used before publish date.', 'kera' ),
			esc_url( get_permalink() ),
			$time_string
		);
	}

	if ( 'post' == get_post_type() ) {
		if ( is_singular() || is_multi_author() ) {
			printf( '<span class="byline"><span class="author vcard"><span class="screen-reader-text">%1$s </span><a class="url fn n" href="%2$s">%3$s</a></span></span>',
				_x( 'Author', 'Used before post author name.', 'kera' ),
				esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ),
				get_the_author()
			);
		}

		$categories_list = get_the_category_list( _x( ', ', 'Used between list items, there is a space after the comma.', 'kera' ) );
		if ( $categories_list && kera_tbay_categorized_blog() ) {
			printf( '<span class="cat-links"><span class="screen-reader-text">%1$s </span>%2$s</span>',
				_x( 'Categories', 'Used before category names.', 'kera' ),
				$categories_list
			);
		}

		$tags_list = get_the_tag_list( '', _x( ', ', 'Used between list items, there is a space after the comma.', 'kera' ) );
		if ( $tags_list ) {
			printf( '<span class="tags-links"><span class="screen-reader-text">%1$s </span>%2$s</span>',
				_x( 'Tags', 'Used before tag names.', 'kera' ),
				$tags_list
			);
		}
	}

	if ( is_attachment() && wp_attachment_is_image() ) {
		// Retrieve attachment metadata.
		$metadata = wp_get_attachment_metadata();

		printf( '<span class="full-size-link"><span class="screen-reader-text">%1$s </span><a href="%2$s">%3$s &times; %4$s</a></span>',
			_x( 'Full size', 'Used before full size attachment link.', 'kera' ),
			esc_url( wp_get_attachment_url() ),
			$metadata['width'],
			$metadata['height']
		);
	}

	if ( ! is_single() && ! post_password_required() && ( comments_open() || get_comments_number() ) ) {
		echo '<span class="comments-link">';
		/* translators: %s: post title */
		comments_popup_link( sprintf( esc_html__( 'Leave a comment<span class="screen-reader-text"> on %s</span>', 'kera' ), get_the_title() ) );
		echo '</span>';
	}
}
endif;

/**
 * Determine whether blog/site has more than one category.
 *
 * @since Kera 1.0
 *
 * @return bool True of there is more than one category, false otherwise.
 */
function kera_tbay_categorized_blog() {
	if ( false === ( $all_the_cool_cats = get_transient( 'kera_tbay_categories' ) ) ) {
		// Create an array of all the categories that are attached to posts.
		$all_the_cool_cats = get_categories( array(
			'fields'     => 'ids',
			'hide_empty' => 1,

			// We only need to know if there is more than one category.
			'number'     => 2,
		) );

		// Count the number of categories that are attached to the posts.
		$all_the_cool_cats = count( $all_the_cool_cats );

		set_transient( 'kera_tbay_categories', $all_the_cool_cats );
	}

	if ( $all_the_cool_cats > 1 ) {
		// This blog has more than 1 category so kera_tbay_categorized_blog should return true.
		return true;
	} else {
		// This blog has only 1 category so kera_tbay_categorized_blog should return false.
		return false;
	}
}

/**
 * Displays the post thumbnail.
 *
 * @since Kera 1.0
 * @return void
 */
function kera_tbay_post_thumbnail() {
    if (post_password_required() || is_attachment() || !has_post_thumbnail()) {
        return;
    }

    $image_size = apply_filters('kera_archive_blog_size_image', 'medium');

    if (is_singular()) {
        echo '<div class="post-thumbnail">';
        the_post_thumbnail('full', ['alt' => esc_attr(get_the_title())]);
        echo '</div>';
    } else {
        echo '<a class="post-' . esc_attr($image_size) . '" href="' . esc_url(get_permalink()) . '" aria-hidden="true">';
        the_post_thumbnail($image_size, ['alt' => esc_attr(get_the_title())]);
        echo '</a>';
    }
}

/**
 * Displays post categories with caching.
 *
 * @param WP_Post $post The post object.
 * @return void
 */
function kera_tbay_post_categories($post) {
    $transient_name = 'kera_post_categories_' . $post->ID;
    $output = get_transient($transient_name);

    if (false === $output) {
        $categories = get_categories(['include' => wp_get_post_categories($post->ID)]);
        $total = count($categories);
        $output = '';

        foreach ($categories as $index => $category) {
            $output .= '<a href="' . esc_url(get_category_link($category->term_id)) . '" class="categories-name">';
            $output .= esc_html($category->name);
            $output .= ($index + 1 < $total) ? ', ' : '';
            $output .= '</a>';
        }

        set_transient($transient_name, $output, WEEK_IN_SECONDS);
    }

    echo $output;
}

if ( ! function_exists( 'kera_tbay_get_link_url' ) ) :
/**
 * Return the post URL.
 *
 * Falls back to the post permalink if no URL is found in the post.
 *
 * @since Kera 1.0
 *
 * @see get_url_in_content()
 *
 * @return string The Link format URL.
 */
function kera_tbay_get_link_url() {
	$has_url = get_url_in_content( get_the_content() );

	return $has_url ? $has_url : apply_filters( 'the_permalink', get_permalink() );
}
endif;

if ( ! function_exists( 'kera_tbay_excerpt_more' ) && ! is_admin() ) :
/**
 * Replaces "[...]" (appended to automatically generated excerpts) with ... and a 'Continue reading' link.
 *
 * @since Kera 1.0
 *
 * @return string 'Continue reading' link prepended with an ellipsis.
 */
function kera_tbay_excerpt_more( $more ) {
	$link = sprintf( '<a href="%1$s" class="more-link">%2$s</a>',
		esc_url( get_permalink( get_the_ID() ) ),
		/* translators: %s: Name of current post */
		sprintf( esc_html__( 'Continue reading %s', 'kera' ), '<span class="screen-reader-text">' . get_the_title( get_the_ID() ) . '</span>' )
		);
	return ' &hellip; ' . $link;
}
add_filter( 'excerpt_more', 'kera_tbay_excerpt_more' );
endif;

/**
 * ------------------------------------------------------------------------------------------------
 * Display meta comment for a specific post
 * ------------------------------------------------------------------------------------------------
 */
if( ! function_exists( 'kera_post_meta_comment' )) {
	function kera_post_meta_comment( $comments = true, $comments_text = false ) {
		if (!$comments) return;
		$text_domain =   esc_html__(' comments','kera');    
		if( get_comments_number() == 1) {
		    $text_domain = esc_html__(' comment','kera');
		}

		if( !$comments_text ) $text_domain ='';
		
		if( $comments && comments_open() ): ?>
				<?php comments_popup_link( 
					'0' .'<span>'. $text_domain .'</span>',  
					'1' .'<span>'. $text_domain .'</span>',  
					'%' .'<span>'. $text_domain .'</span>'
				); ?>
			</span>
		<?php endif; 
	}
}
/**
 * Displays meta information for a specific post.
 *
 * @param array $atts Attributes for meta display.
 * @return void
 */
function kera_post_meta($atts = []) {
    if (get_post_type() !== 'post') {
        return;
    }

    $atts = shortcode_atts([
        'date'        => 1,
        'author'      => 1,
        'author_img'  => 1,
        'comments'    => 0,
        'comments_text' => 0,
        'tags'        => 0,
        'cats'        => 1,
        'edit'        => 1,
    ], $atts, 'kera_post_meta');

    ob_start();
    ?>
    <ul class="entry-meta-list">
        <?php if ($atts['author']) : ?>
            <li class="entry-author">
                <?php if ($atts['author_img']) : ?>
                    <?php echo get_avatar(kera_tbay_get_id_author_post(), 70, '', esc_attr(get_the_author())); ?>
                <?php endif; ?>
                <?php the_author_posts_link(); ?>
            </li>
        <?php endif; ?>

        <?php if ($atts['date']) : ?>
            <li class="entry-date">
                <?php echo is_single() ? kera_time_single_post() : kera_time_link(); ?>
            </li>
        <?php endif; ?>

        <?php if ($atts['comments'] && comments_open()) : ?>
            <li><?php kera_post_meta_comment($atts['comments'], $atts['comments_text']); ?></li>
        <?php endif; ?>

        <?php if ($atts['tags'] && get_the_tag_list()) : ?>
            <li class="entry-tags"><?php echo get_the_tag_list(''); ?></li>
        <?php endif; ?>

        <?php if ($atts['cats'] && get_the_category_list(', ')) : ?>
            <li class="entry-category"><?php kera_the_post_category_full(); ?></li>
        <?php endif; ?>

        <?php if ($atts['edit'] && is_user_logged_in()) : ?>
            <li class="edit-link"><?php edit_post_link(esc_html__('Edit', 'kera')); ?></li>
        <?php endif; ?>
    </ul>
    <?php
    echo apply_filters('kera_post_meta_output', ob_get_clean(), $atts);
}

/**
 * ------------------------------------------------------------------------------------------------
 * The archive title
 * ------------------------------------------------------------------------------------------------
 */
if( ! function_exists( 'kera_post_archive_the_title' )) {
	function kera_post_archive_the_title() {
	    if (get_the_title()) {
	    ?>
	        <h3 class="entry-title">
			   <a href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
	        </h3>
	    <?php
		}	            

	}
}

/**
 * ------------------------------------------------------------------------------------------------
 * The short description archive
 * ------------------------------------------------------------------------------------------------
 */
if( ! function_exists( 'kera_post_archive_the_short_description' )) {
	function kera_post_archive_the_short_description() {
		if (empty(get_the_excerpt()) || get_the_excerpt() == '&nbsp;') {
            return;
        }
		if ( has_excerpt()) { ?>
			<div class="entry-description">
				<?php echo kera_tbay_substring(get_the_excerpt(), 30, '[...]' ); ?>
			</div>
		<?php } else {
			?>
				<div class="entry-description">
					<?php echo kera_tbay_substring(get_the_content(), 20, '...' ); ?>
				</div>
			<?php
		}         

	}
}
/**
 * ------------------------------------------------------------------------------------------------
 * The read more archive
 * ------------------------------------------------------------------------------------------------
 */
if( ! function_exists( 'kera_post_archive_the_read_more' )) {
	function kera_post_archive_the_read_more() {
		?>
			<?php $custom_readmore	= kera_tbay_get_config('text_readmore', esc_html__('Continue Reading', 'kera')); ?>
			<div class="more">
				<a href="<?php the_permalink(); ?>" class="readmore" title="<?php echo esc_attr($custom_readmore); ?>"><?php echo esc_html($custom_readmore); ?></a>
			</div>
		<?php
	}
}

if ( ! function_exists( 'kera_product_ajax_search_sku' ) ) {
	function kera_product_ajax_search_sku( $where ) {
        if ( !defined('TBAY_ELEMENTOR_ACTIVED') ) return $where;

		if ( ! empty( $_REQUEST['query'] ) ) {
			$s = sanitize_text_field( $_REQUEST['query'] );

            return kera_sku_search_query( $where, $s );
		}

		return $where;
	}
}

if (! function_exists('kera_search_title_query')) {
    function kera_search_title_query($search, $wp_query)
    {
        if (! empty($search) && ! empty($wp_query->query_vars['search_terms'])) {
            global $wpdb;

            $q = $wp_query->query_vars;
            $n = ! empty($q['exact']) ? '' : '%';

            $search = array();

            foreach (( array ) $q['search_terms'] as $term) {
                $search[] = $wpdb->prepare("$wpdb->posts.post_title LIKE %s", $n . $wpdb->esc_like($term) . $n);
            }

            if (! is_user_logged_in()) {
                $search[] = "$wpdb->posts.post_password = ''";
            }

            $search = ' AND ' . implode(' AND ', $search);
        }

        return $search;
    }
}

/**
 * ------------------------------------------------------------------------------------------------
 * Ajax search
 * ------------------------------------------------------------------------------------------------
 */
if ( ! function_exists( 'kera_init_product_search' ) ) {
	function kera_init_product_search() {
        if( kera_is_Woocommerce_activated() ) {
            if ( apply_filters( 'kera_search_query_in', kera_tbay_get_config('search_query_in', 'title') === 'all' ) ) {
                add_filter( 'posts_search', 'kera_product_search_sku', 9 );
            } else {
                add_filter('posts_search', 'kera_product_search_title', 20, 2);
            }
        }
	}
	add_action( 'init', 'kera_init_product_search', 99 ); 
}

if ( ! function_exists( 'kera_product_search_sku' ) ) {
	add_filter( 'posts_search', 'kera_product_search_sku', 9 );
	function kera_product_search_sku( $where ) {
        if( !(defined('TBAY_ELEMENTOR_ACTIVED') && TBAY_ELEMENTOR_ACTIVED) ) return $where;

		global $wp;
		
		if ( ( is_admin() ) //if ((is_admin() && 'edit.php' != $pagenow) 
				|| !is_search()  
				|| !isset( $wp->query_vars['s'] ) 
				//post_types can also be arrays..
				|| (isset( $wp->query_vars['post_type'] ) && 'product' != $wp->query_vars['post_type'] )
				|| (isset( $wp->query_vars['post_type'] ) && is_array( $wp->query_vars['post_type'] ) && !in_array( 'product', $wp->query_vars['post_type'] ) ) 
				) {
			return $where;
		}

		$s = $wp->query_vars['s'];

        return kera_sku_search_query( $where, $s );
	}
}

if ( ! function_exists( 'kera_sku_search_query' ) ) {
	function kera_sku_search_query( $where, $s ) {
		global $wpdb;

		$search_ids = array();
		$terms = explode( ',', $s );

		foreach ( $terms as $term ) {
			//Include the search by id if admin area.
			if ( is_admin() && is_numeric( $term ) ) {
				$search_ids[] = $term;
			}
			// search for variations with a matching sku and return the parent.

			$sku_to_parent_id = $wpdb->get_col( $wpdb->prepare( "SELECT p.post_parent as post_id FROM {$wpdb->posts} as p join {$wpdb->wc_product_meta_lookup} ml on p.ID = ml.product_id and ml.sku LIKE '%%%s%%' where p.post_parent <> 0 group by p.post_parent", wc_clean( $term ) ) );

			//Search for a regular product that matches the sku.
			$clean_term = wc_clean( $term );
			$sku_to_id = $wpdb->get_results( "SELECT product_id FROM {$wpdb->wc_product_meta_lookup} WHERE sku LIKE '%{$clean_term}%';", ARRAY_N );

			$sku_to_id_results = array();
			if ( is_array( $sku_to_id ) ) {
				foreach ( $sku_to_id as $id ) {
					$sku_to_id_results[] = $id[0];
				}
			}

			$search_ids = array_merge( $search_ids, $sku_to_id_results, $sku_to_parent_id );
		}

		$search_ids = array_filter( array_map( 'absint', $search_ids ) );

		if ( sizeof( $search_ids ) > 0 ) {
			$where = str_replace( ')))', ")) OR ( {$wpdb->posts}.ID IN (" . implode( ',', $search_ids ) . ")))", $where );
		}

		return $where;
	}
}

if (! function_exists('kera_product_search_title')) {
    function kera_product_search_title($where, $wp_query)
    {
        if( !(defined('TBAY_ELEMENTOR_ACTIVED') && TBAY_ELEMENTOR_ACTIVED) ) return $where;

        $where = kera_search_title_query($where, $wp_query);

        return $where;
    }
}

if ( ! function_exists( 'wp_body_open' ) ) :
	/**
	 * Fire the wp_body_open action.
	 *
	 * Added for backwards compatibility to support pre 5.2.0 WordPress versions.
	 *
	 * @since Kera 1.5
	 */
	function wp_body_open() {
		/**
		 * Triggered after the opening <body> tag.
		 *
		 * @since Kera 1.5
		 */
		do_action( 'wp_body_open' );
	}
endif;


if ( ! function_exists( 'kera_time_link' ) ) :
/**
 * Gets a nicely formatted string for the published date.
 */
function kera_time_link() {
	$time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
	$time_string = sprintf( $time_string,
		get_the_date( DATE_W3C ), 
		get_the_date(),
		get_the_modified_date( DATE_W3C ),
		get_the_modified_date()
	);

	// Wrap the time string in a link, and preface it with 'Posted on'.
	return sprintf( 
		/* translators: %s: post date */
		__( '%sPosted on%s %s', 'kera' ), '<span class="screen-reader-text">', '</span>',
		'<a href="' . esc_url( get_permalink() ) . '" rel="bookmark">'. $time_string . '</a>'
	);
}
endif;

if ( ! function_exists( 'kera_time_single_post' ) ) :
/**
 * Gets a nicely formatted string for the published date.
 */
function kera_time_single_post() {
	$time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
	$time_string = sprintf( $time_string,
		get_the_date( DATE_W3C ), 
		get_the_date(),
		get_the_modified_date( DATE_W3C ),
		get_the_modified_date()
	);

	// Wrap the time string in a link, and preface it with 'Posted on'.
	return sprintf( 
		/* translators: %s: post date */
		__( '%sPosted on%s %s', 'kera' ), '<span class="screen-reader-text">', '</span>',
		$time_string
	);
}
endif;

if ( ! function_exists( 'kera_tbay_icon_post_formats' ) ) {
	function kera_tbay_icon_post_formats() {
		$post_id 			= get_the_ID();
		$post_format 	= get_post_format($post_id);
		
		echo '<span class="post-type">';
		switch ($post_format) {			
			case 'aside':
				echo '<i class="tb-icon tb-icon-document"></i>';
				break;
			case 'audio':
				echo '<i class="tb-icon tb-icon-music-note2"></i>';
				break;				
			case 'gallery':
				echo '<i class="tb-icon tb-icon-pictures"></i>';
				break;	
			case 'image':
				echo '<i class="tb-icon tb-icon-picture2"></i>';
				break;									
			case 'video':
				echo '<i class="tb-icon tb-icon-clapboard-play"></i>';
				break;				
			
			default:
				echo '<i class="tb-icon tb-icon-pushpin"></i>';
				break;
			}
		echo '</span>';
	}
}