<?php

namespace App\Console\Commands;

use App\Models\License;
use App\Models\Product;
use App\Services\InvoiceService;
use App\Services\EmailService;
use Illuminate\Console\Command;
use Carbon\Carbon;

class GenerateRenewalInvoices extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'licenses:generate-renewal-invoices {--days=7 : Number of days before expiry to generate invoices}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate renewal invoices for licenses that are about to expire';

    protected InvoiceService $invoiceService;
    protected EmailService $emailService;

    public function __construct(InvoiceService $invoiceService, EmailService $emailService)
    {
        parent::__construct();
        $this->invoiceService = $invoiceService;
        $this->emailService = $emailService;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $daysBeforeExpiry = (int) $this->option('days');
        $expiryDate = Carbon::now()->addDays($daysBeforeExpiry);

        $this->info("Generating renewal invoices for licenses expiring within {$daysBeforeExpiry} days...");

        // Find licenses that are about to expire and don't have pending renewal invoices
        $licenses = License::with(['user', 'product', 'invoices'])
            ->where('status', 'active')
            ->where('license_expires_at', '<=', $expiryDate)
            ->where('license_expires_at', '>', Carbon::now())
            ->get()
            ->filter(function ($license) {
                // Check if there's already a pending renewal invoice
                return !$license->invoices()
                    ->where('type', 'renewal')
                    ->where('status', 'pending')
                    ->exists();
            });

        $generatedCount = 0;
        $emailSentCount = 0;

        foreach ($licenses as $license) {
            try {
                // Generate renewal invoice
                $invoice = $this->generateRenewalInvoice($license);
                
                if ($invoice) {
                    $generatedCount++;
                    $this->line("Generated renewal invoice for license {$license->license_key} (Product: {$license->product->name})");

                    // Send email notifications
                    if ($this->sendRenewalNotifications($license, $invoice)) {
                        $emailSentCount++;
                    }
                }
            } catch (\Exception $e) {
                $this->error("Failed to generate renewal invoice for license {$license->license_key}: " . $e->getMessage());
            }
        }

        $this->info("Generated {$generatedCount} renewal invoices and sent {$emailSentCount} email notifications.");
        
        return Command::SUCCESS;
    }

    /**
     * Generate renewal invoice for a license
     */
    protected function generateRenewalInvoice(License $license): ?\App\Models\Invoice
    {
        $product = $license->product;
        
        if (!$product) {
            return null;
        }

        // Calculate renewal price based on product settings
        $renewalPrice = $product->renewal_price ?? $product->price ?? 0;
        
        if ($renewalPrice <= 0) {
            $this->warn("No renewal price set for product {$product->name}, skipping invoice generation.");
            return null;
        }

        // Calculate new expiry date based on renewal period
        $newExpiryDate = $this->calculateNewExpiryDate($license, $product);
        
        // Create renewal invoice
        $invoice = $this->invoiceService->createRenewalInvoice($license, [
            'amount' => $renewalPrice,
            'description' => "Renewal for {$product->name} - License {$license->license_key}",
            'due_date' => Carbon::now()->addDays(30), // 30 days to pay
            'new_expiry_date' => $newExpiryDate,
        ]);

        return $invoice;
    }

    /**
     * Calculate new expiry date based on renewal period
     */
    protected function calculateNewExpiryDate(License $license, Product $product): Carbon
    {
        $currentExpiry = $license->license_expires_at ?? Carbon::now();
        $renewalPeriod = $product->renewal_period;

        switch ($renewalPeriod) {
            case 'monthly':
                return $currentExpiry->copy()->addMonth();
            case 'quarterly':
                return $currentExpiry->copy()->addMonths(3);
            case 'semi-annual':
                return $currentExpiry->copy()->addMonths(6);
            case 'annual':
                return $currentExpiry->copy()->addYear();
            case 'three-years':
                return $currentExpiry->copy()->addYears(3);
            case 'lifetime':
                return $currentExpiry->copy()->addYears(100);
            default:
                // Default to product duration_days
                $durationDays = $product->duration_days ?? 365;
                return $currentExpiry->copy()->addDays($durationDays);
        }
    }

    /**
     * Send renewal notifications
     */
    protected function sendRenewalNotifications(License $license, \App\Models\Invoice $invoice): bool
    {
        try {
            // Send email to customer
            if ($license->user) {
                $this->emailService->sendRenewalReminder($license->user, [
                    'license_key' => $license->license_key,
                    'product_name' => $license->product->name,
                    'expires_at' => $license->license_expires_at->format('Y-m-d'),
                    'invoice_amount' => $invoice->amount,
                    'invoice_due_date' => $invoice->due_date->format('Y-m-d'),
                    'invoice_id' => $invoice->id,
                ]);
            }

            // Send email to admin
            $this->emailService->sendAdminRenewalReminder([
                'license_key' => $license->license_key,
                'product_name' => $license->product->name,
                'customer_name' => $license->user ? $license->user->name : 'Unknown User',
                'customer_email' => $license->user ? $license->user->email : 'No email provided',
                'expires_at' => $license->license_expires_at->format('Y-m-d'),
                'invoice_amount' => $invoice->amount,
                'invoice_id' => $invoice->id,
            ]);

            return true;
        } catch (\Exception $e) {
            $this->error("Failed to send renewal notifications for license {$license->license_key}: " . $e->getMessage());
            return false;
        }
    }
}