<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Product;
use App\Models\Setting;
use App\Services\LicenseGeneratorService;

class TestLicenseGeneration extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'test:license-generation';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Test license generation with database tokens';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('=== Testing License Generation with Database Tokens ===');
        
        try {
            // Check settings
            $setting = Setting::first();
            if (!$setting) {
                $this->error('No settings found in database!');
                return 1;
            }
            
            $this->info("API Token from DB: " . ($setting->license_api_token ? 'Found' : 'Not found'));
            $this->info("Envato Token from DB: " . ($setting->envato_personal_token ? 'Found' : 'Not found'));
            
            // Get first product
            $product = Product::first();
            if (!$product) {
                $this->error('No products found in database!');
                return 1;
            }
            
            $this->info("Product: {$product->name}");
            $this->info("Slug: {$product->slug}");
            $this->info("Programming Language: {$product->programmingLanguage->name}");
            
            // Generate license file
            $generator = new LicenseGeneratorService();
            $filePath = $generator->generateLicenseFile($product);
            
            $this->info("Generated file: {$filePath}");
            
            // Check if file exists
            $fullPath = storage_path("app/public/{$filePath}");
            if (file_exists($fullPath)) {
                $this->info("✅ File exists and was created successfully!");
                
                // Show first few lines of the file
                $content = file_get_contents($fullPath);
                $lines = explode("\n", $content);
                
                $this->info("\nFirst 10 lines of generated file:");
                $this->info("================================");
                for ($i = 0; $i < min(10, count($lines)); $i++) {
                    $this->line(($i + 1) . ": " . $lines[$i]);
                }
                
                // Check if tokens are included
                if (strpos($content, $setting->license_api_token) !== false) {
                    $this->info("\n✅ API Token found in generated file!");
                } else {
                    $this->error("\n❌ API Token NOT found in generated file!");
                }
                
                if ($setting->envato_personal_token && strpos($content, $setting->envato_personal_token) !== false) {
                    $this->info("✅ Envato Token found in generated file!");
                } else {
                    $this->warn("⚠️ Envato Token not found (may be empty)");
                }
                
            } else {
                $this->error("❌ File was not created!");
                return 1;
            }
            
        } catch (\Exception $e) {
            $this->error("Error: " . $e->getMessage());
            $this->error("Stack trace:\n" . $e->getTraceAsString());
            return 1;
        }
        
        $this->info("\n=== Test Complete ===");
        return 0;
    }
}
