<?php

if (!function_exists('is_active_route')) {
    /**
     * Check if the current route matches the given route name
     *
     * @param string $routeName
     * @return bool
     */
    function is_active_route(string $routeName): bool
    {
        return request()->routeIs($routeName);
    }
}

if (!function_exists('is_active_route_pattern')) {
    /**
     * Check if the current route matches the given pattern
     *
     * @param string $pattern
     * @return bool
     */
    function is_active_route_pattern(string $pattern): bool
    {
        return request()->routeIs($pattern);
    }
}

if (!function_exists('get_breadcrumbs')) {
    /**
     * Generate breadcrumbs for the current route
     *
     * @return array
     */
    function get_breadcrumbs(): array
    {
        $route = request()->route();
        $breadcrumbs = [];

        if ($route) {
            $routeName = $route->getName();
            $segments = explode('.', $routeName);

            $currentPath = '';
            foreach ($segments as $segment) {
                $currentPath .= ($currentPath ? '.' : '') . $segment;
                $breadcrumbs[] = [
                    'name' => ucfirst(str_replace(['_', '-'], ' ', $segment)),
                    'url' => route($currentPath),
                    'active' => $currentPath === $routeName
                ];
            }
        }

        return $breadcrumbs;
    }
}

if (!function_exists('get_navigation_tree')) {
    /**
     * Get navigation tree structure
     *
     * @return array
     */
    function get_navigation_tree(): array
    {
        return [
            [
                'name' => 'Dashboard',
                'route' => 'admin.dashboard',
                'icon' => 'fas fa-tachometer-alt',
                'children' => []
            ],
            [
                'name' => 'Products',
                'route' => 'admin.products.index',
                'icon' => 'fas fa-box',
                'children' => [
                    ['name' => 'All Products', 'route' => 'admin.products.index'],
                    ['name' => 'Product', 'route' => 'admin.products.create'],
                    ['name' => 'Categories', 'route' => 'admin.product-categories.index'],
                ]
            ],
            [
                'name' => 'Licenses',
                'route' => 'admin.licenses.index',
                'icon' => 'fas fa-key',
                'children' => [
                    ['name' => 'All Licenses', 'route' => 'admin.licenses.index'],
                    ['name' => 'License Logs', 'route' => 'admin.license-logs.index'],
                ]
            ],
            [
                'name' => 'Customers',
                'route' => 'admin.customers.index',
                'icon' => 'fas fa-users',
                'children' => []
            ],
            [
                'name' => 'Support',
                'route' => 'admin.tickets.index',
                'icon' => 'fas fa-headset',
                'children' => [
                    ['name' => 'All Tickets', 'route' => 'admin.tickets.index'],
                    ['name' => 'Categories', 'route' => 'admin.ticket-categories.index'],
                ]
            ],
            [
                'name' => 'Knowledge Base',
                'route' => 'admin.kb.index',
                'icon' => 'fas fa-book',
                'children' => [
                    ['name' => 'Articles', 'route' => 'admin.kb.index'],
                    ['name' => 'Categories', 'route' => 'admin.kb-categories.index'],
                ]
            ],
            [
                'name' => 'Settings',
                'route' => 'admin.settings.index',
                'icon' => 'fas fa-cog',
                'children' => []
            ]
        ];
    }
}

if (!function_exists('get_available_languages')) {
    /**
     * Get available languages from the lang directory
     *
     * @return array
     */
    function get_available_languages(): array
    {
        $languages = [];
        $langPath = resource_path('lang');
        
        if (is_dir($langPath)) {
            $directories = array_diff(scandir($langPath), ['.', '..']);
            
            foreach ($directories as $dir) {
                if (is_dir($langPath . DIRECTORY_SEPARATOR . $dir)) {
                    $languages[] = [
                        'code' => $dir,
                        'name' => get_language_name($dir),
                        'flag' => get_language_flag($dir),
                        'native_name' => get_language_native_name($dir)
                    ];
                }
            }
        }
        
        return $languages;
    }
}

if (!function_exists('get_language_name')) {
    /**
     * Get language name by code
     *
     * @param string $code
     * @return string
     */
    function get_language_name(string $code): string
    {
        $names = [
            'en' => 'English',
            'ar' => 'Arabic',
            'fr' => 'French',
            'es' => 'Spanish',
            'de' => 'German',
            'it' => 'Italian',
            'pt' => 'Portuguese',
            'ru' => 'Russian',
            'zh' => 'Chinese',
            'ja' => 'Japanese',
            'ko' => 'Korean',
            'tr' => 'Turkish',
            'nl' => 'Dutch',
            'sv' => 'Swedish',
            'no' => 'Norwegian',
            'da' => 'Danish',
            'fi' => 'Finnish',
            'pl' => 'Polish',
            'cs' => 'Czech',
            'hu' => 'Hungarian',
            'ro' => 'Romanian',
            'bg' => 'Bulgarian',
            'hr' => 'Croatian',
            'sk' => 'Slovak',
            'sl' => 'Slovenian',
            'et' => 'Estonian',
            'lv' => 'Latvian',
            'lt' => 'Lithuanian',
            'el' => 'Greek',
            'he' => 'Hebrew',
            'th' => 'Thai',
            'vi' => 'Vietnamese',
            'id' => 'Indonesian',
            'ms' => 'Malay',
            'tl' => 'Filipino',
            'hi' => 'Hindi',
            'bn' => 'Bengali',
            'ta' => 'Tamil',
            'te' => 'Telugu',
            'ml' => 'Malayalam',
            'kn' => 'Kannada',
            'gu' => 'Gujarati',
            'pa' => 'Punjabi',
            'mr' => 'Marathi',
            'ne' => 'Nepali',
            'si' => 'Sinhala',
            'my' => 'Burmese',
            'km' => 'Khmer',
            'lo' => 'Lao',
            'ka' => 'Georgian',
            'am' => 'Amharic',
            'sw' => 'Swahili',
            'zu' => 'Zulu',
            'af' => 'Afrikaans',
            'sq' => 'Albanian',
            'az' => 'Azerbaijani',
            'be' => 'Belarusian',
            'bs' => 'Bosnian',
            'ca' => 'Catalan',
            'cy' => 'Welsh',
            'eu' => 'Basque',
            'fa' => 'Persian',
            'gl' => 'Galician',
            'is' => 'Icelandic',
            'mk' => 'Macedonian',
            'mt' => 'Maltese',
            'sr' => 'Serbian',
            'uk' => 'Ukrainian',
            'ur' => 'Urdu',
            'uz' => 'Uzbek',
            'kk' => 'Kazakh',
            'ky' => 'Kyrgyz',
            'tg' => 'Tajik',
            'mn' => 'Mongolian',
            'bo' => 'Tibetan',
            'dz' => 'Dzongkha',
            'or' => 'Odia',
            'as' => 'Assamese',
            'mni' => 'Manipuri',
            'kok' => 'Konkani',
            'mai' => 'Maithili',
            'sat' => 'Santali',
            'brx' => 'Bodo',
            'gom' => 'Goan Konkani',
            'ks' => 'Kashmiri',
            'sd' => 'Sindhi',
            'doi' => 'Dogri',
            'mni-Mtei' => 'Meitei',
            'lus' => 'Mizo',
            'njo' => 'Ao',
            'njz' => 'Nyishi',
            'grt' => 'Garo',
            'kha' => 'Khasi',
            'mni-Beng' => 'Manipuri (Bengali)',
            'mni-Latn' => 'Manipuri (Latin)',
            'mni-Mtei' => 'Manipuri (Meitei)',
            'mni-Beng' => 'Manipuri (Bengali)',
            'mni-Latn' => 'Manipuri (Latin)',
            'mni-Mtei' => 'Manipuri (Meitei)',
        ];
        
        return $names[$code] ?? ucfirst($code);
    }
}

if (!function_exists('get_language_flag')) {
    /**
     * Get language flag emoji by code
     *
     * @param string $code
     * @return string
     */
    function get_language_flag(string $code): string
    {
        $flags = [
            'en' => '🇺🇸',
            'ar' => '🇸🇦',
            'fr' => '🇫🇷',
            'es' => '🇪🇸',
            'de' => '🇩🇪',
            'it' => '🇮🇹',
            'pt' => '🇵🇹',
            'ru' => '🇷🇺',
            'zh' => '🇨🇳',
            'ja' => '🇯🇵',
            'ko' => '🇰🇷',
            'tr' => '🇹🇷',
            'nl' => '🇳🇱',
            'sv' => '🇸🇪',
            'no' => '🇳🇴',
            'da' => '🇩🇰',
            'fi' => '🇫🇮',
            'pl' => '🇵🇱',
            'cs' => '🇨🇿',
            'hu' => '🇭🇺',
            'ro' => '🇷🇴',
            'bg' => '🇧🇬',
            'hr' => '🇭🇷',
            'sk' => '🇸🇰',
            'sl' => '🇸🇮',
            'et' => '🇪🇪',
            'lv' => '🇱🇻',
            'lt' => '🇱🇹',
            'el' => '🇬🇷',
            'he' => '🇮🇱',
            'th' => '🇹🇭',
            'vi' => '🇻🇳',
            'id' => '🇮🇩',
            'ms' => '🇲🇾',
            'tl' => '🇵🇭',
            'hi' => '🇮🇳',
            'bn' => '🇧🇩',
            'ta' => '🇮🇳',
            'te' => '🇮🇳',
            'ml' => '🇮🇳',
            'kn' => '🇮🇳',
            'gu' => '🇮🇳',
            'pa' => '🇮🇳',
            'mr' => '🇮🇳',
            'ne' => '🇳🇵',
            'si' => '🇱🇰',
            'my' => '🇲🇲',
            'km' => '🇰🇭',
            'lo' => '🇱🇦',
            'ka' => '🇬🇪',
            'am' => '🇪🇹',
            'sw' => '🇹🇿',
            'zu' => '🇿🇦',
            'af' => '🇿🇦',
            'sq' => '🇦🇱',
            'az' => '🇦🇿',
            'be' => '🇧🇾',
            'bs' => '🇧🇦',
            'ca' => '🇪🇸',
            'cy' => '🇬🇧',
            'eu' => '🇪🇸',
            'fa' => '🇮🇷',
            'gl' => '🇪🇸',
            'is' => '🇮🇸',
            'mk' => '🇲🇰',
            'mt' => '🇲🇹',
            'sr' => '🇷🇸',
            'uk' => '🇺🇦',
            'ur' => '🇵🇰',
            'uz' => '🇺🇿',
            'kk' => '🇰🇿',
            'ky' => '🇰🇬',
            'tg' => '🇹🇯',
            'mn' => '🇲🇳',
            'bo' => '🇨🇳',
            'dz' => '🇧🇹',
            'or' => '🇮🇳',
            'as' => '🇮🇳',
            'mni' => '🇮🇳',
            'kok' => '🇮🇳',
            'mai' => '🇮🇳',
            'sat' => '🇮🇳',
            'brx' => '🇮🇳',
            'gom' => '🇮🇳',
            'ks' => '🇮🇳',
            'sd' => '🇵🇰',
            'doi' => '🇮🇳',
            'mni-Mtei' => '🇮🇳',
            'lus' => '🇮🇳',
            'njo' => '🇮🇳',
            'njz' => '🇮🇳',
            'grt' => '🇮🇳',
            'kha' => '🇮🇳',
            'mni-Beng' => '🇮🇳',
            'mni-Latn' => '🇮🇳',
            'mni-Mtei' => '🇮🇳',
        ];
        
        return $flags[$code] ?? '🌐';
    }
}

if (!function_exists('get_language_native_name')) {
    /**
     * Get language native name by code
     *
     * @param string $code
     * @return string
     */
    function get_language_native_name(string $code): string
    {
        $nativeNames = [
            'en' => 'English',
            'ar' => 'العربية',
            'fr' => 'Français',
            'es' => 'Español',
            'de' => 'Deutsch',
            'it' => 'Italiano',
            'pt' => 'Português',
            'ru' => 'Русский',
            'zh' => '中文',
            'ja' => '日本語',
            'ko' => '한국어',
            'tr' => 'Türkçe',
            'nl' => 'Nederlands',
            'sv' => 'Svenska',
            'no' => 'Norsk',
            'da' => 'Dansk',
            'fi' => 'Suomi',
            'pl' => 'Polski',
            'cs' => 'Čeština',
            'hu' => 'Magyar',
            'ro' => 'Română',
            'bg' => 'Български',
            'hr' => 'Hrvatski',
            'sk' => 'Slovenčina',
            'sl' => 'Slovenščina',
            'et' => 'Eesti',
            'lv' => 'Latviešu',
            'lt' => 'Lietuvių',
            'el' => 'Ελληνικά',
            'he' => 'עברית',
            'th' => 'ไทย',
            'vi' => 'Tiếng Việt',
            'id' => 'Bahasa Indonesia',
            'ms' => 'Bahasa Melayu',
            'tl' => 'Filipino',
            'hi' => 'हिन्दी',
            'bn' => 'বাংলা',
            'ta' => 'தமிழ்',
            'te' => 'తెలుగు',
            'ml' => 'മലയാളം',
            'kn' => 'ಕನ್ನಡ',
            'gu' => 'ગુજરાતી',
            'pa' => 'ਪੰਜਾਬੀ',
            'mr' => 'मराठी',
            'ne' => 'नेपाली',
            'si' => 'සිංහල',
            'my' => 'မြန်မာ',
            'km' => 'ខ្មែរ',
            'lo' => 'ລາວ',
            'ka' => 'ქართული',
            'am' => 'አማርኛ',
            'sw' => 'Kiswahili',
            'zu' => 'IsiZulu',
            'af' => 'Afrikaans',
            'sq' => 'Shqip',
            'az' => 'Azərbaycan',
            'be' => 'Беларуская',
            'bs' => 'Bosanski',
            'ca' => 'Català',
            'cy' => 'Cymraeg',
            'eu' => 'Euskera',
            'fa' => 'فارسی',
            'gl' => 'Galego',
            'is' => 'Íslenska',
            'mk' => 'Македонски',
            'mt' => 'Malti',
            'sr' => 'Српски',
            'uk' => 'Українська',
            'ur' => 'اردو',
            'uz' => 'Oʻzbek',
            'kk' => 'Қазақ',
            'ky' => 'Кыргызча',
            'tg' => 'Тоҷикӣ',
            'mn' => 'Монгол',
            'bo' => 'བོད་ཡིག',
            'dz' => 'རྫོང་ཁ',
            'or' => 'ଓଡ଼ିଆ',
            'as' => 'অসমীয়া',
            'mni' => 'ꯃꯤꯇꯩꯂꯣꯟ',
            'kok' => 'कोंकणी',
            'mai' => 'मैथिली',
            'sat' => 'ᱥᱟᱱᱛᱟᱲᱤ',
            'brx' => 'बड़ो',
            'gom' => 'कोंकणी',
            'ks' => 'کٲشُر',
            'sd' => 'سنڌي',
            'doi' => 'डोगरी',
            'mni-Mtei' => 'ꯃꯤꯇꯩꯂꯣꯟ',
            'lus' => 'Mizo',
            'njo' => 'Ao',
            'njz' => 'Nyishi',
            'grt' => 'Garo',
            'kha' => 'Khasi',
            'mni-Beng' => 'মণিপুরী',
            'mni-Latn' => 'Manipuri',
            'mni-Mtei' => 'ꯃꯤꯇꯩꯂꯣꯟ',
        ];
        
        return $nativeNames[$code] ?? get_language_name($code);
    }
}