<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\LicenseVerificationLog;
use App\Services\LicenseVerificationLogger;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\JsonResponse;

class LicenseVerificationLogController extends Controller
{
    public function index(Request $request): View
    {
        $query = LicenseVerificationLog::query();

        // Apply filters
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('source')) {
            $query->where('verification_source', $request->source);
        }

        if ($request->filled('domain')) {
            $query->where('domain', 'like', '%' . $request->domain . '%');
        }

        if ($request->filled('ip')) {
            $query->where('ip_address', 'like', '%' . $request->ip . '%');
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $logs = $query->orderBy('created_at', 'desc')->paginate(20);

        // Get statistics
        $stats = LicenseVerificationLogger::getStats(30);

        // Get suspicious activity
        $suspiciousActivity = LicenseVerificationLogger::getSuspiciousActivity(24, 3);

        // Get unique sources and domains for filters
        $sources = LicenseVerificationLog::distinct()->pluck('verification_source')->filter();
        $domains = LicenseVerificationLog::distinct()->pluck('domain')->filter();

        return view('admin.license-verification-logs.index', compact(
            'logs', 'stats', 'suspiciousActivity', 'sources', 'domains'
        ));
    }

    public function getStats(Request $request): JsonResponse
    {
        $days = $request->get('days', 30);
        $stats = LicenseVerificationLogger::getStats($days);
        
        return response()->json($stats);
    }

    public function getSuspiciousActivity(Request $request): JsonResponse
    {
        $hours = $request->get('hours', 24);
        $minAttempts = $request->get('min_attempts', 3);
        $activity = LicenseVerificationLogger::getSuspiciousActivity($hours, $minAttempts);
        
        return response()->json($activity);
    }

    public function show(LicenseVerificationLog $log): View
    {
        return view('admin.license-verification-logs.show', compact('log'));
    }

    public function cleanOldLogs(Request $request): JsonResponse
    {
        $daysToKeep = $request->get('days', 90);
        $cleanedCount = LicenseVerificationLogger::cleanOldLogs($daysToKeep);
        
        return response()->json([
            'success' => true,
            'message' => "Successfully cleaned {$cleanedCount} old log entries",
            'cleaned_count' => $cleanedCount
        ]);
    }

    public function export(Request $request)
    {
        $query = LicenseVerificationLog::query();

        // Apply same filters as index
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('source')) {
            $query->where('verification_source', $request->source);
        }

        if ($request->filled('domain')) {
            $query->where('domain', 'like', '%' . $request->domain . '%');
        }

        if ($request->filled('ip')) {
            $query->where('ip_address', 'like', '%' . $request->ip . '%');
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $logs = $query->orderBy('created_at', 'desc')->get();

        $filename = 'license_verification_logs_' . date('Y-m-d_H-i-s') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($logs) {
            $file = fopen('php://output', 'w');
            
            // CSV Headers
            fputcsv($file, [
                'ID', 'Purchase Code Hash', 'Domain', 'IP Address', 'Status', 
                'Source', 'Is Valid', 'Response Message', 'Created At'
            ]);

            // CSV Data
            foreach ($logs as $log) {
                fputcsv($file, [
                    $log->id,
                    $log->purchase_code_hash,
                    $log->domain,
                    $log->ip_address,
                    $log->status,
                    $log->verification_source,
                    $log->is_valid ? 'Yes' : 'No',
                    $log->response_message,
                    $log->created_at->format('Y-m-d H:i:s')
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
